# @package      hubzero-chuse
# @file         cblist.tcl
# @author       Derrick Kearney <dsk@purdue.edu>
# @copyright    Copyright (c) 2005-2012 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2005-2012 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#

# ----------------------------------------------------------------------
#  COMPONENT: cblist - checkbox list
#
# ======================================================================
package require Itk

option add *Checkboxlist.xScrollMode auto widgetDefault
option add *Checkboxlist.yScrollMode auto widgetDefault
option add *Checkboxlist.xScrollSide auto widgetDefault
option add *Checkboxlist.yScrollSide auto widgetDefault
option add *Checkboxlist.command "" widgetDefault
option add *Checkboxlist.onImage "" widgetDefault
option add *Checkboxlist.offImage "" widgetDefault
option add *Checkboxlist.bgColors [list #ffffff #efefef] widgetDefault
option add *Checkboxlist.mouseOverColor "#cfd9ed" widgetDefault

itcl::class Rappture::Checkboxlist {
    inherit itk::Widget

    itk_option define -xscrollmode xScrollMode XScrollMode ""
    itk_option define -yscrollmode yScrollMode YScrollMode ""
    itk_option define -xscrollside xScrollSide XScrollSide ""
    itk_option define -yscrollside yScrollSide YScrollSide ""
    itk_option define -command command Command ""
    itk_option define -onimage onImage OnImage ""
    itk_option define -offimage offImage OffImage ""
    itk_option define -bgcolors bgColors BgColors ""
    itk_option define -mouseovercolor mouseOverColor MouseOverColor ""


    constructor {args} { # defined below }
#    destructor { # defined below }

    public method add_element { name status txt detail \
                                onChoose onUnChoose icon tags}
    public method choose_ele {name action}
    public method unchoose_ele {name action}
    public method hilight_matching_element_prev {}
    public method hilight_matching_element_next {}
    public method hilight_element_text {name}
    public method show_tagged {{tag ""}}
    public method highlight_matching {{re ""}}
    public method active_tags_add {tags}
    public method active_tags_remove {{tags ""}}

    private variable _bgcoloridx        ;# index of current background color
    private variable _x0                ;# upper left x coord for element box
    private variable _y0                ;# upper left y coord for element box
    private variable _x1                ;# lower right x coord for element box
    private variable _y1                ;# lower right y coord for element box
    private variable _yInc              ;# used for calculating location of next element
    private variable _xImgOffset        ;# x coord increment from the edge to place image
    private variable _xIconOffset       ;# x coord increment from the edge to place icon
    private variable _xTxtOffset        ;# x coord increment from the edge to place text label
    private variable _allNames          ;# list of elements added to the widget
    private variable _activeNames       ;# list of elements currently displayed
    private variable _matchingNames     ;# list of elements matching re
    private variable _activeTags        ;# list of tags that should be active
    private variable _tags              ;# array of tags, each tag has
                                         # a list of names associated
                                         # with it
    private variable _textWidth         ;# text width of the longest entry
    private variable _selected          ;# last entry user <Button-1>'d
                                         # or matched search
    private variable _re                ;# last search regular expression
    private variable _iconWidth         ;# width of an icon
    private variable _iconHeight        ;# height of an icon
    private variable cblist             ;# array of all info for each entry


    private method _coords_reset {}
    private method _toggle {varName}
    private method _toggle_element_bg {name action}
    private method _toggle_img {name onChoose onUnChoose}
    private method _set_ele_status {name status action}
    private method _get_checkbox_image {status}
    private method _get_next_bgcolor {}
    private method _show_detail {name}
    private method _place_element {name}
    private method _place_elements {}
    private method _scroll_updown {entries {units "units"}}
    private method _scroll_moveto {entries}
    private method _escape_regexp_sequences {re}
    private method _canvas_see {item}
    private method _finalize {}
}

itk::usual Checkboxlist {
    keep -background -activebackground -activerelief
    keep -cursor
    keep -highlightcolor -highlightthickness
    keep -troughcolor
}

# ----------------------------------------------------------------------
# CONSTRUCTOR
# ----------------------------------------------------------------------
itcl::body Rappture::Checkboxlist::constructor {args} {
    # not sure what to do with this yet, i turned all the $win to $this
    # frame $win -class CheckButtonList -borderwidth 0

    # user should pack the frame when they are ready
    # pack $win -expand yes -fill both

    itk_component add scroller {
        # why doesnt the scroller seem to honor these options?
        # i have to tell the widget the yscrollsize at definition
        Rappture::Scroller $itk_interior.scroller
    } {
        usual
        keep -xscrollmode
        keep -yscrollmode
        keep -xscrollside
        keep -yscrollside
    }
    pack $itk_component(scroller) -expand yes -fill both

    itk_component add c {
        canvas $itk_component(scroller).c \
            -borderwidth 0 \
            -highlightthickness 0 \
            -selectbackground #00ed00 \
            -selectforeground #000000 \
            -selectborderwidth 0
    } {
        usual
        ignore -borderwidth
        ignore -highlightthickness
        ignore -selectbackground
        ignore -selectforeground
        ignore -selectborderwidth
    }
    # do i need to pack this?
    pack $itk_component(c) -expand yes -fill both

    $itk_component(scroller) contents $itk_component(c)

    set _bgcoloridx -1
    _coords_reset
    set _yInc [expr {$_y1-$_y0}]
    set _xImgOffset 10
    set _xIconOffset 10
    set _xTxtOffset 10
    set _allNames ""
    set _activeNames ""
    set _matchingNames ""
    set _activeTags ""
    set _textWidth 0
    set _selected ""
    set _re ""
    set _iconWidth 16
    set _iconHeight 16

    bind $itk_component(c) <KeyPress-Up> \
        [itcl::code $this _scroll_updown -1 units]

    bind $itk_component(c) <KeyPress-Down> \
        [itcl::code $this _scroll_updown 1 units]

    bind $itk_component(c) <Shift-KeyPress-Up> \
        [itcl::code $this _scroll_updown -5 units]

    bind $itk_component(c) <Shift-KeyPress-Down> \
        [itcl::code $this _scroll_updown 5 units]

    bind $itk_component(c) <KeyPress-Prior> \
        [itcl::code $this _scroll_updown -1 page]

    bind $itk_component(c) <KeyPress-Next> \
        [itcl::code $this _scroll_updown 1 page]

    bind $itk_component(c) <KeyPress-Home> \
        [itcl::code $this _scroll_moveto 0]

    bind $itk_component(c) <KeyPress-End> \
        [itcl::code $this _scroll_moveto 1]


    bind $itk_component(c) <Enter> [list focus $itk_component(c)]

    eval itk_initialize $args
}

# ----------------------------------------------------------------------
# DESTRUCTOR
# ----------------------------------------------------------------------
#itcl::body Rappture::Checkboxlist::destructor {} {
#}

# ----------------------------------------------------------------------
# USAGE: _coords_reset
#
# ----------------------------------------------------------------------
itcl::body Rappture::Checkboxlist::_coords_reset {} {
    set _x0 0
    set _y0 0
    set _x1 0
    set _y1 20
}

# ----------------------------------------------------------------------
# USAGE: _toggle
#
# ----------------------------------------------------------------------
itcl::body Rappture::Checkboxlist::_toggle {varName} {
    # http://wiki.tcl.tk/4517
    upvar 1 $varName var
    set var [expr {$var ? 0 : 1}]
}

itcl::body Rappture::Checkboxlist::_toggle_element_bg {name action} {
    set color ""
    switch $action {
        "Enter" { set color $itk_option(-mouseovercolor) }
        "Leave" {
            if {[string compare $name $_selected] != 0} {
                set color $cblist($name-bgcolor)
            } else {
                set color $itk_option(-mouseovercolor)
            }
        }
        default { set color $cblist($name-bgcolor) }
    }

    $itk_component(c) itemconfigure $name-box -fill $color
}

itcl::body Rappture::Checkboxlist::_toggle_img {name onChoose onUnChoose} {

    set status [_toggle cblist($name-status)]
    set action ""

    switch $status {
        0       {set action $onUnChoose}
        1       {set action $onChoose}
        default {set action ""}
    }

    _set_ele_status $name $status $action
}

itcl::body Rappture::Checkboxlist::choose_ele {name action} {
    set status 1
    _set_ele_status $name $status $action
}

itcl::body Rappture::Checkboxlist::unchoose_ele {name action} {
    set status 0
    _set_ele_status $name $status $action
}

itcl::body Rappture::Checkboxlist::_set_ele_status {name status action} {
    set cblist($name-status) $status
    $itk_component(c) itemconfigure $name-img \
        -image [_get_checkbox_image $status]
    if {[string compare $action ""] != 0} {
        eval $action $name
    }
}

itcl::body Rappture::Checkboxlist::_get_checkbox_image {status} {
    if {$status} {
        return $itk_option(-onimage)
    }
    return $itk_option(-offimage)
}

itcl::body Rappture::Checkboxlist::_get_next_bgcolor {} {
    incr _bgcoloridx
    if {$_bgcoloridx >= [llength $itk_option(-bgcolors)] ||
        $_bgcoloridx < 0} {
        set _bgcoloridx 0
    }
    return [lindex $itk_option(-bgcolors) $_bgcoloridx]
}

itcl::body Rappture::Checkboxlist::_show_detail {name} {

    # unhighlight any previous selected element
    if {[string compare "" $_selected] != 0} {
        $itk_component(c) itemconfigure $_selected-box \
            -fill $cblist($_selected-bgcolor)
    }

    # highlight the newly selected element
    $itk_component(c) itemconfigure $name-box \
        -fill $itk_option(-mouseovercolor)
    set _selected $name

    # display detail
    if {[string compare $itk_option(-command) ""] != 0} {
        eval $itk_option(-command) $cblist($name-detail)
    }
}

itcl::body Rappture::Checkboxlist::add_element {name status txt detail onChoose onUnChoose icon tags} {
    lappend _allNames $name
    lappend cblist($name-txt) $txt
    lappend cblist($name-detail) $detail
    lappend cblist($name-status) [expr {!!$status}]
    lappend cblist($name-bgcolor) ""
    lappend cblist($name-onChoose) $onChoose
    lappend cblist($name-onUnChoose) $onUnChoose
    if {[string compare $icon ""] == 0} {
        set icon [image create photo -width $_iconWidth -height $_iconHeight]
    }
    lappend cblist($name-icon) $icon
    lappend cblist($name-tags) $tags
    foreach tag $tags {
        if {[info exists _tags($tag)] == 0} {
            set _tags($tag) $name
        } else {
            lappend _tags($tag) $name
        }
    }

}

itcl::body Rappture::Checkboxlist::_place_element {name} {
    set cblist($name-bgcolor) [_get_next_bgcolor]
    $itk_component(c) create rect [list $_x0 $_y0 $_x1 $_y1] \
        -width 0 \
        -fill $cblist($name-bgcolor) \
        -tag [list $name-box $name $name-hilight $name-region]

    $itk_component(c) bind $name-hilight <Enter> \
        [itcl::code $this _toggle_element_bg $name Enter]

    $itk_component(c) bind $name-hilight <Leave> \
        [itcl::code $this _toggle_element_bg $name Leave]

    # add element on/off check box
    set im [_get_checkbox_image $cblist($name-status)]
    set imWidth [image width $im]

    set im_X [expr {($_x0+$_xImgOffset+($imWidth/2))}]
    set im_Y [expr {($_y0+$_y1)/2}]

    $itk_component(c) create image $im_X $im_Y \
        -image $im \
        -tag [list $name-img $name $name-hilight]

    $itk_component(c) bind $name-img <Button-1> \
        [itcl::code $this _toggle_img $name \
            $cblist($name-onChoose) \
            $cblist($name-onUnChoose)]
    foreach {x0 y0 x1 y1} [$itk_component(c) bbox $name-img] break

    # add element icon
    # FIXME: this should be calculated once when
    #        the cblist is created of when the
    #        element is added to the list. then
    #        it can be used again in cblist_finalize
    set icon_X [expr {$_x0+$x1+$_xIconOffset+($_iconWidth/2)}]
    set icon_Y [expr {($_y0+$_y1)/2}]

    $itk_component(c) create image $icon_X $icon_Y \
        -image $cblist($name-icon) \
        -tag [list $name-icon $name $name-hilight]
    foreach {x0 y0 x1 y1} [$itk_component(c) bbox $name-icon] break

    # add element name
    set txt_X [expr {($_x0+$x1+$_xTxtOffset)}]
    set txt_Y [expr {($_y0+$_y1)/2}]
    $itk_component(c) create text $txt_X $txt_Y \
        -text $cblist($name-txt) \
        -tag [list $name-txt $name $name-hilight $name-region] -anchor w

    $itk_component(c) bind $name-region <Button-1> \
        [itcl::code $this _show_detail $name]

    foreach {x0 y0 x1 y1} [$itk_component(c) bbox $name-txt] break
    set txt_width [expr {$x1 - $x0}]
    if {$_textWidth < $txt_width} {
        set _textWidth $txt_width
    }

    incr _y0 $_yInc
    incr _y1 $_yInc
}

itcl::body Rappture::Checkboxlist::_place_elements {} {
    set _bgcoloridx -1
    foreach name [lsort $_activeNames] {
        _place_element $name
    }
}

itcl::body Rappture::Checkboxlist::_scroll_updown {entries {units "units"}} {
    $itk_component(c) yview scroll $entries $units
}

itcl::body Rappture::Checkboxlist::_scroll_moveto {entries} {
    $itk_component(c) yview moveto $entries
}

itcl::body Rappture::Checkboxlist::_escape_regexp_sequences {re} {
    return [string map {
                \\ \\\\ \
                \* \\*  \
                \+ \\+  \
                \? \\?  \
                \. \\.  \
                \^ \\^  \
                \$ \\$  \
                \{ \\{  \
                \} \\}  \
                \( \\(  \
                \) \\)  \
                \[ \\[  \
                \] \\]  \
                } $re ]
}

itcl::body Rappture::Checkboxlist::hilight_matching_element_prev {} {
    # names from the last search are stored in $_matchingNames
    # find the presently selected name's index in that list and
    # subtract one if the new index < 0, subtract from the end
    # of the list.
    # highlight previous entry.

    set cur [$itk_component(c) select item]

    if {[string compare "" $cur] == 0} {
        # nothing selected, do nothing
        return
    }

    # remove trailing -txt from item name
    set cur [lsearch -inline -glob [$itk_component(c) gettags $cur] "*-txt"]
    set cur [string replace $cur [expr [string length $cur]-4] end]
    set curIdx [lsearch $_matchingNames $cur]

    set newIdx [expr $curIdx - 1]
    if {$newIdx < 0} {
        # wrap around to the end of the list
        set newIdx [expr [llength $_matchingNames] - 1]
    }

    # scroll to and hilight relevant text
    set cur [lindex $_matchingNames $newIdx]
    _canvas_see $cur
    hilight_element_text $cur
}

itcl::body Rappture::Checkboxlist::hilight_matching_element_next {} {
    # names from the last search are stored in $_matchingNames
    # find the presently selected name's index in that list and
    # subtract one if the new index < 0, subtract from the end
    # of the list.
    # highlight next entry.

    set cur [$itk_component(c) select item]

    if {[string compare "" $cur] == 0} {
        # nothing selected, do nothing
        return
    }

    # remove trailing -txt from item name
    set cur [lsearch -inline -glob [$itk_component(c) gettags $cur] "*-txt"]
    set cur [string replace $cur [expr [string length $cur]-4] end]
    set curIdx [lsearch $_matchingNames $cur]

    set newIdx [expr $curIdx + 1]
    if {$newIdx >= [llength $_matchingNames)]} {
        # wrap around to the beginning of the list
        set newIdx 0
    }

    # scroll to and hilight relevant text
    set cur [lindex $_matchingNames $newIdx]
    _canvas_see $cur
    hilight_element_text $cur
}

itcl::body Rappture::Checkboxlist::hilight_element_text {name} {
    if {[regexp -indices -- $_re $name m] > 0} {
        # there was a match
        # hilight it on the canvas.
        $itk_component(c) select from $name-txt [lindex $m 0]
        $itk_component(c) select to $name-txt [lindex $m 1]

        # make the element selected and show its detail
        _show_detail $name
    }
}

itcl::body Rappture::Checkboxlist::show_tagged {{tag ""}} {
    if {[string length $tag] != 0} {
        # clear all previous tags
        # only show the tag specified by the user
        active_tags_remove
        active_tags_add $tag
    }

    # clear the canvas, reposition elements
    $itk_component(c) delete all

    # set up the pool of names to search from
    set _activeNames ""
    if {[llength $_activeTags] > 0} {
        # there are active tags, add them to the pool of names
        foreach tag $_activeTags {
            foreach {junk names} [array get _tags $tag] break
            if {[info exists names] != 0} {
                eval lappend _activeNames $names
            }
        }
    } else {
        # there are no active tags, use all names as default
        # FIXME: This should be a -all option instead of default
        set _activeNames $_allNames
    }

    # clear out the matching names from previous searches
    set _matchingNames ""
    _coords_reset

    # put elements on the canvas
    _place_elements
    _finalize
}

itcl::body Rappture::Checkboxlist::highlight_matching {{re ""}} {

    # clear out the previously matching names
    set _matchingNames ""

    # clear any previously hilighted areas
    $itk_component(c) select clear

    if {[string length $re] == 0} {
        # don't perform the search on empty search strings
        return [llength $_matchingNames]
    }

    # search for matching names from the active elements being displayed
    # (?i) makes the regexp case insensitive
    set _re "(?i)[_escape_regexp_sequences $re]"
    foreach name [lsearch -all -inline -regexp $_activeNames $_re] {
        eval lappend _matchingNames $name
    }

    # if there are matches, scroll canvas to the first match
    # else if there was a previously selected item scroll to it
    # else don't scroll anywhere
    if {[llength $_matchingNames] != 0} {
        _canvas_see [lindex $_matchingNames 0]

        # FIXME:
        # hilight matching text of the element
        # because we use the canvas select,
        # we can only hilight one find at a time.
        # we hilight the first item in the list.
        # in the future we can change this to a more
        # user friendly interface which highlights all matches
        hilight_element_text [lindex $_matchingNames 0]
    } elseif {[string length $_selected] > 0} {
        _canvas_see $_selected
    }

    # reselect previously selected element
    if {[lsearch $_allNames $_selected] != -1} {
        _show_detail $_selected
    }

    return [llength $_matchingNames]
}

itcl::body Rappture::Checkboxlist::active_tags_add {tags} {

    foreach tag $tags {
        set idx [lsearch -exact $_activeTags $tag]
        if {$idx == -1} {
            lappend _activeTags $tag
        }
    }
}

itcl::body Rappture::Checkboxlist::active_tags_remove {{tags ""}} {

    if {[llength $tags] == 0} {
        # no tags provided, clear all
        # FIXME: should be a -all option or recognize "all"
        set _activeTags ""
        return
    }

    foreach tag $tags {
        set idx [lsearch -exact $_activeTags $tag]
        if {$idx != -1} {
            set _activeTags [lreplace $_activeTags $idx $idx]
        }
    }
}

itcl::body Rappture::Checkboxlist::_finalize {} {

    # NHTC
    # FIXME: clean up spacing calculations
    if {[llength $_activeNames] > 0} {
        set name [lindex $_activeNames 0]
        foreach {x0 y0 x1 y1} [$itk_component(c) bbox $name-img] break
        # calculate x1 for the icon
        # we account for the space,
        # even if there is no image,
        # so the element names will line up
        set x1 [expr {$x1+$_xIconOffset+$_iconWidth}]
        set _x1 [expr {$_x0+$x1+(2*$_xTxtOffset)+$_textWidth}]

        foreach name $_allNames {
            # resize the background box for each element
            set coords [$itk_component(c) coords $name-box]
            set coords [lreplace $coords 2 2 $_x1]
            $itk_component(c) coords $name-box $coords
        }

        # rescale bbox for cblist canvas
        $itk_component(c) configure -width $_x1 -height [expr {$_y1-$_yInc}]
    }

    $itk_component(c) configure \
        -scrollregion [$itk_component(c) bbox all] \
        -xscrollincrement 0.1i \
        -yscrollincrement 0.2i
}

## from http://tcl.sourceforge.net/faqs/tk/#canvas/see
## "see" method alternative for canvas
## Aligns the named item as best it can in the middle of the screen
##
## c    - a canvas widget
## item - a canvas tagOrId
itcl::body Rappture::Checkboxlist::_canvas_see {item} {
    set box [$itk_component(c) bbox $item]
    if {![llength $box]} return
    ## always properly set -scrollregion
    foreach {x y x1 y1}     $box \
            {top btm}       [$itk_component(c) yview] \
            {left right}    [$itk_component(c) xview] \
            {p q xmax ymax} [$itk_component(c) cget -scrollregion] {
        set xpos [expr {(($x1+$x)/2.0)/$xmax - ($right-$left)/2.0}]
        set ypos [expr {(($y1+$y)/2.0)/$ymax - ($btm-$top)/2.0}]
    }
    $itk_component(c) xview moveto $xpos
    $itk_component(c) yview moveto $ypos
}

# ----------------------------------------------------------------------
# OPTION: -xscrollmode
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::xscrollmode {
    $itk_component(scroller) configure -xscrollmode $itk_option(-xscrollmode)
}

# ----------------------------------------------------------------------
# OPTION: -yscrollmode
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::yscrollmode {
    $itk_component(scroller) configure -yscrollmode $itk_option(-yscrollmode)
}

# ----------------------------------------------------------------------
# OPTION: -xscrollside
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::xscrollside {
    $itk_component(scroller) configure -xscrollside $itk_option(-xscrollside)
}

# ----------------------------------------------------------------------
# OPTION: -yscrollside
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::yscrollside {
    $itk_component(scroller) configure -yscrollside $itk_option(-yscrollside)
}

# ----------------------------------------------------------------------
# OPTION: -command
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::command {
}

# ----------------------------------------------------------------------
# OPTION: -onimage
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::onimage {
    # replace all onimages for active names?
}

# ----------------------------------------------------------------------
# OPTION: -offimage
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::offimage {
    # replace all offimages for active names?
}

# ----------------------------------------------------------------------
# OPTION: -bgcolors
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::bgcolors {
    # replace all background colors for active names?
}

# ----------------------------------------------------------------------
# OPTION: -mouseovercolor
# ----------------------------------------------------------------------
itcl::configbody Rappture::Checkboxlist::mouseovercolor {
    # update the background color for the selected item
}

# ----------------------------------------------------------------------
# OPTION: -width
# ----------------------------------------------------------------------
itcl::configbody Rappture::Scroller::width {
    # check for proper value
    winfo pixels $itk_component(hull) $itk_option(-width)

    $_dispatcher event -idle !fixsize
}

# ----------------------------------------------------------------------
# OPTION: -height
# ----------------------------------------------------------------------
itcl::configbody Rappture::Scroller::height {
    # check for proper value
    winfo pixels $itk_component(hull) $itk_option(-height)

    $_dispatcher event -idle !fixsize
}
