<?php
/**
 * HUBzero CMS
 *
 * Copyright 2005-2011 Purdue University. All rights reserved.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of Purdue University.
 *
 * @package   hubzero-cms
 * @author    Sam Wilson <samwilson@purdue.edu>
 * @copyright Copyright 2005-2011 Purdue University. All rights reserved.
 * @license   http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 */

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die('Restricted access');

/**
 * Primary controller for the Billboards component
 */
class BillboardsControllerBillBoards extends \Hubzero\Component\AdminController
{
	/**
	 * Browse the list of billboards
	 *
	 * @return void
	 */
	public function displayTask()
	{
		// Get configuration
		$app = JFactory::getApplication();
		$config = JFactory::getConfig();

		// Incoming
		$this->view->filters = array();
		$this->view->filters['limit'] = $app->getUserStateFromRequest(
			$this->_option . '.billboards.limit',
			'limit',
			$config->getValue('config.list_limit'),
			'int'
		);
		$this->view->filters['start'] = $app->getUserStateFromRequest(
			$this->_option . '.billboards.limitstart',
			'limitstart',
			0,
			'int'
		);

		// Get a billboard object
		$billboards = new BillboardsBillboard($this->database);

		// Get a record count
		$this->view->total = $billboards->getCount($this->view->filters);

		// Grab all the records
		$this->view->rows = $billboards->getRecords($this->view->filters);

		// Initiate paging
		jimport('joomla.html.pagination');
		$this->view->pageNav = new JPagination(
			$this->view->total,
			$this->view->filters['start'],
			$this->view->filters['limit']
		);

		// Set any errors
		if ($this->getError())
		{
			$this->view->setError($this->getError());
		}

		// Output the HTML
		$this->view->display();
	}

	/**
	 * Create a billboard
	 *
	 * @return void
	 */
	public function addTask()
	{
		$this->editTask();
	}

	/**
	 * Edit a billboard
	 *
	 * @return void
	 */
	public function editTask()
	{
		// Hide the menu, force users to save or cancel
		JRequest::setVar('hidemainmenu', 1);

		// Incoming - expecting an array
		$cid = JRequest::getVar('cid', array(0));
		if (!is_array($cid))
		{
			$cid = array($cid);
		}
		$uid = $cid[0];

		// Load the billboard and the collection (we need the collection to grab the collection name)
		$this->view->row = new BillboardsBillboard($this->database);
		$this->view->row->load($uid);
		$this->view->collection = new BillboardsCollection($this->database);
		$this->view->collection->load($this->view->row->collection_id);

		// Fail if not checked out by 'me'
		if ($this->view->row->checked_out && $this->view->row->checked_out != $this->juser->get('id'))
		{
			$this->setRedirect(
				'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
				JText::_('COM_BILLBOARDS_ERROR_CHECKED_OUT')
			);
			return;
		}

		// Build the html select list for ordering
		$query = $this->view->row->buildOrderingQuery($this->view->row->collection_id);

		// Are we editing an existing entry?
		if ($uid)
		{
			// Yes, we should check it out first
			$this->view->row->checkout($this->juser->get('id'));

			// Build the ordering info
			$this->view->row->ordering = $this->ordering($this->view->row, $uid, $query);
		}
		else
		{
			// Set some defaults
			$this->view->row->ordering = $this->ordering($this->view->row, '', $query);
		}

		// Grab the file location for the background images
		$params = JComponentHelper::getParams('com_billboards');
		$this->view->image_location = $params->get('image_location', '/site/media/images/billboards/');

		if (!is_dir(JPATH_ROOT . DS . ltrim($this->view->image_location, DS)))
		{
			jimport('joomla.file.folder');
			JFolder::create(JPATH_ROOT . DS . ltrim($this->view->image_location, DS));
		}

		// Get the relative image location for building the links to the media manager
		$mparams = JComponentHelper::getParams('com_media');
		$this->view->media_path = $mparams->get('image_path', 'site/media/images');

		// Make sure the image path is in the format that we need (i.e. remove any leading or trailing "/")
		if (substr($this->view->media_path, 0, 1) != DS)
		{
			$this->view->media_path = DS.$this->view->media_path;
		}
		if (substr($this->view->media_path, -1, 1) != DS)
		{
			$this->view->media_path = $this->view->media_path.DS;
		}
		$this->view->media_path = rtrim(str_replace($this->view->media_path, "", $this->view->image_location), DS);

		// Build the collection select list
		$this->view->clist = BillboardsHelperHtml::buildCollectionsList($this->view->row->collection_id);

		// Build the select list for possible learn-more locations
		$this->view->learnmorelocation = BillboardsHelperHtml::buildLearnMoreList($this->view->row->learn_more_location);

		// Set any errors
		if ($this->getError())
		{
			$this->view->setError($this->getError());
		}

		// Output the HTML
		$this->view->setLayout('edit')->display();
	}

	/**
	 * Save a billboard
	 *
	 * @return void
	 */
	public function saveTask()
	{
		// Check for request forgeries
		JRequest::checkToken() or jexit('Invalid Token');

		// Incoming, make sure to allow HTML to pass through
		$billboard = JRequest::getVar('billboard', array(), 'post', 'array', JREQUEST_ALLOWHTML);
		$billboard = array_map('trim', $billboard);
		$row = new BillboardsBillboard($this->database);

		// If this is a new item, let's order it last
		if ($billboard['id'] == 0)
		{
			$new_id = $row->getNextOrdering($billboard['collection_id']);
			$billboard['ordering'] = $new_id;
		}

		// Save the billboard
		if (!$row->bind($billboard))
		{
			JError::raiseError(500, $row->getError());
			return;
		}
		if (!$row->check())
		{
			JError::raiseError(500, $row->getError());
			return;
		}
		if (!$row->store())
		{
			JError::raiseError(500, $row->getError());
			return;
		}

		// Check in the billboard now that we've saved it
		$row->checkin();

		// Redirect
		$this->setRedirect(
			'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
			JText::_('COM_BILLBOARDS_BILLBOARD_SUCCESSFULLY_SAVED')
		);
	}

	/**
	 * Save the new order
	 *
	 * @return void
	 */
	public function saveorderTask()
	{
		// Check for request forgeries
		JRequest::checkToken() or jexit('Invalid Token');

		// Initialize variables
		$cid   = JRequest::getVar('cid', array(), 'post', 'array');
		$order = JRequest::getVar('order', array(), 'post', 'array');
		$total = count($cid);
		$row   = new BillboardsBillboard($this->database);

		// Make sure we have something to work with
		if (empty($cid))
		{
			JError::raiseWarning(500, JText::_('BILLBOARDS_ORDER_PLEASE_SELECT_ITEMS'));
			return;
		}

		// Update ordering values
		for ($i = 0; $i < $total; $i++)
		{
			$row->load($cid[$i]);
			if ($row->ordering != $order[$i])
			{
				$row->ordering = $order[$i];
				if (!$row->store())
				{
					JError::raiseError(500, $row->getError());
					return;
				}
			}
		}

		// Clear the component's cache
		$cache = JFactory::getCache('com_billboards');
		$cache->clean();

		// Redirect
		$this->setRedirect(
			'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
			JText::_('COM_BILLBOARDS_ORDER_SUCCESSFULLY_UPDATED')
		);
	}

	/**
	 * Delete a billboard
	 *
	 * @return void
	 */
	public function deleteTask()
	{
		// Check for request forgeries
		JRequest::checkToken() or jexit('Invalid Token');

		// Incoming (expecting an array)
		$ids = JRequest::getVar('cid', array());
		if (!is_array($ids))
		{
			$ids = array($ids);
		}

		// Make sure we have IDs to work with
		if (count($ids) > 0)
		{
			$billboard = new BillboardsBillboard($this->database);

			// Loop through the array of ID's and delete
			foreach ($ids as $id)
			{
				if (!$billboard->delete($id))
				{
					$this->setRedirect(
						'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
						JText::_('COM_BILLBOARDS_ERROR_CANT_DELETE')
					);
					return;
				}
			}
		}

		// Redirect
		$this->setRedirect(
			'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
			JText::sprintf('COM_BILLBOARDS_BILLBOARD_SUCCESSFULLY_DELETED', count($ids))
		);
	}

	/**
	 * Publish billboards
	 *
	 * @return void
	 */
	public function publishTask()
	{
		$this->toggle(1);
	}

	/**
	 * Unpublish billboards
	 *
	 * @return void
	 */
	public function unpublishTask()
	{
		$this->toggle(0);
	}

	/**
	 * Cancels out of the billboard edit view, makes sure to check the billboard back in for other people to edit
	 *
	 * @return void
	 */
	public function cancelTask()
	{
		// Incoming - we need an id so that we can check it back in
		$billboard = JRequest::getVar('billboard', array(), 'post');

		// Check the billboard back in
		$row = new BillboardsBillboard($this->database);
		$row->bind($billboard);
		$row->checkin();

		// Redirect
		$this->setRedirect(
			'index.php?option=' . $this->_option . '&controller=' . $this->_controller
		);
	}

	/**
	 * Toggle a billboard between published and unpublished.  We're looking for an array of ID's to publish/unpublish
	 *
	 * @param  $publish: 1 to publish and 0 for unpublish
	 * @return void
	 */
	protected function toggle($publish=1)
	{
		// Check for request forgeries
		JRequest::checkToken(array('get', 'post')) or jexit('Invalid Token');

		// Incoming (we're expecting an array)
		$ids = JRequest::getVar('cid', array());
		if (!is_array($ids))
		{
			$ids = array($ids);
		}

		// Loop through the IDs
		foreach ($ids as $id)
		{
			// Load the billboard
			$row = new BillboardsBillboard($this->database);
			$row->load($id);

			// Only alter items not checked out or checked out by 'me'
			if ($row->checked_out == 0 || $row->checked_out == $this->juser->get('id'))
			{
				$row->published = $publish;
				if (!$row->store($publish))
				{
					JError::raiseError(500, $row->getError());
					return;
				}
				// Check it back in
				$row->checkin($id);
			}
			else
			{
				$this->setRedirect(
					'index.php?option=' . $this->_option . '&controller=' . $this->_controller,
					JText::_('COM_BILLBOARDS_ERROR_CHECKED_OUT')
				);
				return;
			}
		}

		// Redirect
		$this->setRedirect(
			'index.php?option=' . $this->_option . '&controller=' . $this->_controller
		);
	}

	/**
	 * Build the select list for ordering of a specified Table
	 *
	 * @return $ordering
	 */
	protected function ordering(&$row, $id, $query, $neworder = 0)
	{
		$db = JFactory::getDBO();

		if ($id)
		{
			$order = JHTML::_('list.genericordering', $query);
			$ordering = JHTML::_('select.genericlist', $order, 'billboard[ordering]', 'class="inputbox" size="1"', 'value', 'text', intval($row->ordering));
		}
		else
		{
			if ($neworder)
			{
				$text = JText::_('COM_BILLBOARDS_DESC');
			}
			else
			{
				$text = JText::_('COM_BILLBOARDS_ASC');
			}
			$ordering = '<input type="hidden" name="billboard[ordering]" value="' . $row->ordering . '" />' . $text;
		}

		return $ordering;
	}
}