<?php
/**
 * HUBzero CMS
 *
 * Copyright 2005-2015 HUBzero Foundation, LLC.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * HUBzero is a registered trademark of Purdue University.
 *
 * @package   hubzero-cms
 * @author    Shawn Rice <zooley@purdue.edu>
 * @copyright Copyright 2005-2015 HUBzero Foundation, LLC.
 * @license   http://opensource.org/licenses/MIT MIT
 */

namespace Components\Oaipmh\Models\Schemas;

use Components\Oaipmh\Models\Xml\Response;
use Components\Oaipmh\Models\Service;
use Components\Oaipmh\Models\Schema;
use Date;

require_once(__DIR__ . '/dublincore.php');

/**
 * Qualified Dublin Core schema handler
 */
class QualifiedDC extends DublinCore
{
	/**
	 * Schema description
	 * 
	 * @var  string
	 */
	public static $schema = 'http://www.bepress.com/assets/xsd/oai_qualified_dc.xsd';

	/**
	 * Schema namespace
	 * 
	 * @var  string
	 */
	public static $ns = 'http://www.bepress.com/OAI/2.0/qualified-dublin-core/';

	/**
	 * Get the schema name
	 *
	 * @return  string
	 */
	public function name()
	{
		return 'Qualified Dublin Core';
	}

	/**
	 * Get the schema prefix
	 *
	 * @return  string
	 */
	public function prefix()
	{
		return 'qdc';
	}

	/**
	 * Does this adapter respond to a mime type
	 *
	 * @param   string   $type  Schema type
	 * @return  boolean
	 */
	public static function handles($type)
	{
		return in_array($type, array(
			'qdc',
			'dcq',
			'oai_qdc',
			'qualified-dublin-core',
			'qualifieddublincore',
			'qualifiedddc',
			__CLASS__
		));
	}

	/**
	 * Process a single record
	 *
	 * @param   object   $result
	 * @param   boolean  $metadata
	 * @return  object   $this
	 */
	public function record($result, $metadata=true)
	{
		if ($metadata)
		{
			$this->response
				->element('record')
					->element('header');
		}
		else
		{
			$this->response
				->element('header');
		}

		if (!empty($result->identifier))
		{
			$this->response->element('identifier', $result->identifier)->end();
		}

		// We want the "T" & "Z" strings in the output NOT the UTC offset (-400)
		$gran = $this->service->get('gran', 'c');
		if ($gran == 'c')
		{
			$gran = 'Y-m-d\Th:i:s\Z';
		}

		$datestamp = strtotime($result->date);
		$datestamp = gmdate($gran, $datestamp);
		if (!empty($datestamp))
		{
			$this->response->element('datestamp', $datestamp)->end();
		}
		if (!empty($result->type))
		{
			$this->response->element('setSpec', $result->type)->end();
		}

		$this->response->end(); // End header

		if ($metadata)
		{
			$this->response
				->element('metadata')
					->element('oai_dc:dc')
						->attr('xmlns:' . self::$prefix, self::$ns)
						->attr('xmlns:dc', 'http://purl.org/dc/elements/1.1/')
						->attr('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance')
						->attr('xsi:schemaLocation', self::$ns . ' ' . self::$schema);

			$dcs = array(
				'title'   => array(
					'alternative'
				),
				'creator' => null,
				'subject' => null,
				'date'    => array(
					'created',
					'valid',
					'available',
					'issued',
					'modified',
					'dateAccepted',
					'dateCopyrighted',
					'dateSubmitted'
				),
				'identifier'  => null,
				'description' => array(
					'tableOfContents',
					'abstract'
				),
				'type'        => null,
				'publisher'   => null,
				'rights'      => null,
				'contributor' => null,
				'relation'    => array(
					'isVersionOf',
					'hasVersion',
					'isReplacedBy',
					'replaces',
					'isRequiredBy',
					'requires',
					'isPartOf',
					'hasPart',
					'isReferencedBy',
					'references',
					'isFormatof',
					'hasFormat',
					'conformsTo'
				),
				'format'   => array(
					'medium',
					'extent'
				),
				'coverage' => array(
					'spatial',
					'temporal'
				),
				'language' => null,
				'source'   => null
			);

			// Loop through DC elements
			foreach ($dcs as $dc => $attrs)
			{
				if (!isset($result->$dc))
				{
					continue;
				}

				if (is_array($result->$dc))
				{
					foreach ($result->$dc as $val)
					{
						$term = '';

						if (is_array($val))
						{
							$res  = $val['value'];
							$term = $val['type'];
							// Make sure it's a valid modifier
							if (!in_array($term, $attrs))
							{
								$term = '';
							}
						}
						else
						{
							$res = $val;
						}

						$this->response->element('dc:' . $dc . ($term ? '.' . $term : ''), $this->prepare($res))->end();
					}
				}
				elseif (!empty($result->$dc))
				{
					if ($dc == 'date')
					{
						$this->response->element('dc:' . $dc, Date::of($result->date)->format($gran))->end();
					}
					else
					{
						$this->response->element('dc:' . $dc, $this->prepare($result->$dc))->end();
					}
				}
			}

			$this->response->end() // End oai_dc:dc
						->end(); // End metadata

			$this->response->end(); // End record
		}

		return $this;
	}
}
