<?php
/**
 * HUBzero CMS
 *
 * Copyright 2005-2015 HUBzero Foundation, LLC.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * HUBzero is a registered trademark of Purdue University.
 *
 * @package   hubzero-cms
 * @author    Kevin Wojkovich <kevinw@purdue.edu>
 * @copyright Copyright 2005-2015 HUBzero Foundation, LLC.
 * @license   http://opensource.org/licenses/MIT MIT
 * @since     Class available since release 1.3.2
 */

namespace Components\Feedaggregator\Models;

use Hubzero\Database\Relational;

/**
 * Hubs database model
 *
 * @uses \Hubzero\Database\Relational
 */
class Post extends Relational
{
	/**
	 * The table namespace
	 *
	 * @var  string
	 **/
	protected $namespace = 'feedaggregator';

	/**
	 * Default order by for model
	 *
	 * @var  string
	 **/
	public $orderBy = 'created';

	/**
	 * Fields and their validation criteria
	 *
	 * @var  array
	 */
	protected $rules = array(
		'title'   => 'notempty',
		'feed_id' => 'positive|notzero'
	);

	/**
	 * Automatic fields to populate every time a row is created
	 *
	 * @var  array
	 */
	public $initiate = array(
		'created',
		'created_by'
	);

	/**
	 * Get parent feed
	 *
	 * @return  object
	 */
	public function feed()
	{
		return $this->belongsToOne('Feed', 'feed_id');
	}

	/**
	 * Get a list of the latest records
	 *
	 * @param   integer  $limit
	 * @param   string   $dateField
	 * @param   string   $sort
	 * @return  object
	 */
	public static function getLatest($limit = 10, $dateField = 'created', $sort = 'DESC')
	{
		$rows = self::all()
			->whereEquals('status', '2')
			->order($dateField, $sort)
			->limit($limit);

		return $rows;
	}

	/**
	 * Get a shortened title
	 *
	 * @param integer $size
	 * @return string
	 */
	public function shortTitle($size = 60)
	{
			if (strlen($this->get('title')) >= $size)
			{
				$string = substr($this->get('title'), 0, $size);
				$string = substr($string, 0, strrpos($string, ' ')) . '...';
				return $string;
			}
			else
			{
				return($this->get('title'));
			}
	}

	/**
	 * Get a human readable status
	 *
	 * @return string
	 */
	public function getStatus()
	{
		switch ($this->get('status'))
		{
			case 0:
				return 'new';
			break;
			case 1:
				return 'under review';
			break;
			case 2:
				return 'approved';
			break;
			case 3:
				return 'removed';
			break;
		} //end switch
	}
}
