<?php
/**
 * HUBzero CMS
 *
 * Copyright 2005-2015 HUBzero Foundation, LLC.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * HUBzero is a registered trademark of Purdue University.
 *
 * @package   hubzero-cms
 * @author    Shawn Rice <zooley@purdue.edu>
 * @copyright Copyright 2005-2015 HUBzero Foundation, LLC.
 * @license   http://opensource.org/licenses/MIT MIT
 */

namespace Components\Groups\Models\Import\Handler;

use Hubzero\Base\Obj;
use User;
use Lang;

/**
 * Managers Groups import handler
 */
class Managers extends Obj
{
	/**
	 * Return a sample for import header and content
	 *
	 * @return  array
	 */
	public function sample()
	{
		return array(
			'header'  => 'managers',
			'content' => 'username or id;username or id;username or id'
		);
	}

	/**
	 * Bind all raw data
	 *
	 * @param   object  $raw
	 * @param   object  $record
	 * @param   string  $mode
	 * @return  object
	 */
	public function bind($raw, $record, $mode = 'UPDATE')
	{
		if (isset($raw->managers) && $raw->managers != '')
		{
			$record->managers = (array)$this->_multiValueField($raw->managers);

			foreach ($record->managers as $i => $gid)
			{
				$gid = trim($gid, '"');
				$gid = trim($gid, "'");

				$record->managers[$i] = $gid;

				$member = User::getInstance($gid);

				if (!$member || !$member->get('id'))
				{
					$this->setError(Lang::txt('COM_GROUPS_IMPORT_ERROR_MEMBER_NOT_FOUND', $gid));
					continue;
				}
			}
		}

		return $record;
	}

	/**
	 * Check Data integrity
	 *
	 * @param   object  $raw
	 * @param   object  $record
	 * @param   string  $mode
	 * @return  object
	 */
	public function check($raw, $record, $mode = 'UPDATE')
	{
		return $record;
	}

	/**
	 * Store data
	 *
	 * @param   object  $raw
	 * @param   object  $record
	 * @param   string  $mode
	 * @return  object
	 */
	public function store($raw, $record, $mode = 'UPDATE')
	{
		if (!isset($record->managers))
		{
			return $record;
		}

		if ($mode == 'PATCH' && !$record->managers)
		{
			return $record;
		}

		// Add user to specified groups
		$added = array();
		foreach ($record->managers as $gid)
		{
			$member = User::getInstance($gid);

			if (!$member || !$member->get('id'))
			{
				$this->setError(Lang::txt('COM_GROUPS_IMPORT_ERROR_MEMBER_NOT_FOUND', $gid));
				continue;
			}

			// No need to add if already in the group
			if (in_array($member->get('id'), $added))
			{
				continue;
			}

			// Track managers added
			$added[] = $member->get('id');
		}

		$record->entry->add('managers', $added);
		//$record->entry->update();

		return $record;
	}

	/**
	 * Split a string into multiple values based on delimiter(s)
	 *
	 * @param   mixed   $data   String or array of field values
	 * @param   string  $delim  List of delimiters, separated by a pipe "|"
	 * @return  array
	 */
	private function _multiValueField($data, $delim=',|;')
	{
		if (is_string($data))
		{
			$data = array_map('trim', preg_split("/($delim)/", $data));
			$data = array_values(array_filter($data));
		}

		return $data;
	}
}
