<?php
/**
 * HUBzero CMS
 *
 * Copyright 2005-2015 HUBzero Foundation, LLC.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * HUBzero is a registered trademark of Purdue University.
 *
 * @package   hubzero-cms
 * @author    Shawn Rice <zooley@purdue.edu>
 * @copyright Copyright 2005-2015 HUBzero Foundation, LLC.
 * @license   http://opensource.org/licenses/MIT MIT
 */

// No direct access
defined('_HZEXEC_') or die();

/**
 * Resources Plugin class for questions and answers
 */
class plgResourcesQuestions extends \Hubzero\Plugin\Plugin
{
	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 *
	 * @var  boolean
	 */
	protected $_autoloadLanguage = true;

	/**
	 * Return the alias and name for this category of content
	 *
	 * @param   object  $model  Current model
	 * @return  array
	 */
	public function &onResourcesAreas($model)
	{
		$areas = array();

		if (isset($model->toolpublished) || isset($model->revision))
		{
			if (isset($model->thistool)
			 && $model->thistool
			 && ($model->revision=='dev' or !$model->toolpublished))
			{
				$model->type->params->set('plg_questions', 0);
			}
		}
		if ($model->type->params->get('plg_questions')
			&& $model->access('view-all'))
		{
			$areas['questions'] = Lang::txt('PLG_RESOURCES_QUESTIONS');
		}

		return $areas;
	}

	/**
	 * Return data on a resource view (this will be some form of HTML)
	 *
	 * @param   object  $model   Current model
	 * @param   string  $option  Name of the component
	 * @param   array   $areas   Active area(s)
	 * @param   string  $rtrn    Data to be returned
	 * @return  array
	 */
	public function onResources($model, $option, $areas, $rtrn='all')
	{
		$arr = array(
			'area'     => $this->_name,
			'html'     => '',
			'metadata' => ''
		);

		// Check if our area is in the array of areas we want to return results for
		if (is_array($areas))
		{
			if (!array_intersect($areas, $this->onResourcesAreas($model))
			 && !array_intersect($areas, array_keys($this->onResourcesAreas($model))))
			{
				$rtrn = 'metadata';
			}
		}
		if (!$model->type->params->get('plg_questions'))
		{
			return $arr;
		}

		$this->database = App::get('db');
		$this->model    = $model;
		$this->option   = $option;

		// Get a needed library
		require_once Component::path('com_answers') . DS . 'models' . DS . 'question.php';

		// Get all the questions for this tool
		$this->filters = array(
			'limit'    => Request::getInt('limit', 0),
			'start'    => Request::getInt('limitstart', 0),
			'tag'      => ($this->model->isTool() ? 'tool:' . $this->model->alias : 'resource:' . $this->model->id),
			'search'   => Request::getVar('q', ''),
			'filterby' => Request::getVar('filterby', ''),
			'sortby'   => Request::getVar('sortby', 'withinplugin'),
			'sort_Dir' => 'desc'
		);

		$this->count = $this->_find()->count();

		// Load component language file
		Lang::load('com_answers') ||
		Lang::load('com_answers', Component::path('com_answers') . DS . 'site');

		// Are we returning HTML?
		if ($rtrn == 'all' || $rtrn == 'html')
		{
			switch (strtolower(Request::getWord('action', 'browse')))
			{
				case 'save':
					$arr['html'] = $this->_save();
				break;

				case 'new':
					$arr['html'] = $this->_new();
				break;

				case 'browse':
				default:
					$arr['html'] = $this->_browse();
				break;
			}
		}

		// Are we returning metadata?
		if ($rtrn == 'all' || $rtrn == 'metadata')
		{
			$view = $this->view('default', 'metadata');
			$view->resource = $this->model;
			$view->count    = $this->count;

			$arr['metadata'] = $view->loadTemplate();
		}

		// Return output
		return $arr;
	}

	/**
	 * Parse a list of filters to return data for
	 *
	 * @return  object
	 */
	private function _find()
	{
		$records = \Components\Answers\Models\Question::all()
			->including(['responses', function ($response)
			{
				$response
					->select('id')
					->select('question_id')
					->where('state', '!=', 2);
			}]);

		if ($this->filters['tag'])
		{
			$cloud = new \Components\Answers\Models\Tags();
			$tags = $cloud->parse($this->filters['tag']);

			$records
				->select('#__answers_questions.*')
				->join('#__tags_object', '#__tags_object.objectid', '#__answers_questions.id')
				->join('#__tags', '#__tags.id', '#__tags_object.tagid')
				->whereEquals('#__tags_object.tbl', 'answers')
				->whereIn('#__tags.tag', $tags);
		}

		if ($this->filters['search'])
		{
			$this->filters['search'] = strtolower((string)$this->filters['search']);

			$records->whereLike('subject', $this->filters['search'], 1)
					->orWhereLike('question', $this->filters['search'], 1)
					->resetDepth();
		}

		if ($this->filters['filterby'] == 'open')
		{
			$records->whereEquals('state', 0);
		}
		if ($this->filters['filterby'] == 'closed')
		{
			$records->whereEquals('state', 1);
		}
		if (!$this->filters['filterby'] || $this->filters['filterby'] == 'both')
		{
			$records->where('state', '<', 2);
		}

		return $records;
	}

	/**
	 * Show a list of questions attached to this resource
	 *
	 * @return  string
	 */
	private function _browse()
	{
		switch ($this->filters['sortby'])
		{
			case 'rewards':
				$order = 'points';
				break;
			case 'votes':
				$order = 'helpful';
				break;
			case 'date':
			default:
				$order = 'created';
				break;
		}

		// Get results
		$results = $this->_find()
			->limit($this->params->get('display_limit', 10))
			->order($order, $this->filters['sort_Dir'])
			->paginated()
			->rows();

		$view = $this->view('default', 'browse')
			->setError($this->getErrors())
			->set('option', $this->option)
			->set('resource', $this->model)
			->set('banking', Component::params('com_members')->get('bankAccounts'))
			->set('infolink', Component::params('com_answers')->get('infolink', '/kb/points/'))
			->set('rows', $results)
			->set('count', $this->count);

		return $view->loadTemplate();
	}

	/**
	 * Display a form for adding a question
	 *
	 * @param   object  $row
	 * @return  string
	 */
	private function _new($row=null)
	{
		// Login required
		if (User::isGuest())
		{
			$rtrn = Request::getVar('REQUEST_URI', Route::url($this->model->link() . '&active=' . $this->_name, false, true), 'server');

			App::redirect(
				Route::url('index.php?option=com_users&view=login&return=' . base64_encode($rtrn)),
				Lang::txt('PLG_RESOURCES_QUESTIONS_LOGIN_TO_ASK_QUESTION'),
				'warning'
			);
			return;
		}

		if (!is_object($row))
		{
			$row  = new \Components\Answers\Models\Question();
		}

		// Are we banking?
		$upconfig = Component::params('com_members');
		$banking = $upconfig->get('bankAccounts');

		if ($banking)
		{
			$BTL = new \Hubzero\Bank\Teller(User::get('id'));
			$funds = $BTL->summary() - $BTL->credit_summary();

			$funds = $funds > 0 ?: 0;
		}

		$view = $this->view('new', 'question')
			->set('option', $this->option)
			->set('resource', $this->model)
			->set('row', $row)
			->set('tag', $this->filters['tag'])
			->set('banking', $banking)
			->set('funds', $funds)
			->setErrors($this->getErrors());

		return $view->loadTemplate();
	}

	/**
	 * Save a question and redirect to the main listing when done
	 *
	 * @return  void
	 */
	private function _save()
	{
		// Login required
		if (User::isGuest())
		{
			return $this->_browse();
		}

		// Check for request forgeries
		Request::checkToken();

		Lang::load('com_answers');

		// Incoming
		$tags   = Request::getVar('tags', '');
		$funds  = Request::getInt('funds', 0);
		$reward = Request::getInt('reward', 0);

		// If offering a reward, do some checks
		if ($reward)
		{
			// Is it an actual number?
			if (!is_numeric($reward))
			{
				App::abort(500, Lang::txt('COM_ANSWERS_REWARD_MUST_BE_NUMERIC'));
			}
			// Are they offering more than they can afford?
			if ($reward > $funds)
			{
				App::abort(500, Lang::txt('COM_ANSWERS_INSUFFICIENT_FUNDS'));
			}
		}

		// Initiate class and bind posted items to database fields
		$fields = Request::getVar('question', array(), 'post', 'none', 2);

		$row = \Components\Answers\Models\Question::oneOrNew($fields['id'])->set($fields);

		if ($reward && $this->banking)
		{
			$row->set('reward', 1);
		}

		// Store new content
		if (!$row->save())
		{
			$this->setError($row->getError());
			return $this->_new($row);
		}

		// Hold the reward for this question if we're banking
		if ($reward && $this->banking)
		{
			$BTL = new \Hubzero\Bank\Teller(User::get('id'));
			$BTL->hold($reward, Lang::txt('COM_ANSWERS_HOLD_REWARD_FOR_BEST_ANSWER'), 'answers', $row->get('id'));
		}

		// Add the tags
		$row->tag($tags);

		// Add the tag to link to the resource
		$tag = ($this->model->isTool() ? 'tool:' . $this->model->alias : 'resource:' . $this->model->id);
		$row->addTag($tag, User::get('id'), ($this->model->isTool() ? 0 : 1));

		// Build list of recipients
		$receivers = array();

		// Get tool contributors (if question is about a tool)
		if ($this->model->isTool())
		{
			require_once Component::path('com_tools') . DS . 'tables' . DS . 'author.php';
			require_once Component::path('com_tools') . DS . 'tables' . DS . 'version.php';

			$TA = new \Components\Tools\Tables\Author($this->database);
			$objV = new \Components\Tools\Tables\Version($this->database);

			$toolname = $this->model->alias;

			$rev = $objV->getCurrentVersionProperty($toolname, 'revision');
			$authors = $TA->getToolAuthors('', 0, $toolname, $rev);

			if (count($authors) > 0)
			{
				foreach ($authors as $author)
				{
					$receivers[] = $author->uidNumber;
				}
			}
		}

		// Get users who need to be notified on every question
		$config = Component::params('com_answers');
		$apu = $config->get('notify_users', '');
		$apu = explode(',', $apu);
		$apu = array_map('trim', $apu);

		if (!empty($apu))
		{
			foreach ($apu as $u)
			{
				$user = User::getInstance($u);

				if ($user->get('id'))
				{
					$receivers[] = $user->get('id');
				}
			}
		}

		$receivers = array_unique($receivers);

		// Send the message
		if (!empty($receivers))
		{
			// Send a message about the new question to authorized users (specified admins or related content authors)
			$from = array(
				'email'     => Config::get('mailfrom'),
				'name'      => Config::get('sitename') . ' ' . Lang::txt('COM_ANSWERS_ANSWERS'),
				'multipart' => md5(date('U'))
			);

			// Build the message subject
			$subject = Lang::txt('COM_ANSWERS_ANSWERS') . ', ' . Lang::txt('new question about content you author or manage');

			// Build the message
			$eview = new \Hubzero\Mail\View(array(
				'base_path' => Component::path('com_answers') . DS . 'site',
				'name'      => 'emails',
				'layout'    => 'question_plaintext'
			));
			$eview->option   = 'com_answers';
			$eview->sitename = Config::get('sitename');
			$eview->question = $row;
			$eview->id       = $row->get('id', 0);
			$eview->boundary = $from['multipart'];

			$message['plaintext'] = $eview->loadTemplate(false);
			$message['plaintext'] = str_replace("\n", "\r\n", $message['plaintext']);

			// HTML message
			$eview->setLayout('question_html');

			$message['multipart'] = $eview->loadTemplate();
			$message['multipart'] = str_replace("\n", "\r\n", $message['multipart']);

			if (!Event::trigger('xmessage.onSendMessage', array('new_question_admin', $subject, $message, $from, $receivers, 'com_answers')))
			{
				$this->setError(Lang::txt('COM_ANSWERS_MESSAGE_FAILED'));
			}
		}

		Event::trigger('system.logActivity', [
			'activity' => [
				'action'      => (isset($fields['id']) && $fields['id'] ? 'updated' : 'created'),
				'scope'       => 'question',
				'scope_id'    => $row->get('id'),
				'anonymous'   => $row->get('anonymous', 0),
				'description' => Lang::txt('PLG_RESOURCES_QUESTIONS_ACTIVITY_QUESTION_' . (isset($fields['id']) && $fields['id'] ? 'UPDATED' : 'CREATED'), '<a href="' . Route::url($row->link()) . '">' . $row->get('subject') . '</a>'),
				'details'     => array(
					'title' => $row->get('title'),
					'url'   => $row->link()
				)
			],
			'recipients' => $receivers
		]);

		// Redirect to the question
		App::redirect(
			Route::url($this->model->link() . '&active=' . $this->_name)
		);
	}
}
