#
# @package      hubzero-filexfer
# @file         utils.tcl
# @author       Michael McLennan <mmclennan@purdue.edu>
# @author       Derrick Kearney <dsk@purdue.edu>
# @author       Nicholas J. Kisseberth <nkissebe@purdue.edu>
# @copyright    Copyright (c) 2004-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

# ----------------------------------------------------------------------
#  FILEXFER UTILITIES
#
#  This file contains utility procedures that are used across the
#  server.tcl and its clients.

namespace eval filexfer {
    variable installdir [file dirname [file normalize [info script]]]
}

# ----------------------------------------------------------------------
#  USAGE: filexfer::connect <port> <handler>
#
#  Used internally to create a connection to the filexfer server.
#  Returns the file descriptor for the socket connection.  Sets up
#  the connection so that if a message comes back from the server,
#  it is handled by the <handler> proc.
# ----------------------------------------------------------------------
proc filexfer::connect {port callback} {
    variable installdir

    if {[catch {socket "127.0.0.1" $port} sid] == 0} {
        fconfigure $sid -buffering line
        fileevent $sid readable [list server_mesg $sid]
        return $sid
    }

    # oops! have to spawn the server...
    exec tclsh [file join $installdir server.tcl]

    for {set tries 10} {$tries > 0} {incr tries -1} {
        after 1000  ;# wait a minute for new server to start

        if {[catch {socket "127.0.0.1" $port} sid] == 0} {
            fconfigure $sid -buffering line
            fileevent $sid readable [list server_mesg $sid]
            return $sid
        }
    }
    error "tried to spawn server, but it doesn't respond"
}

# ----------------------------------------------------------------------
# RESOURCE LOADING
# ----------------------------------------------------------------------
namespace eval filexfer {
    #
    # Set up a safe interpreter for loading filexfer options...
    #
    variable optionParser [interp create -safe]
    foreach cmd [$optionParser eval {info commands}] {
        $optionParser hide $cmd
    }
    # this lets us ignore unrecognized commands in the file:
    $optionParser invokehidden proc unknown {args} {}

    foreach {name proc} {
        filexfer_port filexfer::set_filexfer_port
        filexfer_cookie filexfer::set_filexfer_cookie
        hub_name filexfer::set_hub_name
        session_token filexfer::set_session_token
        hub_template filexfer::set_hub_template
    } {
        $optionParser alias $name $proc
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::resources
#
# Loads a list of resources from the $SESSIONDIR/resources file
# and returns a list of the form {key1 value1 key2 value2 ...}
# for all resources related to filexfer.
# ----------------------------------------------------------------------
proc filexfer::resources {} {
    global env
    variable resources
    variable optionParser

    if {![info exists env(SESSIONDIR)]} {
        error "\$SESSIONDIR undefined"
    }
    set rfile [file join $env(SESSIONDIR) resources]
    if {![file exists $rfile]} {
        error "file $rfile not found"
    }

    set fid [open $rfile r]
    set rinfo [read $fid]
    close $fid

    catch {unset resources}
    $optionParser eval $rinfo

    return [array get resources]
}

# ----------------------------------------------------------------------
# RESOURCE: filexfer_port
# ----------------------------------------------------------------------
proc filexfer::set_filexfer_port {port} {
    variable resources

    if {![string is integer $port]} {
        error "bad value \"$port\" for filexfer_port: should be integer"
    }
    set resources(port) $port
}

# ----------------------------------------------------------------------
# RESOURCE: filexfer_cookie
# ----------------------------------------------------------------------
proc filexfer::set_filexfer_cookie {cookie} {
    variable resources
    set resources(cookie) $cookie
}

# ----------------------------------------------------------------------
# RESOURCE: hub_name
# ----------------------------------------------------------------------
proc filexfer::set_hub_name {text} {
    variable resources
    set resources(hubname) $text
}

proc filexfer::set_hub_template {text} {
    variable resources
    set resources(hubtemplate) $text
}

# ----------------------------------------------------------------------
# RESOURCE: session_token
# ----------------------------------------------------------------------
proc filexfer::set_session_token {text} {
    variable resources
    set resources(session) $text
}
