#!/usr/bin/env python3
#
# @package      hubzero-forge
# @file         finalizetoolGITexternal.py
# @author       Steven M. Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2005-2018 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2005-2018 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
#----------------------------------------------------------------------
#  USAGE:
#    finalizetool ?flags? <project>
#
#    where ?flags? includes:
#      -root /where ...... install into this directory (default /apps)
#      -revision num ..... install a particular git revision
#                          from the trunk (default is latest revision)
#      -as current|dev ... install as current or dev version.  This
#                          sets a symbolic link to the installed
#                          version.  (default is current version)
#
#      -title toolname ... If specified, then use this as the official
#                          title of the tool.  The official title
#                          usually comes from the LDAP, and it gets
#                          substituted into the tool.xml file
#                          (if there is one).
#      -version vnum ..... If specified, then use this as the official
#                          version number for the tool.  The official
#                          version usually comes from the LDAP, and it
#                          gets substituted into the tool.xml file
#                          (if there is one).
#
#      -license f ........ File containing license text or "-" to read
#                          license text from stdin.  Text is saved to
#                          a file called "LICENSE.txt" in the source
#                          tarball produced by this script.
#
#      -hubdir f .. Load configuration information from this
#                          directory, assumed to be HubConfiguration Class
#                          with simple variable assignments.  In
#                          particular, we look for:
#
#----------------------------------------------------------------------

import os
import sys
import stat
import argparse
import glob
import re
import shutil
import subprocess
import pexpect
import traceback
import time
import tempfile
try:
   import nbformat
except:
   pass

HUBCONFIGURATIONFILE = 'hubconfiguration.php'

os.umask(0o022)

# this is an easy way to exec commands and catch errors:
def exit_on_error(command):
   child = subprocess.Popen(command,
                            shell=False,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE,
                            close_fds=True,
                            universal_newlines=True)
   stdOutput,stdError = child.communicate()
   exitStatus = child.returncode
   if exitStatus != 0:
      sys.stderr.write("FAILED: %s\n" % (command))
      if stdError:
         sys.stderr.write("%s\n" % (stdError))
      if stdOutput:
         sys.stderr.write("%s\n" % (stdOutput))

   return(exitStatus,stdOutput,stdError)

#
# use these procedures to exec git commands
#
def gitLocal(gitArgs,
             gitPath=None):
   if gitPath:
      currentPath = os.getcwd()
      os.chdir(gitPath)

   command = ['git'] + gitArgs
   exitStatus,stdOutput,stdError = exit_on_error(command)

   if gitPath:
      os.chdir(currentPath)

   return(exitStatus,stdOutput,stdError)


def gitHttp(gitArgs,
            gitPath=None):
   if gitPath:
      currentPath = os.getcwd()
      os.chdir(gitPath)

   timeout = 300 # timeout for expect

   command = ['git'] + gitArgs

#  print(' '.join(command))
   rePassword = re.compile(b"Password( for '.*')?:")
   result,exitStatus = pexpect.run(' '.join(command),timeout=timeout,withexitstatus=True, \
                                   events={rePassword:"%s\n" % (options['repopw'])})
   result = result.replace(b'\r',b'')

   if gitPath:
      os.chdir(currentPath)

   return(exitStatus,result)


# use this to search for files such as tool.xml:
def find_files(dir,
               fname):
   found = []
   dirlist = [dir]
   while len(dirlist) > 0:
      dir = dirlist.pop(0)
# dir is not a regular file or directory,
# if it is a link to a file or directory under the original
# directory path, the real file will be picked up.
      if not os.path.islink(dir):
         for dirFile in glob.iglob(os.path.join(dir,'*')):
            if   os.path.isdir(dirFile):
               dirlist.append(dirFile)
            elif os.path.basename(dirFile) == fname:
               if not os.path.islink(dirFile):
                  found.append(dirFile)

   return(found)


def find_section(tags,
                 text):
   tag = tags.pop(0)
   tagend = re.sub('^<','</',tag)
   tagalone = re.sub('>$','/>',tag)

   lookFor = "(%s)(.*:?)(%s)" % (tag,tagend)
   found = re.search(lookFor,text,flags=re.IGNORECASE|re.DOTALL)
   if found:
      m0 = found.start(1)
      m1 = found.end(3)
      i0 = found.start(2)
      i1 = found.end(2)
      subtext = text[i0:i1]
      if len(tags) == 0:
         return(m0,m1,subtext)
      s0,s1,subtext = find_section(tags,subtext)
      if s0 < 0:
         return(-1,-1,"")
      return(i0+s0,i0+s1,subtext)
      
   lookFor = "(%s)" % (tagalone)
   found = re.search(lookFor,text,flags=re.IGNORECASE|re.DOTALL)
   if found:
      m0 = found.start(1)
      m1 = found.end(1)
      if len(tags) == 0:
         return(m0,m1,"")
      return(-1,-1,"")

   return(-1,-1,"")


def string_insert(string,
                  index,
                  text):
   if index > 0:
      substitutedString = string[:index]
   else:
      substitutedString = ""
   substitutedString += text
   substitutedString += string[index:]

   return(substitutedString)


def string_replace(string,
                   indexStart,
                   indexEnd,
                   text):
   substitutedString = string.replace(string[indexStart:indexEnd],text)

   return(substitutedString)


def relocate(dir,
             path,
             newdir):
   relpath = path[len(dir)+1:]

   return(os.path.join(newdir,relpath))


def mkdir_ifneeded(dir):
   if not os.path.isdir(dir):
      try:
         os.makedirs(dir)
      except:
         sys.stderr.write(traceback.format_exc())
         sys.exit(1)
#
#----------------------------------------------------------------------
# Parse all command line options
#----------------------------------------------------------------------
#
parser = argparse.ArgumentParser(description="Finalize tool publication.")
parser.add_argument('--root',required=False,default='/apps',help="Install the tool in this directory.")
parser.add_argument('--revision',required=False,default='HEAD',help="Install this git revision of the tool.")
parser.add_argument('--as',dest='linkAs',required=False,default='current',choices=['current','dev','none'],help="Create symbolic link as dev or current version.")
parser.add_argument('--title',required=False,help="Official title of this tool for tool.xml")
parser.add_argument('--version',required=False,help="Official version of this tool for tool.xml")
parser.add_argument('--license',required=False,help="Read license text from this file, or - for stdin")
parser.add_argument('--hubdir',required=True,help="location of hubconfiguration.php")
parser.add_argument('--project',required=True,help="project shortname")
parser.add_argument('--publishOption',required=False,help="publishing option")

args = parser.parse_args()

#print(args)

options = {}
options['root']     = args.root
options['revision'] = args.revision
options['as']       = args.linkAs
options['title']    = args.title
options['version']  = args.version
options['license']  = args.license
options['hubdir']   = args.hubdir
options['project']  = args.project

if not os.path.isdir(options['hubdir']):
   sys.stderr.write("ERROR: specified base directory does not exist\n")
   sys.stderr.write(" at %s\n" % (options['hubdir']))
   sys.exit(5)

###################### configuration scan (poor hack) ######################

def loadHubConfigurationData(configurationFileDirname,
                             configurationFileiBasename):
   hubconfigurationData = {}
   if   not os.path.isdir(configurationFileDirname):
      sys.stderr.write("ERROR: specified base directory does not exist at %s\n" % (configurationFileDirname))
      sys.exit(5)
   else:
      hubconfigurationPath = os.path.join(configurationFileDirname,configurationFileiBasename)
      if not os.path.isfile(hubconfigurationPath):
         sys.stderr.write("ERROR: specified hubconfiguration.php file does exist at %s\n" % (hubconfigurationPath))
         sys.exit(5)
      else:
         reHubconfigurationVar = re.compile("\s*(?:var|public)\s+\$(\w+)\s*=\s*\'*(.*?)\'*\s*\;\s*")
         try:
            with open(hubconfigurationPath,'r') as fpHubconfiguration:
               for line in fpHubconfiguration:
                  match = reHubconfigurationVar.match(line)
                  if match:
                     hubconfigurationData[match.group(1)] = match.group(2).strip(" \'\"\t").replace("\\'","'")
         except:
            sys.stderr.write("ERROR: specified hubconfiguration file %s could not be ingested\n" % (hubconfigurationPath))
            sys.stderr.write(traceback.format_exc())
            sys.exit(5)

   return(hubconfigurationData)


def getcfg(key):
   if key in hubconfigurationData:
      cfg = hubconfigurationData[key]
   else:
      cfg = ""

   return(cfg)

hubconfigurationData = loadHubConfigurationData(args.hubdir,HUBCONFIGURATIONFILE)

#
#----------------------------------------------------------------------
# Grab options from the hubconfiguration file, if there is one.
#----------------------------------------------------------------------
#
options['repouser'] =  getcfg('svn_user')
options['repopw']   =  getcfg('svn_password')

#
# Get the text from the license file, if specified.
#
if options['license'] != "":
   if options['license'] == "-":
      info = sys.stdin.read()
   else:
      fp = open(options['license'],'r')
      info = fp.read()
      fp.close()
    
   options['license'] = os.path.join(os.sep,'tmp',"license%d" % (os.getpid()))
   fp = open(options['license'],'w')
   fp.write(info)
   fp.close()
#
# Build the name of the install directory.  If --revision is "HEAD"
# then figure out the revision number.  Otherwise, use it as-is.
# Build up a combination of the -root and --revision values into
# a name that looks like root/app-name/rXXX
#
tooldir = os.path.join(options['root'],options['project'])
if not os.path.isdir(tooldir):
   sys.stderr.write("== ERROR: Tool \"%s\" is not installed in %s\n" % (options['project'],tooldir))
   sys.stderr.write("          Run \"installtool\" first.\n")
   sys.exit(1)

if options['revision'] == "HEAD":
   maxRev = -1
   for dir in glob.iglob(os.path.join(tooldir,'r*')):
      lookFor = "r([0-9]+)$"
      found = re.search(lookFor,dir)
      if found:
         rev = int(found.group(1))
         if rev > maxRev:
            maxRev = rev
   if maxRev < 0:
      sys.stderr.write("== ERROR: can't find latest rXX directory in %s\n" % (tooldir))
      sys.stderr.write("          You may need to run \"installtool\" first.\n")
      sys.exit(1)
   revision = maxRev
else:
   lookFor = "^[rR]?([0-9]+)$"
   found = re.search(lookFor,options['revision'])
   if found:
      revision = int(found.group(1))
   else:
      sys.stderr.write("== ERROR: bad -revision value \"%s\"\n" % (options['revision']))
      sys.stderr.write("          should be \"rNNN\" or \"HEAD\"\n")
      sys.exit(1)

targetdir = os.path.join(options['root'],options['project'],"r%d" % (revision))
#
# Make sure that the revision directory exists.
#
if not os.path.isdir(targetdir):
   sys.stderr.write("ERROR: target directory for revision %d not found\n" % (revision))
   sys.stderr.write("       looking for $targetdir\n")
   sys.exit(1)
#
# Make a symbolic link to install this final version according
# to the -as argument.  If -as is "", then avoid making a link.
#
if options['as'] != 'none':
   dir = os.path.dirname(targetdir)
   savedir = os.getcwd()
   os.chdir(dir)
   shutil.rmtree(options['as'],True)
   try:
      os.remove(options['as'])
   except:
      pass
   os.symlink(os.path.basename(targetdir),options['as'])
   os.chdir(savedir)
#
# Look for an "invoke" script in the "middleware" directory,
# and if it's not there, create one automatically and print a
# warning about it.
#
invokescript = os.path.join(targetdir,'middleware','invoke')
if not os.path.exists(invokescript):
   sys.stdout.write("== WARNING: Missing middleware/invoke script.\n")
   try:
      fp = open(invokescript,'w')
      fp.write("#!/bin/sh\n\n")
      fp.write("/usr/bin/invoke_app \"$@\" -C rappture -t %s\n" % (options['project']))
      fp.close()
      os.chmod(invokescript,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
   except:
      sys.stderr.write("== ERROR: Attempt to create invoke script failed.\n")
      sys.stderr.write(traceback.format_exc())
   else:
      sys.stdout.write("            Created default middleware/invoke script.\n")
#
# Get information about the repository, and clone a version of the
# source code into a temp directory for packaging into a tarball below.
#
exitStatus,info,stdError = gitLocal(['log','--format=%H','--max-count=1'],gitPath=targetdir)
if exitStatus == 0:
   revisionHash = info.strip()
   exitStatus,info,stdError = gitLocal(['config','--get','remote.origin.url'],gitPath=targetdir)
   if exitStatus == 0:
      repoURL = info.strip()
   else:
      sys.stderr.write("== ERROR: can't determine current git repository origin.\n")
      sys.stderr.write("%s\n" % (stdError))
      sys.exit(1)
else:
   sys.stderr.write("== ERROR: can't determine current git repository revision hash.\n")
   sys.stderr.write("%s\n" % (stdError))
   sys.exit(1)

if args.publishOption == 'simtool':
   exitStatus = 0
   notebookPath = os.path.join(targetdir,'simtool',"%s.ipynb" % (options['project']))
   if os.path.exists(notebookPath):
      try:
         nb = nbformat.read(notebookPath,nbformat.NO_CONVERT)
      except FileNotFoundError as err:
         #[Errno 2] No such file or directory: 'outputList/r1/outputList.ipy'
         sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
         exitStatus = 1
      except PermissionError as err:
         #[Errno 13] Permission denied: 'outputList/r1/outputList.ipynb'
         sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
         exitStatus = 1
      except (AttributeError,KeyError) as err:
         sys.stderr.write("Missing attribute: %s\n" % (err.args[0]))
         exitStatus = 1
      except NameError as err:
         sys.stderr.write("Name error: %s\n" % (err.args[0]))
         exitStatus = 1
      else:
# clear existing metadata
         try:
            del nb['metadata']['simTool_info']
         except (AttributeError,KeyError) as err:
            pass

# add metadata
         nb['metadata']['simTool_info'] = {}
         nb['metadata']['simTool_info']['name']     = options['project']
         nb['metadata']['simTool_info']['revision'] = revision
         nb['metadata']['simTool_info']['state']    = 'published'

#        print(nb['metadata']['simTool_info'])

         try:
            nbformat.write(nb,notebookPath,nbformat.NO_CONVERT)
         except PermissionError as err:
            sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
            exitStatus = 1
   else:
      sys.stderr.write("WARNING: SimTool notebook does not exist.\n")
      exitStatus = 1

   if exitStatus != 0:
      sys.stderr.write("WARNING: Could not stamp SimTool as published.\n")

tmpdir = tempfile.mkdtemp(prefix="finalizetool_",dir=os.path.join(os.sep,'tmp'))
tardir = os.path.join(tmpdir,"%s-r%d" % (options['project'],revision))

exitStatus,info = gitHttp(['clone',repoURL,os.path.basename(tardir)],gitPath=tmpdir)
if exitStatus == 0:
   # directory had better be there now
   if not os.path.isdir(os.path.join(tardir,'.git')):
      sys.stderr.write("== ERROR: git clone failed, .git directory missing\n")
      sys.exit(1)
   else:
      exitStatus,info,stdError = gitLocal(['checkout',revisionHash],gitPath=tardir)
      if exitStatus != 0:
         sys.stderr.write("== ERROR: git checkout %s failed\n" % (revisionHash))
         sys.stderr.write("%s\n" % (stdError))
         sys.exit(1)
else:
   sys.stderr.write("== ERROR: git clone failed\n")
   sys.stderr.write("%s\n" % (info))
   sys.exit(1)
#
# If the distribution contains a tool.xml file, then get information
# about the current version and substitute it into the file.
#
def xmlenc(xmlStr):
   if isinstance(xmlStr,bytes):
      xmlStr = xmlStr.decode("utf-8")
   xmlStr.replace("&","\007")
   xmlStr.replace("<","\&lt;")
   xmlStr.replace(">","\&gt;")
   xmlStr.replace("\007","\&amp;")

   return(xmlStr)


toolXmlFiles = find_files(targetdir,'tool.xml')
for tfile in toolXmlFiles:
   try:
      fp = open(tfile,'r')
      info = fp.read()
      fp.close()
   except:
      sys.stderr.write("can't read file %s\n" % (tfile))
      sys.stderr.write(traceback.format_exc())
      sys.exit(1)

   # find the start of the <tool> section
   t0,t1,text = find_section(['<tool>'],info)
   if t0 >= 0:
      t0 += 6
      # insert the given -version information
      if options['version'] != "":
         update = """<identifier>%s</identifier>""" % (xmlenc(options['version']))

         v0,v1,text = find_section(['<tool>','<version>','<identifier>'],info)
         if v0 >= 0:
            # replace <identifier> information
            info = string_replace(info,v0,v1,update)
         else:
            # can't find <identifier>, so insert one at start of <version>
            i0,i1,vers = find_section(['<tool>','<version>'],info)
            if i0 >= 0:
               i0 += 9
               info = string_insert(info,i0,"%s\n" % (update))
            else:
               sys.stderr.write("== ERROR: installtool goofed for the following file:\n")
               sys.stderr.write("   %s\n" % (tfile))
               sys.stderr.write("Can't find version section in XML to insert version tag \"%s\".\n" % (options['version']))
               sys.stderr.write("This might mean that \"installtool\" script was updated recently and this\n")
               sys.stderr.write("tool was installed with an older version of the script.\n")
               sys.stderr.write("Maybe run \"installtool\" step again?\n")
               sys.exit(1)

      # find/replace the tool <name>
      if options['title'] != "":
         update = """<name>%s</name>""" % (options['title'])
         n0,n1,text = find_section(['<tool>','<name>'],info)
         if n0 >= 0:
            info = string_replace(info,n0,n1,update)
         else:
            info = string_insert(info,t0,"  %s\n" % (update))

      # find/replace the tool <id>
      i0,i1,text = find_section(['<tool>','<id>'],info)
      if i0 < 0 or text != options['project']:
         update = """<id>%s</id>""" % (options['project'])
         if i0 >= 0:
            sys.stderr.write("== ERROR: installtool goofed for the following file:\n")
            sys.stderr.write("   %s\n" % (tfile))
            sys.stderr.write("Expected tool name \"%s\" but found \"%s\".\n" % (options['project'],text))
            sys.stderr.write("This might mean that \"installtool\" script was updated recently and this\n")
            sys.stderr.write("tool was installed with an older version of the script.\n")
            sys.stderr.write("Maybe run \"installtool\" step again?\n")
            sys.exit(1)
         else:
            sys.stderr.write("== ERROR: installtool goofed for the following file:\n")
            sys.stderr.write("   %s\n" % (tfile))
            sys.stderr.write("Expected tool name \"%s\" but found nothing.\n" % (options['project']))
            sys.stderr.write("This might mean that \"installtool\" script was updated recently and this\n")
            sys.stderr.write("tool was installed with an older version of the script.\n")
            sys.stderr.write("Maybe run \"installtool\" step again?\n")
            sys.exit(1)

      try:
         fp = open(tfile,'w')
         fp.write(info)
         fp.close()
      except:
         sys.stderr.write("while updating file %s\n" % (tfile))
         sys.stderr.write(traceback.format_exc())
         sys.exit(1)

      sys.stdout.write("Updated %s\n" % (tfile))

      # copy the updated file into the exported src distribution
      shutil.copy2(tfile,relocate(targetdir,tfile,tardir))
   else:
      sys.stderr.write("== WARNING: Can't find <tool> section in %s\n" % (tfile))
      sys.stderr.write("            Skipping tool.xml update\n")
#
# If there's a license, then save the text into a file called
# "LICENSE.txt" in the main directory.  If the file already exists,
# then warn if it's not identical.
#
if options['license'] != "":
   licfile = os.path.join(targetdir,'LICENSE.txt')
   if not os.path.exists(licfile):
      shutil.copy2(options['license'],licfile)
      exitStatus,info,stdError = gitLocal(['add',licfile],gitPath=targetdir)
      if exitStatus != 0:
         sys.stderr.write("== ERROR: can't add license file to repository.\n")
         sys.stderr.write("%s\n" % (stdError))
         sys.exit(1)
      exitStatus,info,stdError = gitLocal(['commit',licfile,'-m',"\"added license terms in LICENSE.txt\""],gitPath=targetdir)
      if exitStatus != 0:
         sys.stderr.write("== ERROR: can't commit license file to repository.\n")
         sys.stderr.write("%s\n" % (stdError))
         sys.exit(1)
   else:
      sys.stdout.write("== license licfile already exists ==\n")
      exitStatus,stdOutput,stdError = exit_on_error(['diff',licfile,options['license']])
      if exitStatus != 0:
         sys.stdout.write("== WARNING: license file exists and is different:\n")
         sys.stdout.write(stdOutput)
         sys.stdout.write("== WARNING: replacing license file with current version\n")
         shutil.copy2(options['license'],licfile)
         exitStatus,info,stdError = gitLocal(['add',licfile],gitPath=targetdir)
         if exitStatus != 0:
            sys.stderr.write("== ERROR: can't add license file to repository.\n")
            sys.stderr.write("%s\n" % (stdError))
            sys.exit(1)
         exitStatus,info,stdError = gitLocal(['commit',licfile,'-m',"\"modified license terms in LICENSE.txt\""],gitPath=targetdir)
         if exitStatus != 0:
            sys.stderr.write("== ERROR: can't commit modified license file to repository.\n")
            sys.stderr.write("%s\n" % (stdError))
            sys.exit(1)
   shutil.copy2(options['license'],relocate(targetdir,licfile,tardir))
   try:
      os.remove(options['license'])
   except:
      pass
#
# Always close off permission to the "src" directory, and warn if
# we can't find it.  If the code is open source, people should be
# downloading the tarball or else checking out the code from git.
# Only the middleware needs the /apps stuff, and it doesn't need to
# access source code--only executables.
#
srcDir = os.path.join(targetdir,'src')
if os.path.exists(srcDir):
   gitfileMode = os.lstat(srcDir).st_mode
#  remove all other permissions
   gitfileMode &= ~stat.S_IRWXO
   os.chmod(srcDir,gitfileMode)
else:
   sys.stdout.write("== WARNING: can't find src directory in project %s\n" % (options['project']))
   sys.stdout.write("== WARNING: Source code may be unprotected!\n")
#
# Always make a tarball for achival purposes and store it in the
# /tmp directory.  The calling code will copy the file back to a
# storage directory in the web area.  If the code is open source,
# then the web will allow access, and if it's closed source, the
# web will protect it.
#
tarfile = os.path.join(os.sep,'tmp',"%s-r%d.tar.gz" % (options['project'],revision))
try:
   os.remove(tarfile)
except:
   pass

shutil.rmtree(os.path.join(tardir,'.git'),True)
exitStatus,stdOutput,stdError = exit_on_error(['tar','cvzf',tarfile,'--directory',tmpdir,"./%s-r%d" % (options['project'],revision)])
if exitStatus != 0:
   sys.stderr.write("== ERROR: tarball creation failed:\n")
   sys.stderr.write(stdOutput)
   sys.stderr.write(stdError)
   sys.exit(1)

# clean up the src export
shutil.rmtree(tmpdir,True)

tarfileMode = os.lstat(tarfile).st_mode
tarfileMode |= stat.S_IROTH
os.chmod(tarfile,tarfileMode)

# return the name of the tarball for the database
sys.stdout.write("source tarball: %s\n" % (tarfile))

sys.exit(0)
