#!/usr/bin/env python
#
# @package      hubzero-forge
# @file         gensvnapache.py
# @author       Nicholas J. Kisseberth <nkissebe@purdue.edu>
# @author       Steven M. Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2010-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2010-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

# Really quick and dirty port of the perl script of similar name
# but querying project data from MySQL instead of LDAP.
# This should get cleaned up an rewritten to use
# the hubzero python library (which isn't ready
# at the time of this writing).
#

import os
import sys
import argparse
import re
import subprocess
try:
   import configparser
except:
   import ConfigParser as configparser
try:
   import pymysql as mysqldb
except:
   import MySQLdb as mysqldb

#
# @TODO: Add commandline or environment variable overrides to the autodetection
#

parser = argparse.ArgumentParser(description="Create the Apache configuration file controlling access to SVN repositories.")
parser.add_argument('--svnroot',required=True,help="path to SVN repositories")
parser.add_argument('--svnurl',required=True,help="URL path to SVN repository, relative to the document root")
parser.add_argument('--svnconf',required=True,help="path to Apache configuration file")
parser.add_argument('--projectType',required=False,help="project type")

args = parser.parse_args()

#print(args)

svnroot     = args.svnroot
svnurl      = args.svnurl
svnconf     = args.svnconf
projectType = args.projectType

if not os.path.isdir(svnroot):
   sys.stderr.write("Error:  '" + svnroot + "' is not a valid directory.\n")
   sys.exit(2)

#
# Detect whether we are running Apache 2.2 or not so we set proper configuration
# directives later on. We assume 2.2 until proven otherwise. We also assume
# standard Debian or Redhat server locations. If running a custom installation
# this will likely fall back to assuming 2.2.
#
apache_2_2 = True
apacheBin = False

apachePath = os.path.join(os.sep,'usr','sbin','apache2')
if os.path.exists(apachePath):
   apacheBin = apachePath
else:
   apachePath = os.path.join(os.sep,'usr','sbin','httpd')
   if os.path.exists(apachePath):
      apacheBin = apachePath

if apacheBin:
   proc = subprocess.Popen([apacheBin, "-v"], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
   out,err = proc.communicate()

   apacheVersion = re.compile(b'.*Apache/(\d+)\.(\d+)\.(\d+)',flags=re.MULTILINE)

   m = apacheVersion.match(out)

   if m and int(m.group(1)) >= 2 and int(m.group(2)) > 2:
      apache_2_2 = False


def loadHubConfigurationData(configurationFileDirname,
                             configurationFileiBasename):
   hubconfigurationData = {}
   if   not os.path.isdir(configurationFileDirname):
      sys.stderr.write("ERROR: specified base directory does not exist at %s\n" % (configurationFileDirname))
      sys.exit(5)
   else:
      hubconfigurationPath = os.path.join(configurationFileDirname,configurationFileiBasename)
      if not os.path.isfile(hubconfigurationPath):
         sys.stderr.write("ERROR: specified hubconfiguration.php file does exist at %s\n" % (hubconfigurationPath))
         sys.exit(5)
      else:
         reHubconfigurationVar = re.compile("\s*(?:var|public)\s+\$(\w+)\s*=\s*\'*(.*?)\'*\s*\;\s*")
         try:
            with open(hubconfigurationPath,'r') as fpHubconfiguration:
               for line in fpHubconfiguration:
                  match = reHubconfigurationVar.match(line)
                  if match:
                     hubconfigurationData[match.group(1)] = match.group(2).strip(" \'\"\t").replace("\\'","'")
         except:
            sys.stderr.write("ERROR: specified hubconfiguration file %s could not be ingested\n" % (hubconfigurationPath))
            sys.stderr.write(traceback.format_exc())
            sys.exit(5)

   return(hubconfigurationData)


config = configparser.RawConfigParser()
config.read(os.path.join(os.sep,'etc','hubzero.conf'))
section = config.get('DEFAULT','site')
docroot = config.get(section,'DocumentRoot')

data = {}
data['joomla']  = loadHubConfigurationData(docroot,'configuration.php')
data['hubzero'] = loadHubConfigurationData(docroot,'hubconfiguration.php')

mysql_host     = data['joomla']['host']
mysql_user     = data['joomla']['user']
mysql_password = data['joomla']['password']
mysql_db       = data['joomla']['db']

try:
   db = mysqldb.connect(host=mysql_host, user=mysql_user, passwd=mysql_password, db=mysql_db)
except:
   sys.stderr.write("Error: Unable to connect to HUBzero database.\n")
   sys.exit(4)
else:
   cursor = db.cursor()
   if not cursor:
      sys.stderr.write("Error: Unable to obtain HUBzero database cursor.\n")
      sys.exit(5)

ldapbase     = data['hubzero']['hubLDAPBaseDN']
ldaphost     = data['hubzero']['hubLDAPMasterHost']
ldapbinduser = data['hubzero']['hubLDAPSearchUserDN']
ldapbindpass = data['hubzero']['hubLDAPSearchUserPW']
try:
   ldapgroupobjectclass = data['hubzero']['hubLDAPGroupObjectClass']
except:
   ldapgroupobjectclass = 'posixGroup'
finally:
   if ldapgroupobjectclass != 'posixGroup':
      ldapgroupobjectclass = 'hubGroup'

try:
   fpSvnConf = open(svnconf,'w')
except:
   sys.stderr.write("ERROR: Attempt to open %s failed.\n" % (svnconf))
else:
   apacheHeader = """#
# Apache configuration for SVN repositories
#
# This file is generated automatically from MySQL.
#      !!!  DO NOT MODIFY THIS FILE  !!!
#
"""

   fpSvnConf.write(apacheHeader)

   dirlist = os.listdir(svnroot)

   for dir in dirlist:
      if not os.path.isdir(os.path.join(svnroot,dir)):
         continue

      sourcePublic = False
      id = dir
      if projectType != 'tool':
         gid = []
         gid.append(id)
         gid.append('apps')
         name = id
      else:
         gid = set([])
         cursor.execute("SELECT title,codeaccess,tg.cn FROM jos_tool_version AS tv, jos_tool_groups AS tg WHERE tg.toolid=tv.toolid AND tv.toolname='" + id + "' AND state='3' AND (tg.role=1 or tg.role=2)")
         rows = cursor.fetchall()
         if len(rows) == 0:
            continue
         name = id
         for name,sourceOpen,gidref in rows:
            if sourceOpen == '@OPEN':
               sourcePublic = True
            else:
               sourcePublic = False
            gid.add(gidref)

      if os.path.isfile(os.path.join(svnroot,id,'conf','svnauthz.conf')):
         svnauthz = True
      else:
         svnauthz = False

      url = svnurl.replace("@PROJECT@",id)

      fpSvnConf.write("<Location %s>\n" % (url))
      fpSvnConf.write("\tRemoveEncoding .tgz\n")
      fpSvnConf.write("\tRemoveEncoding .gz\n")
      fpSvnConf.write("\tRemoveEncoding .Z\n")
      fpSvnConf.write("\tDAV svn\n")
      fpSvnConf.write("\tSVNPath %s\n" % (os.path.join(svnroot,id)))
      fpSvnConf.write("\tAuthType Basic\n")
      fpSvnConf.write("\tAuthBasicProvider ldap\n")
      fpSvnConf.write("\tAuthName \"%s Subversion Repository\"\n" % (name.replace('"',r'\"')))

      if apache_2_2:
         fpSvnConf.write("\tAuthzLDAPAuthoritative on\n")
      if ldapbinduser != "":
         fpSvnConf.write("\tAuthLDAPBindDN \"%s\"\n" % (ldapbinduser))
         fpSvnConf.write("\tAuthLDAPBindPassword \"%s\"\n" % (ldapbindpass))
      if ldapgroupobjectclass == 'hubGroup':
         fpSvnConf.write("\tAuthLDAPGroupAttributeIsDN on\n")
         fpSvnConf.write("\tAuthLDAPGroupAttribute owner\n")
         fpSvnConf.write("\tAuthLDAPGroupAttribute member\n")
      else:
         fpSvnConf.write("\tAuthLDAPGroupAttributeIsDN off\n")
         fpSvnConf.write("\tAuthLDAPGroupAttribute memberUid\n")

      fpSvnConf.write("\tAuthLDAPURL %s/ou=users,%s\n" % (ldaphost,ldapbase))

      if svnauthz:
         fpSvnConf.write("\tAuthzSVNAccessFile %s\n" % (os.path.join(svnroot,id,'conf','svnauthz.conf')))
         fpSvnConf.write("\tRequire valid-user\n")
         if sourcePublic:
            fpSvnConf.write("\tSatisfy Any\n")
      else:
         if sourcePublic:
            fpSvnConf.write("\t<LimitExcept GET PROPFIND OPTIONS REPORT>\n")
         for group in gid:
            if sourcePublic:
               fpSvnConf.write("\t")
            if ldapgroupobjectclass == 'hubGroup':
               fpSvnConf.write("\tRequire ldap-group gid=%s,ou=groups,%s\n" % (group,ldapbase))
            else:
               fpSvnConf.write("\tRequire ldap-group cn=%s,ou=groups,%s\n" % (group,ldapbase))
         if sourcePublic:
            fpSvnConf.write("\t</LimitExcept>\n")

      fpSvnConf.write("</Location>\n")
      fpSvnConf.write("\n")

   fpSvnConf.close()

db.close()

