#!/usr/bin/env python3
#
# @package      hubzero-forge
# @file         installtoolGITexternal.py
# @author       Steven M. Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2005-2018 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2005-2018 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
#----------------------------------------------------------------------
#  USAGE:
#    installtool ?flags? <project>
#
#    where ?flags? includes:
#      -hub example.com .. doing install for this hub
#      -root /where ...... install into this directory (default /apps)
#      -revision num ..... install a particular subversion revision
#                          from the trunk (default is latest revision)
#      -as current|dev ... install as current or dev version.  This
#                          sets a symbolic link to the installed
#                          version.  (default is no link)
#
#      -type app ......... Type of project--either "app" or "raw".
#                          App projects are all named "app-project".
#                          For raw projects, the project name is used
#                          directly.
#
#      -hubdir f ......... Load configuration information from this
#                          directory, assumed to be HubConfiguration Class
#                          with simple variable assignments.  In
#                          particular, we look for:
#                            $forgeURL ......... example.com
#      -gitURL ........... URL for external GIT repository
#
#----------------------------------------------------------------------

import os
import sys
import stat
import argparse
import glob
import re
import shutil
import subprocess
import pexpect
import traceback
import time
try:
   import nbformat
except:
   pass

HUBCONFIGURATIONFILE = 'hubconfiguration.php'
GITHOMEDIRECTORY     = os.path.join(os.sep,'opt','gitExternal','tools')

os.umask(0o022)

# this is an easy way to exec commands and catch errors:
def exit_on_error(command):
   child = subprocess.Popen(command,
                            shell=False,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE,
                            close_fds=True,
                            universal_newlines=True)
   stdOutput,stdError = child.communicate()
   exitStatus = child.returncode
   if exitStatus != 0:
      sys.stderr.write("FAILED: %s\n" % (command))
      if stdError:
         sys.stderr.write("%s\n" % (stdError))
      if stdOutput:
         sys.stderr.write("%s\n" % (stdOutput))

   return(exitStatus,stdOutput,stdError)

#
# use these procedures to exec git commands
#
def gitLocal(gitArgs,
             gitPath=None):
   if gitPath:
      currentPath = os.getcwd()
      os.chdir(gitPath)

   command = ['git'] + gitArgs
   exitStatus,stdOutput,stdError = exit_on_error(command)

   if gitPath:
      os.chdir(currentPath)

   return(exitStatus,stdOutput,stdError)


def gitHttp(gitArgs,
            gitPath=None):
   if gitPath:
      currentPath = os.getcwd()
      os.chdir(gitPath)

   timeout = 300 # timeout for expect

   command = ['git'] + gitArgs

#  print(' '.join(command))
   rePassword = re.compile(b"Password( for '.*')?:")
   result,exitStatus = pexpect.run(' '.join(command),timeout=timeout,withexitstatus=True, \
                                   events={rePassword:"%s\n" % (options['repopw'])})
   result = result.replace(b'\r',b'')

   if gitPath:
      os.chdir(currentPath)

   return(exitStatus,result)


# use this to search for files such as tool.xml:
def find_files(dir,
               fname):
   found = []
   dirlist = [dir]
   while len(dirlist) > 0:
      dir = dirlist.pop(0)
# dir is not a regular file or directory,
# if it is a link to a file or directory under the original
# directory path, the real file will be picked up.
      if not os.path.islink(dir):
         for dirFile in glob.iglob(os.path.join(dir,'*')):
            if   os.path.isdir(dirFile):
               dirlist.append(dirFile)
            elif os.path.basename(dirFile) == fname:
               if not os.path.islink(dirFile):
                  found.append(dirFile)

   return(found)


def find_section(tags,
                 text):
   tag = tags.pop(0)
   tagend = re.sub('^<','</',tag)
   tagalone = re.sub('>$','/>',tag)

   lookFor = "(%s)(.*:?)(%s)" % (tag,tagend)
   found = re.search(lookFor,text,flags=re.IGNORECASE|re.DOTALL)
   if found:
      m0 = found.start(1)
      m1 = found.end(3)
      i0 = found.start(2)
      i1 = found.end(2)
      subtext = text[i0:i1]
      if len(tags) == 0:
         return(m0,m1,subtext)
      s0,s1,subtext = find_section(tags,subtext)
      if s0 < 0:
         return(-1,-1,"")
      return(i0+s0,i0+s1,subtext)
      
   lookFor = "(%s)" % (tagalone)
   found = re.search(lookFor,text,flags=re.IGNORECASE|re.DOTALL)
   if found:
      m0 = found.start(1)
      m1 = found.end(1)
      if len(tags) == 0:
         return(m0,m1,"")
      return(-1,-1,"")

   return(-1,-1,"")


def string_insert(string,
                  index,
                  text):
   if index > 0:
      substitutedString = string[:index]
   else:
      substitutedString = ""
   substitutedString += text
   substitutedString += string[index:]

   return(substitutedString)


def string_replace(string,
                   indexStart,
                   indexEnd,
                   text):
   substitutedString = string.replace(string[indexStart:indexEnd],text)

   return(substitutedString)


#
#----------------------------------------------------------------------
# Parse all command line options
#----------------------------------------------------------------------
#
parser = argparse.ArgumentParser(description="Install git repository for any tool development project on a HUBzero site.")
parser.add_argument('--hub',required=False,help="domain name.")
parser.add_argument('--root',required=False,default='/apps',help="Install the tool in this directory.")
parser.add_argument('--revision',required=False,default='HEAD',help="Install this git revision of the tool.")
parser.add_argument('--from',dest='coFrom',required=False,default='trunk',help="Check out from trunk, branch, or tag.")
parser.add_argument('--as',dest='linkAs',required=False,default='dev',choices=['current','dev','none'],help="Create symbolic link as dev or current version.")
parser.add_argument('--type',dest='projectType',required=False,default='app',choices=['app','raw'],help="Type of the project either app or raw name.")
parser.add_argument('--hubdir',required=True,help="location of hubconfiguration.php")
parser.add_argument('--project',required=True,help="project shortname")
parser.add_argument('--gitURL',required=True,help="URL for external GIT repository")
parser.add_argument('--publishOption',required=False,help="publishing option")

args = parser.parse_args()

#print(args)

options = {}
options['hub']      = args.hub
options['root']     = args.root
options['revision'] = args.revision
options['as']       = args.linkAs
options['type']     = args.projectType
options['hubdir']   = args.hubdir
options['project']  = args.project
options['gitURL']   = args.gitURL

if not os.path.isdir(options['hubdir']):
   sys.stderr.write("ERROR: specified base directory does not exist\n")
   sys.stderr.write(" at %s\n" % (options['hubdir']))
   sys.exit(5)

###################### configuration scan (poor hack) ######################

def loadHubConfigurationData(configurationFileDirname,
                             configurationFileiBasename):
   hubconfigurationData = {}
   if   not os.path.isdir(configurationFileDirname):
      sys.stderr.write("ERROR: specified base directory does not exist at %s\n" % (configurationFileDirname))
      sys.exit(5)
   else:
      hubconfigurationPath = os.path.join(configurationFileDirname,configurationFileiBasename)
      if not os.path.isfile(hubconfigurationPath):
         sys.stderr.write("ERROR: specified hubconfiguration.php file does exist at %s\n" % (hubconfigurationPath))
         sys.exit(5)
      else:
         reHubconfigurationVar = re.compile("\s*(?:var|public)\s+\$(\w+)\s*=\s*\'*(.*?)\'*\s*\;\s*")
         try:
            with open(hubconfigurationPath,'r') as fpHubconfiguration:
               for line in fpHubconfiguration:
                  match = reHubconfigurationVar.match(line)
                  if match:
                     hubconfigurationData[match.group(1)] = match.group(2).strip(" \'\"\t").replace("\\'","'")
         except:
            sys.stderr.write("ERROR: specified hubconfiguration file %s could not be ingested\n" % (hubconfigurationPath))
            sys.stderr.write(traceback.format_exc())
            sys.exit(5)

   return(hubconfigurationData)


def getcfg(key):
   if key in hubconfigurationData:
      cfg = hubconfigurationData[key]
   else:
      cfg = ""

   return(cfg)

hubconfigurationData = loadHubConfigurationData(args.hubdir,HUBCONFIGURATIONFILE)

#
#----------------------------------------------------------------------
# Grab options from the hubconfiguration file, if there is one.
#----------------------------------------------------------------------
#

if not options['hub']:
   options['hub'] = getcfg('forgeURL')

options['repouser'] =  getcfg('svn_user')
options['repopw']   =  getcfg('svn_password')

if   options['type'] == 'app':
   prefix = "app" + options['project']
elif options['type'] == 'raw':
   prefix = options['project']

#
# Build the name of the install directory.  If -revision is "HEAD"
# then figure out the revision number.  Otherwise, use it as-is.
# Build up a combination of the -root and -revision values into
# a name that looks like root/app-name/rXXX
#
repoPath = os.path.join(GITHOMEDIRECTORY,options['project'])
protocol,forgeURL = options['hub'].split('://')
hubURL = "%s://%s@%s" % (protocol,options['repouser'],forgeURL)
repoURL  = '/'.join((hubURL,'tools',prefix,'gitExternal',prefix))

if options['revision'] == "HEAD":
   exitStatus,info,stdError = gitLocal(['rev-list',"--all"],gitPath=repoPath)
   if exitStatus == 0:
      revisionList = info.split()
      nRevisions = len(revisionList)
      revision = nRevisions
      revisionHash = revisionList[0]
   else:
      sys.stderr.write("error: can't find the HEAD revision in git.\n")
      sys.stderr.write("%s\n" % (stdError))
      sys.exit(1)
else:
   exitStatus,info,stdError = gitLocal(['rev-list',"--all"],gitPath=repoPath)
   if exitStatus == 0:
      revisionList = info.split()
      nRevisions = len(revisionList)
      if options['revision'] in revisionList:
         revision = nRevisions-revisionList.index(options['revision'])
         revisionHash = options['revision']
      else:
         sys.stderr.write("error: can't determine specified revision in git.\n")
         sys.stderr.write("got: %s\n" % (revisionList))
         sys.exit(1)
   else:
      sys.stderr.write("error: can't find specified revision in git.\n")
      sys.stderr.write("%s\n" % (stdError))
      sys.exit(1)

targetdir = os.path.join(options['root'],options['project'],"r%d" % (revision))

#
# Create the directory, if needed.  If the directory appears to
# have stuff in it, then "update" its contents.  Otherwise, do
# a "checkout" to get the contents.
#

def mkdir_ifneeded(dir):
   if not os.path.isdir(dir):
      try:
         os.makedirs(dir)
      except:
         sys.stderr.write(traceback.format_exc())
         sys.exit(1)

if os.path.isdir(targetdir):
   shutil.rmtree(targetdir,True)

mkdir_ifneeded(os.path.dirname(targetdir))

exitStatus,info = gitHttp(['clone',repoURL,targetdir],gitPath=os.path.dirname(targetdir))
if exitStatus == 0:
   # directory had better be there now
   if not os.path.isdir(os.path.join(targetdir,'.git')):
      sys.stderr.write("== ERROR: git clone failed, .git directory missing\n")
      sys.exit(1)
   else:
      exitStatus,info,stdError = gitLocal(['checkout',revisionHash],gitPath=targetdir)
      if exitStatus != 0:
         sys.stderr.write("== ERROR: git checkout %s failed\n" % (revisionHash))
         sys.stderr.write("%s\n" % (stdError))
         sys.exit(1)
else:
   sys.stderr.write("== ERROR: git clone failed\n")
   sys.stderr.write("%s\n" % (info))
   sys.exit(1)
#
# Make a symbolic link to install this final version according
# to the -as argument.  If -as is "", then avoid making a link.
#
if options['as'] != 'none':
   dir = os.path.dirname(targetdir)
   savedir = os.getcwd()
   os.chdir(dir)
   shutil.rmtree(options['as'],True)
   try:
      os.remove(options['as'])
   except:
      pass
   os.symlink(os.path.basename(targetdir),options['as'])
   os.chdir(savedir)
#
# Look for an "invoke" script in the "middleware" directory,
# and if it's not there, create one automatically and print a
# warning about it.
#
if not os.path.exists(os.path.join(targetdir,'middleware')):
   sys.stdout.write("== WARNING: Missing middleware directory.\n")
else:
   invokescript = os.path.join(targetdir,'middleware','invoke')
   if not os.path.exists(invokescript):
      sys.stdout.write("== WARNING: Missing middleware/invoke script.\n")
      try:
         fp = open(invokescript,'w')
         fp.write("#!/bin/sh\n\n")
         fp.write("/usr/bin/invoke_app \"$@\" -C rappture -t %s\n" % (options['project']))
         fp.close()
         os.chmod(invokescript,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
      except:
         sys.stderr.write("== ERROR: Attempt to create invoke script failed.\n")
         sys.stderr.write(traceback.format_exc())
      else:
         sys.stdout.write("            Created default middleware/invoke script.\n")
#
# if a src directory exists hide it from the world. open source code is
# treated elsewhere
#
srcDir = os.path.join(targetdir,'src')
if os.path.exists(srcDir):
   gitfileMode = os.lstat(srcDir).st_mode
#  remove all other permissions
   gitfileMode &= ~stat.S_IRWXO
   os.chmod(srcDir,gitfileMode)

#
# If the distribution contains a tool.xml file, then get information
# about the current version and substitute it into the file.
#
def xmlenc(xmlStr):
   if isinstance(xmlStr,bytes):
      xmlStr = xmlStr.decode("utf-8")
   xmlStr.replace("&","\007")
   xmlStr.replace("<","\&lt;")
   xmlStr.replace(">","\&gt;")
   xmlStr.replace("\007","\&amp;")

   return(xmlStr)


toolXmlFiles = find_files(targetdir,'tool.xml')
for tfile in toolXmlFiles:
   exitStatus,info,stdEr = gitLocal(['log','--max-count=1',tfile],gitPath=targetdir)
   if exitStatus == 0:
      if isinstance(info,bytes):
         lookFor = b"commit ([a-z,0-9]+)"
      else:
         lookFor = "commit ([a-z,0-9]+)"
      found = re.search(lookFor,info,flags=re.IGNORECASE|re.MULTILINE)
      if found:
         gitLastModifiedHash = found.group(1)
      else:
         gitLastModifiedHash = ""

      if isinstance(info,bytes):
         lookFor = b"\nDate: *([^\r\n]+)"
      else:
         lookFor = "\nDate: *([^\r\n]+)"
      found = re.search(lookFor,info,flags=re.IGNORECASE|re.MULTILINE)
      if found:
         gitLastModifiedDate = found.group(1)
      else:
         gitLastModifiedDate = ""
   else:
      sys.stderr.write("== ERROR: git log %s failed\n" % (tfile))
      sys.stderr.write("%s\n" % (stdError))
      sys.exit(1)
   #
   # If we have all of the information, substitute into the tool.xml file.
   #
   if gitLastModifiedHash and gitLastModifiedDate:
      try:
         fp = open(tfile,'r')
         info = fp.read()
         fp.close()
      except:
         sys.stderr.write("can't read file %s\n" % (tfile))
         sys.stderr.write(traceback.format_exc())
         sys.exit(1)
      else:
         # find the start of the <tool> section
         t0,t1,text = find_section(['<tool>'],info)
         if t0 >= 0:
            t0 += 6
            version = "???"
            i0,i1,version = find_section(['<tool>','<version>','<identifier>'],info)
            date = time.strftime("%Y-%m-%d %H:%M:%S %Z")
            update = """<version>
    <identifier>%s</identifier>
    <application>
      <revision>%s</revision>
      <modified>%s</modified>
      <installed>%s</installed>
      <directory id="top">%s</directory>
      <directory id="tool">%s</directory>
    </application>
  </version>""" % (xmlenc(version),xmlenc(gitLastModifiedHash),xmlenc(gitLastModifiedDate),xmlenc(date),
                   xmlenc(targetdir),xmlenc(os.path.dirname(tfile)))

            v0,v1,text = find_section(['<tool>','<version>'],info)
            if v0 >= 0:
                info = string_replace(info,v0,v1,update)
            else:
                info = string_insert(info,t0,"  %s\n" % (update))

            # find/replace the tool <name> -- use project name for now
            n0,n1,text = find_section(['<tool>','<name>'],info)
            update = """<name>%s</name>""" % (options['project'])
            if n0 >= 0:
               info = string_replace(info,n0,n1,update)
            else:
               info = string_insert(info,t0,"  %s\n" % (update))

            # find/replace the tool <id>
            i0,i1,text = find_section(['<tool>','<id>'],info)
            update = """<id>%s</id>""" % (options['project'])
            if i0 >= 0:
               info = string_replace(info,i0,i1,update)
            else:
               info = string_insert(info,t0,"  %s\n" % (update))

            try:
               fp = open(tfile,'w')
               fp.write(info)
               fp.close()
            except:
               sys.stderr.write("== ERROR: while updating file %s\n" % (tfile))
               sys.stderr.write(traceback.format_exc())
               sys.exit(1)
            sys.stdout.write("Updated %s\n" % (tfile))
         else:
            sys.stderr.write("WARNING: Can't find <tool> section in %s\n" % (tfile))
            sys.stderr.write("         Skipping tool.xml update\n")
   else:
      sys.stderr.write("WARNING: Can't find tool revision/date in git log.\n")
      sys.stderr.write("         Skipping tool.xml update\n")

if args.publishOption == 'simtool':
   exitStatus = 0
   notebookPath = os.path.join(targetdir,'simtool',"%s.ipynb" % (options['project']))
   if os.path.exists(notebookPath):
      try:
         nb = nbformat.read(notebookPath,nbformat.NO_CONVERT)
      except FileNotFoundError as err:
         #[Errno 2] No such file or directory: 'outputList/r1/outputList.ipy'
         sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
         exitStatus = 1
      except PermissionError as err:
         #[Errno 13] Permission denied: 'outputList/r1/outputList.ipynb'
         sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
         exitStatus = 1
      except (AttributeError,KeyError) as err:
         sys.stderr.write("Missing attribute: %s\n" % (err.args[0]))
         exitStatus = 1
      except NameError as err:
         sys.stderr.write("Name error: %s\n" % (err.args[0]))
         exitStatus = 1
      else:
# clear existing metadata
         try:
            del nb['metadata']['simTool_info']
         except (AttributeError,KeyError) as err:
            pass

# add metadata
         nb['metadata']['simTool_info'] = {}
         nb['metadata']['simTool_info']['name']     = options['project']
         nb['metadata']['simTool_info']['revision'] = revision
         nb['metadata']['simTool_info']['state']    = 'installed'

#        print(nb['metadata']['simTool_info'])

         try:
            nbformat.write(nb,notebookPath,nbformat.NO_CONVERT)
         except PermissionError as err:
            sys.stderr.write("%s: %s\n" % (err.strerror,err.filename))
            exitStatus = 1
   else:
      sys.stderr.write("WARNING: SimTool notebook does not exist.\n")
      exitStatus = 1

   if exitStatus != 0:
      sys.stderr.write("WARNING: Could not stamp SimTool as installed.\n")

# return the current git commit hash as the result
sys.stdout.write("installed revision: %d commitHash: %s\n" % (revision,revisionHash))

sys.exit(0)

