#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <ifaddrs.h>
#include <getopt.h>
#include <rfb/rfbclient.h>
#include <stdarg.h>

static unsigned char *rfbPassword = NULL;

#define EN0	0		/* MODE == encrypt */
#define DE1	1		/* MODE == decrypt */
 
static void deskey(unsigned char *, short);
static void des(unsigned char *, unsigned char *);

static char *GetPassword(rfbClient *client)
{
	unsigned char *dest = NULL;
	unsigned char d3desObfuscationKey[9] = {23,82,107,6,35,78,88,7,0};

	dest = malloc(9);

	if (dest == NULL)
		return NULL;

	memset(dest, 0, 9);

	deskey(d3desObfuscationKey, DE1);

	des(rfbPassword, dest);

	dest[8] = 0;

	return dest;
}

int
main(int argc, char **argv)
{
	char *rfbAuthFilename = NULL;
	char *rfbHost = NULL;
	char *rfbPortStr = NULL;
	long  rfbPort = 0;
	int   rfbWidth = 0;
	int   rfbHeight = 0;
	int   verbose = 0;
	int   debug = 0;
	int   quiet = 0;

	unsigned char b[6];
	rfbClient *client = NULL;
	int result = 0;
	int c = 0;
	char *endptr = NULL;

	while (1) 
	{
		static struct option long_options[] = {
			{ "version", no_argument, 0, 'r' },
			{ "help", no_argument, 0, '?' },
			{ "verbose", no_argument, 0, 'v' },
			{ "quiet", no_argument, 0, 'q' },
			{ "debug", no_argument, 0, 'd' },
			{ "rfbhost", required_argument, 0, 'h' },
			{ "rfbport", required_argument, 0, 'p' },
			{ "rfbpasswd", required_argument, 0, 'w' },
			{ "rfbauth", required_argument, 0, 'a' },
			{ 0, 0, 0, 0 }
		};

		int option_index = 0;

		c = getopt_long(argc, (char **) argv, "dr?vqh:p:w:a:", long_options, &option_index);

		if (c == -1)
		{
			break;
		}

		switch (c) 
		{
			case 'r': 
				fprintf(stdout, "hzvncresize 1.0\n");
				fprintf(stdout, "Copyright (c) 2012 HUBzero Foundation, LLC.\n");
				fprintf(stdout, "This is free software: you can redistribute it and/or modify it under the terms\n");
				fprintf(stdout, "of the GNU General Public License version 3 or later.\n");
				fprintf(stdout, "<http://gnu.org/licenses/gpl.html>\n");
				fprintf(stdout, "\n");
				fprintf(stdout, "There is NO WARRANTY, to the extent permitted by law.\n");
				fprintf(stdout, "\n");
				fprintf(stdout, "Written by Nicholas J. Kisseberth.\n");
				return 0;

			case '?': 
				fprintf(stdout, "Usage: hzvncresize [OPTIONS] width height\n");
				fprintf(stdout, "Send a HUBzero frame buffer resize request to a HUBzero VNC server.\n");
				fprintf(stdout, "\n");
				fprintf(stdout, "Mandatory arguments to long options are mandatory for short options too.\n");
				fprintf(stdout, "  -v, --verbose               print more information about connection to stderr\n");
				fprintf(stdout, "  -q, --quiet                 print no information about connection to stderr\n");
				fprintf(stdout, "  -d, --debug                 print debugging information to stderr\n");
				fprintf(stdout, "  -h, --rfbhost hostname      host or IP address of VNC server (default: localhost)\n");
				fprintf(stdout, "  -p, --rfbport port#         port VNC server is listening on (default: 5000)\n");
				fprintf(stdout, "  -w, --rfbpasswd password    hex encoded vnc password\n");
				fprintf(stdout, "  -a, --rfbauth filename      file containing vnc server password (default: ~/.vnc/passwd)\n");
				fprintf(stdout, "\n");
				fprintf(stdout, "  -?, --help                  display this help and exit\n");
				fprintf(stdout, "  -r, --version               output version information and exit\n");
				fprintf(stdout, "\n");
				fprintf(stdout, "width is new framebuffer width\n");
				fprintf(stdout, "height is new framebuffer height\n");
				return 0;

			case 'v': 
				verbose = 1;
				quiet = 0;
				break;

			case 'd': 
				debug = 1;
				verbose = 1;
				quiet = 0;
				break;

			case 'q': 
				quiet = 1;
				verbose = 0;
				debug = 0;
				break;

			case 'h': 
				rfbHost = optarg;
				break;

			case 'p':
				rfbPortStr = optarg;
				break;

			case 'w': 
				rfbPassword = optarg;
				break;

			case 'a':
				rfbAuthFilename = optarg;
				break;
		}
	}

	if (optind < argc)
	{
		endptr = NULL;			
		errno  = 0;

		rfbWidth = strtol(argv[optind], &endptr, 10);

		if (errno != 0 || endptr == NULL || *endptr != 0 || rfbWidth <=0 || rfbWidth > 65535)
		{
			if (!quiet)
			{
				fprintf(stderr,"Invalid geometry width [%s] specified.\n", argv[optind]);
			}
			return 1;
		}

		optind++;
	}
	
	if (optind < argc)
	{
		endptr = NULL;
		errno  = 0;

		rfbHeight = strtol(argv[optind], &endptr, 10);

		if (errno != 0 || endptr == NULL || *endptr != 0 || rfbHeight <=0 || rfbHeight > 65535)
		{
			if (!quiet)
			{
				fprintf(stderr,"Invalid geometry height [%s] specified.\n", argv[optind]);
			}
			return 1;
		}

		optind++;
	}

	if (optind < argc)
	{
		if (!quiet)
		{
			fprintf(stderr,"%s: unrecognized option '%s'\n", argv[0], argv[optind]);
			fprintf(stderr,"Try '%s: --help' for more information.\n", argv[0]);
		}
		return 1;
	}

	if (rfbWidth == 0 || rfbHeight == 0)
	{
		fprintf(stderr,"No geometry was specified.\n");
		return 1;
	}

	if (rfbHost == NULL)
	{
		rfbHost = "localhost";
	}

	if (rfbPortStr)
	{
		char *endptr = NULL;
					
		errno = 0;

		rfbPort = strtol(rfbPortStr, &endptr, 10);

		if (errno != 0 || endptr == NULL || *endptr != 0 || rfbPort <=0 || rfbPort > 65535)
		{
			if (!quiet)
			{
				fprintf(stderr,"Invalid port [%s] specified for rfbPort.\n", rfbPortStr);
			}

			return 1;
		}
	}

	if (rfbPort == 0)
	{
		rfbPort = 5000;
	}

	if (rfbPassword != NULL)
	{
		char *s = rfbPassword;
		char *d = rfbPassword;

		while(*s)
		{
			c = *s++;

			if (c < '0')
			{
				break;
			}
			else if (c < ':')
			{
				result = (c - '0') * 16;
			}
			else if (c < 'A')
			{
				break;
			}
			else if (c < 'G')
			{
				result = ((c - 'A') + 10) * 16;
			}
			else if (c < 'a')
			{
				break;
			}
			else if (c < 'g')
			{
				result = ((c - 'a') + 10) * 16;
			}
			else
			{
				break;
			}

			c = *s++;

			if (c < '0')
			{
				break;
			}
			else if (c < ':')
			{
				result += (c - '0');
			}
			else if (c < 'A')
			{
				break;
			}
			else if (c < 'G')
			{
				result += ((c - 'A') + 10);
			}
			else if (c < 'a')
			{
				break;
			}
			else if (c < 'g')
			{
				result += ((c - 'a') + 10);
			}
			else
			{
				break;
			}
				
			*d++ = result;

			result = 0;
		}

		if (*s != 0 || result != 0) 
		{
			if (!quiet)
			{
				fprintf(stderr,"Invalid password '%s' specified (not hex?).\n", rfbPassword);
			}

			return 1;
		}

		*d = 0;

		fprintf(stderr,"rfbPassword = %s\n", rfbPassword);
	}

	if (rfbPassword == NULL && rfbAuthFilename == NULL)
	{
		struct ifaddrs *myaddrs, *ifa;
		void *in_addr;
		char buf[64];
		
		char *base = "/var/run/Xvnc/passwd";
		char *suffix = "";

		if(getifaddrs(&myaddrs) != 0)
		{
			if (!quiet)
			{
				fprintf(stderr, "Unable to determine IP address, getifaddrs() failed.\n");
			}

			return 1;
		}

		for (ifa = myaddrs; ifa != NULL; ifa = ifa->ifa_next)
		{
			if (ifa->ifa_addr == NULL)
			{
				continue;
			}

			if (!(ifa->ifa_flags & IFF_UP))
			{
				continue;
			}

			if (ifa->ifa_flags & IFF_LOOPBACK)
			{
				continue;
			}

			switch (ifa->ifa_addr->sa_family)
			{
				case AF_INET:
				{
					struct sockaddr_in *s4 = (struct sockaddr_in *)ifa->ifa_addr;

					if (s4->sin_addr.s_addr == 0x100007F) /* 127.0.0.1 */
						continue;

					in_addr = &s4->sin_addr;
					break;
				}

				case AF_INET6:
				{
					struct sockaddr_in6 *s6 = (struct sockaddr_in6 *)ifa->ifa_addr;
					in_addr = &s6->sin6_addr;
					break;
				}

				default:
					continue;
			}

			if (!inet_ntop(ifa->ifa_addr->sa_family, in_addr, buf, sizeof(buf)))
			{
				if (!quiet)
				{
					fprintf(stderr, "Unable to determine IP address, inet_ntop() failed.\n");
				}
			}
		}

		freeifaddrs(myaddrs);

		rfbAuthFilename = malloc( strlen(base) + strlen(buf) + 4 );
		sprintf(rfbAuthFilename, "%s-%s:0", base, buf);
	}

	if (rfbAuthFilename != NULL && rfbPassword == NULL)
	{
		FILE *fp = NULL;
		int n = 0;

		rfbPassword = malloc(9);

		if (rfbPassword == NULL)
		{
			if (!quiet)
			{
				fprintf(stderr, "Failed to allocate memory for rfbPassword with malloc().\n");
			}

			return 1;
		}

		fp = fopen(rfbAuthFilename, "r");

		if (fp == NULL)
		{
			if (!quiet)
			{
				fprintf(stderr, "Failed to open password file '%s'.\n", rfbAuthFilename);
			}

			return 1;
		}

		n = fread(rfbPassword, 1, 8, fp);

		if (n != 8)
		{
			if (!quiet)
			{
				fprintf(stderr, "Failed to read 8 characters from password file '%s'.\n", rfbAuthFilename);
			}

			return 1;
		}
	}

	if (!debug)
	{
		rfbEnableClientLogging = 0;
	}

	client = rfbGetClient(8,3,4);

	if (client == NULL)
	{
		if (!quiet)
		{
			fprintf(stderr, "Failed to get rfb client structuture with rfbGetClient()\n");
		}

		return 1;
	}

	client->serverHost = strdup(rfbHost);

	if (client->serverHost == NULL)
	{
		if (!quiet)
		{
			fprintf(stderr, "Failed to allocate memory for serverHost using strdup().\n");
		}

		return 1;
	}

	client->serverPort  = rfbPort;
	client->GetPassword = GetPassword;

	if (!rfbInitClient(client,0,NULL))
	{
		if (!quiet)
		{
			fprintf(stderr, "Failed to initialize rfb client with rfbInitClient()\n");
		}

		return 1;
	}

	b[0] = 0x20;
	b[1] = 42;
	b[2] = ((rfbWidth>>8) & 0x0ff);
	b[3] = (rfbWidth & 0x0ff);
	b[4] = ((rfbHeight>>8) & 0x0ff);
	b[5] = (rfbHeight & 0x0ff);

	result = WriteToRFBServer(client, b, 6);

	HandleRFBServerMessage(client);

	rfbClientCleanup(client);

	if (!result) 
	{
		if (!quiet)
		{
			fprintf(stderr, "hzvncresize: write error (%d: %s)", errno, strerror(errno));
		}

		return 2;
	}

	return 0;
}

/* D3DES (V5.09) - 
 *
 * A portable, public domain, version of the Data Encryption Standard.
 *
 * Written with Symantec's THINK (Lightspeed) C by Richard Outerbridge.
 * Thanks to: Dan Hoey for his excellent Initial and Inverse permutation
 * code;  Jim Gillogly & Phil Karn for the DES key schedule code; Dennis
 * Ferguson, Eric Young and Dana How for comparing notes; and Ray Lau,
 * for humouring me on. 
 *
 * THIS SOFTWARE PLACED IN THE PUBLIC DOMAIN BY THE AUTHOUR
 * 920825 19:42 EDST
 *
 * Copyright (c) 1988,1989,1990,1991,1992 by Richard Outerbridge.
 * (GEnie : OUTER; CIS : [71755,204]) Graven Imagery, 1992.
 */
 
static void scrunch(unsigned char *, unsigned long *);
static void unscrun(unsigned long *, unsigned char *);
static void desfunc(unsigned long *, unsigned long *);
static void cookey(unsigned long *);
 
static unsigned long KnL[32] = { 0L };
 
static unsigned short bytebit[8]	= 
{
	01, 02, 04, 010, 020, 040, 0100, 0200 
};

static unsigned long bigbyte[24] = 
{
	0x800000L,	0x400000L,	0x200000L, 	0x100000L,
	0x80000L,	0x40000L,	0x20000L,	0x10000L,
	0x8000L,	0x4000L,	0x2000L,	0x1000L,
	0x800L,		0x400L,		0x200L,		0x100L,
	0x80L,		0x40L,		0x20L,		0x10L,
	0x8L,		0x4L,		0x2L,		0x1L	
};
 
/* Use the key schedule specified in the Standard (ANSI X3.92-1981). */
 
static unsigned char pc1[56] = 
{
	56, 48, 40, 32, 24, 16,  8,	 0, 57, 49, 41, 33, 25, 17,
	 9,  1, 58, 50, 42, 34, 26,	18, 10,  2, 59, 51, 43, 35,
	62, 54, 46, 38, 30, 22, 14,	 6, 61, 53, 45, 37, 29, 21,
	13,  5, 60, 52, 44, 36, 28,	20, 12,  4, 27, 19, 11,  3 
};
 
static unsigned char totrot[16] = 
{
	1,2,4,6,8,10,12,14,15,17,19,21,23,25,27,28 
};
 
static unsigned char pc2[48] = 
{
	13, 16, 10, 23,  0,  4,	 2, 27, 14,  5, 20,  9,
	22, 18, 11,  3, 25,  7,	15,  6, 26, 19, 12,  1,
	40, 51, 30, 36, 46, 54,	29, 39, 50, 44, 32, 47,
	43, 48, 38, 55, 33, 52,	45, 41, 49, 35, 28, 31 
};
 
static void deskey(unsigned char *key, short edf)	/* Thanks to James Gillogly & Phil Karn! */
{
	int i, j, l, m, n;
	unsigned char pc1m[56], pcr[56];
	unsigned long kn[32];
 
	for ( j = 0; j < 56; j++ ) 
	{
		l = pc1[j];
		m = l & 07;
		pc1m[j] = (key[l >> 3] & bytebit[m]) ? 1 : 0;
	}

	for( i = 0; i < 16; i++ ) 
	{
		if( edf == DE1 ) 
		{
			m = (15 - i) << 1;
		}
		else 
		{
			m = i << 1;
		}

		n = m + 1;
		kn[m] = kn[n] = 0L;

		for( j = 0; j < 28; j++ ) 
		{
			l = j + totrot[i];

			if( l < 28 )
			{
				pcr[j] = pc1m[l];
			}
			else 
			{
				pcr[j] = pc1m[l - 28];
			}
		}

		for( j = 28; j < 56; j++ ) 
		{
			l = j + totrot[i];

			if( l < 56 )
			{
				pcr[j] = pc1m[l];
			}
			else 
			{
				pcr[j] = pc1m[l - 28];
			}
		}

		for( j = 0; j < 24; j++ ) 
		{
			if( pcr[pc2[j]] )
			{
				kn[m] |= bigbyte[j];
			}

			if( pcr[pc2[j+24]] )
			{
				kn[n] |= bigbyte[j];
			}
		}
	}

	cookey(kn);

	return;
}
 
static void usekey(unsigned long *from)
{
	unsigned long *to, *endp;
 
	to = KnL, endp = &KnL[32];

	while( to < endp )
	{
		*to++ = *from++;
	}

	return;
}

static void cookey(unsigned long *raw1)
{
	unsigned long *cook, *raw0;
	unsigned long dough[32];
	int i;
 
	cook = dough;

	for( i = 0; i < 16; i++, raw1++ ) 
	{
		raw0 = raw1++;
		*cook	 = (*raw0 & 0x00fc0000L) << 6;
		*cook	|= (*raw0 & 0x00000fc0L) << 10;
		*cook	|= (*raw1 & 0x00fc0000L) >> 10;
		*cook++	|= (*raw1 & 0x00000fc0L) >> 6;
		*cook	 = (*raw0 & 0x0003f000L) << 12;
		*cook	|= (*raw0 & 0x0000003fL) << 16;
		*cook	|= (*raw1 & 0x0003f000L) >> 4;
		*cook++	|= (*raw1 & 0x0000003fL);
	}

	usekey(dough);

	return;
}
 
void cpkey(unsigned long *into)
{
	unsigned long *from, *endp;
 
	from = KnL, endp = &KnL[32];

	while( from < endp )
	{
		*into++ = *from++;
	}

	return;
}

static void des(unsigned char *inblock, unsigned char *outblock)
{
	unsigned long work[2];
 
	scrunch(inblock, work);
	desfunc(work, KnL);
	unscrun(work, outblock);

	return;
}

static void scrunch(unsigned char *outof, unsigned long *into)
{
	*into 	 = (*outof++ & 0xffL) << 24;
	*into 	|= (*outof++ & 0xffL) << 16;
	*into 	|= (*outof++ & 0xffL) << 8;
	*into++ |= (*outof++ & 0xffL);
	*into 	 = (*outof++ & 0xffL) << 24;
	*into 	|= (*outof++ & 0xffL) << 16;
	*into 	|= (*outof++ & 0xffL) << 8;
	*into	|= (*outof   & 0xffL);

	return;
}
 
static void unscrun(unsigned long *outof, unsigned char *into)
{
	*into++ = (*outof >> 24) & 0xffL;
	*into++ = (*outof >> 16) & 0xffL;
	*into++ = (*outof >>  8) & 0xffL;
	*into++ =  *outof++	 & 0xffL;
	*into++ = (*outof >> 24) & 0xffL;
	*into++ = (*outof >> 16) & 0xffL;
	*into++ = (*outof >>  8) & 0xffL;
	*into   =  *outof	 & 0xffL;

	return;
}

static unsigned long SP1[64] = 
{
	0x01010400L, 0x00000000L, 0x00010000L, 0x01010404L,
	0x01010004L, 0x00010404L, 0x00000004L, 0x00010000L,
	0x00000400L, 0x01010400L, 0x01010404L, 0x00000400L,
	0x01000404L, 0x01010004L, 0x01000000L, 0x00000004L,
	0x00000404L, 0x01000400L, 0x01000400L, 0x00010400L,
	0x00010400L, 0x01010000L, 0x01010000L, 0x01000404L,
	0x00010004L, 0x01000004L, 0x01000004L, 0x00010004L,
	0x00000000L, 0x00000404L, 0x00010404L, 0x01000000L,
	0x00010000L, 0x01010404L, 0x00000004L, 0x01010000L,
	0x01010400L, 0x01000000L, 0x01000000L, 0x00000400L,
	0x01010004L, 0x00010000L, 0x00010400L, 0x01000004L,
	0x00000400L, 0x00000004L, 0x01000404L, 0x00010404L,
	0x01010404L, 0x00010004L, 0x01010000L, 0x01000404L,
	0x01000004L, 0x00000404L, 0x00010404L, 0x01010400L,
	0x00000404L, 0x01000400L, 0x01000400L, 0x00000000L,
	0x00010004L, 0x00010400L, 0x00000000L, 0x01010004L 
};
 
static unsigned long SP2[64] = 
{
	0x80108020L, 0x80008000L, 0x00008000L, 0x00108020L,
	0x00100000L, 0x00000020L, 0x80100020L, 0x80008020L,
	0x80000020L, 0x80108020L, 0x80108000L, 0x80000000L,
	0x80008000L, 0x00100000L, 0x00000020L, 0x80100020L,
	0x00108000L, 0x00100020L, 0x80008020L, 0x00000000L,
	0x80000000L, 0x00008000L, 0x00108020L, 0x80100000L,
	0x00100020L, 0x80000020L, 0x00000000L, 0x00108000L,
	0x00008020L, 0x80108000L, 0x80100000L, 0x00008020L,
	0x00000000L, 0x00108020L, 0x80100020L, 0x00100000L,
	0x80008020L, 0x80100000L, 0x80108000L, 0x00008000L,
	0x80100000L, 0x80008000L, 0x00000020L, 0x80108020L,
	0x00108020L, 0x00000020L, 0x00008000L, 0x80000000L,
	0x00008020L, 0x80108000L, 0x00100000L, 0x80000020L,
	0x00100020L, 0x80008020L, 0x80000020L, 0x00100020L,
	0x00108000L, 0x00000000L, 0x80008000L, 0x00008020L,
	0x80000000L, 0x80100020L, 0x80108020L, 0x00108000L 
};
 
static unsigned long SP3[64] = 
{
	0x00000208L, 0x08020200L, 0x00000000L, 0x08020008L,
	0x08000200L, 0x00000000L, 0x00020208L, 0x08000200L,
	0x00020008L, 0x08000008L, 0x08000008L, 0x00020000L,
	0x08020208L, 0x00020008L, 0x08020000L, 0x00000208L,
	0x08000000L, 0x00000008L, 0x08020200L, 0x00000200L,
	0x00020200L, 0x08020000L, 0x08020008L, 0x00020208L,
	0x08000208L, 0x00020200L, 0x00020000L, 0x08000208L,
	0x00000008L, 0x08020208L, 0x00000200L, 0x08000000L,
	0x08020200L, 0x08000000L, 0x00020008L, 0x00000208L,
	0x00020000L, 0x08020200L, 0x08000200L, 0x00000000L,
	0x00000200L, 0x00020008L, 0x08020208L, 0x08000200L,
	0x08000008L, 0x00000200L, 0x00000000L, 0x08020008L,
	0x08000208L, 0x00020000L, 0x08000000L, 0x08020208L,
	0x00000008L, 0x00020208L, 0x00020200L, 0x08000008L,
	0x08020000L, 0x08000208L, 0x00000208L, 0x08020000L,
	0x00020208L, 0x00000008L, 0x08020008L, 0x00020200L 
};
 
static unsigned long SP4[64] = 
{
	0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L,
	0x00802080L, 0x00800081L, 0x00800001L, 0x00002001L,
	0x00000000L, 0x00802000L, 0x00802000L, 0x00802081L,
	0x00000081L, 0x00000000L, 0x00800080L, 0x00800001L,
	0x00000001L, 0x00002000L, 0x00800000L, 0x00802001L,
	0x00000080L, 0x00800000L, 0x00002001L, 0x00002080L,
	0x00800081L, 0x00000001L, 0x00002080L, 0x00800080L,
	0x00002000L, 0x00802080L, 0x00802081L, 0x00000081L,
	0x00800080L, 0x00800001L, 0x00802000L, 0x00802081L,
	0x00000081L, 0x00000000L, 0x00000000L, 0x00802000L,
	0x00002080L, 0x00800080L, 0x00800081L, 0x00000001L,
	0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L,
	0x00802081L, 0x00000081L, 0x00000001L, 0x00002000L,
	0x00800001L, 0x00002001L, 0x00802080L, 0x00800081L,
	0x00002001L, 0x00002080L, 0x00800000L, 0x00802001L,
	0x00000080L, 0x00800000L, 0x00002000L, 0x00802080L 
};
 
static unsigned long SP5[64] = 
{
	0x00000100L, 0x02080100L, 0x02080000L, 0x42000100L,
	0x00080000L, 0x00000100L, 0x40000000L, 0x02080000L,
	0x40080100L, 0x00080000L, 0x02000100L, 0x40080100L,
	0x42000100L, 0x42080000L, 0x00080100L, 0x40000000L,
	0x02000000L, 0x40080000L, 0x40080000L, 0x00000000L,
	0x40000100L, 0x42080100L, 0x42080100L, 0x02000100L,
	0x42080000L, 0x40000100L, 0x00000000L, 0x42000000L,
	0x02080100L, 0x02000000L, 0x42000000L, 0x00080100L,
	0x00080000L, 0x42000100L, 0x00000100L, 0x02000000L,
	0x40000000L, 0x02080000L, 0x42000100L, 0x40080100L,
	0x02000100L, 0x40000000L, 0x42080000L, 0x02080100L,
	0x40080100L, 0x00000100L, 0x02000000L, 0x42080000L,
	0x42080100L, 0x00080100L, 0x42000000L, 0x42080100L,
	0x02080000L, 0x00000000L, 0x40080000L, 0x42000000L,
	0x00080100L, 0x02000100L, 0x40000100L, 0x00080000L,
	0x00000000L, 0x40080000L, 0x02080100L, 0x40000100L 
};
 
static unsigned long SP6[64] = 
{
	0x20000010L, 0x20400000L, 0x00004000L, 0x20404010L,
	0x20400000L, 0x00000010L, 0x20404010L, 0x00400000L,
	0x20004000L, 0x00404010L, 0x00400000L, 0x20000010L,
	0x00400010L, 0x20004000L, 0x20000000L, 0x00004010L,
	0x00000000L, 0x00400010L, 0x20004010L, 0x00004000L,
	0x00404000L, 0x20004010L, 0x00000010L, 0x20400010L,
	0x20400010L, 0x00000000L, 0x00404010L, 0x20404000L,
	0x00004010L, 0x00404000L, 0x20404000L, 0x20000000L,
	0x20004000L, 0x00000010L, 0x20400010L, 0x00404000L,
	0x20404010L, 0x00400000L, 0x00004010L, 0x20000010L,
	0x00400000L, 0x20004000L, 0x20000000L, 0x00004010L,
	0x20000010L, 0x20404010L, 0x00404000L, 0x20400000L,
	0x00404010L, 0x20404000L, 0x00000000L, 0x20400010L,
	0x00000010L, 0x00004000L, 0x20400000L, 0x00404010L,
	0x00004000L, 0x00400010L, 0x20004010L, 0x00000000L,
	0x20404000L, 0x20000000L, 0x00400010L, 0x20004010L 
};
 
static unsigned long SP7[64] = 
{
	0x00200000L, 0x04200002L, 0x04000802L, 0x00000000L,
	0x00000800L, 0x04000802L, 0x00200802L, 0x04200800L,
	0x04200802L, 0x00200000L, 0x00000000L, 0x04000002L,
	0x00000002L, 0x04000000L, 0x04200002L, 0x00000802L,
	0x04000800L, 0x00200802L, 0x00200002L, 0x04000800L,
	0x04000002L, 0x04200000L, 0x04200800L, 0x00200002L,
	0x04200000L, 0x00000800L, 0x00000802L, 0x04200802L,
	0x00200800L, 0x00000002L, 0x04000000L, 0x00200800L,
	0x04000000L, 0x00200800L, 0x00200000L, 0x04000802L,
	0x04000802L, 0x04200002L, 0x04200002L, 0x00000002L,
	0x00200002L, 0x04000000L, 0x04000800L, 0x00200000L,
	0x04200800L, 0x00000802L, 0x00200802L, 0x04200800L,
	0x00000802L, 0x04000002L, 0x04200802L, 0x04200000L,
	0x00200800L, 0x00000000L, 0x00000002L, 0x04200802L,
	0x00000000L, 0x00200802L, 0x04200000L, 0x00000800L,
	0x04000002L, 0x04000800L, 0x00000800L, 0x00200002L 
};
 
static unsigned long SP8[64] = 
{
	0x10001040L, 0x00001000L, 0x00040000L, 0x10041040L,
	0x10000000L, 0x10001040L, 0x00000040L, 0x10000000L,
	0x00040040L, 0x10040000L, 0x10041040L, 0x00041000L,
	0x10041000L, 0x00041040L, 0x00001000L, 0x00000040L,
	0x10040000L, 0x10000040L, 0x10001000L, 0x00001040L,
	0x00041000L, 0x00040040L, 0x10040040L, 0x10041000L,
	0x00001040L, 0x00000000L, 0x00000000L, 0x10040040L,
	0x10000040L, 0x10001000L, 0x00041040L, 0x00040000L,
	0x00041040L, 0x00040000L, 0x10041000L, 0x00001000L,
	0x00000040L, 0x10040040L, 0x00001000L, 0x00041040L,
	0x10001000L, 0x00000040L, 0x10000040L, 0x10040000L,
	0x10040040L, 0x10000000L, 0x00040000L, 0x10001040L,
	0x00000000L, 0x10041040L, 0x00040040L, 0x10000040L,
	0x10040000L, 0x10001000L, 0x10001040L, 0x00000000L,
	0x10041040L, 0x00041000L, 0x00041000L, 0x00001040L,
	0x00001040L, 0x00040040L, 0x10000000L, 0x10041000L 
};
 
static void desfunc(unsigned long *block, unsigned long *keys)
{
	unsigned long fval, work, right, leftt;
	int round;
	
	leftt = block[0];
	right = block[1];
	work = ((leftt >> 4) ^ right) & 0x0f0f0f0fL;
	right ^= work;
	leftt ^= (work << 4);
	work = ((leftt >> 16) ^ right) & 0x0000ffffL;
	right ^= work;
	leftt ^= (work << 16);
	work = ((right >> 2) ^ leftt) & 0x33333333L;
	leftt ^= work;
	right ^= (work << 2);
	work = ((right >> 8) ^ leftt) & 0x00ff00ffL;
	leftt ^= work;
	right ^= (work << 8);
	right = ((right << 1) | ((right >> 31) & 1L)) & 0xffffffffL;
	work = (leftt ^ right) & 0xaaaaaaaaL;
	leftt ^= work;
	right ^= work;
	leftt = ((leftt << 1) | ((leftt >> 31) & 1L)) & 0xffffffffL;
	
	for( round = 0; round < 8; round++ ) 
	{
		work  = (right << 28) | (right >> 4);
		work ^= *keys++;
		fval  = SP7[ work		 & 0x3fL];
		fval |= SP5[(work >>  8) & 0x3fL];
		fval |= SP3[(work >> 16) & 0x3fL];
		fval |= SP1[(work >> 24) & 0x3fL];
		work  = right ^ *keys++;
		fval |= SP8[ work		 & 0x3fL];
		fval |= SP6[(work >>  8) & 0x3fL];
		fval |= SP4[(work >> 16) & 0x3fL];
		fval |= SP2[(work >> 24) & 0x3fL];
		leftt ^= fval;
		work  = (leftt << 28) | (leftt >> 4);
		work ^= *keys++;
		fval  = SP7[ work		 & 0x3fL];
		fval |= SP5[(work >>  8) & 0x3fL];
		fval |= SP3[(work >> 16) & 0x3fL];
		fval |= SP1[(work >> 24) & 0x3fL];
		work  = leftt ^ *keys++;
		fval |= SP8[ work		 & 0x3fL];
		fval |= SP6[(work >>  8) & 0x3fL];
		fval |= SP4[(work >> 16) & 0x3fL];
		fval |= SP2[(work >> 24) & 0x3fL];
		right ^= fval;
	}
		
	right = (right << 31) | (right >> 1);
	work = (leftt ^ right) & 0xaaaaaaaaL;
	leftt ^= work;
	right ^= work;
	leftt = (leftt << 31) | (leftt >> 1);
	work = ((leftt >> 8) ^ right) & 0x00ff00ffL;
	right ^= work;
	leftt ^= (work << 8);
	work = ((leftt >> 2) ^ right) & 0x33333333L;
	right ^= work;
	leftt ^= (work << 2);
	work = ((right >> 16) ^ leftt) & 0x0000ffffL;
	leftt ^= work;
	right ^= (work << 16);
	work = ((right >> 4) ^ leftt) & 0x0f0f0f0fL;
	leftt ^= work;
	right ^= (work << 4);
	*block++ = right;
	*block = leftt;

	return;
}
