# @package      hubzero-mw2-common
# @file         mw_db.py
# @copyright    Copyright (c) 2016-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

"""
Communicate with MySQL database while setting, holding and releasing the 
appropriate locks, and checking for errors
"""

import MySQLdb
import time
from errors import MaxwellError
from log import log

DEBUG = False

class MW_DB:
  """Wrapper for database calls.  The main motivation for making this into a class
  (besides tidying up and enabling unit testing) is to get rid of a global, "lock_held".
  """

  def __init__(self, mysql_host, mysql_user, mysql_password, mysql_db, mysql_connect_attempts = 5):
    """Create database connection."""
    self.lock_held = None
    self.host = mysql_host
    self.user = mysql_user
    self.passwd = mysql_password
    self.dbname = mysql_db
    self.retries = mysql_connect_attempts
    self.c = None
    self.db = None
    self.__new_cursor()

  def cursor(self):
    self.c = self.db.cursor()
  
  def force_new_cursor(self):
    self.close()
    self.new_cursor()

  def __new_cursor(self):
    if self.db is None or self.c is None:
      self.db = MySQLdb.connect(host=self.host, user=self.user, passwd=self.passwd, db=self.dbname)
      if DEBUG:
        log("MySQLdb.connect succeeded")
      self.c = self.db.cursor()

  def new_cursor(self):
    """Try to connect to database as long as we can log the attempts"""
    while True:
      try:
        self.__new_cursor()
        break
      except MySQLdb.MySQLError, e:
        log("Exception trying to connect: %s" % e)
      time.sleep(1)
    
  def close(self):
    self.db.close()
    self.c = None
    self.db = None

  def getall(self, cmd, params):
    self.c.execute(cmd, params)
    return self.c.fetchall()

  def getrow(self, cmd, params):
    self.c.execute(cmd, params)
    return self.c.fetchone()

  def getsingle(self, cmd, params):
    self.c.execute(cmd, params)
    row = self.c.fetchone()
    if row is None:
      return None
    return row[0]

  #
  # Transaction support: requires an engine like InnoDB
  #
  """Example usage elsewhere:
    while True:
      try:
        db.start_transaction()
        # SQL queries      
        db.commit()
        break
      except MySQLdb.MySQLError, e:
        log("exception: %s" % e)
        db.rollback()
  """

  def start_transaction(self):
    # the default isolation level allows state machine violations (with basic "SELECT")
    # Use "SELECT ... FOR UPDATE" to get the necessary locks.
    # 'SET TRANSACTION ISOLATION LEVEL SERIALIZABLE' would work but:
    # "Percona-XtraDB-Cluster doesn't recommend using SERIALIZABLE isolation with pxc_strict_mode = ENFORCING"
    # "SELECT ... LOCK IN SHARE MODE" also works, and it's the same behavior as SERIALIZABLE
    # "FOR UPDATE" is nicer because it waits and resumes automatically instead 
    # of throwing an exception like SERIALIZABLE
    self.__new_cursor()
    self.c.execute('START TRANSACTION', None)

  def rollback(self):
    """Handle exceptions here while doing rollback because rollback is already done while handling an exception"""
    try:
      self.c.execute('ROLLBACK', None)
      self.close()
      time.sleep(1)
    except MySQLdb.MySQLError, e:
      log("Rollback failed: %s" % e)

  def commit(self):
    # caller needs to handle exceptions and retry transaction
    self.c.execute('COMMIT', None)

  def simple_tr(self, cmd, params):
    """Wrap the sql argument inside a transaction being retried forever"""
    while True:
      try:
        self.start_transaction()
        self.c.execute(cmd, params)
        self.commit()
        break
      except MySQLdb.MySQLError, e:
        log("exception in simple_tr: %s" % e)
        self.rollback()

  def getsingle_tr(self, cmd, params):
    row = None
    while True:
      try:
        self.start_transaction()
        self.c.execute(cmd, params)
        row = self.c.fetchone()
        self.commit()
        break
      except MySQLdb.MySQLError, e:
        log("exception in simple_tr: %s" % e)
        self.rollback()
    if row is None:
      return None
    return row[0]

  def callback_tr(self, callback, args):
    """Experimental.  callback argument is a function that will run multiple SQL queries.  
    The callback function is called with the provided list [] of arguments"""
    while True:
      self.new_cursor()
      try:
        self.start_transaction()
        callback(*args)
        self.commit()
        break
      except MySQLdb.MySQLError, e:
        log("exception in callback_tr: %s" % e)
        self.rollback()

  #
  # Advisory lock support: alternative to using transactions, e.g., when using MyISAM engine
  # The following is specific to MySQL... Only one lock at a time, identified by string
  #

  def lock(self, name):
    """Get a MySQL advisory lock. Does not return a value."""

    if name == self.lock_held:
      return

    if self.lock_held is not None:
      # can only hold one advisory lock at a time!  Works globally, but if this process sets a new
      # lock, the previous is lost.
      raise MaxwellError("Already got lock '%s' when trying to get '%s'" % (self.lock_held, name))

    iteration = 0
    # get_lock returns 1 if lock was obtained, 0 or NULL otherwise
    lock_r = self.getsingle("""SELECT get_lock('%s',5)""" % name, [])
    while lock_r != 1:
      self.close()
      log("Contention while waiting on lock %s" % name)
      iteration += 1
      if iteration > 100:
        raise MaxwellError("can't get lock, giving up")
      time.sleep(1)
      self.new_cursor()
      lock_r = self.getsingle("""SELECT get_lock('%s',5)""" % name, [])
        
    self.lock_held = name

  def unlock(self, name):
    """Release a MySQL advisory lock. Does not return a value."""

    if self.lock_held != name:
      raise MaxwellError("lock held is '%s', not '%s'", (self.lock_held, name))
    self.lock_held = None

    try:
      lock_r = self.getsingle("SELECT release_lock('%s')" % name, [])
      # release_lock returns 1 if lock was released, 0 or NULL otherwise
      if lock_r != 1:
        log("Error c.unlocking %s" % name)
    except MySQLdb.MySQLError, ex:
      # log and continue, hoping for the best, as we're done with this lock.
      # if anything went wrong, the damage has already been done.
      log("Error c.unlocking '%s': '%s'." % (name, ex))


