# @package      hubzero-mw2-common
# @file         user_account.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import pwd
import time
from log import log
from errors import MaxwellError
from constants import FILE_CONFIG_FILE, LIB_PATH, VERBOSE, CONTAINER_K

DEBUG = False

class User_account():
  """ Get, manipulate and format account information for a given user.
    Allow root to act as that user by relinquishing privileges.
  """
  def __init__(self, user, container_conf = {}):
    self.user = user
    if user == "anonymous":
      self.uid = 1234
      self.homedir = container_conf["HOME_DIR"]+"/"+'anonymous'
    else:
      count = 0
      while count < 10:
        try:
          info = pwd.getpwnam(self.user)
          self.uid = info[2]
          self.gid = info[3]
          self.homedir = info[5]
          self.shell = info[6]
          count = 100
        except StandardError:
          log("No account information for '%s'." % self.user)
          count += 1
          time.sleep(count)
      if count == 10:
        raise MaxwellError("Giving up: account info for '%s'." % self.user)

    if DEBUG:
      log("got account information for '%s': uid: %d, gid %d" % (self.user, self.uid, self.gid))

  def create_home(self):
    if not os.path.isdir(self.homedir):
      log("creating home directory %s" % self.homedir)
      os.mkdir(self.homedir, 0700)
      os.chown(self.homedir, self.uid, self.gid)

  def home_quota(self, container_conf = {}):
    # setting up home directory and quotas should only be done on a file server!
    # REVISIT: create a locking mechanism to prevent race conditions
    # to avoid trying to use the directory before it's been completely setup, in
    # the case of multiple sessions being started by the user simultaneously, for
    # the first time.
    # REVISIT: as it uses FILE_CONFIG_FILE perhaps it should be in the maxwell_fs script
    if not os.path.isfile(self.homedir + "/.environ"):
      if VERBOSE:
        log("missing .environ in home directory %s" % self.homedir)
      import subprocess
      # need to create home directory using root privileges
      self.create_home()
      if not os.path.isfile(os.path.dirname(FILE_CONFIG_FILE) + "/template.environ"):
        log("missing template.environ in service directory %s" % os.path.dirname(FILE_CONFIG_FILE))

      args = ['/bin/su', '-', self.user, '-s', '/bin/dash', '-c',
        '/bin/cp %s %s' % (os.path.dirname(FILE_CONFIG_FILE) + "/template.environ", self.homedir + "/.environ")]
      if DEBUG:
        log("Executing %s" % " ".join(args))
      retcode = subprocess.call(args)
      if retcode != 0:
        log("Unable to copy template.environ")
      # set user quota -- deprecated
      # old call holds outdated information.  Web server will set quota with correct information
      # args = [LIB_PATH + "/set_quotas", self.user]
      # subprocess.check_call(args)
      #
      # Are we on a file server that's a SAMBA server too, to support Windows execution hosts?
      # Samba server: create a user password so that Windows hosts can mount the home directory
      # get configuration for containers
      CONTAINER_MERGED = CONTAINER_K
      CONTAINER_MERGED.update(container_conf)
      if "WIN_SMB_SECRET" in CONTAINER_MERGED and os.path.isfile('/usr/bin/smbpasswd'):
        import hashlib
        # Generate Samba md160 password hash and samba account (used for access of home dirs on windows execution hosts
        # Generate user Samba password
        # Requirement: user MUST exist in ldap before adding them to the Samba database
        h = hashlib.new('ripemd160')  # Chosen because it generates 40 character digests
        h.update(CONTAINER_MERGED["WIN_SMB_SECRET"] + self.user)
        sambaPassword = h.hexdigest()
        # Create user account with special password in Samba
        args = ['/usr/bin/smbpasswd', '-a', self.user, '-D', '1', '-s']
        process = subprocess.Popen(
          args,
          stdin = subprocess.PIPE,
          stdout = subprocess.PIPE,
          stderr = subprocess.PIPE
        )
        (stdout1, stderr1) = process.communicate("%s\n%s\n" % (sambaPassword, sambaPassword))
        if process.returncode != 0:
          log("SAMBA error for user '%s'" % self.user)
          log("STDERR-ADD-USER-TO-SAMBA: '%s'" % stderr1)
        if VERBOSE and len(stdout1) > 0:
          log("STDOUT-ADD-USER-TO-SAMBA: '%s'" % stdout1)

  def session_dir(self, session_id):
    return self.homedir + "/data/sessions/%s" % session_id

  def results_dir(self, session_id):
    return self.homedir + "/data/results/%s" % session_id

  def params_path(self, session_id):
    return self.session_dir(session_id) + "/parameters.hz"

  def env(self, session_id, timeout, params):
    e = [
      "HOME=\"%s\"" % self.homedir,
      "LANG=en_US.UTF-8",
      "LANGUAGE=en_US.UTF-8",
      "LC_ALL=en_US.UTF-8",
      "LOGNAME=\"%s\"" % self.user,
      "PATH=\"%s\"" % "/bin:/usr/bin:/usr/bin/X11:/sbin:/usr/sbin",
      "SESSION=\"%s\"" % session_id,
      "SESSIONDIR=\"%s\"" % self.session_dir(session_id),
      "RESULTSDIR=\"%s\"" % self.results_dir(session_id),
      "TIMEOUT=\"%s\"" % str(timeout),
      "USER=\"%s\"" % self.user
    ]
    if params:
      return e + ["TOOL_PARAMETERS=\"%s\"" % self.params_path(session_id)]
    else:
      return e

  def passwd_entry(self):
    return "%s:!:%d:%d::%s:%s" % (self.user, self.uid, self.gid, self.homedir, self.shell)

  def shadow_entry(self):
    # format:   login:password:last:min:maxdays:warning:max_inactive:exp:flag
    # 13281: is in 2006, 99999 gives hundreds of years before it needs to be changed
    return "%s:!:13281:0:99999:7:::" % (self.user)

  def relinquish(self):
    """Relinquish elevated privileges;  adopt user identity for file operations"""
    try:
      os.setregid(self.gid, self.gid)
      os.setreuid(self.uid, self.uid)
      if DEBUG:
        log("set uid to %d and gid to %d" % (self.uid, self.gid))
    except OSError:
      raise MaxwellError("unable to change uid and gid")

    still_root = True
    try:
      #os.setuid(0)
      os.setreuid(0, 0)
    except OSError:
      still_root = False
    if still_root:
      raise MaxwellError("Was able to revert to root!")

  def groups(self):
    # this should, but does not, return primary groups that are listed without members
    # for example 'public' is not returned.
    #  >>> grp.getgrnam('public')
    #  ('public', '*', 3000, [])
    #import grp
    #return [g.gr_name for g in grp.getgrall() if self.user in g.gr_mem]
    # this does it:
    f = os.popen("id %s |sed -e 's/^.*groups=//' -e 's/[0-9]*[(]\([^)]*\)[)]/\\1/g'" % self.user)
    groups = f.read()
    groups = groups.strip()
    f.close()
    return groups.split(',')

  def initgroups(self):
    """As of Python 2.6.5 os.initgroups() didn't exist yet.
    Therefore, we have to do it ourselves.
    """
    # This should but doesn't include primary groups
    #import grp
    #grplist = [g.gr_gid for g in grp.getgrall() if self.user in g.gr_mem]
    pipe = os.popen("/usr/bin/id %s" % self.user, "r")
    line = pipe.readline()
    # line looks like uid=123(username) gid=123(group) groups=123(grp1),...
    arr = line.split(' ')
    if len(arr) != 3:
      raise KeyError("User not found")

    # Element 2 looks like groups=123(grp1),456(grp2),...
    arr = arr[2].split('=')

    # Now split it up by commas
    arr = arr[1].split(',')

    # Form the list of gid numbers.
    grplist = []
    for g in arr:
      gid = int(g.split('(')[0])
      grplist += [ gid ]

    os.setregid(self.gid, self.gid)
    return os.setgroups(grplist)
