# @package      hubzero-mw2-common
# @file         display.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
Used by the client to manipulate displays and the MySQL table of the same name.
Displays are an abstraction of the virtualization and isolation mechanism used to create tool sessions.
Displays are used by the client to keep track of virtualization environments (containers) 
and to request operations on execution hosts.
"""
import os
from MySQLdb import MySQLError
from constants import BASE_LOG_PATH, VERBOSE
from errors import DisplayError, PrivateError
from vnc_pass import VNC_pass
from support import genpasswd
from log import log
from host import Host

DEBUG = True
class Display:
  """Is used on web servers only, not on execution hosts.
  displays and containers refer to the same thing, but displays are from the perspective of the 
  web server, and containers from the perspective of the execution host.
  A display
  Display states:
  'starting': the container hasn't finished starting yet
  'used':  the container on the execution host is running a tool
  'ready':  there should be a matching container on the execution host
  'broken':  don't use that container, something went wrong
  'absent' : there is no container running
  'stopping': the container hasn't finished stopping yet -- note that session may have been deleted already (zombie check)


  absent => starting (atomic)
  ready => used, purged
  purged => stopping
  starting => broken, used, ready
  used => stopping (atomic), broken, ready (if tool failed to start)
  stopping => broken, absent, or ready if Windows
  invariant:  session must exist for displays in state starting, used;  otherwise delete_zombie will harvest
  invariant (needs to be checked?): container must exist on execution host for displays in states ready, starting, used

  Removed functionality: Windows containers are never destroyed, and instead should go from stopping => ready
  """
  def __init__(self, host, dispnum, vncpass, status = ''):
    """host + dispnum is unique"""
    self.host = host # a host object
    self.dispnum = int(dispnum) # an integer
    self.vncpass = vncpass # a string
    self.sessnum = 0
    self.status = status

  def __set_status(self, db, status, sessnum=None):
    """Update host usage count after changing the status of a display, all wrapped in a transaction.
    This is because decisions are made in other threads based on the usage count"""
    self.status = status
    if VERBOSE:
      log("setting status of display %d on host %s to %s" % (self.dispnum, self.host.hostname, status))
    while True:
      try:
        db.start_transaction()
        if sessnum == None:
          if DEBUG:
            log("""UPDATE display SET status=%s WHERE hostname=%s AND dispnum=%s""" % (status, self.host.hostname, str(self.dispnum)))
          db.c.execute("""
UPDATE display SET status=%s
WHERE hostname=%s AND dispnum=%s""",
          [status, self.host.hostname, self.dispnum])
        else:
          if DEBUG:
            log("""UPDATE display SET sessnum=%s, status=%s WHERE hostname=%s AND dispnum=%s""" % (sessnum, status, self.host.hostname, str(self.dispnum)))
          db.c.execute("""
UPDATE display SET sessnum=%s, status=%s
WHERE hostname=%s AND dispnum=%s""",
          [sessnum, status, self.host.hostname, self.dispnum])
        self.host.count_uses(db)
        db.commit()
        break
      except MySQLError, e:
        log("Exception setting display state: %s" % e)
        db.rollback()

  def absent(self, db):
    if self.status == "broken":
      return
    if self.status != 'stopping':
      log("display %d status was '%s' instead of expected 'stopping'" % (self.dispnum, self.status))
    else: 
      self.__set_status(db, 'absent', 0)

  def ready(self, db):
    """
    starting => ready
    used => ready if session start aborted
    """
    if self.status == "broken":
      return
    if self.status != "starting" and self.status != "used":
      raise DisplayError("display %d status was '%s' instead of expected 'starting' or 'used'" % (self.dispnum, self.status))
    self.__set_status(db, 'ready', 0)

  def used(self, db, sessnum):
    """
    starting, ready => used
    """
    if self.status == "broken":
      return
    if self.status != "ready" and self.status != "starting":
      raise DisplayError("display %d status was '%s' instead of expected 'ready or starting'" % (self.dispnum, self.status))
    self.__set_status(db, 'used', sessnum)

  def broken(self, db):
    """
    from any state
    """
    self.__set_status(db, 'broken')

  def stopping(self, db):
    """used, purged => stopping"""
    if self.status == "broken":
      return
    if self.status != "used" and self.status != "purged":
      log("display %d status was '%s' instead of expected 'used or purged'" % (self.dispnum, self.status))
    self.__set_status(db, 'stopping')

  def stop(self, db):
    """stop the display."""
    self.stopping(db)
    self.tell_stop(db)
    self.absent(db)

  def get_sessnum(self, db):
    return db.getsingle_tr("""
      SELECT sessnum FROM display
      WHERE hostname=%s AND dispnum=%s""",
      [self.host.hostname, self.dispnum])

  def get_status(self, db):
    return db.getsingle_tr("""
      SELECT status FROM display
      WHERE hostname=%s AND dispnum=%s""",
      [self.host.hostname, self.dispnum])

  def log_file(self):
    """return path to log file for this display"""
    return "%s/vnc-%s:%d" % (BASE_LOG_PATH, self.host.hostname, self.dispnum)

  def start(self, db, app, sessnum):
    """Setup this display for a particular app, using geometry, etc. information.  Set a new password."""
    
    # Generate a password for the new VNC display.
    if DEBUG:
      log("starting display for sessnum %d" % sessnum)

    vgen = VNC_pass()
    raw_vncpass = vgen.encrypt(genpasswd(8))
    self.vncpass = vgen.tohex(raw_vncpass)

    db.simple_tr("""
UPDATE display
SET geometry=%s, depth=%s, sessnum=%s, vncpass=%s
WHERE hostname=%s AND dispnum=%s""",
          [app.geometry, app.depth, sessnum, self.vncpass, self.host.hostname, self.dispnum])
    # disconnect from database while communicating with execution host
    db.close()
    if VERBOSE:
      log("startvnc %s %s %s" % (str(self.dispnum), str(app.geometry), str(app.depth)))
    vnc_res = self.host.service("startvnc", [str(self.dispnum), str(app.geometry), str(app.depth)], raw_vncpass)
    db.new_cursor()
    if vnc_res == 0x62:
      log("Display %s:%d is already running, trying to stop it." % (self.host.hostname, self.dispnum))
      # If we succeed in clearing the display, just return normal failure
      # and don't mark the display as broken.
      self.stop(db)
      raise DisplayError("Tried to start already running display %s:%d." % (self.host.hostname, self.dispnum))
    elif vnc_res == 0xff:
      self.broken(db)
      raise DisplayError("Unknown host %s in creating display." % self.host.hostname)
    elif vnc_res == 0x7f:
      # bash: /opt/mw/bin/maxwell_service: No such file or directory
      # either host is not configured correctly, or this script isn't.
      self.broken(db)
      raise DisplayError("Incorrect maxwell-service path for %s" % (self.host.hostname))
    elif vnc_res == 2:
      self.broken(db)
      raise DisplayError("Maxwell Service error when creating display %s:%d.  Consult the service log file on the execution host." % (self.host.hostname, self.dispnum))
    elif vnc_res != 0:
      self.broken(db)
      raise DisplayError("Unknown error: %s when creating display %s:%d." % (str(vnc_res), self.host.hostname, self.dispnum))
    if sessnum == 0:
      # this is a standby container
      self.ready(db)
    else:
      self.used(db, sessnum)
    if DEBUG:
      log("started display %s:%d for sessnum %d" % (self.host.hostname, self.dispnum, sessnum))

  def tell_stop(self, db):
    """
    Tell execution host to stop container.
    """
    # don't hold database connection while doing remote operation
    db.close()
    stop_res = self.host.service("stopvnc", [str(self.dispnum)])
    db.new_cursor()
    if stop_res != 0:
      self.broken(db)
      raise DisplayError("Stopping display %s:%d exit code was %d" % (self.host.hostname, self.dispnum, stop_res))

  @staticmethod
  def insert(db, host):
    """Return a display in 'starting' state, from a new table row.
    While holding the display lock, get the highest display # for a given host.
    Then insert a row.  Displays are not auto-incremented because
    the primary key is (host, display) for the display table.
    host:  a Host object"""

    while True:
      try:
        db.start_transaction()
        # find an unused display number
        # does this host have any displays already?  If so what's the highest number?
        max_disp = db.getsingle("""
          SELECT MAX(dispnum) FROM display
          WHERE hostname=%s FOR UPDATE""",
          [host.hostname])
        first_display = host.get_first_display(db)
        if max_disp is None:
          dispnum = first_display
        else:
          # is the next highest OK to use?
          dispnum = int(max_disp) + 1
          # What is the maximum display number? 
          # < 65536 because only two bytes available for IP address (see container.py)
          # < maximum number of uses + first_display
          display_limit = host.max_uses(db) + first_display
          if dispnum >= display_limit: # 
            # try to fill gaps because container id range is restricted
            min_disp = db.getsingle("""SELECT MIN(dispnum) FROM display WHERE hostname=%s FOR UPDATE""",
              [host.hostname])
            if min_disp > first_display:
              dispnum = first_display
            else:
              min_disp = db.getsingle("""
                SELECT l.dispnum +1 as start FROM display as l
                LEFT OUTER JOIN display as r on l.hostname = r.hostname AND l.dispnum +1 = r.dispnum
                WHERE l.hostname=%s AND r.dispnum IS NULL ORDER BY start ASC LIMIT 1 FOR UPDATE""",
                [host.hostname])
              dispnum = int(min_disp)
              if dispnum >= display_limit or dispnum < first_display:
                raise DisplayError("All display numbers used up")
        db.c.execute("""
          INSERT INTO display(hostname,dispnum,status)
          VALUES(%s, %s, %s)""",
          [host.hostname, dispnum, 'starting'])
          
        # Update host count before unlocking, to make sure other sessions go to different hosts as appropriate
        host.count_uses(db)
        db.commit()
        break
      except MySQLError, e:
        log("exception: %s" % e)
        db.rollback()

    d = Display(host, dispnum, "")
    d.status = 'starting'
    return d

  @staticmethod
  def get_ready_count(db, host, appinfo, strict_geo = True):
    """return count of ready displays on that host, that have the appropriate geometry."""
    if DEBUG:
      log("getting count of ready displays")
    if strict_geo:
      return db.getsingle_tr("""
        SELECT count(*)
        FROM display
        WHERE display.hostname=%s
        AND display.status='ready'
        AND display.geometry=%s AND display.depth=%s""", \
          [host.hostname, appinfo.geometry, appinfo.depth])
    else:
      return db.getsingle_tr("""
        SELECT count(*)
        FROM display
        WHERE display.hostname=%s
        AND display.status='ready'""", \
        [host.hostname])

  @staticmethod
  def get_Display(db, appinfo, sessnum, host_k={}, zone = None, strict_geo = False):
    """
    1. Find a host, taking max_uses into account
    2. on that host, find a 'ready' display
    3. If found, mark it used, resize display
       else, call Display.make_Display on the chosen host
    4. Return None on failure
    """
    if DEBUG:
      log("getting display for sessnum %d" % sessnum)
      
    # try 100 times, rolling back transaction if there's a MySQL exception
    d = None
    geometry = None
    dispnum = 0
    while True:
      try:
        db.start_transaction()
        # include host selection in transaction because that could change if the transaction is retried
        h = Host.get_host(db, appinfo.hostreq, host_k, zone)
        if h is None:
          # this will not get fixed by trying again, so raise an exception
          raise DisplayError("Cannot find a suitable host on which to run '%s'" % appinfo.appname)
        # get a standby display from that host, or create a new one
        if DEBUG:
          log("got host %s" % h.hostname)
        row = db.getrow("""
SELECT dispnum, vncpass, geometry FROM display
WHERE hostname=%s AND status='ready'
ORDER BY display.dispnum ASC LIMIT 1 FOR UPDATE""",
          [h.hostname])
        if row is None:
          db.commit()
          break
        dispnum = row[0]
        vncpass = row[1]
        geometry = row[2]
        if DEBUG:
          log("got display %s" % dispnum)
        db.c.execute("""
UPDATE display SET status='used', sessnum=%s
WHERE hostname=%s AND dispnum=%s""",
          [sessnum, h.hostname, dispnum])
        h.count_uses(db)
        db.commit()
        # host, dispnum, vncpass, status = ''):
        if DEBUG:
          log("Display(h, %s, %s, 'used')" % (dispnum, vncpass))
        d = Display(h, dispnum, vncpass, 'used')
        break
      except MySQLError, e:
        log("exception: %s" % e)
        db.rollback()
    if d is None:
      return Display.make_Display(db, h, appinfo, sessnum)
    if geometry != appinfo.geometry:
      # resize the display we got
      log("%s  geo %s" % (dispnum, appinfo.geometry))
      exitcode = h.service("resize", ['%s' % dispnum, appinfo.geometry])
      if exitcode != 0:
        d = Display(h, dispnum, 'nothing')
        d.broken(db)
        raise DisplayError("Unable to resize display")
    if DEBUG:
      log("get_Display returning")
    return d

  @staticmethod
  def purge(db, hostname):
    """Put ready displays into 'purged' state, from which they can be shut down at leisure
    """
    if hostname is None:
      db.simple_tr("""
UPDATE display SET status = 'purged'
WHERE status = 'ready'""", 
        [])
    else:
      db.simple_tr("""
UPDATE display SET status = 'purged'
WHERE (status = 'ready' OR (status = 'starting' AND display.sessnum = 0)) AND hostname = %s""",
        [hostname])

    rows = db.getall("""
SELECT hostname, dispnum, geometry, depth FROM display
WHERE status = 'purged'""", 
      [])
    for row in rows:
      if VERBOSE:
        log("stopping ready display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        d.stop(db)
        appinfo = type("", (), dict(geometry=row[2], depth=row[3]))()
        Display.make_Display(db, h, appinfo, 0)
      except DisplayError:
        pass

  @staticmethod
  def verify(db):
    """verify that the status of the container matches our records
       - check if perhaps 'stopping' displays have already stopped
       - check that 'used' displays are really running
    """
    from session import Session
    rows = db.getall("""
      SELECT hostname, dispnum, sessnum FROM display
      WHERE status = 'stopping'
    """, [])
    for row in rows:
      if VERBOSE:
        log("checking display: %s:%d" % (row[0], row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      d.status = 'stopping'
      vnc_res = h.ask_service("status", [str(row[1])])
      if vnc_res.find('running') == -1:
        s = Session(row[2])
        try:
          s.stop_session(db, "failed")
        except PrivateError, pe:
          d.absent(db)
    rows = db.getall("""
      SELECT hostname, dispnum, sessnum FROM display
      WHERE status = 'used'
    """, [])
    for row in rows:
      if VERBOSE:
        log("checking display: %s:%d" % (row[0], row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      vnc_res = h.ask_service("status", [str(row[1])])
      if vnc_res.find('running') == -1:
        log("display: %s:%d not running: %s" % (row[0], row[1], vnc_res))
        s = Session(row[2])
        try:
          s.stop_session(db, "failed")
        except PrivateError, pe:
          log("%s" % pe)
          
    # verify that absent displays are really stopped
    db.simple_tr("""
          UPDATE display SET status = 'purged'
          WHERE status = 'absent'
        """, [])
    rows = db.getall("""
      SELECT hostname, dispnum FROM display
      WHERE status = 'purged'
    """, [])
    for row in rows:
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      if h.is_windows:
        d.__set_status(db, 'absent', 0) # can't use ready method due to state checks
      else:
        try:
          d.status = 'purged'
          d.stop(db)
        except DisplayError:
          pass
    

  @staticmethod
  def bork(db):
    """Repair broken displays.  Set their status to "purged", then run tests
      Try to stop broken displays, then restart them
      Make sure absent displays really are stopped
      Find displays stuck in "starting" state without matching sessions
    """
    import time

    # looking for displays stuck in starting state.
    db.simple_tr("""
      UPDATE display SET status = 'prepurged'
      WHERE status = 'starting' AND display.sessnum = 0
    """, [])
    log("sleeping to see if starting displays are stuck")
    time.sleep(20)
    # by now all the starting displays that were prepurged should have finished 
    # starting and be in the ready state.
    # any that are still in the 'prepurged' state are broken and need to be stopped.
    rows = db.getall("""
      SELECT hostname, dispnum, geometry, depth FROM display
      WHERE status = 'prepurged'
    """, [])
    for row in rows:
      if VERBOSE:
        log("cleaning display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        d.stop(db)
      except DisplayError:
        pass

    # looking for displays stuck in stopping state.
    db.simple_tr("""
      UPDATE display SET status = 'prepurged'
      WHERE status = 'stopping'
    """, [])
    log("sleeping to see if 'stopping' displays are stuck")
    time.sleep(20)
    # by now all the starting displays that were prepurged should have finished 
    # stopping and be in the absent state or possibly reused already.
    # any that are still in the 'prepurged' state are broken and need to be stopped.
    rows = db.getall("""
      SELECT hostname, dispnum, geometry, depth FROM display
      WHERE status = 'prepurged'
    """, [])
    for row in rows:
      if VERBOSE:
        log("cleaning display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        d.stop(db)
      except DisplayError:
        pass
          
    # fix displays in broken state
    rows = db.getall("""
SELECT hostname, dispnum, geometry FROM display
WHERE status = 'broken'
    """, [])
    for row in rows:
      if VERBOSE:
        log("stopping broken display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        # stop the display without changing status
        d.tell_stop(db)
      except DisplayError:
        log("stop failed")
        continue
      try:
        # try to start the display again
        if row[2] == '':
          geo = '800x600'
        else:
          geo = row[2]
        appinfo = type("", (), dict(geometry=geo, depth=24))()
        d.__set_status(db, "starting")
        d.start(db, appinfo, 0)
      except DisplayError:
        log("start failed")
    # active displays but matching session has ended.
    rows = db.getall("""
SELECT hostname, display.dispnum FROM display 
JOIN sessionlog ON display.sessnum = sessionlog.sessnum 
WHERE display.status = 'starting' OR display.status = 'used'
    """, [])
    for row in rows:
      if VERBOSE:
        log("stopping zombie display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        d.stop(db)
      except DisplayError:
        pass
    # starting display for a session, but session doesn't exist
    rows = db.getall("""
SELECT hostname, display.dispnum FROM display 
LEFT OUTER JOIN session ON display.sessnum = session.sessnum 
WHERE session.sessnum is NULL AND display.sessnum != 0 AND display.status = 'starting'
    """, [])
    for row in rows:
      if VERBOSE:
        log("stopping abandoned display: %s:%d" % (row[0],row[1]))
      h = Host(row[0], {})
      d = Display(h, row[1], '')
      try:
        d.status = 'purged'
        d.stop(db)
        db.commit()
      except DisplayError:
        pass
    

  @staticmethod
  def make_Display(db, host, appinfo, sessnum):
    """if sessnum is >0, return a display in 'used' state for a session.
    else return a display in the 'ready' state.
    Either try to start a container from an absent display, or create a new one."""
    if host.hostname == '':
      raise DisplayError("no hostname provided")
    d = None
    if DEBUG:
      log("make_Display(%s, appinfo, %s)" % (host.hostname, sessnum))
    while True:
      try:
        db.start_transaction()
        # SQL queries
        dispnum = db.getsingle("""
          SELECT dispnum FROM display
          WHERE hostname=%s AND status='absent'
          LIMIT 1 FOR UPDATE""",
          [host.hostname])
        if dispnum is not None:
          db.c.execute("""
UPDATE display SET status='starting'
WHERE hostname=%s AND dispnum=%s""",
          [host.hostname, dispnum])
          db.commit()
          # host, dispnum, vncpass, status
          d = Display(host, dispnum, '', 'starting')
        else:
          db.commit()
        break
      except MySQLError, e:
        log("exception: %s" % e)
        db.rollback()
    if d is None:
      d = Display.insert(db, host)
    log("set status of display %d on host %s to 'starting'" % (d.dispnum, host.hostname))
    d.start(db, appinfo, sessnum)
    return d

  @staticmethod
  def stop_zombie(db, host_k={}):
    """
    call this with a lock to avoid multiple stop attempts...
    get displays that say they belong to a session that doesn't exist.  Exclude displays
    in the 'stopping', starting, ready or absent states (no valid session attached).  Useful for cleaning up
    after testing."""
    Display.stop_orphan(db, host_k)
    row = db.getrow("""
        SELECT hostname, display.dispnum FROM display LEFT OUTER JOIN session ON
        display.sessnum = session.sessnum
        WHERE session.sessnum is NULL
        AND display.status = 'used'""", [])
    if row is None:
      return
    h = Host(row[0], host_k)
    d = Display(h, row[1], "")
    log("I think display %s on host %s is a zombie, trying to stop it" % (row[1], row[0]))
    d.stop(db)

  @staticmethod
  def stop_orphan(db, host_k={}):
    """
    Check for displays stuck in stopping state."""
    tr_counter = 0
    while tr_counter < 10:
      try:
        db.start_transaction()
        row = db.getrow("""
            SELECT hostname, display.dispnum, display.sessnum FROM display LEFT OUTER JOIN sessionlog ON
            display.sessnum = sessionlog.sessnum
            WHERE sessionlog.sessnum is NULL
            AND display.status = 'stopping' LIMIT 1 FOR UPDATE""", [])
        if row is None:
          db.commit()
          return
        h = Host(row[0], host_k)
        d = Display(h, row[1], "")
        db.c.execute("""
          INSERT INTO sessionlog
            (exechost, dispnum, sessnum, status)
          VALUES (%s, %s, %s, %s)""" ,
          [row[0], row[1], row[2], '1024']
        )
        db.commit()
        break
      except MySQLError, e:
        log("MySQL Exception %s" % e)
        db.rollback()
        time.sleep(1)
      tr_counter += 1
    if tr_counter >= 10:
      raise DisplayError('Transaction failed 10 times, while trying to stop a session')
    log("Display %s on host %s stuck in stopping state, trying to stop it" % (row[1], row[0]))
    d.stop(db)

  @staticmethod
  def from_sessnum(db, sessnum):
    """
    Given a session number, return a Display object.  Implicitly support or start a transaction with 'FOR UPDATE'.
    """
    row = db.getrow("""
      SELECT hostname, dispnum, sessnum, status, vncpass FROM display WHERE sessnum=%s
      FOR UPDATE""", [sessnum]
    )
    if row is None:
      return None
    h = Host(row[0])
    d = Display(h, row[1], "")
    d.sessnum = row[2]
    d.status = row[3]
    d.vncpass = row[4]
    return d


