/**
 * @file       cache.c
 * @copyright  Copyright (c) 2016-2020 The Regents of the University of California.
 * @license    http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2016-2020 The Regents of the University of California.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of The Regents of the University of California.
 */

#include <limits.h>
#include <stddef.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <unistd.h>
#include <fuse.h>

#include "path.h"
#include "globals.h"
#include "lock.h"
#include "git.h"



/* This function opens a file and caches the fd in open_files.
 */
int open_cache(path_t *rpath, int flags)
{
  struct fuse_context *fc = fuse_get_context();
  char tmp_path[PATH_MAX];
  int fd;

  debug("open_cache(%s)", rpath->full_path);

  if (fc == NULL) return -EACCES;

  update_mwb();

  dump_open_files();
  // Find the file in the open_files list
  path_list_t **p = &open_files;
  while (*p) {
    if (strcmp((*p)->path->full_path, rpath->full_path) == 0 &&
        (*p)->uid == fc->uid) {
      // It's already there
      debug("open_cache(): found file in open_files, returning fd %d", (*p)->fd);
      return (*p)->fd;
    }
    else {
      p = &((*p)->next);
    }
  }

  debug("open_cache(): open_files insert");
#ifdef SHADOW_WRITES

/* Build the path for the temporary file. If using git, inject the files
 * subdirectory.
 */
#ifdef USE_GIT
  if (rpath->file_path[0] != 0) {
    snprintf(tmp_path, sizeof(tmp_path), "%s/%s/files/%s/.%s:%d", source_dir,
             rpath->repo_name, rpath->file_path, rpath->file_name, fc->uid);
  }
  else {
    snprintf(tmp_path, sizeof(tmp_path), "%s/%s/files/.%s:%d", source_dir,
             rpath->repo_name, rpath->file_name, fc->uid);
  }
#else
  if (rpath->file_path[0] != 0) {
    snprintf(tmp_path, sizeof(tmp_path), "%s/%s/%s/.%s:%d", source_dir,
             rpath->repo_name, rpath->file_path, rpath->file_name, fc->uid);
  }
  else {
    snprintf(tmp_path, sizeof(tmp_path), "%s/%s/.%s:%d", source_dir,
             rpath->repo_name, rpath->file_name, fc->uid);
  }
#endif

  debug("open_cache(): tmp_path = '%s'", tmp_path);

  /* tmp_path shouldn't exist. If it did exist, it would already be in the
   * open_files list.
   */
  if (access(tmp_path, F_OK) == 0) {
    error("open_copy(): tmp file '%s' already exists, but not in open_files?!",
           tmp_path);
    return -1;
  }

  // Copy the originating file to the temporary path
  int res;
  res = copy_file(rpath->full_path, tmp_path);
  if (res != 0) {
    return res;
  }
#else
  strncpy(tmp_path, rpath->full_path, sizeof(tmp_path)-1);
  tmp_path[sizeof(tmp_path)-1] = '\0';

  debug("open_cache(): tmp_path = '%s'", tmp_path);
#endif

  /* Try for read/write access first, even if not included in flags
   */
  int orig_flags = flags;
  if (flags & O_RDWR) {
    flags &= ~O_RDWR;
    debug("open_cache(): Removing flag O_RDWR");
  }
  if (flags & O_RDONLY) {
    flags &= ~O_RDONLY;
    debug("open_cache(): Removing flag O_RDONLY");
  }
  if (flags & O_WRONLY) {
    flags &= ~O_WRONLY;
    debug("open_cache(): Removing flag O_WRONLY");
  }

  flags |= O_RDWR;

  fd = open(tmp_path, flags);
  if (fd < 0) {
    // Oh well
    debug("open_cache(): unable to open RDWR, trying with user flags");
    fd = open(tmp_path, orig_flags);
    if (fd < 0) {
      error("open_cache(): failed to open '%s'", tmp_path);
      return -errno;
    }
  }

  *p = malloc(sizeof(path_list_t));
  if (!*p) {
    error("open_cache(): malloc failure for path_list_t insertion");
    close(fd);
    return -EACCES;
  }
  (*p)->path = dup_path(rpath);
#ifdef SHADOW_WRITES
  (*p)->tmp_path = strdup(tmp_path);
#endif
  (*p)->uid = fc->uid;
  (*p)->next = NULL;
  (*p)->fd = fd;

  dump_open_files();
  return fd;
}

int flush_file(path_list_t *p)
{
  int res;

  debug("flush_file(%s)", p->path->full_path);

  if (lock_repo(p->path->repo_name) != 0) {
    return -EACCES;
  }

  res = close(p->fd);
  if (res != 0) {
    error("flush_file(): unable to close fd %d for path %s", p->fd,
          p->path->full_path);
    unlock_repo(p->path->repo_name);
    return res;
  }

#ifdef SHADOW_WRITES
  res = unlink(p->path->full_path);
  if (res != 0) {
    error("flush_file(): failed to unlink '%s'", p->path->full_path);
    unlock_repo(p->path->repo_name);
    return -errno;
  }

  res = rename(p->tmp_path, p->path->full_path);
  if (res != 0) {
    error("flush_file(): rename(%s, %s) failed errno %d", p->tmp_path,
          p->path->full_path, errno);
    unlock_repo(p->path->repo_name);
    return -errno;
  }
#endif

#ifdef USE_GIT
  git(GIT_ADD, p->path, NULL, p->uid);
#endif

  update_mwb();

  unlock_repo(p->path->repo_name);

  return 0;
}

