/**
 * @file       lock.c
 * @copyright  Copyright (c) 2016-2020 The Regents of the University of California.
 * @license    http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2016-2020 The Regents of the University of California.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of The Regents of the University of California.
 */

#include <limits.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "lock.h"
#include "globals.h"



/* lock_repo() provides basic repository-granularity locking to prevent
 * simultaneous git fuckups
 */
int lock_repo(const char *repo_name) {
#ifdef LOCKING
  char lock_name[PATH_MAX+9] = {0};
  int res;
  int i;

  debug("lock_repo(%s)", repo_name);

  snprintf(lock_name, PATH_MAX+9, "%s/%s/.lock42", source_dir, repo_name);

  debug("lock_name %s", lock_name);

  // Laugh it up, fuzzball
  for (i = 0; i < LOCK_ATTEMPTS; i++) {
    res = mkdir(lock_name, 0700);
    if (res != 0) {
      debug("sleeping on lock %s, res %d, %d", lock_name, res, errno);
      sleep(1);
    }
    else { 
      return 0;
    }
  }

  error("lock_repo(%s), lock_name %s failed", repo_name, lock_name);
  return -1;
#else
  return 0;
#endif
}

int unlock_repo(const char *repo_name) {
#ifdef LOCKING
  char lock_name[PATH_MAX+9];

  debug("unlock_repo(%s)", repo_name);

  snprintf(lock_name, PATH_MAX+9, "%s/%s/.lock42", source_dir, repo_name);

  debug("lock_name %s", lock_name);

  if (rmdir(lock_name) != 0) {
    error("unlock_repo(%s), lock_name %s failed", repo_name, lock_name);
    return -1;
  }
#endif
  return 0;
}

