# @package      hubzero-python
# @file         user.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import datetime
import hubzero.config.passwords
import hubzero.utilities.group
import hubzero.utilities.misc
import hubzero.config.webconfig
import hubzero.data.db
import ldap
import ldap.modlist
import os


def _userExists(db, username):
	sql = 'select id from `jos_users` where username = %s'
	data = (username)

	userID = db.execSelectQueryScalar(sql, data)
	
	if userID is None:
		return False
	else:
		return True


def _del_jos_users(db, username):
	sql = 'select id from `jos_users` where username = %s'
	data = (username)
	userID = db.execSelectQueryScalar(sql, data)	

	sql = 'delete from `jos_users` where userid = %s'
	data = (userID)
	userID = db.execSelectQueryScalar(sql, data)	

	return userID


def _del_jos_core_acl_aro(db, userID):

	sql = 'select id from `jos_core_acl_aro` where `value` = %s'
	data = (userID)
	aroID = db.execSelectQueryScalar(sql, data)	

	sql = "DELETE FROM `jos_core_acl_aro` where `value` = %s"
	data = (userID)
	db.execQuery(sql, data)
	
	return aroID


def _del_jos_core_acl_groups_aro_map(db, aroID):
	print "aroID = " + str(aroID)
	sql = "DELETE FROM `jos_core_acl_groups_aro_map` WHERE `aro_id` = %s"
	data = (aroID)
	db.execQuery(sql, data)


def _del_jos_xprofiles(db, uidNumber):
	sql = "DELETE FROM `jos_xprofiles` WHERE `uidNumber` = %s"
	data = (uidNumber)
	db.execQuery(sql, data)


def _insert_jos_users(db, joomlauserid, name, username, email, pw, jgid, jgidNumber):

	pwhash = hubzero.config.passwords.MD5PasswordHash(pw)

	if joomlauserid == -1:
		sql = "INSERT INTO `jos_users` (`name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
		sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
		data = (name, username, email, pwhash , jgid, 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '');
		userID = db.execInsertQuery(sql, data)
	else:
		sql = "INSERT INTO `jos_users` (`id`, `name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
		sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
		data = (joomlauserid, name, username, email, pwhash, jgid, 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '');
		userID = db.execInsertQuery(sql, data)

	return userID


def _insert_jos_core_acl_aro(db, userID, fullname):
	sql = "INSERT INTO `jos_core_acl_aro` (`section_value`, `value`, `order_value`, `name`, `hidden`) "
	sql += "VALUES (%s, %s, %s, %s, %s)"
	data = ( 'users',  userID, 0, fullname, 0)
	aroID = db.execInsertQuery(sql, data)
	return aroID
	

def _insert_jos_core_acl_groups_aro_map(db, jgidNumber, aroID):
	sql = "INSERT INTO `jos_core_acl_groups_aro_map` (`group_id`, `section_value`, `aro_id`) "
	sql += "VALUES (%s, %s, %s)"
	data = ( jgidNumber, '', aroID)
	db.execQuery(sql, data)


def _insert_jos_xprofiles(db, hubname, uid, fname, mname, lname, username, email, pw, joomlagid, joomlagidNumber, gid, gidNumber, loginShell, homeDir):
	
	sql = "INSERT INTO `jos_xprofiles` (`uidNumber`, `name`, `username`, `email`, `registerDate`, `gidNumber`, `homeDirectory`, `loginShell`, "
	sql += "`ftpShell`, `userPassword`, `gid`, `orgtype`, `organization`, `countryresident`, `countryorigin`, `gender`, `url`, `reason`, "
	sql += "`mailPreferenceOption`, `usageAgreement`, `jobsAllowed`, `modifiedDate`, `emailConfirmed`, `regIP`, `regHost`, `nativeTribe`, `phone`, "
	sql += "`proxyPassword`, `proxyUidNumber`, `givenName`, `middleName`, `surname`, `picture`, `vip`, `public`, `params`, `note`, `shadowExpire`) "
	sql += "VALUES (" + '%s,'*37 + "%s );" 
	
	data = (uid, fname + ' ' + lname, username, email, datetime.datetime.now(), gidNumber, homeDir, loginShell, 
	        '/usr/lib/sftp-server', hubzero.config.passwords.MD5PasswordHash(pw), gid, '', '', '', '', '', '', '', 
	        0, 0, 3, '0000-00-00 00:00:00', 1, '', '', '', '',
	        '', '', fname, mname, lname, '', 0, 0, '', '', None)
	db.execQuery(sql, data)


def _insert_jos_users_password(db, userid, pw):
	sql = "INSERT INTO jos_users_password (`user_id`, `passhash`) " 
	sql += "VALUES (%s, %s)"
	
	data = (userid, hubzero.config.passwords.MD5PasswordHash(pw))
	db.execQuery(sql, data)


def _add_ldap_user(username, gecos, password, uidNumber, homeDir, loginShell, gid, gidNumber):

	# get ldapManagerUserDN and PW
	
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW= hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")	

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "uid=" + username + ",ou=users," + hubLDAPBaseDN
	attrs = {}
	attrs['objectclass'] = ['top', 'account', 'posixAccount', 'shadowAccount']  
	attrs['cn'] = username

	if password != '*':
		attrs['userPassword'] = hubzero.config.passwords.MD5PasswordHash(password)
	else:
		attrs['userPassword'] = '{MD5}*' # can user generate a value that will MD5 hash to '*'?
	
	attrs['uid'] = username
	attrs['uidNumber'] = str(uidNumber)
	attrs['homeDirectory'] = homeDir
	attrs['cn'] = gecos
	attrs['loginShell'] = loginShell
	
	# TODO: grab this from the com_members component parm field
	attrs['gidNumber'] = str(gidNumber)
	#attrs['gid'] = gid
	
	print "add ldap user dn: " + dn

	ldif = ldap.modlist.addModlist(attrs)
	l.add_s(dn, ldif)


def _delete_ldap_user(username):
	# get ldapManagerUserDN and PW
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW= hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")	

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)
	dn = "uid=" + username + ",ou=users," + hubLDAPBaseDN
	print "delete ldap user dn: " + dn
	l.delete_s(dn)


def addhubuser(hubname, username, gecos, email, pw, pwdisabled, joomlagid, 
               joomlagidNumber, gid, gidNumber, joomlauserid, loginShell, nocreatehome, homeDir, createGroup, 
               skipldap):

	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	
	db =  hubzero.data.db.mySql("localhost", dbName, dbUserName, dbPW)

	if _userExists(db, username):
		print "User already exists: " + username
		return (-1)


	# parse gecos, it's a massive mess, but it's a comma delimited field used in the /etc/passwd file to store personal info
	# it's comma delimited, and name is the first item in the list... right now we don't care about what else gets passed,
	# but the first entry
	gecosFields = gecos.split(",")
	fullName = gecosFields[0]
	fullNameList = gecosFields[0].split(" ")
	
	if len(fullName) == 3:
		firstName = fullName[0]
		middleName = fullName[1]
		lastName = fullName[2]
	elif len(fullName) == 2:
		firstName = fullName[0]
		middleName = ""
		lastName = fullName[1]
	elif len(fullName) == 1:
		firstName = fullName[0]
		middleName = ""
		lastName = ""
	else:
		firstName = ""
		middleName = ""
		lastName = ""
	

	# usually homeDir will be blank, but not always
	if homeDir == "":
		homeDir = '/home/' + hubname + "/" + username


	if pwdisabled: 
		pw = "*"

	# create linux group for user named same as username (need group insert here for xprofiles addition below)
	if createGroup:
		gidNumber = hubzero.utilities.group._insert_jos_xgroups(db, username, username + " group")	
		gid = username # group name same as user


	userID = _insert_jos_users(db, joomlauserid, fullName, username, email, pw, joomlagid, joomlagidNumber)
	aroID = _insert_jos_core_acl_aro(db, userID, fullName)
	_insert_jos_core_acl_groups_aro_map(db, joomlagidNumber, aroID)
	_insert_jos_xprofiles(db, hubname, userID, firstName, middleName, lastName, username, email, pw, joomlagid, joomlagidNumber, gid, gidNumber, loginShell, homeDir)
	_insert_jos_users_password(db, userID, pw)


	# create linux group for user named same as username (need xgroups insert here because we need userid)
	if createGroup:

		hubzero.utilities.group._insert_jos_xgroups_members(db, gidNumber, userID)
		hubzero.utilities.group._add_ldap_group(username, username + " group", gidNumber)
	else:
		gidNumber = 100 # default to the users group
		
	
	if not skipldap:
		_add_ldap_user(username, gecos, pw, userID, homeDir, loginShell, gid, gidNumber)


	# create home directory
	if not nocreatehome:
		rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(["mkdir", homeDir, "--mode", "0750"])
		print procStdOut
		if rc:
			print procStdErr
		
		print "creating homedir: " + homeDir
		
		# set ownership
		os.chown(homeDir, userID, gidNumber)
		
		# change permissions user full, noone else has anything
		rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(["chmod", '700', homeDir])
		if rc:
			print procStdErr
		
	return(0)

	
def delhubuser(username):
	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.mySql("localhost", dbName, dbUserName, dbPW)
	
	sql = 'select id from `jos_users` where username = %s'
	data = (username)
	userID = db.execSelectQueryScalar(sql, data)

	print "deleting user (" + username + ") from database"
	
	aroID = _del_jos_core_acl_aro(db, userID)
	_del_jos_core_acl_groups_aro_map(db, aroID)
	_del_jos_xprofiles(db, userID)
	_del_jos_users(db, username)
	
	_delete_ldap_user(username)
