# @package      hubzero-python
# @file         db.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

from contextlib import contextmanager

import contextlib
import exceptions
import itertools
import sys
import warnings

# Guess there is a chance this won't be installed
try:
	import MySQLdb
	from _mysql_exceptions import ProgrammingError, OperationalError
except ImportError:
	try:
		import pymysql
		pymysql.install_as_MySQLdb()
                import MySQLdb
		from pymysql import ProgrammingError, OperationalError
	except ImportError:
		sys.stderr.write("Error: Cannot import MySQLdb or pymysql library")
		sys.exit(1)

warnings.filterwarnings('error', category=MySQLdb.Warning)

class MySQLConnection:
	""" Very lightweight MySQL library """

	_db = None
	_dbHost = ""
	_dbName = ""
	_dbPW = ""
	_dbUsername = ""


	def _db_connect(self):
		"""internal connect method"""
		self._db = MySQLdb.connect(host=self._dbHost, 
		                          user=self._dbUsername, 
		                          db=self._dbName,
		                          passwd=self._dbPW,
                                          charset='utf8')

	def __init__(self, dbHost, dbName, dbUsername, dbPW):
		""" Constructor """
		self._dbHost = dbHost
		self._dbName = dbName
		self._dbPW = dbPW
		self._dbUsername = dbUsername
		self._db_connect()
		self._db.autocommit(True)


	def __del__(self):
		""" Destructor """
		self.close()
	
	
	def close(self):
		if self._db is not None:
			self._db.close()
			self._db = None


	def _cursor(self):
		return self._db.cursor()


	def _execute(self, cursor, sql, parms):
		"""
		we do this in multiple places and it's nice to be consistent with handling errors
		"""
		return cursor.execute(sql, parms)


	@contextmanager
	def lock(name, spinLockLoop = 10):
		"""
		name - name of lock
		spinLockLoop - how many times should we try to get lock before throwing an exception
		
		use context manager decorator so we can use 'with lock("lockname")' calls to manage and close locks automatically,
		even if there is an exception anywhere inside the with block
	
		example usage:
		
		with db.lock("lock1):
		    db.query(query1)
			db.query(query2)
			db.query(query3)
			...
		# lock auto released here once outside with section
		
		"""
		cursor = self._cursor
		
		gotLock = 0
		loopIteration = 0
		
		while not got_lock:
			loopIteration += 1
			if iteration > 10:
				raise MaxwellError("can't get lock, giving up")
			else:
				raise Exception("Contention while waiting on lock %s" % name)

			# Todo, finish here, see what  get lock returns when it fails, then test
		
			self._execute(cursor, "SELECT GET_LOCK(%s,%s)", name, 5)
		
		yield # return to caller, then after with block is done, continue below

		got_lock = self._execute(cursor, "SELECT RELEASE_LOCK(%s)", name)
		
		

	def query(self, query, parms):
		""" Returns a list of rows for the given query and parameters """
		cursor = self._cursor()
		try:
			self._execute(cursor, query, parms)

			columnNames = []
			for column in cursor.description:
				columnNames.append(column[0])

			# return list of dictWithAttributes objects, zip with column names
			# dictWithAttributes custom object allows us to to obj.prop style notation with value returned by this function
			# and avoid object["prop"] style ugly lookups
			return [dictWithAttributes(zip(columnNames, row)) for row in cursor]		
		finally:
			if cursor: cursor.close()

	
	def query_lastrowid(self, sql, parms):
		"""
		Single insert query that returns the pkid of the last inserted row
		"""
		cursor = self._cursor()
		try:
			self._execute(cursor, sql, parms)
			return cursor.lastrowid
		finally:
			if cursor: cursor.close() 

		
	def query_rowcount(self, sql, parms):
		"""
		exec query and return the rowcount
		"""
		cursor = self._cursor()
		try:
			self._execute(cursor, sql, parms)
			return cursor.rowcount
		finally:
			if cursor: cursor.close()


	def query_selectscalar(self, sql, parms):
		"""
		When you're looking for an easy way to get only a single value from a query 
		"""
		try:
			cursor = self._cursor()
			self._execute(cursor, sql, parms)
			r = cursor.fetchall()
	
			if len(r) != 1:
				return None
			else:
				return r[0][0]
		finally:
			if cursor: cursor.close()

		
class dictWithAttributes(dict):
	"""
	helps support object.property style notation in returned query results
	"""
	def __getattr__(self, name):
		return self[name]
