# @package      hubzero-submit-client
# @file         LocalBatchPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
import logging

try:
   from Pegasus.DAX3 import *
except:
   from hubzero.submit.DAX3 import *

class LocalBatchPEGASUS:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                executable,
                arguments,
                pegasusTemplates,
                timeHistoryLogs):
      self.logger            = logging.getLogger(__name__)
      self.hubUserName       = hubUserName
      self.hubUserId         = hubUserId
      self.runName           = runName
      self.localJobId        = localJobId
      self.instanceId        = instanceId
      self.instanceDirectory = instanceDirectory
      self.scratchDirectory  = scratchDirectory
      self.useSetup          = useSetup
      self.pegasusVersion    = pegasusVersion
      self.pegasusHome       = pegasusHome
      self.timestampStart    = os.path.join(self.instanceDirectory,timeHistoryLogs['timestampStart'])
      self.timestampFinish   = os.path.join(self.instanceDirectory,timeHistoryLogs['timestampFinish'])
      self.timeResults       = os.path.join(self.instanceDirectory,timeHistoryLogs['timeResults'])
      self.checkDaxPath      = ""

      basename = os.path.basename(executable)
      if basename.startswith('pegasus-'):
         self.daxPath             = ""
         self.executable          = executable
         userArguments = arguments.split()
         scriptArguments = []
         for userArgument in userArguments:
            if os.path.isfile(userArgument):
               absolutePath = os.path.abspath(userArgument)
               scriptArguments.append(absolutePath)
            else:
               scriptArguments.append(userArgument)
         self.arguments           = ' '.join(scriptArguments)
         self.checkDaxExecutables = True
      else:
         self.daxPath             = os.path.join(self.instanceDirectory,"%s_%s.dax" % (self.localJobId,self.instanceId))
         self.executable          = 'pegasus-plan'
         self.arguments           = "--dax %s" % (self.daxPath)
         self.checkDaxExecutables = False

      self.pegasusTemplates = pegasusTemplates

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# LocalBatchPEGASUS:makeSerialTemplate
#
exitStatus=0

mkdir -p INSTANCEDIRECTORY/work
mkdir -p SCRATCHDIRECTORY/work

PEGASUSSETUP

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

#Discarded Arguments      = DISCARDED
#Not Recognized Arguments = NOTRECOGNIZED

date +"%s" > TS_START

${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS \\
             EXECUTABLE --conf INSTANCEDIRECTORY/JOBID_INSTANCEID.pegasusrc \\
             --sites local \\
             --dir SCRATCHDIRECTORY \\
             --relative-dir scratch \\
             --relative-submit-dir work \\
             --output local \\
             ARGUMENTS \\
             --quiet \\
             --submit < /dev/null
exitStatus=$?

date +"%s" > TS_FINISH

exit ${exitStatus}
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_runName                = re.compile("RUNNAME")
      re_jobId                  = re.compile("JOBID")
      re_instanceId             = re.compile("INSTANCEID")
      re_instanceDirectory      = re.compile("INSTANCEDIRECTORY")
      re_scratchDirectory       = re.compile("SCRATCHDIRECTORY")
      re_pegasusSetup           = re.compile("PEGASUSSETUP")
      re_executable             = re.compile("EXECUTABLE")
      re_arguments              = re.compile("ARGUMENTS")
      re_discardedArguments     = re.compile("DISCARDED")
      re_notRecognizedArguments = re.compile("NOTRECOGNIZED")
      re_tsStart                = re.compile("TS_START")
      re_tsFinish               = re.compile("TS_FINISH")
      re_timeResults            = re.compile("TIME_RESULTS")
      re_hubUserName            = re.compile("HUBUSERNAME")
      re_hubUserId              = re.compile("HUBUSERID")

      template = self.__makeSerialTemplate()

      try:
         pegasusSetupCommand = ". %s\nuse -e -r pegasus-%s" % (self.useSetup,self.pegasusVersion)
         pegasusSetupCommandPath = "INSTANCEDIRECTORY/work/pegasusSetup.sh"
         template = re_pegasusSetup.sub("cat > %s << EOF\n%s\nEOF\n. %s\n" % (pegasusSetupCommandPath,pegasusSetupCommand,
                                                                              pegasusSetupCommandPath),template)
      except:
         template = re_pegasusSetup.sub("",template)

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_instanceDirectory.sub(self.instanceDirectory,template)
      template = re_scratchDirectory.sub(self.scratchDirectory,template)
      template = re_executable.sub(self.executable,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      if os.path.exists(self.pegasusTemplates['rc']):
         fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
         if fpRCTemplate:
            searchString = os.path.basename(self.executable) + '.arguments'
            settings = fpRCTemplate.readlines()
            for setting in settings:
#pegasus-plan.arguments = --nocleanup
               if setting.count(searchString) > 0:
                  try:
                     parameter,value = setting.split('=')
                     auxiliaryArguments.append(value.strip())
                  except:
                     pass
            fpRCTemplate.close()

      userArguments = self.arguments.split()
      location = 0
      while location < len(userArguments):
         if userArguments[location].startswith('-'):
            recognizeArgument = True
            saveDaxPath       = False
            if   userArguments[location].startswith('-D'):
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-d' or userArguments[location] == '--dax':
               keepArgument = True
               if self.checkDaxExecutables:
                  saveDaxPath = True
            elif userArguments[location] == '-b' or userArguments[location] == '--basename':
               keepArgument = True
            elif userArguments[location] == '-c' or userArguments[location] == '--cache':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-C' or userArguments[location] == '--cluster':
               keepArgument = True
            elif userArguments[location] == '--conf':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '--dir':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-f' or userArguments[location] == '--force':
               keepArgument = True
            elif userArguments[location] == '--force-replan':
               keepArgument = True
            elif userArguments[location] == '-g' or userArguments[location] == '--group':
               keepArgument = True
            elif userArguments[location] == '-h' or userArguments[location] == '--help':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '--inherited-rc-files':
               keepArgument = True
            elif userArguments[location] == '-j' or userArguments[location] == '--j':
               keepArgument = True
            elif userArguments[location] == '-n' or userArguments[location] == '--nocleanup':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-o' or userArguments[location] == '--output':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-q' or userArguments[location] == '--quiet':
               keepArgument = True
            elif userArguments[location] == '--relative-submit-dir':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-s' or userArguments[location] == '--sites':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-S' or userArguments[location] == '--submit':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location].startswith('-v') or userArguments[location] == '--verbose':
               keepArgument = True
            elif userArguments[location] == '-V' or userArguments[location] == '--version':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            else:
               keepArgument = False
               recognizeArgument = False
               notRecognizedArguments.append(userArguments[location])

            if keepArgument:
               auxiliaryArguments.append(userArguments[location])
            location += 1
            while location < len(userArguments) and not userArguments[location].startswith('-'):
               if   keepArgument:
                  auxiliaryArguments.append(userArguments[location])
                  if saveDaxPath:
                     self.checkDaxPath = userArguments[location]
               elif recognizeArgument:
                  discardedArguments.append(userArguments[location])
               else:
                  notRecognizedArguments.append(userArguments[location])
               location += 1
         else:
            location += 1

      arguments = ' '.join(auxiliaryArguments)
      template  = re_arguments.sub(arguments.strip(),template)
      arguments = ' '.join(discardedArguments)
      template  = re_discardedArguments.sub(arguments.strip(),template)
      arguments = ' '.join(notRecognizedArguments)
      template  = re_notRecognizedArguments.sub(arguments.strip(),template)
      template  = re_hubUserName.sub(self.hubUserName,template)
      template  = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __buildCatalogs(self):
      # setup regex's for the template
      re_runName           = re.compile("RUNNAME")
      re_jobId             = re.compile("JOBID")
      re_instanceId        = re.compile("INSTANCEID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_scratchDirectory  = re.compile("SCRATCHDIRECTORY")
      re_pegasusHome       = re.compile("PEGASUSHOME")

      for templateType in self.pegasusTemplates:
         template = self.pegasusTemplates[templateType]
         if template != "":
            template = re_jobId.sub(self.localJobId,template)
            template = re_instanceId.sub(self.instanceId,template)
            template = re_instanceDirectory.sub(self.instanceDirectory,template)
            template = re_scratchDirectory.sub(self.scratchDirectory,template)
            template = re_pegasusHome.sub(self.pegasusHome,template)
            if   templateType == 'rc':
               pegasusFile = "%s_%s.pegasusrc" % (self.localJobId,self.instanceId)
            elif templateType == 'sites':
               pegasusFile = "%s_%s_sites.xml" % (self.localJobId,self.instanceId)
            elif templateType == 'tc':
               pegasusFile = "%s_%s_tc.txt" % (self.localJobId,self.instanceId)
            pegasusPath = os.path.join(self.instanceDirectory,pegasusFile)
            fpPegasusFile = open(pegasusPath,'w')
            if fpPegasusFile:
               fpPegasusFile.write(template)
               fpPegasusFile.close()


   def buildBatchScript(self):
      batchLogName    = ""
      batchScriptName = "%s_%s.pegasus" % (self.localJobId,self.instanceId)
      batchScript     = self.__buildSerialFile()

      self.__buildCatalogs()

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


