# @package      hubzero-submit-client
# @file         ClientIdAuthAttributes.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import sys
import os
import getpass
import pwd
import logging

class ClientIdAuthAttributes:
   def __init__(self):
      self.logger            = logging.getLogger(__name__)
      self.sessionAttributes = {}
      self.userAttributes    = {}

      if 'SESSIONDIR' in os.environ:
         resourcePath = os.path.join(os.environ['SESSIONDIR'],'resources')
         self.getSessionAttributes(resourcePath)

      if not 'token' in self.sessionAttributes:
         if 'HOME' in os.environ:
            resourcePath = os.path.join(os.environ['HOME'],'.default_resources')
            self.getSessionAttributes(resourcePath)


   def getSessionAttributes(self,
                            resourcePath):
      del self.sessionAttributes
      self.sessionAttributes = {}

      if os.path.exists(resourcePath):
         try:
            fpResource = open(resourcePath,'r')
            try:
               eof = False
               while not eof:
                  record = fpResource.readline()
                  if record != "":
                     if   record.startswith('session_token '):
                        self.sessionAttributes['token'] = record.split()[1]
                     elif record.startswith('username '):
                        self.sessionAttributes['userName'] = record.split()[1]
                     elif record.startswith('sessionid '):
                        self.sessionAttributes['sessionId'] = record.split()[1]
                  else:
                     eof = True
            except:
               pass
            finally:
               fpResource.close()
         except (IOError,OSError):
            pass


   def getUserAttributes(self):
      del self.userAttributes
      self.userAttributes = {}

      self.userAttributes['userName'] = getpass.getuser()
      haveToken = 'token' in self.sessionAttributes
      if not haveToken:
         si = sys.stdin
         try:
            sys.stdin = open('/dev/tty')
            try:
               username = raw_input("Username: (%s) " % (self.userAttributes['userName']))
               if username != "":
                  self.userAttributes['userName'] = username
               password = getpass.getpass("Password: ")
               if password != "":
                  self.userAttributes['password'] = password
            except (IOError,OSError):
               pass
         except (IOError,OSError):
            pass
         sys.stdin = si

      try:
         (login,pw,uid,gid,name,homedir,shell) = pwd.getpwnam(self.userAttributes['userName'])
         self.userAttributes['userId'] = uid
      except:
         pass


   def haveIdAuthAttributes(self):
      haveAttributes = False
      if   'token' in self.sessionAttributes:
         haveAttributes = True
      elif 'userName' in self.userAttributes and 'password' in self.userAttributes:
         haveAttributes = True

      return(haveAttributes)


   def getSignonAttributes(self):
      signonAttributes = {}
      signonAttributes['userName']     = self.userAttributes['userName']
      if 'password' in self.userAttributes:
         signonAttributes['password']  = self.userAttributes['password']
      if 'userId' in self.userAttributes:
         signonAttributes['userId']    = self.userAttributes['userId']
      if 'sessionId' in self.sessionAttributes:
         signonAttributes['sessionId'] = self.sessionAttributes['sessionId']
      if 'token' in self.sessionAttributes:
         signonAttributes['token']     = self.sessionAttributes['token']

      return(signonAttributes)


   def clearSessionToken(self):
      if 'token' in self.sessionAttributes:
         del self.sessionAttributes['token']


