# @package      hubzero-submit-client
# @file         LocalWorkflowPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

from Pegasus.DAX3 import *

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate

class LocalWorkflowPEGASUS:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                jobDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                localJobId,
                instanceId,
                pegasusTemplates,
                pegasusSiteInfo,
                submissionScriptsInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.jobDirectory          = jobDirectory
      self.scratchDirectory      = scratchDirectory
      self.useSetup              = useSetup
      self.pegasusVersion        = pegasusVersion
      self.pegasusHome           = pegasusHome
      self.localJobId            = localJobId
      if int(instanceId) > 0:
         self.instanceId         = instanceId
      else:
         self.instanceId         = None

      self.daxPath               = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable            = 'pegasus-plan'
      self.arguments             = "--dax %s" % (self.daxPath)

      self.pegasusTemplates      = pegasusTemplates
      self.pegasusSiteInfo       = pegasusSiteInfo
      self.submissionScriptsInfo = submissionScriptsInfo
      self.timestampStart        = os.path.join(self.jobDirectory,timeHistoryLogs['timestampStart'])
      self.timestampFinish       = os.path.join(self.jobDirectory,timeHistoryLogs['timestampFinish'])
      self.timeResults           = os.path.join(self.jobDirectory,timeHistoryLogs['timeResults'])


   def __buildWorkflowScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Workflow','PEGASUS','plan')

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if 'rc' in self.pegasusTemplates:
         searchString = os.path.basename(self.executable) + '.arguments'
         settings = self.pegasusTemplates['rc'].split('\n')
         for setting in settings:
#pegasus-plan.arguments = --nocleanup
            if setting.count(searchString) > 0:
               try:
                  parameter,value = setting.split('=')
                  auxiliaryArguments.append(value.strip())
               except:
                  pass

      substitutions = {}
      substitutions["USESETUP"]          = self.useSetup
      substitutions["PEGASUSVERSION"]    = self.pegasusVersion
      substitutions["JOBID"]             = self.localJobId
      substitutions["RUNNAME"]           = self.runName
      substitutions["INSTANCEDIRECTORY"] = self.jobDirectory
      substitutions["SCRATCHDIRECTORY"]  = self.scratchDirectory
      substitutions["EXECUTABLE"]        = self.executable
      arguments = ' '.join(auxiliaryArguments)
      substitutions["ARGUMENTS"]         = arguments.strip()
      arguments = ' '.join(discardedArguments)
      substitutions["DISCARDED"]         = arguments.strip()
      arguments = ' '.join(notRecognizedArguments)
      substitutions["NOTRECOGNIZED"]     = arguments.strip()
      substitutions["TS_START"]          = self.timestampStart
      substitutions["TS_FINISH"]         = self.timestampFinish
      substitutions["TIME_RESULTS"]      = self.timeResults
      substitutions["HUBUSERNAME"]       = self.hubUserName
      substitutions["HUBUSERID"]         = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildCatalogs(self):
      re_instanceIdEmpty = re.compile("_@@{INSTANCEID}")
      substitutions = {}
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEID"]           = self.instanceId
      substitutions["INSTANCEDIRECTORY"]    = self.jobDirectory
      substitutions["BASESCRATCHDIRECTORY"] = os.path.basename(self.scratchDirectory)
      substitutions["SCRATCHDIRECTORY"]     = self.scratchDirectory
      substitutions["PEGASUSHOME"]          = self.pegasusHome

      for templateType in self.pegasusTemplates:
         template = self.pegasusTemplates[templateType]
         if self.instanceId:
            substitutions["INSTANCEID"] = self.instanceId
         else:
            template = re_instanceIdEmpty.sub("",template)
            substitutions["INSTANCEID"] = "00"

         pegasusTemplate = ParameterTemplate(template)
         try:
            pegasusText = pegasusTemplate.substitute_recur(substitutions)
         except KeyError,err:
            pegasusText = ""
            self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
         except TypeError,err:
            pegasusText = ""
            self.logger.log(logging.ERROR,getLogMessage("Pegasus template substitution failed:\n%s\n" % (template)))

         if   templateType == 'rc':
            pegasusFile = "%s.pegasusrc" % (self.localJobId)
         elif templateType == 'sites':
            pegasusFile = "%s_sites.xml" % (self.localJobId)
         pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
         try:
            fpPegasusFile = open(pegasusPath,'w')
            try:
               fpPegasusFile.write(pegasusText)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (pegasusPath)))
            finally:
               fpPegasusFile.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusPath)))


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(self.jobDirectory)
      for dirFile in dirFiles:
         dirPath = os.path.join(self.jobDirectory,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      dax = ADAG(self.localJobId)

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(self.jobDirectory,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles  = filter(reAppScriptPath.search,dirFiles)

         if len(appScriptFiles) == 1:
            arch      = self.pegasusSiteInfo['arch']
            osFlavor  = self.pegasusSiteInfo['osFlavor'].lower()
            osVersion = self.pegasusSiteInfo['osVersion']
            osRelease = self.pegasusSiteInfo['osRelease']

            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            appScriptPath = "file://" + appScriptPath
            appScript = Executable(name=appScriptName,arch="%s" % (arch),
                                                      os="%s" % (osFlavor),
                                                      osrelease="%s" % (osRelease),
                                                      osversion="%s" % (osVersion),installed=True)
            appScript.addPFN(PFN(appScriptPath,"local"))
            dax.addExecutable(appScript)

            jobId = "%s_%s" % (self.runName,instanceId)
            job = Job(name=appScriptName,id=jobId)
            dax.addJob(job)

      try:
         fpDAX = open(self.daxPath,'w')
         try:
            dax.writeXML(fpDAX)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.daxPath)))
         finally:
            fpDAX.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.daxPath)))


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()
      workflowScriptExecutable = True

      return(workflowScriptName,workflowScript,workflowScriptExecutable)


