# @package      hubzero-submit-client
# @file         SubmitCommand.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import sys
import logging
import traceback
import shlex
import copy
import json
import textwrap
from io import IOBase

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage, logSetMessageFile
from hubzero.submit.SubmitClient import SubmitClient

class SubmitCommand:
   LOGDIRECTORY = os.getcwd()
   HUBLOGFILE   = '.submit.log'
   HUBLOGPATH   = os.path.join(LOGDIRECTORY,HUBLOGFILE)

   CONFIGURATIONDIRECTORY = os.path.join(os.sep,'etc','submit')
   CONFIGURATIONFILE      = 'submit-client.conf'

   TAILNLINESDEFAULT = 10
   SEPARATORDEFAULT  = ","

   def __init__(self,
                configurationDirectory=CONFIGURATIONDIRECTORY,
                hubLogPath=HUBLOGPATH):
      self.__openLogger(hubLogPath)
      self.configFilePath = os.path.join(configurationDirectory,self.CONFIGURATIONFILE)

      try:
         instantiator          = traceback.extract_stack()[-2]
      except:
         self.instantiatorPath = "submit"
      else:
         self.instantiatorPath = os.path.realpath(os.path.abspath(instantiator[0]))

      self.submitClient = None

      self.args         = []
      self.savedCommand = {}

      self.help                 = False
      self.helpManagers         = False
      self.helpTools            = False
      self.helpVenues           = False
      self.helpExamples         = False
      self.version              = False
      self.versionClient        = False
      self.versionServer        = False
      self.versionDistributor   = False
      self.debug                = False
      self.local                = False
      self.queryJobs            = []
      self.killJobs             = []
      self.venueStatus          = ""
      self.venues               = []
      self.inputFiles           = []
      self.parameters           = []
      self.dataFile             = ""
      self.separator            = self.SEPARATORDEFAULT
      self.separators           = []
      self.nCores               = None
      self.ppn                  = None
      self.nStripes             = None
      self.wallTime             = None
      self.environmentVariables = {}
      self.runName              = ""
      self.manager              = ""
      self.redundancy           = None
      self.reportMetrics        = False
      self.detach               = False
      self.attachId             = None
      self.wait                 = False
      self.quota                = True
      self.tailNlinesDefault    = self.TAILNLINESDEFAULT
      self.tailStdout           = False
      self.tailStdoutNlines     = self.tailNlinesDefault
      self.tailStderr           = False
      self.tailStderrNlines     = self.tailNlinesDefault
      self.tailFiles            = []
      self.progress             = None
      self.command              = ""
      self.commandArguments     = []
      self.stdin                = None


   def __openLogger(self,
                    hubLogPath):
      class EmptyFilter(logging.Filter):
         """
         This is a filter which rejects empty messages

         """

         def filter(self,record):
            if record.getMessage() == "":
               emptyRecord = True
            else:
               emptyRecord = False

            return(not emptyRecord)

      self.logger = logging.getLogger('')
      self.logger.setLevel(logging.ERROR)

      if sys.stderr.isatty():
         logHandler = logging.StreamHandler(sys.stderr)
         logSetMessageFile(sys.stderr)
      else:
         logHandler = logging.FileHandler(hubLogPath)

      emptyFilter = EmptyFilter()
      logHandler.addFilter(emptyFilter)

      logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
      logHandler.setFormatter(logFormatter)
      self.logger.addHandler(logHandler)


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(str(message))
         sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(str(message))
         sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def setHelp(self,
               detail=None):
      """
      Request general help or a variety of more specific help.
      detail       submit arguments
      None         --help
      managers     --help managers
      tools        --help tools
      venues       --help venues
      examples     --help examples
      """
      setError = False
      if detail:
         if   detail == 'managers':
            self.helpManagers = True
         elif detail == 'tools':
            self.helpTools    = True
         elif detail == 'venues':
            self.helpVenues   = True
         elif detail == 'examples':
            self.helpExamples = True
         else:
            self.logger.log(logging.ERROR,getLogMessage("invalid help detail requested"))
            setError = True
      else:
         self.help = True

      return(setError)


   def resetHelp(self,
                 detail=None):
      """
      Disable requests for general help or a variety of more specific help.
      See setHelp() for detail values.
      """
      resetError = False
      if detail:
         if   detail == 'managers':
            self.helpManagers = False
         elif detail == 'tools':
            self.helpTools    = False
         elif detail == 'venues':
            self.helpVenues   = False
         elif detail == 'examples':
            self.helpExamples = False
         else:
            self.logger.log(logging.ERROR,getLogMessage("invalid help detail requested"))
            resetError = True
      else:
         self.help         = False
         self.helpManagers = False
         self.helpTools    = False
         self.helpVenues   = False
         self.helpExamples = False

      return(resetError)


   def __addHelpArgs(self):
      if self.help:
         self.args += ['--help']
      if self.helpManagers:
         self.args += ['--help','managers']
      if self.helpTools:
         self.args += ['--help','tools']
      if self.helpVenues:
         self.args += ['--help','venues']
      if self.helpExamples:
         self.args += ['--help','examples']


   def __saveHelp(self):
      self.savedCommand['help'] = []
      if self.help:
         self.savedCommand['help'].append('none')
      if self.helpManagers:
         self.savedCommand['help'].append('managers')
      if self.helpTools:
         self.savedCommand['help'].append('tools')
      if self.helpVenues:
         self.savedCommand['help'].append('venues')
      if self.helpExamples:
         self.savedCommand['help'].append('examples')
      if len(self.savedCommand['help']) == 0:
         del self.savedCommand['help']


   def __loadHelp(self):
      self.resetHelp()
      if 'help' in self.savedCommand:
         if 'none' in self.savedCommand['help']:
            self.setHelp()
         if 'managers' in self.savedCommand['help']:
            self.setHelp(detail='managers')
         if 'tools' in self.savedCommand['help']:
            self.setHelp(detail='tools')
         if 'venues' in self.savedCommand['help']:
            self.setHelp(detail='venues')
         if 'examples' in self.savedCommand['help']:
            self.setHelp(detail='examples')


   def setVersion(self,
                  detail=None):
      """
      Request complete or partial version information.
      detail       submit arguments
      None         --version
      client       --version client
      server       --version server
      distributor  --version distributor
      """
      setError = False
      if detail:
         if   detail == 'client':
            self.versionClient      = True
            self.version            = False
         elif detail == 'server':
            self.versionServer      = True
            self.version            = False
         elif detail == 'distributor':
            self.versionDistributor = True
            self.version            = False
         else:
            self.logger.log(logging.ERROR,getLogMessage("invalid version detail requested"))
            setError = True
      else:
         self.version            = True
         self.versionClient      = False
         self.versionServer      = False
         self.versionDistributor = False

      return(setError)


   def resetVersion(self,
                    detail=None):
      """
      Disable requests for version information.
      See setVersion() for detail values.
      """
      resetError = False
      if detail:
         if   detail == 'client':
            self.versionClient      = False
         elif detail == 'server':
            self.versionServer      = False
         elif detail == 'distributor':
            self.versionDistributor = False
         else:
            self.logger.log(logging.ERROR,getLogMessage("invalid version detail requested"))
            resetError = True
      else:
         self.version = False

      return(resetError)


   def __addVersionArgs(self):
      if self.version:
         self.args += ['--version']
      if self.versionClient:
         self.args += ['--version','client']
      if self.versionServer:
         self.args += ['--version','server']
      if self.versionDistributor:
         self.args += ['--version','distributor']


   def __saveVersion(self):
      self.savedCommand['version'] = []
      if self.version:
         self.savedCommand['version'].append('none')
      if self.versionClient:
         self.savedCommand['version'].append('client')
      if self.versionServer:
         self.savedCommand['version'].append('server')
      if self.versionDistributor:
         self.savedCommand['version'].append('distributor')
      if len(self.savedCommand['version']) == 0:
         del self.savedCommand['version']


   def __loadVersion(self):
      self.resetVersion()
      if 'version' in self.savedCommand:
         if 'none' in self.savedCommand['version']:
            self.setVersion()
         if 'client' in self.savedCommand['version']:
            self.setVersion(detail='client')
         if 'server' in self.savedCommand['version']:
            self.setVersion(detail='server')
         if 'distributor' in self.savedCommand['version']:
            self.setVersion(detail='distributor')


   def setDebug(self,
                debug=True):
      """
      Turn submit debug reporting on or off.
      """
      setError = False
      if isinstance(debug,bool):
         self.debug = debug
      else:
         self.logger.log(logging.ERROR,getLogMessage("debug must be a boolean"))
         setError = True

      return(setError)


   def resetDebug(self):
      """
      Turn submit debug reporting off.
      """
      resetError = False
      self.debug = False

      return(resetError)


   def __addDebugArg(self):
      if self.debug:
         self.args += ['--debug']


   def __saveDebug(self):
      if self.debug:
         self.savedCommand['debug'] = self.debug


   def __loadDebug(self):
      self.resetDebug()
      if 'debug' in self.savedCommand:
         self.debug = self.savedCommand['debug']


   def setLocal(self,
                local=True):
      """
      Turn submit local execution on or off.
      """
      setError = False
      if isinstance(local,bool):
         self.local = local
      else:
         self.logger.log(logging.ERROR,getLogMessage("local must be a boolean"))
         setError = True

      return(setError)


   def resetLocal(self):
      """
      Turn submit local execution off.
      """
      resetError = False
      self.local = False

      return(resetError)


   def __addLocalArg(self):
      if self.local:
         self.args += ['--local']


   def __saveLocal(self):
      if self.local:
         self.savedCommand['local'] = self.local


   def __loadLocal(self):
      self.resetLocal()
      if 'local' in self.savedCommand:
         self.local = self.savedCommand['local']


   def setQueryJobs(self,
                    queryJobs):
      """
      Do a query for the status of previously submitted runs.
      queryJobs can be given as a single integer id or list
      of integer ids.
      """
      setError = False
      if   isinstance(queryJobs,int):
         self.queryJobs = [queryJobs]
      elif isinstance(queryJobs,list):
         self.queryJobs = copy.copy(queryJobs)
      else:
         self.logger.log(logging.ERROR,getLogMessage("queryJobs must be an integer or list"))
         setError = True

      return(setError)


   def resetQueryJobs(self):
      """
      Disable query for the status of previously submitted runs.
      """
      resetError = False
      self.queryJobs = []

      return(resetError)


   def __addQueryJobsArgs(self):
      if self.queryJobs:
         for queryJob in self.queryJobs:
            self.args += ['--status',str(queryJob)]


   def __saveQueryJobs(self):
      if self.queryJobs:
         self.savedCommand['queryJobs'] = copy.copy(self.queryJobs)


   def __loadQueryJobs(self):
      self.resetQueryJobs()
      if 'queryJobs' in self.savedCommand:
         self.setQueryJobs(self.savedCommand['queryJobs'])


   def setKillJobs(self,
                   killJobs):
      """
      Kill previously submitted runs.
      killJobs can be given as a single integer id or list
      of integer ids.
      """
      setError = False
      if   isinstance(killJobs,int):
         self.killJobs = [killJobs]
      elif isinstance(killJobs,list):
         self.killJobs = copy.copy(killJobs)
      else:
         self.logger.log(logging.ERROR,getLogMessage("killJobs must be an integer or list"))
         setError = True

      return(setError)


   def resetKillJobs(self):
      """
      Disable killing of previously submitted runs.
      """
      resetError = False
      self.killJobs = []

      return(resetError)


   def __addKillJobsArgs(self):
      if self.killJobs:
         for killJob in self.killJobs:
            self.args += ['--kill',str(killJob)]


   def __saveKillJobs(self):
      if self.killJobs:
         self.savedCommand['killJobs'] = copy.copy(self.killJobs)


   def __loadKillJobs(self):
      self.resetKillJobs()
      if 'killJobs' in self.savedCommand:
         self.setKillJobs(self.savedCommand['killJobs'])


   try:
      isinstance("", basestring)
      def __isArgString(self,
                        arg):
         return isinstance(arg,basestring)
   except NameError:
      def __isArgString(self,
                        arg):
         return isinstance(arg,str)


   def setVenueStatus(self,
                      venueStatus):
      """
      Request status of submit venue named venueStatus.
      """
      setError = False
      if self.__isArgString(venueStatus):
         self.venueStatus = venueStatus
      else:
         self.logger.log(logging.ERROR,getLogMessage("venueStatus must be a string"))
         setError = True

      return(setError)


   def resetVenueStatus(self):
      """
      Disable request of submit venue status.
      """
      resetError = False
      self.venueStatus = ""

      return(resetError)


   def __addVenueStatusArgs(self):
      if self.venueStatus:
         self.args += ['--venueStatus',self.venueStatus]


   def __saveVenueStatus(self):
      if self.venueStatus:
         self.savedCommand['venueStatus'] = self.venueStatus


   def __loadVenueStatus(self):
      self.resetVenueStatus()
      if 'venueStatus' in self.savedCommand:
         self.setVenueStatus(self.savedCommand['venueStatus'])


   def setVenue(self,
                venue):
      """
      Request that run be submitted to "venue".
      """
      setError = False
      if self.__isArgString(venue):
         self.venues = [venue]
      else:
         self.logger.log(logging.ERROR,getLogMessage("venue must be a string"))
         setError = True

      return(setError)


   def __loadVenue(self):
      self.resetVenues()
      if 'venue' in self.savedCommand:
         self.setVenues(self.savedCommand['venue'])


   def setVenues(self,
                 venues):
      """
      Supply set of venues for submit run.
      venues can be specified as a single destination or
      of list of destinations.
      """
      setError = False
      if   self.__isArgString(venues):
         self.venues = [venues]
      elif isinstance(venues,list):
         self.venues = copy.copy(venues)
      else:
         self.logger.log(logging.ERROR,getLogMessage("venues must be a string or list"))
         setError = True

      return(setError)


   def resetVenues(self):
      """
      Remove venues from submit run.
      """
      resetError = False
      self.venues = []

      return(resetError)


   def __addVenuesArgs(self):
      if self.venues:
         for venue in self.venues:
            self.args += ['--venue',venue]


   def __saveVenues(self):
      if self.venues:
         self.savedCommand['venues'] = copy.copy(self.venues)


   def __loadVenues(self):
      self.resetVenues()
      if   'venues' in self.savedCommand:
         self.setVenues(self.savedCommand['venues'])
      elif 'venue' in self.savedCommand:
         self.setVenues(self.savedCommand['venue'])


   def setInputFiles(self,
                     inputFiles):
      """
      Supply set of input files for submit run.
      inputFiles can be specified as a single filename or
      of list of filenames.
      """
      setError = False
      if   self.__isArgString(inputFiles):
         self.inputFiles = [inputFiles]
      elif isinstance(inputFiles,list):
         self.inputFiles = copy.copy(inputFiles)
      else:
         self.logger.log(logging.ERROR,getLogMessage("inputFiles must be a string or list"))
         setError = True

      return(setError)


   def resetInputFiles(self):
      """
      Remove inputfiles from submit run.
      """
      resetError = False
      self.inputFiles = []

      return(resetError)


   def __addInputFilesArgs(self):
      if self.inputFiles:
         for inputFile in self.inputFiles:
            self.args += ['--inputfile',inputFile]


   def __saveInputFiles(self):
      if self.inputFiles:
         self.savedCommand['inputfiles'] = copy.copy(self.inputFiles)


   def __loadInputFiles(self):
      self.resetInputFiles()
      if 'inputfiles' in self.savedCommand:
         self.setInputFiles(self.savedCommand['inputfiles'])


   def setDefaultSeparator(self,
                           separatorDefault):
      """
      Set the default parameter separator.
      """
      setError = False
      if self.__isArgString(separatorDefault):
         if len(self.separator) > 1:
            self.logger.log(logging.ERROR,getLogMessage("separatorDefault must be a character"))
            setError = True
         else:
            self.separator = separatorDefault
      else:
         self.logger.log(logging.ERROR,getLogMessage("separatorDefault must be a character"))
         setError = True

      return(setError)


   def resetDefaultSeparator(self):
      """
      Set the default parameter separator to the system default.
      """
      resetError = False
      self.separator = self.SEPARATORDEFAULT

      return(resetError)


   def __saveDefaultSeparator(self):
      if self.separator != self.SEPARATORDEFAULT:
         self.savedCommand['defaultSeparator'] = self.separator


   def __loadDefaultSeparator(self):
      self.resetDefaultSeparator()
      if 'defaultSeparator' in self.savedCommand:
         self.setDefaultSeparator(self.savedCommand['defaultSeparator'])


   def setParameters(self,
                     parameters,
                     separator=None):
      """
      Supply set of parameters for submit run.
      parameters can be specified as a single string or
      of list of strings.  Previously set parameters are
      removed.
      """
      setError = False

      if separator:
         if self.__isArgString(separator):
            if len(separator) > 1:
               self.logger.log(logging.ERROR,getLogMessage("separator must be a character"))
               setError = True
         else:
            self.logger.log(logging.ERROR,getLogMessage("separator must be a character"))
            setError = True
      else:
         separator = self.separator

      if   self.__isArgString(parameters):
         if not setError:
            self.parameters = [parameters]
            self.separators = [separator]
      elif isinstance(parameters,list):
         if not setError:
            self.parameters = copy.copy(parameters)
            self.separators = len(parameters)*[separator]
      else:
         self.logger.log(logging.ERROR,getLogMessage("parameters must be a string or list"))
         setError = True

      return(setError)


   def addParameters(self,
                     parameters,
                     separator=None):
      """
      Supply additional set of parameters for submit run.
      parameters can be specified as a single string or
      of list of strings.
      """
      setError = False

      if separator:
         if self.__isArgString(separator):
            if len(separator) > 1:
               self.logger.log(logging.ERROR,getLogMessage("separator must be a character"))
               setError = True
         else:
            self.logger.log(logging.ERROR,getLogMessage("separator must be a character"))
            setError = True
      else:
         separator = self.separator

      if   self.__isArgString(parameters):
         if not setError:
            self.parameters += [parameters]
            self.separators += [separator]
      elif isinstance(parameters,list):
         if not setError:
            self.parameters += copy.copy(parameters)
            self.separators += len(parameters)*[separator]
      else:
         self.logger.log(logging.ERROR,getLogMessage("parameters must be a string or list"))
         setError = True

      return(setError)


   def resetParameters(self):
      """
      Remove parameters from submit run.
      """
      resetError = False
      self.parameters = []
      self.separators = []

      return(resetError)


   def __addParametersArgs(self):
      if self.parameters:
         previousSeparator = self.SEPARATORDEFAULT
         for separator,parameter in zip(self.separators,self.parameters):
            if separator != previousSeparator:
               self.args += ['--separator',separator]
            self.args += ['--parameters',parameter]
            previousSeparator = separator


   def __saveParameters(self):
      if self.parameters:
         if self.separators:
            self.savedCommand['separators'] = copy.copy(self.separators)
            self.savedCommand['parameters'] = copy.copy(self.parameters)


   def __loadParameters(self):
      self.resetParameters()
      if 'parameters' in self.savedCommand:
         if 'separators' in self.savedCommand:
            self.separators = copy.copy(self.savedCommand['separators'])
            self.parameters = copy.copy(self.savedCommand['parameters'])


   def setDataFile(self,
                   dataFile):
      """
      Supply parameter datafile for submit run.
      dataFile is a single filename.
      """
      setError = False
      if self.__isArgString(dataFile):
         self.dataFile = dataFile
      else:
         self.logger.log(logging.ERROR,getLogMessage("dataFile must be a string"))
         setError = True

      return(setError)


   def resetDataFile(self):
      """
      Remove parameter datafile for submit run.
      """
      resetError = False
      self.dataFile = ""

      return(resetError)


   def __addDataFileArgs(self):
      if self.dataFile:
         self.args += ['--data',self.dataFile]


   def __saveDataFile(self):
      if self.dataFile:
         self.savedCommand['data'] = self.dataFile


   def __loadDataFile(self):
      self.resetDataFile()
      if 'data' in self.savedCommand:
          self.setDataFile(self.savedCommand['data'])


   def setNcores(self,
                 nCores):
      """
      Set number of cores requested for submit run.
      If only one core is required no setting is necessary.
      """
      setError = False
      if isinstance(nCores,int):
         self.nCores = nCores
      else:
         self.logger.log(logging.ERROR,getLogMessage("nCores must be an integer"))
         setError = True

      return(setError)


   def resetNcores(self):
      """
      Set number of cores requested for submit run to the default value.
      """
      resetError = False
      self.nCores = None

      return(resetError)


   def __addNcoresArgs(self):
      if self.nCores:
         self.args += ['--nCpus',str(self.nCores)]


   def __saveNcores(self):
      if self.nCores:
         self.savedCommand['nCores'] = self.nCores


   def __loadNcores(self):
      self.resetNcores()
      if 'nCores' in self.savedCommand:
         self.setNcores(self.savedCommand['nCores'])


   def setPPN(self,
              ppn):
      """
      Set number of cores per node requested for submit run.
      If number of cores per node is not set for the submit run
      a default number of cores per node is set based on the
      venue configuration.  This default value is typical based
      on the node hardware.
      """
      setError = False
      if isinstance(ppn,int):
         self.ppn = ppn
      else:
         self.logger.log(logging.ERROR,getLogMessage("ppn must be an integer"))
         setError = True

      return(setError)


   def resetPPN(self):
      """
      Set number of cores per node requested for submit run to the default value.
      """
      resetError = False
      self.ppn = None

      return(resetError)


   def __addPPNArgs(self):
      if self.ppn:
         self.args += ['--ppn',str(self.ppn)]


   def __savePPN(self):
      if self.ppn:
         self.savedCommand['ppn'] =  self.ppn


   def __loadPPN(self):
      self.resetPPN()
      if 'ppn' in self.savedCommand:
         self.setPPN(self.savedCommand['ppn'])


   def setNstripes(self,
                   nStripes):
      """
      Set number of parallel stripes to be used in a local
      parametric sweep run.
      """
      setError = False
      if isinstance(nStripes,int):
         self.nStripes = nStripes
      else:
         self.logger.log(logging.ERROR,getLogMessage("nStripes must be an integer"))
         setError = True

      return(setError)


   def resetNstripes(self):
      """
      Set number of parallel stripes to be used in a local
      parametric sweep run to the default value.
      """
      resetError = False
      self.nStripes = None

      return(resetError)


   def __addNstripesArgs(self):
      if self.nStripes:
         self.args += ['--stripes',str(self.nStripes)]


   def __saveNstripes(self):
      if self.nStripes:
         self.savedCommand['nStripes'] = self.nStripes


   def __loadNstripes(self):
      self.resetNstripes()
      if 'nStripes' in self.savedCommand:
         self.setNstripes(self.savedCommand['nStripes'])


   def setWallTime(self,
                   wallTime):
      """
      Set the wall time limit for batch queue runs.
      wallTime can be given as integer or floating
      point number of minutes.
      """
      setError = False
      if   isinstance(wallTime,float):
         self.wallTime = wallTime
      elif isinstance(wallTime,int):
         self.wallTime = wallTime
      else:
         self.logger.log(logging.ERROR,getLogMessage("wallTime must be number"))
         setError = True

      return(setError)


   def resetWallTime(self):
      """
      Set wall time requested for submit run to the default value.
      """
      resetError = False
      self.wallTime = None

      return(resetError)


   def __addWallTimeArgs(self):
      if self.wallTime:
         self.args += ['--wallTime',str(self.wallTime)]


   def __saveWallTime(self):
      if self.wallTime:
         self.savedCommand['wallTime'] = self.wallTime


   def __loadWallTime(self):
      self.resetWallTime()
      if 'wallTime' in self.savedCommand:
         self.setWallTime(self.savedCommand['wallTime'])


   def setEnvironmentVariables(self,
                               environmentVariables):
      """
      Provide a set of environment variables to be set
      for submit run.  environmentVariables should be a
      dictionary with keys being the environment variable
      names.
      """
      setError = False
      if isinstance(environmentVariables,dict):
         self.environmentVariables = copy.copy(environmentVariables)
      else:
         self.logger.log(logging.ERROR,getLogMessage("environmentVariables must be a dictionary"))
         setError = True

      return(setError)


   def resetEnvironmentVariables(self):
      """
      Remove all environment variables set for submit run.
      """
      resetError = False
      self.environmentVariables = {}

      return(resetError)


   def __addEnvironmentVariablesArgs(self):
      if self.environmentVariables:
         for environmentVariable,value in self.environmentVariables.items():
            if value:
               self.args += ['--env','='.join([environmentVariable,value])]
            else:
               self.args += ['--env',environmentVariable]


   def __saveEnvironmentVariables(self):
      if self.environmentVariables:
         self.savedCommand['environmentVariables'] = copy.copy(self.environmentVariables)


   def __loadEnvironmentVariables(self):
      self.resetEnvironmentVariables()
      if 'environmentVariables' in self.savedCommand:
         self.environmentVariables = copy.copy(self.savedCommand['environmentVariables'])


   def setRunName(self,
                  runName):
      """
      Set submit run name.
      Default run name is the auto generated submit jobId.
      The runName is used to set the standard output and
      standard error filenames.
      """
      setError = False
      if self.__isArgString(runName):
         self.runName = runName
      else:
         self.logger.log(logging.ERROR,getLogMessage("runName must be a string"))
         setError = True

      return(setError)


   def resetRunName(self):
      """
      Set submit run name to default name.
      """
      resetError = False
      self.runName = ""

      return(resetError)


   def __addRunNameArgs(self):
      if self.runName:
         self.args += ['--runName',self.runName]


   def __saveRunName(self):
      if self.runName:
         self.savedCommand['runName'] = self.runName


   def __loadRunName(self):
      self.resetRunName()
      if 'runName' in self.savedCommand:
         self.setRunName(self.savedCommand['runName'])


   def setManager(self,
                  manager):
      """
      Set manager for submit run.
      By default manager is set by the venue or tool configuration.
      """
      setError = False
      if self.__isArgString(manager):
         self.manager = manager
      else:
         self.logger.log(logging.ERROR,getLogMessage("manager must be a string"))
         setError = True

      return(setError)


   def resetManager(self):
      """
      Set manager for submit run to the default manager.
      """
      resetError = False
      self.manager = ""

      return(resetError)


   def __addManagerArgs(self):
      if self.manager:
         self.args += ['--manager',self.manager]


   def __saveManager(self):
      if self.manager:
         self.savedCommand['manager'] = self.manager


   def __loadManager(self):
      self.resetManager()
      if 'manager' in self.savedCommand:
         self.setManager(self.savedCommand['manager'])


   def setRedundancy(self,
                     redundancy):
      """
      Set redundancy factor for submit run.
      The default redundancy factor is set by submit configuration.
      """
      setError = False
      if isinstance(redundancy,int):
         self.redundancy = redundancy
      else:
         self.logger.log(logging.ERROR,getLogMessage("redundancy must be an integer"))
         setError = True

      return(setError)


   def resetRedundancy(self):
      """
      Set redundancy factor for submit run to the default value.
      """
      resetError = False
      self.redundancy = None

      return(resetError)


   def __addRedundancyArgs(self):
      if self.redundancy:
         self.args += ['--redundancy',str(self.redundancy)]


   def __saveRedundancy(self):
      if self.redundancy:
         self.savedCommand['redundancy'] = self.redundancy


   def __loadRedundancy(self):
      self.resetRedundancy()
      if 'redundancy' in self.savedCommand:
         self.setRedundancy(self.savedCommand['redundancy'])


   def setReportMetrics(self,
                        reportMetrics=True):
      """
      Turn submit resource metrics reporting on or off.
      """
      setError = False
      if isinstance(reportMetrics,bool):
         self.reportMetrics = reportMetrics
      else:
         self.logger.log(logging.ERROR,getLogMessage("reportMetrics must be a boolean"))
         setError = True

      return(setError)


   def resetReportMetrics(self):
      """
      Turn submit resource metrics reporting off.
      """
      resetError = False
      self.reportMetrics = False

      return(resetError)


   def __addReportMetricsArg(self):
      if self.reportMetrics:
         self.args += ['--metrics']


   def __saveReportMetrics(self):
      if self.reportMetrics:
         self.savedCommand['reportMetrics'] = self.reportMetrics


   def __loadReportMetrics(self):
      self.resetReportMetrics()
      if 'reportMetrics' in self.savedCommand:
         self.setReportMetrics(self.savedCommand['reportMetrics'])


   def setDetach(self,
                 detach=True):
      """
      Enable or disable detachment from submit run.
      If a job is detached you can continue with other
      operations and not wait for the submit to complete.
      Job status will need to be monitored to determine
      when completion occurs.  A detached can be reattached
      at a later time.
      """
      setError = False
      if isinstance(detach,bool):
         self.detach = detach
      else:
         self.logger.log(logging.ERROR,getLogMessage("detach must be a boolean"))
         setError = True

      return(setError)


   def resetDetach(self):
      """
      Disable detachment from submit run.
      """
      resetError = False
      self.detach = False

      return(resetError)


   def __addDetachArg(self):
      if self.detach:
         self.args += ['--detach']


   def __saveDetach(self):
      if self.detach:
         self.savedCommand['detach'] = self.detach


   def __loadDetach(self):
      self.resetDetach()
      if 'detach' in self.savedCommand:
         self.setDetach(self.savedCommand['detach'])


   def setAttachId(self,
                   attachId):
      """
      Reattach to a previously detached submit run.
      """
      setError = False
      if isinstance(attachId,int):
         self.attachId = attachId
      else:
         self.logger.log(logging.ERROR,getLogMessage("attachId must be an integer"))
         setError = True

      return(setError)


   def resetAttachId(self):
      """
      Do not reattach to a previously detached submit run.
      """
      resetError = False
      self.attachId = None

      return(resetError)


   def __addAttachIdArgs(self):
      if self.attachId:
         self.args += ['--attach',self.attachId]


   def __saveAttachId(self):
      if self.attachId:
         self.savedCommand['attachId'] = self.attachId


   def __loadAttachId(self):
      self.resetAttachId()
      if 'attachId' in self.savedCommand:
         self.setAttachId(self.savedCommand['attachId'])


   def setWait(self,
               wait=True):
      """
      Wait for period of reduced run submission rate to submit run.
      Users have a limited rate at which they are allowed to submit jobs.
      The measured rate of run submission is measured over time with
      more attention payed to the most recent submission.
      """
      setError = False
      if isinstance(wait,bool):
         self.wait = wait
      else:
         self.logger.log(logging.ERROR,getLogMessage("wait must be a boolean"))
         setError = True

      return(setError)


   def resetWait(self):
      """
      Do not wait for period of reduced run submission rate to submit run.
      If your measured rate of job submission is to high your request for
      a new submit run will be denied.
      """
      resetError = False
      self.wait = False

      return(resetError)


   def __addWaitArg(self):
      if self.wait:
         self.args += ['--wait']


   def __saveWait(self):
      if self.wait:
         self.savedCommand['wait'] = self.wait


   def __loadWait(self):
      self.resetWait()
      if 'wait' in self.savedCommand:
         self.setWait(self.savedCommand['wait'])


   def setQuota(self,
                quota):
      """
      Enable or disable quota limit for submit run.
      If enabled your HUB disk quota is used to limit data
      generation on the remote resource.  This property is
      enabled by default.
      """ 
      setError = False
      if isinstance(quota,bool):
         self.quota = quota
      else:
         self.logger.log(logging.ERROR,getLogMessage("quota must be a boolean"))
         setError = True

      return(setError)


   def resetQuota(self):
      """
      Enable quota limit for submit run.
      """
      resetError = False
      self.quota = True

      return(resetError)


   def __addQuotaArg(self):
      if not self.quota:
         self.args += ['--noquota']


   def __saveQuota(self):
      if not self.quota:
         self.savedCommand['quota'] = self.quota


   def __loadQuota(self):
      self.resetQuota()
      if 'quota' in self.savedCommand:
         self.setQuota(self.savedCommand['quota'])


   def setDefaultTailNlines(self,
                            tailNlinesDefault):
      """
      Set the default number of lines to report when
      tailing output files.
      """
      setError = False
      if isinstance(tailNlinesDefault,int):
         self.tailNlinesDefault = tailNlinesDefault
      else:
         self.logger.log(logging.ERROR,getLogMessage("tailNlinesDefault must be an integer"))
         setError = True

      return(setError)


   def resetDefaultTailNlines(self):
      """
      Set the default number of lines to report when
      tailing output files to the system default.
      """
      resetError = False
      self.tailNlinesDefault = self.TAILNLINESDEFAULT

      return(resetError)


   def __saveDefaultTailNlines(self):
      if self.tailNlinesDefault != self.TAILNLINESDEFAULT:
         self.savedCommand['tailNlinesDefault'] = self.tailNlinesDefault


   def __loadDefaultTailNlines(self):
      self.resetDefaultTailNlines()
      if 'tailNlinesDefault' in self.savedCommand:
         self.setDefaultTailNlines(self.savedCommand['tailNlinesDefault'])


   def setTailStdout(self,
                     tailStdout=True,
                     tailStdoutNlines=None):
      """
      Enable or disable real time reporting of standard output file.
      """
      setError = False
      if isinstance(tailStdout,bool):
         self.tailStdout = tailStdout
         if self.tailStdout:
            if tailStdoutNlines:
               self.tailStdoutNlines = tailStdoutNlines
            else:
               self.tailStdoutNlines = self.tailNlinesDefault
      else:
         self.logger.log(logging.ERROR,getLogMessage("tailStdout must be a boolean"))
         setError = True

      return(setError)


   def resetTailStdout(self):
      """
      Disable real time reporting of standard output file.
      """
      resetError = False
      self.tailStdout = False

      return(resetError)


   def __addTailStdoutArg(self):
      if self.tailStdout:
         self.args += ['--tailStdout','%d' % (self.tailStdoutNlines)]


   def __saveTailStdout(self):
      if self.tailStdout:
         self.savedCommand['tailStdout']       = self.tailStdout
         self.savedCommand['tailStdoutNlines'] = self.tailStdoutNlines


   def __loadTailStdout(self):
      self.resetTailStdout()
      if 'tailStdout' in self.savedCommand:
         if 'tailStdoutNlines' in self.savedCommand:
            self.setTailStdout(self.savedCommand['tailStdout'],
                               self.savedCommand['tailStdoutNlines'])
         else:
            self.setTailStdout(self.savedCommand['tailStdout'])


   def setTailStderr(self,
                     tailStderr=True,
                     tailStderrNlines=None):
      """
      Enable or disable real time reporting of standard error file.
      """
      setError = False
      if isinstance(tailStderr,bool):
         self.tailStderr = tailStderr
         if self.tailStderr:
            if tailStderrNlines:
               self.tailStderrNlines = tailStderrNlines
            else:
               self.tailStderrNlines = self.tailNlinesDefault
      else:
         self.logger.log(logging.ERROR,getLogMessage("tailStderr must be a boolean"))
         setError = True

      return(setError)


   def resetTailStderr(self):
      """
      Disable real time reporting of standard error file.
      """
      resetError = False
      self.tailStderr = False

      return(resetError)


   def __addTailStderrArg(self):
      if self.tailStderr:
         self.args += ['--tailStderr','%d' % (self.tailStderrNlines)]


   def __saveTailStderr(self):
      if self.tailStderr:
         self.savedCommand['tailStderr']       = self.tailStderr
         self.savedCommand['tailStderrNlines'] = self.tailStderrNlines


   def __loadTailStderr(self):
      self.resetTailStderr()
      if 'tailStderr' in self.savedCommand:
         if 'tailStderrNlines' in self.savedCommand:
            self.setTailStderr(self.savedCommand['tailStderr'],
                               self.savedCommand['tailStderrNlines'])
         else:
            self.setTailStderr(self.savedCommand['tailStderr'])


   def setTailFiles(self,
                    tailFiles):
      """
      Enable real time reporting of files other than
      standard output and standard error.
      tailFiles can be specified as a single filename
      or a list of filenames.  To specify the number (#)
      of lines to report add :# to the filename.
      """
      setError = False
      if   self.__isArgString(tailFiles):
         self.tailFiles = [tailFiles]
      elif isinstance(tailFiles,list):
         self.tailFiles = copy.copy(tailFiles)
      else:
         self.logger.log(logging.ERROR,getLogMessage("tailFiles must be a string or list"))
         setError = True

      return(setError)


   def resetTailFiles(self):
      """
      Disable real time reporting of files other than
      standard output and standard error.
      """
      resetError = False
      self.tailFiles = []

      return(resetError)


   def __addTailFilesArgs(self):
      if self.tailFiles:
         for tailFile in self.tailFiles:
            self.args += ['--tail',tailFile]


   def __saveTailFiles(self):
      if self.tailFiles:
         self.savedCommand['tailFiles'] = copy.copy(self.tailFiles)


   def __loadTailFiles(self):
      self.resetTailFiles()
      if 'tailFiles' in self.savedCommand:
         self.setTailFiles(self.savedCommand['tailFiles'])


   def setProgress(self,
                   detail=None):
      """
      Specify progress reporting style.
      detail       submit arguments
      None 
      curses       --progress curses  
      submit       --progress submit
      text         --progress text
      pegasus      --progress pegasus
      silent       --progress silent
      """
      setError = False
      if detail:
         if   detail == 'auto':
            self.progress = None
         elif detail == 'curses':
            self.progress = detail
         elif detail == 'submit':
            self.progress = detail
         elif detail == 'text':
            self.progress = detail
         elif detail == 'pegasus':
            self.progress = detail
         elif detail == 'silent':
            self.progress = detail
         else:
            self.logger.log(logging.ERROR,getLogMessage("invalid progress detail requested"))
            setError = True
      else:
         self.progress = None

      return(setError)


   def resetProgress(self):
      """
      Specify progress reporting to the default style.
      """
      resetError = False
      self.progress = None

      return(resetError)


   def __addProgressArgs(self):
      if   self.progress == 'curses':
         self.args += ['--progress','curses']
      elif self.progress == 'submit':
         self.args += ['--progress','submit']
      elif self.progress == 'text':
         self.args += ['--progress','text']
      elif self.progress == 'pegasus':
         self.args += ['--progress','pegasus']
      elif self.progress == 'silent':
         self.args += ['--progress','silent']


   def __saveProgress(self):
      if self.progress:
         self.savedCommand['progress'] = self.progress


   def __loadProgress(self):
      self.resetProgress()
      if 'progress' in self.savedCommand:
         self.setProgress(detail=self.savedCommand['progress'])


   def setCommand(self,
                  command):
      """
      Specify command to be run on remote resource.
      command can be given as a single string or a list
      of strings.  A single string will be split to form
      a list.  The first list element is taken to be the
      command and additional elements are taken to be
      command arguments.
      """
      setError = False
      commandLine = None
      if   self.__isArgString(command):
         try:
            commandLine = shlex.split(command)
         except:
            self.logger.log(logging.ERROR,getLogMessage("command is not a valid string"))
            setError = True
      elif isinstance(command,list):
         commandLine = command
      else:
         self.logger.log(logging.ERROR,getLogMessage("command must be a string or list"))
         setError = True

      if commandLine:
         self.command = copy.copy(commandLine[0])
         self.commandArguments = copy.copy(commandLine[1:])

      return(setError)


   def resetCommand(self):
      """
      Remove previous command and command argument settings.
      """
      resetError = False
      self.command = ""
      self.commandArguments = []

      return(resetError)


   def __addCommandArg(self):
      if self.command:
         self.args += [self.command]


   def __saveCommand(self):
      if self.command:
         self.savedCommand['command'] = self.command


   def __loadCommand(self):
      self.resetCommand()
      if 'command' in self.savedCommand:
         self.setCommand(self.savedCommand['command'])


   def setCommandArguments(self,
                           commandArguments):
      """
      Specify command arguments to be used on remote resource.
      commandArguments can be given as a single string or a list
      of strings.  A single string will be split to form
      a list.
      """
      setError = False
      if   self.__isArgString(commandArguments):
         try:
            self.commandArguments = shlex.split(commandArguments)
         except:
            self.logger.log(logging.ERROR,getLogMessage("commandArguments is not a valid string"))
            setError = True
      elif isinstance(commandArguments,list):
         self.commandArguments = copy.copy(commandArguments)
      else:
         self.logger.log(logging.ERROR,getLogMessage("commandArguments must be a string or list"))
         setError = True

      return(setError)


   def resetCommandArguments(self):
      """
      Remove previous command argument settings.
      """
      resetError = False
      self.commandArguments = []

      return(resetError)


   def __addCommandArgumentsArgs(self):
      if self.commandArguments:
         self.args += self.commandArguments


   def __saveCommandArguments(self):
      if self.commandArguments:
         self.savedCommand['commandArguments'] = copy.copy(self.commandArguments)


   def __loadCommandArguments(self):
      self.resetCommandArguments()
      if 'commandArguments' in self.savedCommand:
         self.setCommandArguments(self.savedCommand['commandArguments'])


   def setStdin(self,
                stdinPath):
      """
      Specify path to stdin file.
      """
      setError = False
      if self.__isArgString(stdinPath):
         self.stdin = stdinPath
      else:
         self.logger.log(logging.ERROR,getLogMessage("stdinPath must be a string"))
         setError = True

      return(setError)


   def resetStdin(self):
      """
      Remove previous stdin settings.
      """
      resetError = False
      self.stdin = None

      return(resetError)


   def __saveStdin(self):
      if self.stdin:
         self.savedCommand['stdin'] = self.stdin


   def __loadStdin(self):
      self.resetStdin()
      if 'stdin' in self.savedCommand:
         self.setStdin(self.savedCommand['stdin'])


   def __buildCommandArgumentsArgs(self,
                                   args):
      del self.args
      self.args = []

      if args:
         if isinstance(args,list):
            self.args = args
         else:
            self.logger.log(logging.ERROR,getLogMessage("args must be a list"))
      else:
         self.__addHelpArgs()
         if len(self.args) == 0:
            self.__addVersionArgs()
            if len(self.args) == 0:
               self.__addDebugArg()
               self.__addQueryJobsArgs()
               self.__addKillJobsArgs()
               self.__addVenueStatusArgs()
               self.__addReportMetricsArg()
               self.__addRedundancyArgs()
               self.__addRunNameArgs()
               if self.local:
                  self.__addLocalArg()
                  self.__addNstripesArgs()
               else:
                  self.__addVenuesArgs()
                  self.__addNcoresArgs()
                  self.__addPPNArgs()
                  self.__addWallTimeArgs()
               self.__addParametersArgs()
               self.__addInputFilesArgs()
               self.__addDetachArg()
               self.__addAttachIdArgs()
               self.__addWaitArg()
               self.__addDataFileArgs()
               self.__addEnvironmentVariablesArgs()
               self.__addManagerArgs()
               self.__addTailStdoutArg()
               self.__addTailStderrArg()
               self.__addTailFilesArgs()
               self.__addProgressArgs()
               self.__addQuotaArg()
               self.__addCommandArg()
               self.__addCommandArgumentsArgs()


   def resetSubmitCommand(self):
      """
      Reset submit command settings to default values.
      """
      self.__resetHelp()
      self.__resetVersion()
      self.__resetDebug()
      self.__resetLocal()
      self.__resetQueryJobs()
      self.__resetKillJobs()
      self.__resetVenueStatus()
      self.__resetVenues()
      self.__resetInputFiles()
      self.__resetDefaultSeparator()
      self.__resetParameters()
      self.__resetDataFile()
      self.__resetNcores()
      self.__resetPPN()
      self.__resetNstripes()
      self.__resetWallTime()
      self.__resetEnvironmentVariables()
      self.__resetRunName()
      self.__resetManager()
      self.__resetRedundancy()
      self.__resetReportMetrics()
      self.__resetDetach()
      self.__resetAttachId()
      self.__resetWait()
      self.__resetQuota()
      self.__resetDefaultTailNlines()
      self.__resetTailStdout()
      self.__resetTailStderr()
      self.__resetTailFiles()
      self.__resetProgress()
      self.__resetCommand()
      self.__resetCommandArguments()
      self.__resetStdin()


   def saveSubmitCommand(self,
                         submitCommandJSONFile=None,
                         indent=None,
                         sortKeys=False):
      """
      Save JSON file containing submit command settings.
      """
      saveError = False
      del self.savedCommand
      self.savedCommand = {}

      self.__saveHelp()
      self.__saveVersion()
      self.__saveDebug()
      self.__saveLocal()
      self.__saveQueryJobs()
      self.__saveKillJobs()
      self.__saveVenueStatus()
      self.__saveVenues()
      self.__saveInputFiles()
      self.__saveDefaultSeparator()
      self.__saveParameters()
      self.__saveDataFile()
      self.__saveNcores()
      self.__savePPN()
      self.__saveNstripes()
      self.__saveWallTime()
      self.__saveEnvironmentVariables()
      self.__saveRunName()
      self.__saveManager()
      self.__saveRedundancy()
      self.__saveReportMetrics()
      self.__saveDetach()
      self.__saveAttachId()
      self.__saveWait()
      self.__saveQuota()
      self.__saveDefaultTailNlines()
      self.__saveTailStdout()
      self.__saveTailStderr()
      self.__saveTailFiles()
      self.__saveProgress()
      self.__saveCommand()
      self.__saveCommandArguments()
      self.__saveStdin()

      jsonMessage = json.dumps(self.savedCommand,indent=indent,sort_keys=sortKeys) + '\n'
      if submitCommandJSONFile:
         try:
            fpJSONfile = open(submitCommandJSONFile,'w')
            try:
               fpJSONfile.write(jsonMessage)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (submitCommandJSONFile)))
               saveError = True
            finally:
               fpJSONfile.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (submitCommandJSONFile)))
            saveError = True
      else:
         self.__writeToStdout(jsonMessage)

      return(saveError)


   def loadSubmitCommand(self,
                         submitCommandJSONFile):
      """
      Load JSON file containing submit command settings.
      """
      loadError = False
      del self.savedCommand
      self.savedCommand = {}
      try:
         fpJSONfile = open(submitCommandJSONFile,'r')
         try:
            self.savedCommand = json.load(fpJSONfile)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (submitCommandJSONFile)))
            loadError = True
         except ValueError:
            self.logger.log(logging.ERROR,getLogMessage("%s JSON object could not be decoded" % (submitCommandJSONFile)))
            loadError = True
         finally:
            fpJSONfile.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (submitCommandJSONFile)))
         loadError = True

      if not loadError:
         self.__loadHelp()
         self.__loadVersion()
         self.__loadDebug()
         self.__loadLocal()
         self.__loadQueryJobs()
         self.__loadKillJobs()
         self.__loadVenueStatus()
         self.__loadVenues()
         self.__loadInputFiles()
         self.__loadDefaultSeparator()
         self.__loadParameters()
         self.__loadDataFile()
         self.__loadNcores()
         self.__loadPPN()
         self.__loadNstripes()
         self.__loadWallTime()
         self.__loadEnvironmentVariables()
         self.__loadRunName()
         self.__loadManager()
         self.__loadRedundancy()
         self.__loadReportMetrics()
         self.__loadDetach()
         self.__loadAttachId()
         self.__loadWait()
         self.__loadQuota()
         self.__loadDefaultTailNlines()
         self.__loadTailStdout()
         self.__loadTailStderr()
         self.__loadTailFiles()
         self.__loadProgress()
         self.__loadCommand()
         self.__loadCommandArguments()
         self.__loadStdin()

      return(loadError)


   def show(self,
            args=None,
            textWidth=80):
      """
      Show submit command as determined from previous settings.
      If args is supplied previous settings are ignored but not
      replaced or overwritten.
      """
      self.__buildCommandArgumentsArgs(args)

      if self.args:
         command = ['submit'] + self.args
      else:
         command = ['submit']

      commandLine = ' '.join(command)
      prettyCommand = textwrap.fill(commandLine,textWidth,subsequent_indent='       ',
                                    break_long_words=False,break_on_hyphens=False)
      self.__writeToStdout(prettyCommand + '\n')


   def submit(self,
              args=None,
              stdin=None):
      """
      Execute submit command as determined from previous settings.
      If args is supplied previous settings are ignored but not
      replaced or overwritten.
      """
      result = None
      exitCode = 0

      userStdin = stdin
      if not userStdin:
         if args:
            userStdin = sys.stdin
         else:
            userStdin = self.stdin
      if not userStdin:
         userStdin = os.devnull

      submitStdin = None
      try:
         isFile = isinstance(userStdin,file)
      except:
         isFile = isinstance(userStdin,IOBase)

      if   isFile:
         submitStdin = userStdin
      elif isinstance(userStdin,int):
         submitStdin = os.open(userStdin,'r',closefd=False)
      elif type(userStdin).__name__ == 'OutStream':
         if userStdin.closed:
            stdinPath = os.devnull
            try:
               fpStdin = open(stdinPath,'r')
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("File %s could not be opened.\n" % (stdinPath)))
               exitCode = 1
            else:
               submitStdin = fpStdin
         else:
            submitStdin = userStdin
      else:
         if   userStdin == None:
            stdinPath = os.devnull
         else:
            try:
               stdinPath = str(userStdin)
            except UnicodeEncodeError:
               try:
                  stdinPath = unicode(userStdin).encode('unicode_escape')
               except:
                  stdinPath = None
            except:
               stdinPath = None
         if stdinPath:
            if stdinPath == os.devnull or os.path.isfile(stdinPath):
               try:
                  fpStdin = open(stdinPath,'r')
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("File %s could not be opened.\n" % (stdinPath)))
                  exitCode = 1
               else:
                  submitStdin = fpStdin
            else:
               self.logger.log(logging.ERROR,getLogMessage("File %s does not exist.\n" % (stdinPath)))
               exitCode = 1
         else:
            self.logger.log(logging.ERROR,getLogMessage("Bad argument type specified for stdin.\n"))
            exitCode = 1

      if not exitCode:
         if self.submitClient:
            del self.submitClient
         stdinSaved = sys.stdin
         sys.stdin = submitStdin
         self.submitClient = SubmitClient(self.configFilePath)

         self.__buildCommandArgumentsArgs(args)

         if self.args:
            if self.submitClient.configure():
               self.submitClient.setCommandArguments([self.instantiatorPath] + self.args)
               (parseCommandError,contactServer) = self.submitClient.parseCommandArguments()
               if not parseCommandError:
                  if not self.submitClient.reportDebugOutput():
                     logSetMessageFile(None)
                  else:
                     self.logger.setLevel(logging.DEBUG)
                  if contactServer:
                     self.submitClient.getUserAttributes()
                     if self.submitClient.connectToServer('execute'):
                        result = self.submitClient.run()
                     else:
                        exitCode = 1
               else:
                  self.submitClient.sendUserErrorNotification("Command line argument parsing failed.")
                  exitCode = 1
            else:
               self.submitClient.sendUserErrorNotification("submit configuration failed.")
               exitCode = 1
         else:
            self.submitClient.sendUserErrorNotification("No command line provided.")
            exitCode = 1
         sys.stdin = stdinSaved

      if not result:
         result = {}
         result['exitCode']           = exitCode
         result['serverDisconnected'] = False

      return(result)


