# @package      hubzero-submit-client
# @file         LocalWorkflowPARALLEL.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate

class LocalWorkflowPARALLEL:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                jobDirectory,
                scratchDirectory,
                useSetup,
                nStripes,
                localJobId,
                instanceId,
                submissionScriptsInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.jobDirectory          = jobDirectory
      self.scratchDirectory      = scratchDirectory
      self.useSetup              = useSetup
      self.nStripes              = nStripes
      self.localJobId            = localJobId
      if int(instanceId) > 0:
         self.instanceId         = instanceId
      else:
         self.instanceId         = None

      self.makefilePath          = os.path.join(self.jobDirectory,"%s.make" % (self.localJobId))
      self.executable            = os.path.join(os.sep,"usr","bin","make")
      self.arguments             = "-j %d -f %s all" % (self.nStripes,self.makefilePath)

      self.submissionScriptsInfo = submissionScriptsInfo
      self.timestampStart        = os.path.join(self.jobDirectory,timeHistoryLogs['timestampStart'])
      self.timestampFinish       = os.path.join(self.jobDirectory,timeHistoryLogs['timestampFinish'])
      self.timeResults           = os.path.join(self.jobDirectory,timeHistoryLogs['timeResults'])


   def __buildWorkflowScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Workflow','PARALLEL','make')

      auxiliaryArguments = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      substitutions = {}
      substitutions["JOBID"]             = self.localJobId
      substitutions["RUNNAME"]           = self.runName
      substitutions["INSTANCEDIRECTORY"] = self.jobDirectory
      substitutions["SCRATCHDIRECTORY"]  = self.scratchDirectory
      substitutions["EXECUTABLE"]        = self.executable
      arguments = ' '.join(auxiliaryArguments)
      substitutions["ARGUMENTS"]         = arguments.strip()
      substitutions["TS_START"]          = self.timestampStart
      substitutions["TS_FINISH"]         = self.timestampFinish
      substitutions["TIME_RESULTS"]      = self.timeResults
      substitutions["HUBUSERNAME"]       = self.hubUserName
      substitutions["HUBUSERID"]         = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildMakefile(self):
      instanceIds = []
      dirFiles = os.listdir(self.jobDirectory)
      for dirFile in dirFiles:
         dirPath = os.path.join(self.jobDirectory,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      instanceTargets = ""
      allTargets      = "all:"
      targetsOnLine   = 0

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(self.jobDirectory,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles = list(filter(reAppScriptPath.search,dirFiles))

         if len(appScriptFiles) == 1:
            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            jobId            = "%s_%s" % (self.localJobId,instanceId)
            targetStdout     = os.path.join(instanceId,"%s_%s.stdout" % (self.runName,instanceId))
            targetExecutable = os.path.join(instanceId,"%s.sh" % (jobId))
           
            instanceTargets += """%s:\n\t@echo "Executing JOB %s"\n\t@%s\n\t@echo "Completed JOB %s"\n\n""" % \
                                                                    (targetStdout,jobId,targetExecutable,jobId)
            allTargets      += """ %s""" % (targetStdout)
            targetsOnLine   += 1
            if targetsOnLine == 3:
               allTargets   += """ \\\n    """
               targetsOnLine = 0
      allTargets = allTargets.strip()
      allTargets = allTargets.strip(" \\")

      try:
         fpMakefile = open(self.makefilePath,'w')
         try:
            fpMakefile.write("%s%s\n" % (instanceTargets,allTargets))
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.makefilePath)))
         finally:
            fpMakefile.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.makefilePath)))


   def buildWorkflowScript(self):
      workflowScriptName = "%s.parallel" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildMakefile()
      workflowScriptExecutable = True

      return(workflowScriptName,workflowScript,workflowScriptExecutable)


