# @package      hubzero-submit-client
# @file         SubmitClient.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
__version__ = '3.7.4'

import os
import sys
import fcntl
import re
import uuid
import hashlib
import shutil
import shlex
import select
import socket
import time
import signal
import struct
import resource
import subprocess
import traceback
import math
import random
import stat
import tempfile
import csv
import json
import pprint
import base64
from errno import EPIPE
from io import IOBase
import logging
import warnings

def showWarning(message, category, filename, lineno, file=None, line=None):
   return

_default_show_warning = warnings.showwarning

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage, logSetMessageFile, logSetJobId
from hubzero.submit.LogMessage             import isLogMessageFileStdout, isLogMessageFileStderr, isLoggingOn
from hubzero.submit.CommandParser          import CommandParser
from hubzero.submit.ParameterTemplate      import ParameterTemplate
from hubzero.submit.LocalBatchAppScript    import LocalBatchAppScript
from hubzero.submit.JobScanner             import JobScanner
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.TarCommand             import buildCreate as buildCreateTarCommand
from hubzero.submit.UnboundConnection      import UnboundConnection
from hubzero.submit.ClientIdAuthAttributes import ClientIdAuthAttributes
from hubzero.submit.SubmissionScriptsInfo  import SubmissionScriptsInfo

TIMESTAMPINPUTBUILT  = ".__timestamp_inputbuilt"
TIMESTAMPINPUTSTAGED = ".__timestamp_inputstaged"
TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"
EXITCODE             = ".__exit_code"

if sys.version_info > (3,):
   long = int
ZERO = long(0)

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class SubmitClient:
   AUTHORIZEDACTIONNONE             = 0
   AUTHORIZEDACTIONEXECUTE          = 1 << 0
   AUTHORIZEDACTIONHEARTBEAT        = 1 << 1
   AUTHORIZEDACTIONSENDMETRICS      = 1 << 2
   AUTHORIZEDACTIONHARVEST          = 1 << 3
   AUTHORIZEDACTIONSENDMEASUREMENTS = 1 << 4

   def __init__(self,
                configFilePath):
      self.logger         = logging.getLogger(__name__)
      self.configData     = {}
      self.configFilePath = configFilePath

      self.clientIdAuthAttributes = ClientIdAuthAttributes()
      self.serverConnection       = None
      self.commandParser          = None
      self.localWorkflowPEGASUS   = None
      self.localWorkflowPARALLEL  = None
      self.jobScanner             = None

      self.serverData                        = {}
      self.serverData['serverId']            = None
      self.serverData['version']             = ""
      self.serverData['authz']               = False
      self.serverData['jobId']               = 0
      self.serverData['jobToken']            = ""
      self.serverData['submitterClass']      = 1
      self.serverData['event']               = ""
      self.serverData['createdSessions']     = []
      self.serverData['heartbeatInterval']   = 0
      self.serverData['pegasusRC']           = ""
      self.serverData['pegasusSites']        = ""
      self.serverData['localPegasusSite']    = {}
      self.serverData['submissionScripts']   = None
      self.serverData['childHasExited']      = False
      self.serverData['exited']              = False
      self.serverData['isBuildingOutputTar'] = False

      self.clientData                          = {}
      self.clientData['clientId']              = uuid.uuid4()
      self.clientData['runCompleted']          = False
      self.clientData['args']                  = None
      self.clientData['hostFQDN']              = socket.getfqdn()
      self.clientData['version']               = __version__
      self.clientData['operationMode']         = 0
      self.clientData['authorizedAction']      = self.AUTHORIZEDACTIONNONE
      self.clientData['localJobId']            = "00000000"
      self.clientData['userName']              = None
      self.clientData['userId']                = None
      self.clientData['disableJobMonitoring']  = False
      self.clientData['jobMonitoringWait']     = False
      self.clientData['jobMonitoringComplete'] = False
      self.clientData['statusModifiedTime']    = 0.
      self.clientData['runProgressMessage']    = ""
      self.clientData['detach']                = False
      self.clientData['attachId']              = ZERO
      self.clientData['cacheSquid']            = ""
      self.clientData['pegasusExists']         = False
      self.clientData['pegasusVersion']        = None
      self.clientData['pegasusHome']           = None
      self.clientData['pegasusSetupError']     = ""
      self.clientData['reportMetrics']         = False
#     self.clientData['isClientTTY']           = True
      self.clientData['isClientTTY']           = not sys.stdin.closed and sys.stdin.isatty() and \
                                                 not sys.stdout.closed and sys.stdout.isatty()
      self.clientData['progressReport']        = 'text'
      self.clientData['readyToExit']           = False
      self.clientData['isExportingInputTar']   = False
      self.clientData['isExportingSubmitTar']  = False
      self.clientData['isServerReadyForIO']    = False
      self.clientData['selectTimeout']         = 1*60.
      self.clientData['selectTimeoutSpread']   = 0.15
      self.clientData['disconnectMax']         = 3*self.clientData['selectTimeout']
      self.clientData['suspended']             = False
      self.clientData['transferFiles']         = []
      self.clientData['commandFiles']          = []
      self.clientData['inputObjects']          = {}
      self.clientData['outputObjects']         = {}
      self.clientData['isParametric']          = False
      self.clientData['isParallel']            = False
      self.clientData['nStripes']              = 1
      self.clientData['localExecution']        = False
      self.clientData['isPegasusCommand']      = False
      self.clientData['localWaiting']          = False
      self.clientData['runProgressMessage']    = ""
      self.clientData['childPid']              = None
      self.clientData['childReadStdout']       = None
      self.clientData['childReadStderr']       = None
      self.clientData['localStartTime']        = time.time()
      self.clientData['localFinishTime']       = None
      self.clientData['childHasExited']        = False
      self.clientData['childKillPending']      = False
      self.clientData['exitCode']              = 0
      self.clientData['cpuTime']               = 0.
      self.clientData['realTime']              = 0.
      self.clientData['jobStates']             = {}
      self.clientData['jobScanPath']           = ""
      self.clientData['jobScannerStarted']     = False
      self.clientData['runStatusPath']         = ""
      self.clientData['enteredCommand']        = ""
      self.clientData['event']                 = "[simulation]"
      self.clientData['startDate']             = None
      self.clientData['finishDate']            = None
      self.clientData['runName']               = ""
      self.clientData['jobPath']               = os.getcwd()
      self.clientData['importFilesComplete']   = False
      self.clientData['isImportingOutputTar']  = False
      self.clientData['ignoreMountPoints']     = [ os.sep + mp for mp in ['boot','dev','etc','proc','run','sys']]
      self.clientData['mountPoints']           = {}
      self.clientData['environment']           = ""
      self.clientData['measurementsData']      = None

      if   'APPTAINER_BIND' in os.environ:
         bindMounts = os.environ['APPTAINER_BIND'].split(',')
         for mountPoint in bindMounts:
            self.clientData['mountPoints'][mountPoint] = {'device':':'.join(['ABM',mountPoint]),
                                                          'inputFiles':[],
                                                          'submitCommandFiles':[]}
      elif 'SINGULARITY_BIND' in os.environ:
         bindMounts = os.environ['SINGULARITY_BIND'].split(',')
         for bindMount in bindMounts:
            device,mountPoint = bindMount.split(':')
            self.clientData['mountPoints'][mountPoint] = {'device':':'.join(['SBM',device]),
                                                          'inputFiles':[],
                                                          'submitCommandFiles':[]}
      else:
         dfCommand = ['df','--portability','--all']
         child = subprocess.Popen(dfCommand,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True,
                                  universal_newlines=True)
         dfStdOutput,dfStdError = child.communicate()
         dfExitStatus = child.returncode
         if dfExitStatus == 0:
            mounts = ''.join(dfStdOutput).strip().split('\n')[1:]
            for mount in mounts:
               device     = mount.split()[0]
               mountPoint = mount.split()[-1]

               if mountPoint in self.clientData['ignoreMountPoints']:
                  pass
               else:
                  skipMountPoint = False
                  for ignoreMountPoint in self.clientData['ignoreMountPoints']:
                     if mountPoint.startswith(ignoreMountPoint + os.sep):
                        skipMountPoint = True
                        break

                  if not skipMountPoint:
                     if ':' in device:
                        self.clientData['mountPoints'][mountPoint] = {'device':device,
                                                                      'inputFiles':[],
                                                                      'submitCommandFiles':[]}
                     else:
                        self.clientData['mountPoints'][mountPoint] = {'device':':'.join([self.clientData['hostFQDN'],device]),
                                                                      'inputFiles':[],
                                                                      'submitCommandFiles':[]}
         else:
            self.logger.log(logging.ERROR,getLogMessage("df command failed:\n%s\n" % (dfStdError)))

      self.clientData['workDirectory']           = os.getcwd()
      self.clientData['workDirectoryProperties'] = self.__getFileProperties(self.clientData['workDirectory'])
      if self.clientData['workDirectoryProperties']['inode'] == 0:
         self.clientData['exitCode'] = 1
      if not self.clientData['workDirectoryProperties']['mountDevice']:
         self.clientData['exitCode'] = 1

      try:
         fd,signaturePath = tempfile.mkstemp(prefix='.submitSignature_',dir=self.clientData['workDirectory'])
         os.close(fd)
         with open(signaturePath,'w') as fp:
            fp.write("%d" % (int(time.time())))
      except (IOError,OSError):
         self.clientData['signatureProperties'] = {}
         self.logger.log(logging.ERROR,getLogMessage("Could not create signature file"))
         self.clientData['exitCode'] = 1
      else:
         self.clientData['signatureProperties'] = self.__getFileProperties(signaturePath)

      self.clientData['importObjects']       = {}
      self.clientData['exportObjects']       = {}
      self.clientData['closePendingObjects'] = []
      self.clientData['filesToRemove']       = []
      self.clientData['emptyFilesToRemove']  = []

      self.readChildExitSignalPipe  = None
      self.writeChildExitSignalPipe = None
      try:
         self.readChildExitSignalPipe,self.writeChildExitSignalPipe = os.pipe2(os.O_NONBLOCK | os.O_CLOEXEC)
      except:
         try:
            self.readChildExitSignalPipe,self.writeChildExitSignalPipe = os.pipe()
            for fd in [self.readChildExitSignalPipe,self.writeChildExitSignalPipe]:
               fdFlags = fcntl.fcntl(fd, fcntl.F_GETFL)
               fcntl.fcntl(fd, fcntl.F_SETFL, fdFlags | os.O_NONBLOCK | os.O_CLOEXEC)
         except:
            self.readChildExitSignalPipe  = None
            self.writeChildExitSignalPipe = None

      signal.signal(signal.SIGINT,self.handleTerminationSignal)
      signal.signal(signal.SIGHUP,self.handleTerminationSignal)
      signal.signal(signal.SIGQUIT,self.handleDetachSignal)
      signal.signal(signal.SIGABRT,self.handleTerminationSignal)
#     signal.signal(signal.SIGUSR2,self.handleBreakConnectionSignal)
      signal.signal(signal.SIGTERM,self.handleTerminationSignal)
      signal.signal(signal.SIGCONT,self.handleContinueSignal)
      signal.signal(signal.SIGTSTP,self.handleSuspendSignal)
      signal.signal(signal.SIGXCPU,self.handleTerminationSignal)
      signal.signal(signal.SIGCHLD,self.handleLocalCompleteSignal)


   def __getFileProperties(self,
                           filePath):
      fileProperties = {}
      fileProperties['filePath']    = os.path.abspath(filePath)
      fileProperties['mountPoint']  = ""
      fileProperties['mountDevice'] = ""
      if os.path.exists(fileProperties['filePath']):
         if len(self.clientData['mountPoints']) > 0:
            mountPoint = fileProperties['filePath']
            while mountPoint:
               if os.path.ismount(mountPoint):
                  fileProperties['mountPoint']  = mountPoint
                  fileProperties['mountDevice'] = self.clientData['mountPoints'][mountPoint]['device']
                  break
               else:
                  if mountPoint != os.sep:
                     mountPoint = os.path.dirname(mountPoint)
                  else:
                     break

         if not os.path.isdir(fileProperties['filePath']):
            md5Hash = hashlib.md5()
            with open(fileProperties['filePath'],'rb') as f:
               # Read and update hash in chunks of 4K
               for block in iter(lambda: f.read(4096),b""):
                  md5Hash.update(block)
               fileProperties['checksum'] = md5Hash.hexdigest()
         else:
            fileProperties['checksum'] = ""

         fileProperties['fileSize'] = os.lstat(fileProperties['filePath']).st_size
         fileProperties['inode']    = os.lstat(fileProperties['filePath']).st_ino
      else:
         fileProperties['checksum'] = ""
         fileProperties['fileSize'] = 0
         fileProperties['inode']    = 0

      return fileProperties


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(str(message))
      except UnicodeEncodeError:
         try:
            sys.stdout.write(str(message.encode('ascii','replace').decode('utf-8')))
         except IOError as e:
            if not e.args[0] in [EPIPE]:
               self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))
         else:
            sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))
      else:
         sys.stdout.flush()


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(str(message))
      except UnicodeEncodeError:
         try:
            sys.stderr.write(str(message.encode('ascii','replace').decode('utf-8')))
         except IOError as e:
            if not e.args[0] in [EPIPE]:
               self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))
         else:
            sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))
      else:
         sys.stderr.flush()


   def sendUserErrorNotification(self,
                                 message):
      if   isLogMessageFileStdout():
         self.__writeToStdout(message + '\n')
      elif isLogMessageFileStderr():
         self.__writeToStderr(message + '\n')
      elif isLoggingOn():
         self.logger.log(logging.ERROR,getLogMessage(message))
      else:
         self.__writeToStderr(message + '\n')


   def killChild(self):
      if self.clientData['childPid'] and not self.clientData['childKillPending']:
         self.clientData['childKillPending'] = True
         try:
            os.kill(self.clientData['childPid'],signal.SIGINT)
            for retry in iterRange(0,12):
               time.sleep(10)
               os.kill(self.clientData['childPid'],0)
               if not self.clientData['childKillPending']:
                  break
            if self.clientData['childKillPending']:
               os.kill(self.clientData['childPid'],signal.SIGTERM)
               for retry in iterRange(0,6):
                  time.sleep(10)
                  os.kill(self.clientData['childPid'],0)
                  if not self.clientData['childKillPending']:
                     break
            if self.clientData['childKillPending']:
               os.kill(self.clientData['childPid'],signal.SIGKILL)
         except:
            pass


   def handleSuspendSignal(self,
                           signalNumber,
                           frame):
      if self.serverConnection and self.serverConnection.isConnected():
         clientMessage = {'messageType':'clientSuspended'}
         self.serverConnection.postJsonMessage(clientMessage)


   def handleContinueSignal(self,
                           signalNumber,
                           frame):
      if self.serverConnection and self.serverConnection.isConnected():
         clientMessage = {'messageType':'clientContinued'}
         self.serverConnection.postJsonMessage(clientMessage)


   def handleTerminationSignal(self,
                               signalNumber,
                               frame):
      if not self.clientData['readyToExit']:
         if self.clientData['localExecution']:
            if self.clientData['childPid']:
               self.killChild()
            else:
               exitCode = (1 << 7) | signalNumber
               self.clientData['exitCode'] = exitCode
         else:
            if self.serverConnection and self.serverConnection.isConnected():
               clientMessage = {'messageType':'clientSignal',
                                'signal':signalNumber}
               self.serverConnection.postJsonMessage(clientMessage)
            else:
               self.clientData['readyToExit'] = True
               exitCode = (1 << 7) | signalNumber
               self.clientData['exitCode'] = exitCode
               self.exit(exitCode)


   def handleLocalCompleteSignal(self,
                                 signalNumber,
                                 frame):
      self.clientData['childKillPending'] = False
      if self.clientData['childPid'] and self.writeChildExitSignalPipe:
         textLength = os.write(self.writeChildExitSignalPipe,struct.pack('i',signalNumber))
         self.writeSignalPipe(self.writeChildExitSignalPipe,signalNumber)


   def handleDetachSignal(self,
                          signalNumber,
                          frame):
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if not self.clientData['localExecution']:
            clientMessage = {'messageType':'detachSignal'}
            self.serverConnection.postJsonMessage(clientMessage)


#  def handleBreakConnectionSignal(self,
#                                  signalNumber,
#                                  frame):
#     if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
#        if not self.clientData['localExecution']:
#           if self.serverConnection:
#              self.serverConnection.closeConnection()


   def setupPegasus(self):
      pegasusVersion    = None
      pegasusHome       = None
      pegasusPythonPath = None

      try:
         warnings.showwarning = showWarning
         try:
            import Pegasus.api
         except:
            from Pegasus.DAX3 import ADAG
         warnings.showwarning = _default_show_warning

         tStart = time.time()
         pegasusCommand = "pegasus-config --version\n" + \
                          "pegasus-config --bin\n" + \
                          "pegasus-config --python"
         child = subprocess.Popen(pegasusCommand,shell=True,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True,
                                  universal_newlines=True)
         pegasusStdOutput,pegasusStdError = child.communicate()
         pegasusExitStatus = child.returncode
         if pegasusExitStatus == 0:
            try:
               pegasusVersion,pegasusBin,pegasusPythonPath = ''.join(pegasusStdOutput).strip().split()
               pegasusHome = os.path.dirname(pegasusBin)
            except:
               pass

         tFinish = time.time()
#        print("setupPegasus:testing w/external environment took %f secs" % (tFinish-tStart))
      except:
         pass

      if not pegasusVersion or not pegasusHome or not pegasusPythonPath:
         try:
            tStart = time.time()
            pegasusCommand = ". %s\n" % (self.configData['useSetup']) + \
                             "use -e -r pegasus-%s\n" % (self.configData['pegasusVersion']) + \
                             "pegasus-config --version\n" + \
                             "pegasus-config --bin\n" + \
                             "pegasus-config --python"
            child = subprocess.Popen(pegasusCommand,shell=True,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True,
                                     universal_newlines=True)
            pegasusStdOutput,pegasusStdError = child.communicate()
            pegasusExitStatus = child.returncode
            if pegasusExitStatus == 0:
               try:
                  pegasusVersion,pegasusBin,pegasusPythonPath = ''.join(pegasusStdOutput).strip().split()
                  pegasusHome = os.path.dirname(pegasusBin)
                  if not pegasusPythonPath in sys.path:
                     sys.path.insert(0,pegasusPythonPath)
                  warnings.showwarning = showWarning
                  try:
                     import Pegasus.api
                  except:
                     from Pegasus.DAX3 import ADAG
                  warnings.showwarning = _default_show_warning
               except:
                  pegasusVersion    = ""
                  pegasusHome       = ""
                  pegasusPythonPath = ""
                  self.clientData['pegasusSetupError'] = "Pegasus import failed.\n"
            else:
               self.clientData['pegasusSetupError'] = "Pegasus version determination failed.\n%s\n" % (pegasusStdError)
            tFinish = time.time()
#           print("setupPegasus:testing w/internal environment took %f secs" % (tFinish-tStart))
         except:
            pegasusVersion    = ""
            pegasusHome       = ""
            pegasusPythonPath = ""
            self.clientData['pegasusSetupError'] = "Pegasus version determination failed.\n"

      if pegasusVersion and pegasusHome and pegasusPythonPath:
         self.clientData['pegasusExists'] = True
      self.clientData['pegasusVersion']   = pegasusVersion
      self.clientData['pegasusHome']      = pegasusHome


   def scheduleHeartbeat(self):
      signal.signal(signal.SIGALRM,self.heartbeat)
      signal.alarm(self.serverData['heartbeatInterval'])


   def heartbeat(self,
                 signalNumber,
                 frame):
      self.logger.log(logging.INFO,getLogMessage("Client heartbeat"))
      self.connect()
      clientMessage = {'messageType':'pwd',
                       'path':self.clientData['workDirectory'],
                       'properties':self.clientData['workDirectoryProperties'],
                       'signature':self.clientData['signatureProperties']}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'jobId',
                       'jobId':self.serverData['jobId'],
                       'localExecution':self.clientData['localExecution']}
      self.serverConnection.postJsonMessage(clientMessage)
      self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONHEARTBEAT
      self.signon()


   def cancelHeartbeat(self):
      alarmFunction = signal.getsignal(signal.SIGALRM)
      if not alarmFunction in [None,signal.SIG_IGN,signal.SIG_DFL]:
         if alarmFunction == self.heartbeat:
            signal.alarm(0)


   def configure(self):
      configured = False
      if self.clientData['exitCode'] == 0:
         self.clientData['exitCode'] = 2

         sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
         keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
         commentPattern  = re.compile('\s*#.*')
         inClientSection = False

         try:
            fpConfig = open(self.configFilePath,'r')
            try:
               eof = False
               while not eof:
                  record = fpConfig.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   sectionPattern.match(record):
                        sectionName = sectionPattern.match(record).group(2)
                        inClientSection = (sectionName == 'client')
                        if inClientSection:
                           self.configData = {'listenURIs':[],
                                              'maximumConnectionPasses':15,
                                              'submitSSLCA':"/etc/submit/submit_server_ca.crt",
                                              'pegasusVersion':"4.8.0",
                                              'useSetup':"/etc/environ.sh",
                                              'doubleDashTerminator':False,
                                              'maximumLocalStripes':1
                                             }
                     elif inClientSection:
                        if keyValuePattern.match(record):
                           key,value = keyValuePattern.match(record).group(2,4)
                           if key in self.configData:
                              if   isinstance(self.configData[key],list):
                                 self.configData[key] = [e.strip() for e in value.split(',')]
                              elif isinstance(self.configData[key],bool):
                                 self.configData[key] = bool(value.lower() == 'true')
                              elif isinstance(self.configData[key],float):
                                 self.configData[key] = float(value)
                              elif isinstance(self.configData[key],int):
                                 self.configData[key] = int(value)
                              elif isinstance(self.configData[key],dict):
                                 try:
                                    sampleKey   = list(self.configData[key].keys())[0]
                                    sampleValue = self.configData[key][sampleKey]
                                 except:
                                    sampleKey   = "key"
                                    sampleValue = "value"
                                 self.configData[key] = {}
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.configData[key][dictKey] = dictValue
                              else:
                                 self.configData[key] = value
                           else:
                              self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
                  else:
                     eof = True
                     configured = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (self.configFilePath)))
            finally:
               fpConfig.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.configFilePath)))

         if configured:
            if len(self.configData['listenURIs']) == 0:
               self.logger.log(logging.ERROR,getLogMessage("listenURIs missing from %s" % (self.configFilePath)))
               configured = False

            self.setupPegasus()

            if 'SUBMIT_DOUBLEDASH' in os.environ:
               if   os.environ['SUBMIT_DOUBLEDASH'].lower() == 'true':
                  self.configData['doubleDashTerminator'] = True
               elif os.environ['SUBMIT_DOUBLEDASH'].lower() == 'false':
                  self.configData['doubleDashTerminator'] = False

      return(configured)


   def exit(self,
            exitCode=0):
      if self.clientData['readyToExit']:
         if not self.jobScanner:
            self.clientData['runCompleted'] = True
      else:
         if self.serverConnection and self.serverConnection.isConnected():
            clientMessage = {'messageType':'clientExit',
                             'exitCode':exitCode}
            self.serverConnection.postJsonMessage(clientMessage)
            self.clientData['readyToExit'] = True
         else:
            self.clientData['exitCode'] = exitCode
            self.clientData['runCompleted'] = True


   def getUserAttributes(self):
      return(self.clientIdAuthAttributes.getUserAttributes())


   def haveIdAuthAttributes(self):
      return(self.clientIdAuthAttributes.haveIdAuthAttributes())


   def connect(self):
      isConnected = False
      if self.haveIdAuthAttributes():
         self.serverConnection = UnboundConnection(UnboundConnection.TLSREQUIREMENTNONE,
                                                   listenURIs=self.configData['listenURIs'],
                                                   maximumConnectionPasses=self.configData['maximumConnectionPasses'],
                                                   submitSSLCA=self.configData['submitSSLCA'])
         isConnected = self.serverConnection.isConnected()

      return(isConnected)


   def sendExecutionContext(self):
      clientMessage = {'messageType':'clientId',
                       'clientId':self.clientData['clientId'].hex}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'clientVersion',
                       'version':self.clientData['version']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'doubleDashTerminator',
                       'doubleDashTerminator':self.configData['doubleDashTerminator']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'args',
                       'args':self.clientData['args']}
      self.serverConnection.postJsonMessage(clientMessage)

      blackListEnvironmentVariables = ['CLASSPATH','DISPLAY','EDITOR','HOME','LOGNAME','LS_COLORS','MAIL','MANPATH', \
                                       'OLDPWD','PATH','PERLLIB','PYTHONPATH','RESULTSDIR','SESSIONDIR','SHELL','SHLVL', \
                                       'SVN_EDITOR','SUBMIT_DOUBLEDASH','TERM','TIMEOUT','USER','VISUAL', \
                                       'WINDOWID', 'XTERM_LOCALE','XTERM_SHELL','XTERM_VERSION','_']
      environmentVariables = {}
      for environmentVar in os.environ:
         if not environmentVar in blackListEnvironmentVariables:
            if environmentVar.startswith('group_'):
               continue
            if environmentVar.startswith('etc'):
               continue
            if environmentVar.startswith('rpath_'):
               continue
            if environmentVar.startswith('session'):
               continue

            environmentVariables[environmentVar] = os.environ[environmentVar]
      if environmentVariables:
         clientMessage = {'messageType':'vars',
                          'vars':environmentVariables}
         self.serverConnection.postJsonMessage(clientMessage)

      umask = os.umask(0)
      os.umask(umask)
      clientMessage = {'messageType':'umask',
                       'umask':umask}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'pwd',
                       'path':self.clientData['workDirectory'],
                       'properties':self.clientData['workDirectoryProperties'],
                       'signature':self.clientData['signatureProperties']}
      self.serverConnection.postJsonMessage(clientMessage)

#     self.clientData['isClientTTY'] = sys.stdin.isatty() and sys.stdout.isatty()
      clientMessage = {'messageType':'isClientTTY',
                       'isClientTTY':self.clientData['isClientTTY']}
      self.serverConnection.postJsonMessage(clientMessage)

      if self.clientData['pegasusVersion']:
         clientMessage = {'messageType':'pegasusVersion',
                          'version':self.clientData['pegasusVersion']}
      else:
         clientMessage = {'messageType':'pegasusVersion',
                          'version':self.configData['pegasusVersion']}
      self.serverConnection.postJsonMessage(clientMessage)


   def sendLocalContext(self):
      clientMessage = {'messageType':'clientId',
                       'clientId':self.clientData['clientId'].hex}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'clientVersion',
                       'version':self.clientData['version']}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'pwd',
                       'path':self.clientData['workDirectory'],
                       'properties':self.clientData['workDirectoryProperties'],
                       'signature':self.clientData['signatureProperties']}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'jobId',
                       'jobId':self.serverData['jobId'],
                       'localExecution':self.clientData['localExecution']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'jobToken',
                       'token':self.serverData['jobToken']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'operationMode',
                       'operationMode':self.clientData['operationMode']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'createdSessions',
                       'createdSessions':self.serverData['createdSessions']}
      self.serverConnection.postJsonMessage(clientMessage)

      try:
         submitApplicationRevision = os.environ["SUBMIT_APPLICATION_REVISION"]
      except:
         submitApplicationRevision = ""
      if submitApplicationRevision != "":
         clientMessage = {'messageType':'event',
                          'text':submitApplicationRevision}
         self.serverConnection.postJsonMessage(clientMessage)
      else:
         clientMessage = {'messageType':'event',
                          'text':self.serverData['event']}
         self.serverConnection.postJsonMessage(clientMessage)


   def sendHarvestContext(self):
      clientMessage = {'messageType':'clientId',
                       'clientId':self.clientData['clientId'].hex}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'clientVersion',
                       'version':self.clientData['version']}
      self.serverConnection.postJsonMessage(clientMessage)
      harvestArgs = []
      harvestArgs.append(self.clientData['args'][0])
      harvestArgs.append("--harvest")
      if self.serverData['jobId'] != ZERO:
         harvestArgs.append("%d" % (self.serverData['jobId']))
      else:
         harvestArgs.append("%d" % (self.commandParser.getOption('harvestId')))
      harvestArgs.append("--harvestPath")
      harvestArgs.append(os.path.join(os.getcwd(),self.clientData['runName']))
      if self.commandParser.getOption('runName'):
         harvestArgs.append("--runName")
         harvestArgs.append(self.commandParser.getOption('runName'))
      if self.clientData['reportMetrics']:
         harvestArgs.append("--metrics")
      clientMessage = {'messageType':'args',
                       'args':harvestArgs}
      self.serverConnection.postJsonMessage(clientMessage)

      blackListEnvironmentVariables = ['CLASSPATH','DISPLAY','EDITOR','HOME','LOGNAME','LS_COLORS','MAIL','MANPATH', \
                                       'OLDPWD','PATH','PERLLIB','PYTHONPATH','RESULTSDIR','SESSIONDIR','SHELL','SHLVL', \
                                       'SVN_EDITOR','SUBMIT_DOUBLEDASH','TERM','TIMEOUT','USER','VISUAL', \
                                       'WINDOWID','XTERM_LOCALE','XTERM_SHELL','XTERM_VERSION','_']
      environmentVariables = {}
      for environmentVar in os.environ:
         if not environmentVar in blackListEnvironmentVariables:
            if environmentVar.startswith('group_'):
               continue
            if environmentVar.startswith('etc'):
               continue
            if environmentVar.startswith('rpath_'):
               continue
            if environmentVar.startswith('session'):
               continue

            environmentVariables[environmentVar] = os.environ[environmentVar]
      if environmentVariables:
         clientMessage = {'messageType':'vars',
                          'vars':environmentVariables}
         self.serverConnection.postJsonMessage(clientMessage)

      umask = os.umask(0)
      os.umask(umask)
      clientMessage = {'messageType':'umask',
                       'umask':umask}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'pwd',
                       'path':self.clientData['workDirectory'],
                       'properties':self.clientData['workDirectoryProperties'],
                       'signature':self.clientData['signatureProperties']}
      self.serverConnection.postJsonMessage(clientMessage)

      clientMessage = {'messageType':'jobId',
                       'jobId':self.serverData['jobId'],
                       'localExecution':self.clientData['localExecution']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'jobToken',
                       'token':self.serverData['jobToken']}
      self.serverConnection.postJsonMessage(clientMessage)


   def sendMeasurementsContext(self):
      clientMessage = {'messageType':'clientId',
                       'clientId':self.clientData['clientId'].hex}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'clientVersion',
                       'version':self.clientData['version']}
      self.serverConnection.postJsonMessage(clientMessage)
      clientMessage = {'messageType':'operationMode',
                       'operationMode':self.clientData['operationMode']}
      self.serverConnection.postJsonMessage(clientMessage)


   def signon(self):
      signonAttributes = self.clientIdAuthAttributes.getSignonAttributes()
      clientMessage = {'messageType':'userName',
                       'userName':signonAttributes['userName']}
      if 'sudoUserName' in signonAttributes:
         clientMessage['sudoUserName'] = signonAttributes['sudoUserName']
      if 'wsUserName' in signonAttributes:
         clientMessage['wsUserName'] = signonAttributes['wsUserName']
      self.serverConnection.postJsonMessage(clientMessage)

      sendSignonAttributes = ['password','sessionId','sessionToken','cacheHosts','privateFingerPrint']
      for sendSignonAttribute in sendSignonAttributes:
         if sendSignonAttribute in signonAttributes:
            clientMessage = {'messageType':sendSignonAttribute,
                             sendSignonAttribute:signonAttributes[sendSignonAttribute]}
            self.serverConnection.postJsonMessage(clientMessage)

      self.clientData['userName'] = signonAttributes['userName']
      if 'userId' in signonAttributes:
         self.clientData['userId'] = signonAttributes['userId']

      clientMessage = {'messageType':'clientReadyForSignon'}
      self.serverConnection.postJsonMessage(clientMessage)


   def mapSubmitCommandFiles(self):
      exitCode = 0
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         submitCommandFiles = self.commandParser.getSubmitCommandFiles()
         for submitCommandFile in submitCommandFiles:
            fileProperties = self.__getFileProperties(submitCommandFile)
            clientMessage = {'messageType':'submitCommandFile',
                             'path':fileProperties['filePath'],
                             'properties':fileProperties,
                             'argValue':submitCommandFile}
            self.serverConnection.postJsonMessage(clientMessage)

      if exitCode:
         self.exit(exitCode)
      else:
         clientMessage = {'messageType':'submitCommandFileInodesSent'}
         self.serverConnection.postJsonMessage(clientMessage)
         self.clientData['isExportingSubmitTar'] = True


   def connectToServer(self,
                       purpose):
      connectedToServer = False
      if self.haveIdAuthAttributes():
         if self.connect():
            if   purpose == 'execute':
               if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNHARVESTID:
                  self.sendHarvestContext()
                  self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONHARVEST
               else:
                  self.sendExecutionContext()
                  self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONEXECUTE
            elif purpose == 'sendMetrics':
               self.sendLocalContext()
               self.cancelHeartbeat()
               self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONSENDMETRICS
            elif purpose == 'harvest':
               self.sendHarvestContext()
               self.cancelHeartbeat()
               self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONHARVEST
            elif purpose == 'sendMeasurements':
               self.sendMeasurementsContext()
               self.cancelHeartbeat()
               self.clientData['authorizedAction'] = self.AUTHORIZEDACTIONSENDMEASUREMENTS

            if self.clientData['authorizedAction'] != self.AUTHORIZEDACTIONNONE:
               self.signon()
               connectedToServer = True
         else:
            self.sendUserErrorNotification("Connection to submit server failed.")
      else:
         self.sendUserErrorNotification("Could not acquire proper authentication information.")

      return(connectedToServer)


   def executeSubmission(self,
                         instantiatorPath,
                         args):
      result = {}
      result['exitCode']           = self.clientData['exitCode']
      result['serverDisconnected'] = False

      if result['exitCode'] == 0:
         if args:
            if self.configure():
               self.setCommandArguments([instantiatorPath] + args)
               (parseCommandError,contactServer) = self.parseCommandArguments()
               if not parseCommandError:
                  if not self.reportDebugOutput():
                     logSetMessageFile(None)
                  else:
                     self.logger.setLevel(logging.DEBUG)
                  if contactServer:
                     self.getUserAttributes()
                     if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNMEASUREMENT:
                        connectedToServer = self.connectToServer('sendMeasurements')
                     else:
                        connectedToServer = self.connectToServer('execute')

                     if connectedToServer:
                        result = self.run()
                     else:
                        result['exitCode'] = 1
               else:
                  self.sendUserErrorNotification("Command line argument parsing failed.")
                  result['exitCode'] = 1
            else:
               self.sendUserErrorNotification("submit configuration failed.")
               result['exitCode'] = 1
         else:
            self.sendUserErrorNotification("No command line provided.")
            result['exitCode'] = 1

      try:
         if self.clientData['signatureProperties']:
            os.remove(self.clientData['signatureProperties']['filePath'])
      except:
         pass

      return(result)


   def getInputObjects(self):
      serverReader = self.serverConnection.getInputObject()
      childProcessInputObjects = []
      if self.clientData['childReadStdout']:
         childProcessInputObjects.append(self.clientData['childReadStdout'])
      if self.clientData['childReadStderr']:
         childProcessInputObjects.append(self.clientData['childReadStderr'])
      inputSignalPipes = []
      if self.readChildExitSignalPipe:
         inputSignalPipes.append(self.readChildExitSignalPipe)

      return(serverReader,
             list(self.clientData['inputObjects'].keys()),
             list(self.clientData['exportObjects'].keys()),
             childProcessInputObjects,
             inputSignalPipes)


   def getOutputObjects(self):
      serverWriter = self.serverConnection.getOutputObject()

      importObjects = []
      for importObject in self.clientData['importObjects']:
         if self.clientData['importObjects'][importObject]['buffer']:
            importObjects.append(importObject)

      return(serverWriter,importObjects)


   def readFile(self,
                fileObject):
      if   fileObject in self.clientData['inputObjects']:
         inputFile = self.clientData['inputObjects'][fileObject]
         if inputFile == '#STDIN#' and self.clientData['localExecution']:
            try:
               text = os.read(fileObject.fileno(),1024).decode('utf-8')
               if not text:
                  self.closeFile(fileObject)
            except:
               self.logger.log(logging.ERROR,getLogMessage("Exception reading inputObject %s" % \
                                                  (self.clientData['inputObjects'][fileObject])))
               self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
               self.exit(1)
         else:
            try:
               text = os.read(fileObject.fileno(),1024)
            except:
               self.logger.log(logging.ERROR,getLogMessage("Exception reading importObject %s" % \
                                                  (self.clientData['importObjects'][fileObject])))
               self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
               self.exit(1)
            else:
               if not text:
                  clientMessage = {'messageType':'close',
                                   'path':inputFile}
                  self.serverConnection.postJsonMessage(clientMessage)
                  self.closeFile(fileObject)
               else:
                  clientMessage = {'messageType':'write',
                                   'path':inputFile,
                                   'text':base64.b64encode(text).decode('utf-8')}
                  self.serverConnection.postJsonMessage(clientMessage)
      elif fileObject in self.clientData['exportObjects']:
         inputFile = self.clientData['exportObjects'][fileObject]
         inputPath = os.path.join(os.sep,"tmp",inputFile)
#        self.logger.log(logging.DEBUG,getLogMessage("exportObjects: %s" % (inputPath)))
         try:
            text = os.read(fileObject.fileno(),1024)
         except:
            self.logger.log(logging.ERROR,getLogMessage("Exception reading exportObject %s" % \
                                               (self.clientData['exportObjects'][fileObject])))
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
            self.exit(1)
         else:
            if not text:
               clientMessage = {'messageType':'close',
                                'path':inputPath}
               self.serverConnection.postJsonMessage(clientMessage)
               self.closeFile(fileObject)
            else:
               clientMessage = {'messageType':'write',
                                'path':inputPath,
                                'text':base64.b64encode(text).decode('utf-8')}
               self.serverConnection.postJsonMessage(clientMessage)


   def writeFile(self,
                 fileObject):
      if fileObject in self.clientData['importObjects']:
         writeBuffer = self.clientData['importObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
         except:
            self.logger.log(logging.ERROR,getLogMessage("Exception writing importObject %s" % \
                                       (self.clientData['importObjects'][fileObject]['path'])))
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
            self.exit(1)
         else:
            self.clientData['importObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
#           self.logger.log(logging.DEBUG,getLogMessage("importObject: %d bytes written" % (textLength)))


   def closeFile(self,
                 fileObject):
      if fileObject in self.clientData['inputObjects']:
         del self.clientData['inputObjects'][fileObject]

      if fileObject in self.clientData['exportObjects']:
         exportFile = self.clientData['exportObjects'][fileObject]
         exportPath = os.path.join(os.sep,"tmp",exportFile)
#        self.logger.log(logging.DEBUG,getLogMessage("closeFile: %s" % (exportPath)))
         if os.path.exists(exportPath):
            os.remove(exportPath)
         del self.clientData['exportObjects'][fileObject]

      if fileObject in self.clientData['outputObjects']:
         del self.clientData['outputObjects'][fileObject]

      transferTarFile = ""
      if fileObject in self.clientData['importObjects']:
         transferTarFile = self.clientData['importObjects'][fileObject]['path']
         del self.clientData['importObjects'][fileObject]
         if len(self.clientData['importObjects']) == 0:
            self.clientData['importFilesComplete'] = True
            clientMessage = {'messageType':'importFilesComplete'}
            self.serverConnection.postJsonMessage(clientMessage)
            self.serverData['isBuildingOutputTar']  = False
            self.clientData['isImportingOutputTar'] = False
            self.__writeToStdout("Import file transfer complete. %s\n" % (time.ctime()))

      if fileObject == self.clientData['childReadStdout']:
         self.clientData['childReadStdout'] = None
      if fileObject == self.clientData['childReadStderr']:
         self.clientData['childReadStderr'] = None

      try:
         fileObject.close()
         if transferTarFile:
            tarCommand = ['tar','xvf',os.path.join(self.clientData['workDirectory'],transferTarFile)]
#           self.logger.log(logging.DEBUG,getLogMessage("command: %s" % (tarCommand)))
            child = subprocess.Popen(tarCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True,
                                     universal_newlines=True)
            tarStdOutput,tarStdError = child.communicate()
            tarExitStatus = child.returncode
            if tarExitStatus == 0:
               os.remove(os.path.join(self.clientData['workDirectory'],transferTarFile))
            else:
               self.logger.log(logging.ERROR,getLogMessage("Failed to extract transfer tarfile %s" % (transferTarFile)))
               if tarStdOutput:
                  self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
               if tarStdError:
                  self.logger.log(logging.ERROR,getLogMessage(tarStdError))
      except:
         pass


   def checkClosePendingObjects(self):
      markedForDeletion = []
      for fileObject in self.clientData['closePendingObjects']:
         if fileObject in self.clientData['importObjects']:
            if self.clientData['importObjects'][fileObject]['buffer'] == b"":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
      for fileObject in markedForDeletion:
         self.clientData['closePendingObjects'].remove(fileObject)
      del markedForDeletion


   def readSignalPipe(self,
                      fileDescriptor):
      text = os.read(fileDescriptor,4)
      signalNumber = struct.unpack('i',text)


   def writeSignalPipe(self,
                       fileDescriptor,
                       signalNumber):
      textLength = os.write(fileDescriptor,struct.pack('i',signalNumber))


   def showHelpUsage(self,
                     operationMode):
      if operationMode & (self.commandParser.OPERATIONMODEHELPUSAGE | \
                          self.commandParser.OPERATIONMODEHELPEXAMPLES):
         if operationMode & (self.commandParser.OPERATIONMODEHELPTOOLS | \
                             self.commandParser.OPERATIONMODEHELPVENUES | \
                             self.commandParser.OPERATIONMODEHELPMANAGERS | \
                             self.commandParser.OPERATIONMODEVERSIONSERVER | \
                             self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR):
            showHelp = False
         else:
            showHelp = True
      else:
         showHelp = False

      return(showHelp)


   def setCommandArguments(self,
                           args):
      self.commandParser = CommandParser(self.configData['doubleDashTerminator'])
      self.clientData['args'] = self.commandParser.modifyCommandArguments(args)


   def validateArguments(self):
      validArguments = True
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNMEASUREMENT:
         measurementsPath = self.commandParser.getOption('measurementsPath')
         if os.path.exists(measurementsPath):
            try:
               with open(measurementsPath,'r') as fpMeasurements:
                  self.clientData['measurementsData'] = json.load(fpMeasurements)
                  self.clientData['measurementsData']['messageType'] = 'measurements'
            except:
               message = "measurements file %s could not be loaded\n" % (measurementsPath)
               self.__writeToStderr(message)
               validArguments = False
         else:
            message = "measurements file %s does not exist\n" % (measurementsPath)
            self.__writeToStderr(message)
            validArguments = False

      return(validArguments)


   def parseCommandArguments(self):
      parseCommandError = False
      contactServer = False
      self.commandParser.parseArguments(self.clientData['args'][1:])
      self.clientData['operationMode']        = self.commandParser.getOperationMode()
      self.clientData['disableJobMonitoring'] = self.commandParser.getOption('disableJobMonitoring')

      if self.commandParser.validateArguments() and self.validateArguments():
         if self.clientData['operationMode'] & self.commandParser.OPERATIONMODEVERSIONCLIENT:
            self.__writeToStdout("Submit client version: %s\n" % (self.clientData['version']))

         if   self.showHelpUsage(self.clientData['operationMode']):
            self.commandParser.showUsage()
         elif self.clientData['operationMode'] & (self.commandParser.OPERATIONMODERUNSTATUS | \
                                                  self.commandParser.OPERATIONMODERUNKILL | \
                                                  self.commandParser.OPERATIONMODERUNVENUESTATUS | \
                                                  self.commandParser.OPERATIONMODERUNSERVER | \
                                                  self.commandParser.OPERATIONMODERUNDISTRIBUTOR):
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNPROXY:
            self.clientData['attachId'] = self.commandParser.getOption('attachId')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEHIT:
            self.clientData['cacheSquid'] = self.commandParser.getOption('cacheHit')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEMISS:
            self.clientData['cacheSquid'] = self.commandParser.getOption('cacheMiss')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEPUBLISH:
            self.clientData['cacheSquid'] = self.commandParser.getOption('cachePublish')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
            self.commandParser.setSubmitCommandFiles()
            exitCode = self.commandParser.setSweepParameters()
            if exitCode == 0:
               exitCode = self.commandParser.setCSVDataParameters()
               if exitCode == 0:
                  parameterCombinationCount = self.commandParser.getParameterCombinationCount()
                  if parameterCombinationCount > 0:
                     self.clientData['isParametric'] = True

                  if not self.clientData['isParametric']:
                     if self.commandParser.getOption('progressCurses'):
                        message = "curses progress reports are only valid with parametric simulation\n"
                        self.__writeToStderr(message)
                        parseCommandError = True
                  else:
                     if self.commandParser.getOption('progressCurses') and not self.clientData['isClientTTY']:
                        message = "curses progress reports require a tty\n"
                        self.__writeToStderr(message)
                        parseCommandError = True

                  localExecution = self.commandParser.getOption('localExecution')
                  enteredExecutable = self.commandParser.getEnteredExecutable()
                  if   self.clientData['isParametric'] and not self.clientData['pegasusExists'] and localExecution:
                     message = "Pegasus is required to execute a parametric sweep.\n"
                     self.__writeToStderr(message)
                     if self.clientData['pegasusSetupError']:
                        self.__writeToStderr(self.clientData['pegasusSetupError'])
                     parseCommandError = True
                  elif self.clientData['isParametric'] and '@:' in enteredExecutable:
                     enteredExecutable = enteredExecutable.replace('@:','')
                     message = "Parameter substitution is not allowed within the command file: %s\n" % (enteredExecutable)
                     self.__writeToStderr(message)
                     parseCommandError = True
                  elif self.clientData['isParametric'] and '@@' in enteredExecutable:
                     message = "Parameter substitution is not allowed in the command name: %s\n" % (enteredExecutable)
                     self.__writeToStderr(message)
                     parseCommandError = True
                  elif self.clientData['isParametric'] and \
                       self.commandParser.getParametricArgumentCount() == 0 and \
                       self.commandParser.getParametricInputCount() == 0 and \
                       self.commandParser.getParametricEnvironmentCount() == 0:
                     message = "Parameters supplied for %d jobs, " % (parameterCombinationCount) + \
                               "but there are no template files or\ncommand substitutions that would make the jobs unique.  " + \
                               "You should prefix any template files\nwith '@:' or " + \
                               "embed parameter names (in the form '@@name') into command line arguments.\n"
                     self.__writeToStderr(message)
                     parseCommandError = True
                  elif not self.checkInputsExist():
                     parseCommandError = True
                  else:
#                    self.logger.log(logging.DEBUG,getLogMessage("setting localExecution"))
                     self.clientData['localExecution']   = localExecution
                     self.clientData['isPegasusCommand'] = os.path.basename(enteredExecutable).startswith('pegasus-')
                     self.clientData['reportMetrics']    = self.commandParser.getOption('metrics')
                     self.clientData['detach']           = self.commandParser.getOption('detach')
                     if self.commandParser.getOption('runName'):
                        self.clientData['runName']       = self.commandParser.getOption('runName')
                     contactServer = True

                     if self.clientData['localExecution']:
                        if self.clientData['isParametric'] or self.clientData['isPegasusCommand']:
                           if not self.clientData['pegasusExists']:
                              if self.clientData['isParametric']:
                                 message = "Pegasus is required to execute a parametric sweep.\n"
                              else:
                                 message = "Pegasus is required to execute a Pegasus command.\n"
                              self.__writeToStderr(message)
                              if self.clientData['pegasusSetupError']:
                                 self.__writeToStderr(self.clientData['pegasusSetupError'])
                              parseCommandError = True
                              contactServer     = False
                           if self.clientData['runName']:
                              runDirectory = os.path.join(os.getcwd(),self.clientData['runName'])
                              if os.path.exists(runDirectory):
                                 message = "Run directory %s exists\n" % (runDirectory)
                                 self.__writeToStderr(message)
                                 parseCommandError = True
                                 contactServer     = False
                        if   self.clientData['isParametric'] and self.commandParser.getOption('nStripes'):
                           self.clientData['isParallel'] = True
                           self.clientData['nStripes']   = int(self.commandParser.getOption('nStripes'))
                           if self.clientData['nStripes'] > self.configData['maximumLocalStripes']:
                              self.clientData['nStripes'] = self.configData['maximumLocalStripes']
                              self.__writeToStderr("maximumLocalStripes exceeded, using %d stripes\n" % \
                                                                (self.configData['maximumLocalStripes']))
                        elif self.commandParser.getOption('nStripes'):
                           self.__writeToStderr("nStripes only applies to local parametric sweeps\n")
                     else:
                        if self.commandParser.getOption('nStripes'):
                           self.__writeToStderr("nStripes only applies to local parametric sweeps\n")
               else:
                  parseCommandError = True
            else:
               parseCommandError = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNHARVESTID:
            if self.commandParser.getOption('runName'):
               self.clientData['runName'] = self.commandParser.getOption('runName')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNMEASUREMENT:
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNNONE:
            if not self.clientData['operationMode'] & self.commandParser.OPERATIONMODEVERSIONCLIENT:
               if not self.clientData['operationMode'] & self.commandParser.OPERATIONMODEHELPEXAMPLES:
                  message = "No command given to execute\n"
                  self.__writeToStderr(message)
                  parseCommandError = True
      else:
         parseCommandError = True

      if   not self.commandParser.getOption('progress'):
         self.clientData['progressReport'] = 'silent'
      elif self.commandParser.getOption('progressCurses'):
         self.clientData['progressReport'] = 'curses'
      elif self.commandParser.getOption('progressSubmit'):
         self.clientData['progressReport'] = 'submit'
      elif self.commandParser.getOption('progressText'):
         self.clientData['progressReport'] = 'text'
      else:
         if   self.clientData['isParametric'] and self.clientData['isClientTTY']:
            self.clientData['progressReport'] = 'curses'
         elif self.clientData['isParametric'] and self.clientData['localExecution']:
            self.clientData['progressReport'] = 'submit'
         else:
            self.clientData['progressReport'] = 'text'

      return(parseCommandError,contactServer)


   def getOperationMode(self):
      return(self.clientData['operationMode'])


   def reportDebugOutput(self):
      return(self.commandParser.getOption('debug'))


   def __updatePegasusWorkflowStatus(self):
      jobStatusReportOrders = {}
      jobStatusReportOrders['waiting']    = 5
      jobStatusReportOrders['aborted']    = 1
      jobStatusReportOrders['setup']      = 6
      jobStatusReportOrders['setting up'] = 6
      jobStatusReportOrders['failed']     = 3
      jobStatusReportOrders['executing']  = 4
      jobStatusReportOrders['finished']   = 2

      parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
      parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
      if '.' in parameterCombinationsBase:
         parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
      tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
      tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
      copyTmpFile = False

      if os.path.exists(self.clientData['jobScanPath']):
         try:
            fpCSVIn = open(self.clientData['jobScanPath'],'rt')
            try:
               csvReader = csv.reader(fpCSVIn)
               try:
                  fpCSVOut = open(tmpParameterCombinationsPath,'wt')
                  try:
                     csvWriter = csv.writer(fpCSVOut)
                     csvWriter.writerow(('# command: ' + self.clientData['enteredCommand'],))
                     csvWriter.writerow(('# started: ' + self.clientData['startDate'],))
                     if self.clientData['finishDate']:
                        csvWriter.writerow(('# finished: ' + self.clientData['finishDate'],))
                     nCompleted = 0
                     for instance in self.clientData['jobStates']:
                        jobStatusState = self.clientData['jobStates'][instance]
                        if jobStatusState in ['finished','failed','aborted']:
                           nCompleted += 1
                     csvWriter.writerow(('# completed: %d/%d jobs' % (nCompleted,len(self.clientData['jobStates'])),))
                  except csv.Error:
                     self.logger.log(logging.ERROR,getLogMessage("csv writter failed on %s" % (tmpParameterCombinationsPath)))
                  else:
                     parameterNames = []
                     try:
                        while len(parameterNames) <= 1:
                           parameterNames = next(csvReader)
                     except StopIteration:
                        self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (self.clientData['jobScanPath'])))
                     except csv.Error:
                        self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (self.clientData['jobScanPath'])))
                     else:
                        try:
                           csvWriter.writerow(parameterNames)
                        except csv.Error:
                           self.logger.log(logging.ERROR,getLogMessage("csv writter failed on %s" % (tmpParameterCombinationsPath)))
                        else:
                           try:
                              parameterCombinations = {}
                              for parameterCombination in csvReader:
                                 instance = int(parameterCombination[0])
                                 if instance in self.clientData['jobStates']:
                                    jobStatusState       = self.clientData['jobStates'][instance]
                                    jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
                                    parameterCombination[1] = jobStatusState
                                    if not jobStatusReportOrder in parameterCombinations:
                                       parameterCombinations[jobStatusReportOrder] = []
                                    parameterCombinations[jobStatusReportOrder].append(parameterCombination)
                              activeJobStatusReportOrders = list(parameterCombinations.keys())
                              activeJobStatusReportOrders.sort()
                           except csv.Error:
                              self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % \
                                                                     (self.clientData['jobScanPath'])))
                           else:
                              try:
                                 for jobStatusReportOrder in activeJobStatusReportOrders:
                                    for parameterCombination in parameterCombinations[jobStatusReportOrder]:
                                       csvWriter.writerow(parameterCombination)
                              except csv.Error:
                                 self.logger.log(logging.ERROR,getLogMessage("csv writter failed on %s" % \
                                                                           (tmpParameterCombinationsPath)))
                              else:
                                 copyTmpFile = True
                  finally:
                     fpCSVOut.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (tmpParameterCombinationsPath)))
            except csv.Error:
               self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (self.clientData['jobScanPath'])))
            finally:
               fpCSVIn.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.clientData['jobScanPath'])))

      if copyTmpFile:
         os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])

      jobStatusStates = {}
      for jobStatusState in jobStatusReportOrders:
         jobStatusStates[jobStatusState] = 0
      nCompleted = 0
      for instance in self.clientData['jobStates']:
         jobStatusState = self.clientData['jobStates'][instance]
         if not jobStatusState in jobStatusStates:
            jobStatusStates[jobStatusState] = 0
         jobStatusStates[jobStatusState] += 1
         if jobStatusState in ['finished','failed','aborted']:
            nCompleted += 1
      percentDone = 100.*float(nCompleted)/float(len(self.clientData['jobStates']))

      jobStatusReports = {}
      for jobStatusState in jobStatusStates:
         jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
         if not jobStatusReportOrder in jobStatusReports:
            jobStatusReports[jobStatusReportOrder] = {}
         jobStatusReports[jobStatusReportOrder][jobStatusState] = jobStatusStates[jobStatusState]

      activeJobStatusReportOrders = list(jobStatusReports.keys())
      activeJobStatusReportOrders.sort()
      runProgressMessage = "=SUBMIT-PROGRESS=>"
      for jobStatusReportOrder in activeJobStatusReportOrders:
         for jobStatusState in jobStatusReports[jobStatusReportOrder]:
            runProgressMessage += " %s=%d" % (jobStatusState.replace(' ','_'), \
                                              jobStatusReports[jobStatusReportOrder][jobStatusState])
      runProgressMessage += " %%done=%.2f" % (percentDone)
      if self.clientData['progressReport'] == 'submit':
         if runProgressMessage != self.clientData['runProgressMessage']:
            self.__writeToStdout("%s timestamp=%.1f\n" % (runProgressMessage,time.time()))
      self.clientData['runProgressMessage'] = runProgressMessage


   def getWorkflowProgressMessage(self,
                                  parameterCombinationsPath):
      runProgressMessage = ""
      statusCounts = {}
      statusCounts['aborted']    = 0
      statusCounts['finished']   = 0
      statusCounts['failed']     = 0
      statusCounts['executing']  = 0
      statusCounts['waiting']    = 0
      statusCounts['setting_up'] = 0
      statusCounts['setup']      = 0
      if os.path.exists(parameterCombinationsPath):
         try:
            fpCSVIn = open(parameterCombinationsPath,'rt')
            try:
               csvReader = csv.reader(fpCSVIn)
               parameterNames = next(csvReader)
               while len(parameterNames) > 0 and parameterNames[0][0] == '#':
                  parameterNames = next(csvReader)

               nCompleted = 0
               nInstances = 0
               for parameterCombination in csvReader:
                  jobStatusState = parameterCombination[1]
                  try:
                     statusCounts[jobStatusState] += 1
                  except:
                     statusCounts['setting_up'] += 1

                  if jobStatusState in ['finished','failed','aborted']:
                     nCompleted += 1
                  nInstances += 1
               percentDone = 100.*float(nCompleted)/float(nInstances)
            except StopIteration:
               self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (parameterCombinationsPath)))
            except csv.Error:
               self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (parameterCombinationsPath)))
            else:
               runProgressMessage = "=SUBMIT-PROGRESS=>"
               for jobStatusState in ('aborted','finished','failed','executing','waiting','setting_up','setup'):
                  runProgressMessage +=" %s=%d" % (jobStatusState,statusCounts[jobStatusState])
               runProgressMessage += " %%done=%.2f" % (percentDone)
            finally:
               fpCSVIn.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (parameterCombinationsPath)))

      return(runProgressMessage)


   @staticmethod
   def __which(program):
      def isExe(fpath):
         return(os.path.isfile(fpath) and os.access(fpath,os.X_OK))

      fpath,fname = os.path.split(program)
      if fpath:
         if isExe(program):
            return(program)
      else:
         for path in os.environ["PATH"].split(os.pathsep):
            exePath = os.path.join(path,program)
            if isExe(exePath):
               return(exePath)

      return(None)


   def __getStdInputFile(self):
      stdInputFile = os.devnull
      try:
         isFile = isinstance(sys.stdin,file)
      except:
         isFile = isinstance(sys.stdin,IOBase)
      if isFile and not sys.stdin.closed and not sys.stdin.isatty():
         toCheck = []
         stdinFd = sys.stdin.fileno()
         toCheck.append(stdinFd)
         try:
            ready = select.select(toCheck,[],[],0.) # wait for input
         except select.error:
            ready = {}
            ready[0] = []
         if stdinFd in ready[0]:
            content = sys.stdin.read()
            if content != "":
               stdInputFile = os.path.join(self.clientData['jobPath'],".__%s.stdin" % (self.clientData['localJobId']))
               try:
                  stdinFile = open(stdInputFile,'w')
                  try:
                     stdinFile.write(content)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdInputFile)))
                  finally:
                     stdinFile.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdInputFile)))
               try:
                  si = open('/dev/tty')
               except:
                  pass
               else:
                  os.dup2(si.fileno(),sys.stdin.fileno())
            del content

      return(stdInputFile)


   def getUseEnvironment(self):
      useEnvironment = ""
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = list(filter(reChoice.search,list(os.environ.keys())))
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         includedScripts = []
         useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
         useEnvironment += ". %s\n" % (self.configData['useSetup'])
         if "ENVIRON_CONFIG_DIRS" in os.environ:
            useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
         for environmentVar in environmentVars:
            includedScript = os.environ[environmentVar]
            if not includedScript in includedScripts:
               useEnvironment += "use -e -r " + includedScript + "\n"
               includedScripts.append(includedScript)
         del includedScripts

      return(useEnvironment)


   def startLocalParametric(self):
      from hubzero.submit.LocalWorkflowPEGASUS  import LocalWorkflowPEGASUS
      from hubzero.submit.LocalWorkflowPARALLEL import LocalWorkflowPARALLEL
      exitCode = 0

      useEnvironment = self.getUseEnvironment()

      clientInputFiles = []
      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            clientInputFiles.append(inputFile)

      self.clientData['enteredCommand'] = self.commandParser.getEnteredCommand()
      enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = enteredCommandArguments[0]
      if '/' in userExecutable:
         self.clientData['event'] = '/' + os.path.basename(userExecutable)
      else:
         self.clientData['event'] = userExecutable

      self.clientData['jobPath'] = os.path.join(os.getcwd(),self.clientData['runName'])
      if not os.path.isdir(self.clientData['jobPath']):
         os.mkdir(self.clientData['jobPath'])
      self.clientData['jobScanPath'] = os.path.join(self.clientData['jobPath'],'parameterCombinations.csv')
      self.commandParser.writeParameterCombinations(self.clientData['jobScanPath'])

      self.stdinput = self.__getStdInputFile()

      inputError = False
      executableExistsError = False
      nInstanceIdDigits = max(2,int(math.log10(self.commandParser.getParameterCombinationCount())+1))
      instance = 0
      for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.clientData['jobScanPath']):
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)
         self.clientData['jobStates'][instance] = 'setup'

         instanceDirectory = os.path.join(self.clientData['jobPath'],instanceId)
         if not os.path.isdir(instanceDirectory):
            os.makedirs(instanceDirectory)
         instanceInputsPath = instanceDirectory

         exitCode = 0

         enteredCommand = self.commandParser.getEnteredCommand()
         template = ParameterTemplate(enteredCommand)
         try:
            enteredCommand = template.substitute(substitutions)
         except KeyError as e:
            self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
            inputError = True

         inputFiles = []
         for inputFile in clientInputFiles:
            template = ParameterTemplate(inputFile)
            try:
               inputFile = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               inputError = True
            else:
               inputFiles.append(inputFile)

         if len(enteredCommandArguments) > 1:
            for arg in enteredCommandArguments[1:]:
               if not arg.startswith('-'):
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError as e:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                     inputError = True
                  else:
                     if os.path.isfile(arg.replace('@:','')):
                        if not arg in inputFiles:
                           inputFiles.append(arg)
               else:
                  lexArgs = shlex.split(arg)
                  for lexArg in lexArgs:
                     if '=' in lexArg:
                        arglets = lexArg.split('=')
                        for arglet in arglets:
                           template = ParameterTemplate(arglet)
                           try:
                              arglet = template.substitute(substitutions)
                           except KeyError as e:
                              self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                              inputError = True
                           else:
                              if os.path.isfile(arglet.replace('@:','')):
                                 if not arglet in inputFiles:
                                    inputFiles.append(arglet)

         if self.clientData['environment']:
            for environmentVariableValue in self.clientData['environment'].split():
               environmentVariable,value = environmentVariableValue.split('=')
               template = ParameterTemplate(value)
               try:
                  value = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  inputError = True
               else:
                  if os.path.isfile(value.replace('@:','')):
                     if not value in inputFiles:
                        inputFiles.append(value)

         substitutedInputFiles = []
         for inputFile in inputFiles:
            if '@:' in inputFile:
               inputFile = os.path.expanduser(inputFile.replace('@:',''))
               try:
                  fpInputFile = open(inputFile,'r')
                  try:
                     inputText = fpInputFile.readlines()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                  else:
                     template = ParameterTemplate(''.join(inputText))
                     try:
                        inputText = template.substitute(substitutions)
                     except KeyError as e:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                        inputError = True
                     inputName = os.path.basename(inputFile)
                     inputPath = os.path.join(instanceInputsPath,inputName)
                     try:
                        fpInputPath = open(inputPath,'w')
                        try:
                           fpInputPath.writelines(inputText)
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                        else:
                           substitutedInputFiles.append(inputPath)
                        finally:
                           fpInputPath.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                  finally:
                     fpInputFile.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
            else:
               substitutedInputFiles.append(inputFile)
         del inputFiles
         inputFiles = substitutedInputFiles

         remoteArgs = []

         executable = self.__which(enteredCommandArguments[0])
         if executable:
            executable = os.path.realpath(executable)
            remoteArgs.append(executable)
         else:
            if not executableExistsError:
               self.__writeToStderr("Specified command %s is not in your PATH\n" % (enteredCommandArguments[0]))
            inputError = True
            # allow completion of input checks
            remoteArgs.append(enteredCommandArguments[0])
            executable = enteredCommandArguments[0]
            executableExistsError = True

         for arg in enteredCommandArguments[1:]:
            template = ParameterTemplate(arg)
            try:
               arg = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               inputError = True
            else:
               if arg.startswith('-'):
                  lexArgs = shlex.split(arg)
                  for lexArg in lexArgs:
                     if '=' in lexArg:
                        arglets = lexArg.split('=')
                        rarglets = [arglets[0]]
                        for arglet in arglets[1:]:
                           if '@:' in arglet:
                              if os.path.isfile(os.path.expanduser(arglet.replace('@:',''))):
                                 inputName = os.path.basename(arglet.replace('@:',''))
                                 rarglets.append(inputName)
                           else:
                              if os.path.isfile(arglet):
                                 rarg = os.path.abspath(arglet)
                                 rarglets.append(rarg)
                              else:
                                 rarglets.append(arglet)
                        remoteArgs.append('='.join(rarglets))
                     else:
                        if '@:' in lexArg:
                           if os.path.isfile(os.path.expanduser(lexArg.replace('@:',''))):
                              inputName = os.path.basename(lexArg.replace('@:',''))
                              remoteArgs.append(inputName)
                        else:
                           if os.path.isfile(lexArg):
                              inputName = os.path.abspath(lexArg)
                              remoteArgs.append(inputName)
                           else:
                              remoteArgs.append(lexArg)
               else:
                  if '@:' in arg:
                     if os.path.isfile(os.path.expanduser(arg.replace('@:',''))):
                        inputName = os.path.basename(arg.replace('@:',''))
                        remoteArgs.append(inputName)
                  else:
                     if os.path.isfile(arg):
                        rarg = os.path.abspath(arg)
                        remoteArgs.append(rarg)
                     else:
                        remoteArgs.append(arg)

         remoteCommand = " ".join(remoteArgs)
         arguments     = remoteArgs[1:]

         jobEnvironment = self.clientData['environment']
         if self.clientData['environment']:
            jobEnvironments = []
            for environmentVariableValue in self.clientData['environment'].split():
               environmentVariable,value = environmentVariableValue.split('=')

               template = ParameterTemplate(value)
               try:
                  value = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  inputError = True
               else:
                  if '@:' in value:
                     if os.path.isfile(os.path.expanduser(value.replace('@:',''))):
                        inputName = os.path.basename(value.replace('@:',''))
                        inputPath = os.path.join(instanceInputsPath,inputName)
                        jobEnvironments.append('='.join([environmentVariable,inputPath]))
                  else:
                     if os.path.isfile(value):
                        rarg = os.path.abspath(value)
                        jobEnvironments.append('='.join([environmentVariable,rarg]))
                     else:
                        jobEnvironments.append('='.join([environmentVariable,value]))

            jobEnvironment = ' '.join(jobEnvironments)

         timeHistoryLogs = {}
         timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['exitCode']             = "%s.%s_%s" % (EXITCODE,self.clientData['localJobId'],instanceId)

         localBatchAppScript = LocalBatchAppScript(self.clientData['userName'],self.clientData['userId'],
                                                   self.serverData['submitterClass'],
                                                   self.clientData['runName'],instanceDirectory,
                                                   self.clientData['localJobId'],instanceId,
                                                   executable,self.stdinput,arguments,useEnvironment,jobEnvironment,
                                                   self.serverData['submissionScripts'],timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = localBatchAppScript.buildAppScript()
         appScriptPath = os.path.join(self.clientData['jobPath'],instanceId,appScriptName)

         try:
            fpAppScript = open(appScriptPath,'w')
            try:
               fpAppScript.write(appScript)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (appScriptPath)))
               exitCode = 1
            else:
               if appScriptExecutable:
                  os.chmod(appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
               self.clientData['filesToRemove'].append(appScriptPath)
            finally:
               fpAppScript.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (appScriptPath)))
            exitCode = 1
         del localBatchAppScript

         stdFile = os.path.join(instanceDirectory,"%s_%s.stderr" % (self.clientData['runName'],instanceId))
         self.clientData['emptyFilesToRemove'].append(stdFile)

      nInstances = instance

      instance = 0
      instanceId = str(instance).zfill(nInstanceIdDigits)
      timeHistoryLogs = {}
      timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)

      if self.clientData['isParallel']:
         self.localWorkflowPARALLEL = LocalWorkflowPARALLEL(self.clientData['userName'],self.clientData['userId'],
                                                            self.serverData['submitterClass'],self.clientData['runName'],
                                                            self.clientData['jobPath'],self.clientData['jobPath'],
                                                            self.configData['useSetup'],self.clientData['nStripes'],
                                                            self.clientData['localJobId'],instanceId,
                                                            self.serverData['submissionScripts'],timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.localWorkflowPARALLEL.buildWorkflowScript()
         batchScriptPath = os.path.join(self.clientData['jobPath'],batchScriptName)
      else:
         pegasusTemplates = {}
         if self.serverData['pegasusRC'] != "":
            pegasusTemplates['rc'] = self.serverData['pegasusRC']
         if self.serverData['pegasusSites'] != "":
            pegasusTemplates['sites'] = self.serverData['pegasusSites']

         self.localWorkflowPEGASUS = LocalWorkflowPEGASUS(self.clientData['userName'],self.clientData['userId'],
                                                          self.serverData['submitterClass'],self.clientData['runName'],
                                                          self.clientData['jobPath'],self.clientData['jobPath'],
                                                          self.configData['useSetup'],
                                                          self.clientData['pegasusVersion'],self.clientData['pegasusHome'],
                                                          self.clientData['localJobId'],instanceId,
                                                          pegasusTemplates,self.serverData['localPegasusSite'],
                                                          self.serverData['submissionScripts'],timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.localWorkflowPEGASUS.buildWorkflowScript()
         batchScriptPath = os.path.join(self.clientData['jobPath'],batchScriptName)

      try:
         fpBatchScript = open(batchScriptPath,'w')
         try:
            fpBatchScript.write(batchScript)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (batchScriptPath)))
            exitCode = 1
         else:
            if batchScriptExecutable:
               os.chmod(batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            self.clientData['filesToRemove'].append(batchScriptPath)
         finally:
            fpBatchScript.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (batchScriptPath)))
         exitCode = 1

      enteredCommand = self.commandParser.getEnteredCommand()

      stdFile = os.path.join(self.clientData['jobPath'],"%s.stdout" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)
      stdFile = os.path.join(self.clientData['jobPath'],"%s.stderr" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)

      self.clientData['startDate']      = time.strftime("%a %b %e %X %Z %Y")
      self.clientData['localStartTime'] = time.time()
      self.__updatePegasusWorkflowStatus()

      if inputError:
         self.exit(1)
      else:
         self.clientData['childHasExited'] = False
         child = subprocess.Popen(batchScriptPath,shell=False,bufsize=1,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True,
                                  universal_newlines=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr

         clientMessage = {'messageType':'localWait'}
         self.serverConnection.postJsonMessage(clientMessage)

         if not sys.stdin.closed and not sys.stdin in self.clientData['inputObjects']:
            self.clientData['inputObjects'][sys.stdin] = '#STDIN#'

         if self.clientData['progressReport'] == 'curses':
            self.jobScanner = JobScanner(self.clientData['jobScanPath'])
            self.clientData['jobScannerStarted'] = True
            if not self.jobScanner.start():
               self.jobScanner.finish()
               self.jobScanner = None


   def startLocalPegasusCommand(self):
      from hubzero.submit.LocalBatchPEGASUS import LocalBatchPEGASUS
      exitCode = 0

      self.clientData['enteredCommand'] = self.commandParser.getEnteredCommand()
      enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = enteredCommandArguments[0]
      if '/' in userExecutable:
         self.clientData['event'] = '/' + os.path.basename(userExecutable)
      else:
         self.clientData['event'] = userExecutable
      remoteArgs = []
      for arg in enteredCommandArguments[1:]:
         if   arg.startswith('-'):
            remoteArgs.append(arg)
         elif os.path.isfile(arg) or os.path.isdir(arg):
            rarg = os.path.abspath(arg)
            remoteArgs.append(rarg)
         else:
            remoteArgs.append(arg)
      arguments = " ".join(remoteArgs)

      instance = 1
      nInstanceIdDigits = 2
      instanceId = str(instance).zfill(nInstanceIdDigits)
      self.clientData['jobPath'] = os.path.join(os.getcwd(),self.clientData['runName'],instanceId)
      if not os.path.isdir(self.clientData['jobPath']):
         os.makedirs(self.clientData['jobPath'])

      timeHistoryLogs = {}
      timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)

      pegasusTemplates = {}
      if self.serverData['pegasusRC'] != "":
         pegasusTemplates['rc'] = self.serverData['pegasusRC']
      if self.serverData['pegasusSites'] != "":
         pegasusTemplates['sites'] = self.serverData['pegasusSites']

      self.localBatchPEGASUS = LocalBatchPEGASUS(self.clientData['userName'],self.clientData['userId'],
                                                 self.serverData['submitterClass'],self.clientData['runName'],
                                                 self.clientData['localJobId'],instanceId,
                                                 self.clientData['jobPath'],self.clientData['jobPath'],
                                                 self.configData['useSetup'],
                                                 self.clientData['pegasusVersion'],self.clientData['pegasusHome'],
                                                 userExecutable,arguments,
                                                 pegasusTemplates,self.serverData['localPegasusSite'],
                                                 self.serverData['submissionScripts'],timeHistoryLogs)

      batchScriptName,batchScript,batchScriptExecutable = self.localBatchPEGASUS.getBatchScript()
      batchLogName                                      = self.localBatchPEGASUS.getBatchLog()
      batchScriptPath = os.path.join(self.clientData['jobPath'],batchScriptName)

      try:
         fpBatchScript = open(batchScriptPath,'w')
         try:
            fpBatchScript.write(batchScript)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (batchScriptPath)))
            exitCode = 1
         else:
            if batchScriptExecutable:
               os.chmod(batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            self.clientData['filesToRemove'].append(batchScriptPath)
         finally:
            fpBatchScript.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (batchScriptPath)))
         exitCode = 1

      stdFile = os.path.join(self.clientData['jobPath'],"%s.stdout" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)
      stdFile = os.path.join(self.clientData['jobPath'],"%s.stderr" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)

      self.clientData['localStartTime'] = time.time()

      self.clientData['childHasExited'] = False
      child = subprocess.Popen(batchScriptPath,shell=False,bufsize=1,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True,
                               universal_newlines=True)
      self.clientData['childPid']        = child.pid
      self.clientData['childReadStdout'] = child.stdout
      self.clientData['childReadStderr'] = child.stderr

      clientMessage = {'messageType':'localWait'}
      self.serverConnection.postJsonMessage(clientMessage)


   def startLocalExecute(self):
      self.clientData['localStartTime'] = time.time()
      commandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = commandArguments[0]
      if '/' in userExecutable:
         self.clientData['event'] = '/' + os.path.basename(userExecutable)
      else:
         self.clientData['event'] = userExecutable

      for environmentVariableValue in self.clientData['environment'].split():
         environmentVariable,value = environmentVariableValue.split('=')
         os.environ[environmentVariable] = value

      try:
         tStart = time.time()
         self.clientData['childHasExited'] = False
         child = subprocess.Popen(commandArguments,shell=False,bufsize=1,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  env=os.environ,
                                  close_fds=True,
                                  universal_newlines=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr
         clientMessage = {'messageType':'localWait'}
         self.serverConnection.postJsonMessage(clientMessage)
         if self.serverData['heartbeatInterval'] > 0:
            self.scheduleHeartbeat()
         tFinish = time.time()
         self.logger.log(logging.DEBUG,getLogMessage("server request = %s took %f secs" % \
                                                     ('startLocalExecute',tFinish-tStart)))
      except OSError as e:
         self.__writeToStderr("Failed to invoke %s: %s\n" % (commandArguments[0],e.args[1]))
         self.exit(e.args[0])


   def startLocal(self):
      if self.commandParser.getOption('environment'):
         exitCode = 0
         environment = ""
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               message = "Invalid environment variable %s specified." % (environmentVariableValue)
               self.logger.log(logging.ERROR,getLogMessage(message))
               exitCode = 1
            else:
               environment += environmentVariable + "=" + value + " "

         if not exitCode:
            self.clientData['environment'] = environment.strip()
         else:
            self.exit(exitCode)

      os.environ['SUBMIT_JOB'] = "%s" % (self.serverData['jobId'])

      if   self.clientData['isParametric']:
         self.startLocalParametric()
      elif self.clientData['isPegasusCommand']:
         self.startLocalPegasusCommand()
      else:
         self.startLocalExecute()

      signal.signal(signal.SIGTSTP,signal.SIG_DFL)
      signal.signal(signal.SIGCONT,signal.SIG_DFL)


   def finishLocalParametric(self):
      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      parameterCombinationCount = self.commandParser.getParameterCombinationCount()
      if parameterCombinationCount > 0:
         jobCPUtime = 0.
         nInstanceIdDigits = max(2,int(math.log10(parameterCombinationCount)+1))
         for instance in iterRange(1,parameterCombinationCount+1):
            instanceId = str(instance).zfill(nInstanceIdDigits)

            jobStatistic = JobStatistic(1)
            timeFile = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobStartedTime',timePath)
            timeFile = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobFinshedTime',timePath)
            timerFile = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
            timerPath = os.path.join(self.clientData['jobPath'],instanceId,timerFile)
            jobStatistic.recordTimer(timerPath)
            jobStatistic.setWaitingTime()
            jobStatistic.setElapsedRunTime()

            exitCodeFile = "%s.%s_%s" % (EXITCODE,self.clientData['localJobId'],instanceId)
            exitCodePath = os.path.join(self.clientData['jobPath'],instanceId,exitCodeFile)
            if os.path.exists(exitCodePath):
               os.remove(exitCodePath)

            exitCode       = jobStatistic['exitCode']
            cpuTime        = jobStatistic['userTime']+jobStatistic['sysTime']
            jobCPUtime    += cpuTime
            elapsedRunTime = jobStatistic['elapsedRunTime']

            for ftype in '.dax','_sites.xml','_sites.yml','.pegasusrc','_tc.txt','_tc.yml','.make':
               pegasusFile = os.path.join(self.clientData['jobPath'],"%s_%s%s" % (self.clientData['localJobId'],instanceId,ftype))
               if os.path.isfile(pegasusFile):
                  os.remove(pegasusFile)
               else:
                  pegasusFile = os.path.join(self.clientData['jobPath'],"%s%s" % (self.clientData['localJobId'],ftype))
                  if os.path.isfile(pegasusFile):
                     os.remove(pegasusFile)

            clientMessage = {'messageType':'localMetrics',
                             'instanceIndex':instance,
                             'status':exitCode,
                             'cpuTime':cpuTime,
                             'realTime':elapsedRunTime,
                             'event':self.clientData['event']}
            self.serverConnection.postJsonMessage(clientMessage)
            del jobStatistic

         instance = 0
         instanceId = str(instance).zfill(nInstanceIdDigits)
         jobStatistic = JobStatistic(1)
         timeFile = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
         timePath = os.path.join(self.clientData['jobPath'],timeFile)
         jobStatistic.recordTime('jobStartedTime',timePath)
         timeFile = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
         timePath = os.path.join(self.clientData['jobPath'],timeFile)
         jobStatistic.recordTime('jobFinshedTime',timePath)
         timerFile = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
         timerPath = os.path.join(self.clientData['jobPath'],timerFile)
         jobStatistic.recordTimer(timerPath)
         jobStatistic.setWaitingTime()
         jobStatistic.setElapsedRunTime()

         exitCode       = jobStatistic['exitCode']
         cpuTime        = jobStatistic['userTime']+jobStatistic['sysTime']
         cpuTime       -= jobCPUtime
         elapsedRunTime = jobStatistic['elapsedRunTime']

         clientMessage = {'messageType':'localMetrics',
                          'instanceIndex':instance,
                          'status':exitCode,
                          'cpuTime':cpuTime,
                          'realTime':elapsedRunTime,
                          'event':"[sweep]"}
         self.serverConnection.postJsonMessage(clientMessage)
         del jobStatistic

         workDirectoryName = os.path.join(self.clientData['jobPath'],'work')
         if os.path.isdir(workDirectoryName):
            shutil.rmtree(workDirectoryName,True)
         workDirectoryName = os.path.join(self.clientData['jobPath'],'scratch')
         if os.path.isdir(workDirectoryName):
            shutil.rmtree(workDirectoryName,True)

         if self.clientData['progressReport'] == 'text' or self.clientData['progressReport'] == 'pegasus':
            self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (self.clientData['jobPath']))


   def moveTree(self,
                src,
                dst,
                symlinks=False):
      if os.path.isdir(src):
         if os.path.exists(dst):
            if not os.path.isdir(dst):
               self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (dst)))
               return
         else:
            os.mkdir(dst)
         names = os.listdir(src)
         for name in names:
            srcPath = os.path.join(src,name)
            dstPath = os.path.join(dst,name)
            try:
               if symlinks and os.path.islink(srcPath):
                  linkto = os.readlink(srcPath)
                  os.symlink(linkto,dstPath)
               elif os.path.isdir(srcPath):
                  self.moveTree(srcPath,dstPath,symlinks)
               else:
                  os.rename(srcPath,dstPath)
            except (IOError,os.error) as e:
               self.logger.log(logging.ERROR,getLogMessage("moveTree: Can't move %s to %s: %s" % (srcPath,dstPath,str(e.args[0]))))
      else:
         self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (src)))


   def finishLocalPegasusCommand(self):
      nInstanceIdDigits = 2
      instance = 1
      instanceId = str(instance).zfill(nInstanceIdDigits)

      workDirectoryName = os.path.join(self.clientData['jobPath'],'work')
      if os.path.isdir(workDirectoryName):
         shutil.rmtree(workDirectoryName,True)
      workDirectoryName = os.path.join(self.clientData['jobPath'],'scratch')
      if os.path.isdir(workDirectoryName):
         shutil.rmtree(workDirectoryName,True)
      for ftype in '.dax','_sites.xml','_sites.yml','.pegasusrc','_tc.txt','_tc.yml':
         pegasusFile = os.path.join(self.clientData['jobPath'],"%s_%s%s" % (self.clientData['localJobId'],instanceId,ftype))
         if os.path.isfile(pegasusFile):
            os.remove(pegasusFile)
         else:
            pegasusFile = os.path.join(self.clientData['jobPath'],"%s%s" % (self.clientData['localJobId'],ftype))
            if os.path.isfile(pegasusFile):
               os.remove(pegasusFile)

      self.moveTree(self.clientData['jobPath'],os.getcwd())
      runDirectory = os.path.join(os.getcwd(),self.clientData['runName'])
      if os.path.exists(runDirectory):
         shutil.rmtree(runDirectory,True)


   def finishLocal(self):
      signal.signal(signal.SIGCONT,self.handleContinueSignal)
      signal.signal(signal.SIGTSTP,self.handleSuspendSignal)

      if self.clientData['isParametric']:
         self.finishLocalParametric()

      for fileToRemove in self.clientData['filesToRemove']:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.clientData['emptyFilesToRemove']:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if self.clientData['isPegasusCommand']:
         self.finishLocalPegasusCommand()


   def scanLocalOutputFile(self,
                           outputFileObject):
      outChunk = os.read(outputFileObject.fileno(),1024).decode('utf-8')
      if len(outChunk) > 0:
         if self.clientData['isParametric']:
            update = False
            if self.clientData['isParallel']:
               jobStartPattern    = re.compile('Executing JOB ([0-9]+)_([0-9]+)$')
               jobCompletePattern = re.compile('Completed JOB ([0-9]+)_([0-9]+)$')
            else:
               jobStartPattern    = re.compile('.* Executing JOB ([^ ]*) *-n *([^ ]*) *-N *([^ ]*).*/([^ ]*)')
               jobCompletePattern = re.compile('.* JOB *([0-9]+)_([0-9]+)[\.|_]sh_[A-Za-z0-9]*[-_][0-9]*')
            for record in outChunk.split('\n'):
               matchObj = jobStartPattern.match(record)
               if matchObj:
                  if self.clientData['isParallel']:
                     instance = int(matchObj.group(2))
                     self.clientData['jobStates'][instance] = 'executing'
                     update = True
                  else:
                     if matchObj.group(2) == matchObj.group(4):
                        instance = int(re.split('[_-]',matchObj.group(3))[-1])
                        self.clientData['jobStates'][instance] = 'executing'
                        update = True
               else:
                  matchObj = jobCompletePattern.match(record)
                  if matchObj:
                     instance = int(matchObj.group(2))
                     instanceExitCode = 0
                     exitCodeFile = "%s.%s_%s" % (EXITCODE,matchObj.group(1),matchObj.group(2))
                     exitCodePath = os.path.join(self.clientData['jobPath'],matchObj.group(2),exitCodeFile)
                     if os.path.exists(exitCodePath):
                        try:
                           fpExitCode = open(exitCodePath,'r')
                           try:
                              instanceExitCode = int(fpExitCode.readline())
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (exitCodePath)))
                           finally:
                              fpExitCode.close()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (exitCodePath)))
                     if instanceExitCode:
                        self.clientData['jobStates'][instance] = 'failed'
                     else:
                        self.clientData['jobStates'][instance] = 'finished'
                     update = True
            if update:
               self.__updatePegasusWorkflowStatus()
         else:
            if outputFileObject == self.clientData['childReadStdout']:
               self.__writeToStdout(outChunk)
            if outputFileObject == self.clientData['childReadStderr']:
               self.__writeToStderr(outChunk)
      else:
         self.closeFile(outputFileObject)


   def checkInputsExist(self):
      inputsExist = True
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         inputFiles = []
         if self.clientData['isParametric']:
            for substitutions in self.commandParser.getNextParameterCombination():
               if self.commandParser.getOption('inputFiles'):
                  for inputFile in self.commandParser.getOption('inputFiles'):
                     template = ParameterTemplate(inputFile)
                     actualInputFile = os.path.expanduser(template.substitute(substitutions).replace('@:',''))
                     if not actualInputFile in inputFiles:
                        inputFiles.append(actualInputFile)
         else:
            if self.commandParser.getOption('inputFiles'):
               for inputFile in self.commandParser.getOption('inputFiles'):
                  inputFiles.append(inputFile)

         if len(inputFiles) > 0:
            for inputFile in inputFiles:
               if not os.path.exists(inputFile):
                  self.__writeToStderr("Input %s does not exist\n" % (inputFile))
                  inputsExist = False

      return(inputsExist)


   def mapInputFiles(self):
      exitCode = 0
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         inputFiles = []
         enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
         if self.clientData['isParametric']:
            for substitutions in self.commandParser.getNextParameterCombination():
               if self.commandParser.getOption('inputFiles'):
                  for inputFile in self.commandParser.getOption('inputFiles'):
                     template = ParameterTemplate(inputFile)
                     actualInputFile = os.path.expanduser(template.substitute(substitutions).replace('@:',''))
                     if not actualInputFile in inputFiles:
                        inputFiles.append(actualInputFile)

               if self.commandParser.getOption('environment'):
                  for environmentVariableValue in self.commandParser.getOption('environment'):
                     template = ParameterTemplate(environmentVariableValue)
                     try:
                        environmentVariableValue = template.substitute(substitutions)
                     except KeyError as e:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                        exitCode = 1

                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]
                     if environmentVariable and value:
                        actualValue = os.path.expanduser(value.replace('@:',''))
                        if os.path.isfile(actualValue) or os.path.isdir(actualValue):
                           if not actualValue in inputFiles:
                              inputFiles.append(actualValue)

               if len(enteredCommandArguments) > 0:
                  for arg in enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        arg = template.substitute(substitutions)
                     except KeyError as e:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                        exitCode = 1
                     if not arg.startswith('-'):
                        if '=' in arg:
                           arglets = arg.split('=')
                           for arglet in arglets[1:]:
                              actualInputFile = os.path.expanduser(arglet.replace('@:',''))
                              if os.path.isfile(actualInputFile) or os.path.isdir(actualInputFile):
                                 if not actualInputFile in inputFiles:
                                    inputFiles.append(actualInputFile)
                        else:
                           actualInputFile = os.path.expanduser(arg.replace('@:',''))
                           if os.path.isfile(actualInputFile) or os.path.isdir(actualInputFile):
                              if not actualInputFile in inputFiles:
                                 inputFiles.append(actualInputFile)
                     else:
                        lexArgs = shlex.split(arg)
                        for lexArg in lexArgs:
                           if '=' in lexArg:
                              arglets = lexArg.split('=')
                              for arglet in arglets[1:]:
                                 actualInputFile = os.path.expanduser(arglet.replace('@:',''))
                                 if os.path.isfile(actualInputFile) or os.path.isdir(actualInputFile):
                                    if not actualInputFile in inputFiles:
                                       inputFiles.append(actualInputFile)
                           else:
                              actualInputFile = os.path.expanduser(lexArg.replace('@:',''))
                              if os.path.isfile(actualInputFile) or os.path.isdir(actualInputFile):
                                 if not actualInputFile in inputFiles:
                                    inputFiles.append(actualInputFile)
         else:
            if self.commandParser.getOption('inputFiles'):
               for inputFile in self.commandParser.getOption('inputFiles'):
                  inputFiles.append(inputFile)

            if self.commandParser.getOption('environment'):
               for environmentVariableValue in self.commandParser.getOption('environment'):
                  environmentVariable = ""
                  value               = ""
                  nParts = len(environmentVariableValue.split('='))
                  if   nParts == 1:
                     environmentVariable = environmentVariableValue.strip()
                     if environmentVariable in os.environ:
                        value = os.environ[environmentVariable]
                  elif nParts == 2:
                     environmentVariable,value = environmentVariableValue.split('=')
                     environmentVariable = environmentVariable.strip()
                     value               = value.strip()
                     if value == "":
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                  if environmentVariable and value:
                     if os.path.isfile(value) or os.path.isdir(value):
                        if not value in inputFiles:
                           inputFiles.append(value)

            if len(enteredCommandArguments) > 0:
               for arg in enteredCommandArguments:
                  if not arg.startswith('-'):
                     if '=' in arg:
                        arglets = arg.split('=')
                        for arglet in arglets[1:]:
                           if os.path.isfile(arglet) or os.path.isdir(arglet):
                              if not arglet in inputFiles:
                                 inputFiles.append(arglet)
                     else:
                        if os.path.isfile(arg) or os.path.isdir(arg):
                           if not arg in inputFiles:
                              inputFiles.append(arg)
                  else:
                     lexArgs = shlex.split(arg)
                     for lexArg in lexArgs:
                        if '=' in lexArg:
                           arglets = lexArg.split('=')
                           for arglet in arglets[1:]:
                              if os.path.isfile(arglet) or os.path.isdir(arglet):
                                 if not arglet in inputFiles:
                                    inputFiles.append(arglet)
                        else:
                           if os.path.isfile(lexArg) or os.path.isdir(lexArg):
                              if not lexArg in inputFiles:
                                 inputFiles.append(lexArg)

         if len(inputFiles) > 0:
            for inputFile in inputFiles:
               fileProperties = self.__getFileProperties(inputFile)
               clientMessage = {'messageType':'inputFile',
                                'path':fileProperties['filePath'],
                                'properties':fileProperties}
               self.serverConnection.postJsonMessage(clientMessage)

            if len(self.clientData['mountPoints']) > 0:
               self.logger.log(logging.DEBUG,getLogMessage("mapInputFiles:\n%s" % \
                                 (pprint.pformat(self.clientData['mountPoints']))))

      if exitCode:
         self.exit(exitCode)
      else:
         clientMessage = {'messageType':'inputFileInodesSent'}
         self.serverConnection.postJsonMessage(clientMessage)
         self.clientData['isExportingInputTar'] = True


   def wait(self):
      if self.serverConnection.isConnected():
         self.clientData['localWaiting'] = True
         clientMessage = {'messageType':'exit'}
         self.serverConnection.postJsonMessage(clientMessage)


   def processServerRequests(self):
      message = self.serverConnection.pullMessage(0)
      while message:
         args = message.split()
         if args[0] != 'null' and args[0] != 'json':
            self.logger.log(logging.DEBUG,getLogMessage("server request = %s" % (args[0])))

         if args[0] == 'json':
            jsonMessageLength = int(args[1])
            jsonMessage = self.serverConnection.pullMessage(jsonMessageLength)
            if len(jsonMessage) > 0:
               try:
                  serverJsonObject = json.loads(jsonMessage)
               except ValueError:
                  self.logger.log(logging.ERROR,getLogMessage("JSON object %s could not be decoded" % (jsonMessage)))
               else:
                  if serverJsonObject['messageType'] != 'null':
                     tStart = time.time()
                     self.logger.log(logging.DEBUG,getLogMessage("server request = %s" % (serverJsonObject['messageType'])))
                  if   serverJsonObject['messageType'] == 'serverId':
                     self.serverData['serverId'] = uuid.UUID(serverJsonObject['serverId'])
                  elif serverJsonObject['messageType'] == 'serverVersion':
                     self.serverData['version'] = serverJsonObject['version']
                  elif serverJsonObject['messageType'] == 'message':
                     self.logger.log(logging.INFO,getLogMessage(serverJsonObject['text']))
                  elif serverJsonObject['messageType'] == 'serverReadyForSignon':
                     clientMessage = {'messageType':'signon'}
                     if 'encryptedMessage' in serverJsonObject:
                        encryptedMessage = base64.b64decode(serverJsonObject['encryptedMessage'].encode('utf-8'))
                        command = ['openssl','rsautl','-decrypt','-inkey',os.environ['SUBMITPRIVATEKEYPATH']]
                        child = subprocess.Popen(command,
                                                 stdin=subprocess.PIPE,
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.PIPE,
                                                 close_fds=True)
                        stdOutput,stdError = child.communicate(encryptedMessage)
                        exitStatus = child.returncode
                        if exitStatus == 0:
                           try:
                              secretMessage = uuid.UUID(stdOutput)
                           except:
                              secretMessage = uuid.UUID(stdOutput.decode('utf-8'))
                           finally:
                              secretMessage = secretMessage.hex
                           authenticationHash = hashlib.sha256()
                           authenticationHash.update(self.clientData['clientId'].hex.encode('utf-8'))
                           authenticationHash.update(secretMessage.encode('utf-8'))
                           authenticationHash.update(self.serverData['serverId'].hex.encode('utf-8'))
                           clientMessage['authenticationHash'] = authenticationHash.hexdigest()
                     self.serverConnection.postJsonMessage(clientMessage)
                  elif serverJsonObject['messageType'] == 'authz':
                     self.serverData['authz'] = serverJsonObject['success']
                     if not self.serverData['authz']:
                        self.logger.log(logging.ERROR,getLogMessage("Session authentication failed"))
                        self.clientIdAuthAttributes.clearSessionToken()
                        tryAgain = serverJsonObject['retry']
                        if tryAgain:
                           self.getUserAttributes()
                           if self.haveIdAuthAttributes():
                              self.signon()
                     else:
                        if   self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONEXECUTE:
                           if self.clientData['attachId'] == ZERO:
                              self.mapSubmitCommandFiles()
                           else:
                              clientMessage = {'messageType':'parseArguments'}
                              self.serverConnection.postJsonMessage(clientMessage)
                        elif self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONHEARTBEAT:
                           clientMessage = {'messageType':'clientHeartbeat'}
                           self.serverConnection.postJsonMessage(clientMessage)
                           self.scheduleHeartbeat()
                        elif self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONSENDMETRICS:
                           self.finishLocal()
                           clientMessage = {'messageType':'localExit',
                                            'status':self.clientData['exitCode'],
                                            'cpuTime':self.clientData['cpuTime'],
                                            'realTime':self.clientData['realTime'],
                                            'event':self.clientData['event']}
                           if 'RAPPTURE_GENERATED' in os.environ:
                              clientMessage['generated'] = os.environ['RAPPTURE_GENERATED']
                           self.serverConnection.postJsonMessage(clientMessage)
                           self.clientData['readyToExit'] = True
                        elif self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONHARVEST:
                           clientMessage = {'messageType':'parseHarvestArguments'}
                           self.serverConnection.postJsonMessage(clientMessage)
                        elif self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONSENDMEASUREMENTS:
                           self.serverConnection.postJsonMessage(self.clientData['measurementsData'])
#                          self.clientData['readyToExit'] = True

                        self.clientData['authorizedAction'] == self.AUTHORIZEDACTIONNONE
                  elif serverJsonObject['messageType'] == 'writeStdout':
                     self.__writeToStdout(serverJsonObject['text'])
                  elif serverJsonObject['messageType'] == 'writeStderr':
                     self.__writeToStderr(serverJsonObject['text'])
                  elif serverJsonObject['messageType'] == 'write':
                     serverOutputFile = serverJsonObject['file']
                     text = base64.b64decode(serverJsonObject['text'].encode('utf-8'))
                     if len(text) > 0:
                        found = False
                        if not found:
                           for importObject in self.clientData['importObjects']:
                              if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                                 self.clientData['importObjects'][importObject]['buffer'] += text
#                                self.logger.log(logging.DEBUG,getLogMessage("importObject buffer length + %d = %d" % \
#                                           (len(text),len(self.clientData['importObjects'][importObject]['buffer']))))
                                 found = True
                                 break
                        if not found:
                           self.logger.log(logging.ERROR,getLogMessage("File object not found to match %s" % (serverOutputFile)))
                  elif serverJsonObject['messageType'] == 'close':
                     serverOutputFile = serverJsonObject['file']
                     found = False
                     for importObject in self.clientData['importObjects']:
                        if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                           self.clientData['closePendingObjects'].append(importObject)
                           found = True
                           break
                     if not found:
                        self.logger.log(logging.ERROR,getLogMessage("File object not found to match %s" % (serverOutputFile)))
                  elif serverJsonObject['messageType'] == 'jobId':
                     self.serverData['jobId'] = serverJsonObject['jobId']
                     if self.serverData['jobId'] == 0:
                        self.exit(1)
                     else:
                        self.clientData['localJobId'] = "%08d" % (self.serverData['jobId'])
                        logSetJobId(self.serverData['jobId'])
                        if not self.clientData['runName']:
                           self.clientData['runName'] = self.clientData['localJobId']
                     if self.clientData['reportMetrics']:
                        self.__writeToStderr("=SUBMIT-METRICS=> job=%s\n" % (self.clientData['localJobId']))
                  elif serverJsonObject['messageType'] == 'jobToken':
                     self.serverData['jobToken'] = serverJsonObject['token']
                  elif serverJsonObject['messageType'] == 'submitterClass':
                     self.serverData['submitterClass'] = serverJsonObject['class']
                  elif serverJsonObject['messageType'] == 'runName':
                     self.clientData['runName']    = serverJsonObject['runName']
                     self.serverData['jobId']      = serverJsonObject['jobId']
                     self.clientData['localJobId'] = "%08d" % (self.serverData['jobId'])
                     logSetJobId(self.serverData['jobId'])
                  elif serverJsonObject['messageType'] == 'event':
                     self.serverData['event'] = serverJsonObject['text']
                  elif serverJsonObject['messageType'] == 'operationMode':
                     self.clientData['operationMode'] = serverJsonObject['operationMode']
                  elif serverJsonObject['messageType'] == 'createdSessions':
                     self.serverData['createdSessions'] = serverJsonObject['createdSessions']
                  elif serverJsonObject['messageType'] == 'heartbeatInterval':
                     self.serverData['heartbeatInterval'] = serverJsonObject['interval']
                  elif serverJsonObject['messageType'] == 'pegasusRC':
                     self.serverData['pegasusRC'] = serverJsonObject['text']
                  elif serverJsonObject['messageType'] == 'pegasusSites':
                     self.serverData['pegasusSites']     = serverJsonObject['text']
                     self.serverData['localPegasusSite'] = serverJsonObject['localSite']
                  elif serverJsonObject['messageType'] == 'submissionScripts':
                     submissionScripts = serverJsonObject['submissionScripts']
                     self.serverData['submissionScripts'] = SubmissionScriptsInfo('Client',
                                                                                  submissionScripts=submissionScripts)
                  elif serverJsonObject['messageType'] == 'clientSuspended':
                     self.clientData['suspended'] = True
                     signal.signal(signal.SIGCONT,self.handleContinueSignal)
                     signal.signal(signal.SIGTSTP,signal.SIG_DFL)
                     os.kill(os.getpid(),signal.SIGTSTP)
                  elif serverJsonObject['messageType'] == 'clientContinued':
                     self.clientData['suspended'] = False
                     signal.signal(signal.SIGCONT,signal.SIG_DFL)
                     signal.signal(signal.SIGTSTP,self.handleSuspendSignal)
                     os.kill(os.getpid(),signal.SIGCONT)
                  elif serverJsonObject['messageType'] == 'jobScanUpdate':
                     messageText = serverJsonObject['text']
                     if self.clientData['jobScanPath'] == "":
                        jobScanPath = os.path.join(self.clientData['workDirectory'],self.clientData['runName'])
                        if not os.path.exists(jobScanPath):
                           os.makedirs(jobScanPath)
                        jobScanPath = os.path.join(jobScanPath,'parameterCombinations.csv')
                        self.clientData['jobScanPath'] = jobScanPath
                     try:
                        parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
                        parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
                        if '.' in parameterCombinationsBase:
                           parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
                        tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
                        tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
                        renameTmpFile = False
                        try:
                           fpJobScan = open(tmpParameterCombinationsPath,'w')
                           try:
                              fpJobScan.write(messageText)
                              renameTmpFile = True
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % \
                                                                       (tmpParameterCombinationsPath)))
                           finally:
                              fpJobScan.close()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (tmpParameterCombinationsPath)))
                        if renameTmpFile:
                           os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])
                     except:
                        pass
                     if self.clientData['progressReport'] == 'curses':
                        if os.path.exists(self.clientData['jobScanPath']):
                           if not self.jobScanner and not self.clientData['jobScannerStarted']:
                              self.jobScanner = JobScanner(self.clientData['jobScanPath'])
                              self.clientData['jobScannerStarted'] = True
                              if not self.jobScanner.start():
                                 self.jobScanner.finish()
                                 self.jobScanner = None
                  elif serverJsonObject['messageType'] == 'runStatusUpdate':
                     messageText = serverJsonObject['text']
                     saveFile    = serverJsonObject['saveFile']
                     if self.clientData['runStatusPath'] == "":
                        runStatusPath = os.path.join(self.clientData['workDirectory'],self.clientData['runName'])
                        if saveFile and not os.path.exists(runStatusPath):
                           os.makedirs(runStatusPath)
                        runStatusPath = os.path.join(runStatusPath,'pegasusstatus.txt')
                        self.clientData['runStatusPath'] = runStatusPath
                     try:
                        if saveFile:
                           runStatusDir  = os.path.dirname(self.clientData['runStatusPath'])
                           runStatusBase = os.path.basename(self.clientData['runStatusPath'])
                           if '.' in runStatusBase:
                              runStatusBase = runStatusBase.split('.')[0]
                           tmpRunStatusFile = runStatusBase + '.tmp'
                           tmpRunStatusPath = os.path.join(runStatusDir,tmpRunStatusFile)
                           renameTmpFile = False
                           try:
                              fpJobScan = open(tmpRunStatusPath,'w')
                              try:
                                 fpJobScan.write(messageText)
                                 renameTmpFile = True
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (tmpRunStatusPath)))
                              finally:
                                 fpJobScan.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (tmpRunStatusPath)))
                           if renameTmpFile:
                              os.rename(tmpRunStatusPath,self.clientData['runStatusPath'])
                     except:
                        pass
                  elif serverJsonObject['messageType'] == 'serverReadyForIO':
                     clientMessage = {'messageType':'serverReadyForIO'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.clientData['isServerReadyForIO'] = True
                     if self.clientData['localExecution']:
                        self.startLocal()
                     else:
                        if not sys.stdin.closed:
                           self.clientData['inputObjects'][sys.stdin] = '#STDIN#'
                        if self.clientData['detach']:
                           clientMessage = {'messageType':'detachSignal'}
                           self.serverConnection.postJsonMessage(clientMessage)
                  elif serverJsonObject['messageType'] == 'readyToDetach':
                     clientMessage = {'messageType':'detach'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     if self.jobScanner:
                        self.jobScanner.finish()
                        self.jobScanner = None
                     self.clientData['readyToExit'] = True
                     self.clientData['exitCode']    = 0
                     if self.clientData['runName'] and self.serverData['jobId']:
                        notificationMessage = \
                                     "Detaching from run %s.\n" % (self.clientData['runName']) + \
                                     "    Check run status with the command: submit --status %d\n" % (self.serverData['jobId']) + \
                                     "       Terminate run with the command: submit --kill %d\n" % (self.serverData['jobId']) + \
                                     "   Re-connect to run with the command: submit --attach %d\n" % (self.serverData['jobId'])
                        self.__writeToStdout(notificationMessage)
                  elif serverJsonObject['messageType'] == 'attached':
                     notificationMessage = "Attached to run %d.\n" % (self.clientData['attachId'])
                     self.__writeToStdout(notificationMessage)
                     clientMessage = {'messageType':'clientVersion',
                                      'version':self.clientData['version']}
                     self.serverConnection.postJsonMessage(clientMessage)
                  elif serverJsonObject['messageType'] == 'serverReadyForInputMapping':
                     self.mapInputFiles()
                  elif serverJsonObject['messageType'] == 'addExportFile':
                     self.clientData['transferFiles'].append(serverJsonObject['file'])
                  elif serverJsonObject['messageType'] == 'exportFiles':
                     transferTarFile = serverJsonObject['file']
                     transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
                     if transferTarFile.endswith('.gz'):
                        tarCommand = buildCreateTarCommand(transferTarPath,
                                                           self.clientData['transferFiles'],
                                                           useAbsolutePaths=True,
                                                           doGzip=True)
                     else:
                        tarCommand = buildCreateTarCommand(transferTarPath,
                                                           self.clientData['transferFiles'],
                                                           useAbsolutePaths=True,
                                                           doGzip=False)
                     self.__writeToStdout("Export file transfer initiated. %s\n" % (time.ctime()))
#                    self.logger.log(logging.DEBUG,getLogMessage("command: " + str(tarCommand)))
                     try:
                        child = subprocess.Popen(tarCommand,
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.PIPE,
                                                 close_fds=True,
                                                 universal_newlines=True)
                        tarStdOutput,tarStdError = child.communicate()
                        tarExitStatus = child.returncode
                        if tarExitStatus != 0:
                           self.logger.log(logging.ERROR,getLogMessage("Failed to write transfer tarfile %s" % (transferTarPath)))
                           if tarStdOutput:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                           if tarStdError:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                           exitCode = 1
                        else:
                           try:
                              fp = open(transferTarPath,'rb')
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (transferTarPath)))
                           else:
                              self.clientData['exportObjects'][fp] = transferTarFile
                     except OSError as e:
                        self.logger.log(logging.ERROR,getLogMessage("Failed to to create transfer tarfile %s" % (transferTarPath)))
                        self.logger.log(logging.ERROR,getLogMessage(e.args[1]))
                  elif serverJsonObject['messageType'] == 'noExportFiles':
                     clientMessage = {'messageType':'startRemote'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.clientData['isExportingInputTar'] = False
                  elif serverJsonObject['messageType'] == 'exportFilesComplete':
                     clientMessage = {'messageType':'startRemote'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.clientData['isExportingInputTar'] = False
                     self.__writeToStdout("Export file transfer complete. %s\n" % (time.ctime()))
                  elif serverJsonObject['messageType'] == 'addExportCommandFile':
                     self.clientData['commandFiles'].append(serverJsonObject['file'])
                  elif serverJsonObject['messageType'] == 'exportCommandFiles':
                     transferTarFile = serverJsonObject['file']
                     transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
                     if transferTarFile.endswith('.gz'):
                        tarCommand = buildCreateTarCommand(transferTarPath,
                                                           self.clientData['commandFiles'],
                                                           useAbsolutePaths=True,
                                                           doGzip=True)
                     else:
                        tarCommand = buildCreateTarCommand(transferTarPath,
                                                           self.clientData['commandFiles'],
                                                           useAbsolutePaths=True,
                                                           doGzip=False)
                     self.__writeToStdout("Export command file transfer initiated. %s\n" % (time.ctime()))
#                    self.logger.log(logging.DEBUG,getLogMessage("command: " + str(tarCommand)))
                     try:
                        child = subprocess.Popen(tarCommand,
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.PIPE,
                                                 close_fds=True,
                                                 universal_newlines=True)
                        tarStdOutput,tarStdError = child.communicate()
                        tarExitStatus = child.returncode
                        if tarExitStatus != 0:
                           self.logger.log(logging.ERROR,getLogMessage("Failed to write transfer tarfile %s" % (transferTarPath)))
                           if tarStdOutput:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                           if tarStdError:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                           exitCode = 1
                        else:
                           try:
                              fp = open(transferTarPath,'rb')
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (transferTarPath)))
                           else:
                              self.clientData['exportObjects'][fp] = transferTarFile
                     except OSError as e:
                        self.logger.log(logging.ERROR,getLogMessage("Failed to to create transfer tarfile %s" % (transferTarPath)))
                        self.logger.log(logging.ERROR,getLogMessage(e.args[1]))
                  elif serverJsonObject['messageType'] == 'noExportCommandFiles':
                     clientMessage = {'messageType':'parseArguments'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.clientData['isExportingSubmitTar'] = False
                  elif serverJsonObject['messageType'] == 'exportCommandFilesComplete':
                     clientMessage = {'messageType':'parseArguments'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.clientData['isExportingSubmitTar'] = False
                     self.__writeToStdout("Export command file transfer complete. %s\n" % (time.ctime()))
                  elif serverJsonObject['messageType'] == 'argumentsParsed':
                     if   self.clientData['localExecution']:
                        clientMessage = {'messageType':'startLocal'}
                        self.serverConnection.postJsonMessage(clientMessage)
                     elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEHIT:
                        clientMessage = {'messageType':'doCacheHit',
                                         'cacheSquid':self.clientData['cacheSquid']}
                        self.serverConnection.postJsonMessage(clientMessage)
                     elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEMISS:
                        clientMessage = {'messageType':'doCacheMiss',
                                         'cacheSquid':self.clientData['cacheSquid']}
                        self.serverConnection.postJsonMessage(clientMessage)
                     elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNCACHEPUBLISH:
                        clientMessage = {'messageType':'doCachePublish',
                                         'cacheSquid':self.clientData['cacheSquid']}
                        self.serverConnection.postJsonMessage(clientMessage)
                     elif not (self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNPROXY):
                        clientMessage = {'messageType':'setupRemote'}
                        self.serverConnection.postJsonMessage(clientMessage)
                  elif serverJsonObject['messageType'] == 'harvestArgumentsParsed':
                     clientMessage = {'messageType':'startHarvest'}
                     self.serverConnection.postJsonMessage(clientMessage)
                  elif serverJsonObject['messageType'] == 'childHasExited':
                     self.serverData['childHasExited'] = serverJsonObject['childHasExited']
                     if not self.clientData['importFilesComplete']:
                        clientMessage = {'messageType':'clientReadyForIO'}
                        self.serverConnection.postJsonMessage(clientMessage)
                        self.serverData['isBuildingOutputTar']  = True
                        self.clientData['isImportingOutputTar'] = True
                  elif serverJsonObject['messageType'] == 'noImportFile':
                     self.clientData['importFilesComplete'] = True
                     clientMessage = {'messageType':'importFilesComplete'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.serverData['isBuildingOutputTar']  = False
                     self.clientData['isImportingOutputTar'] = False
                  elif serverJsonObject['messageType'] == 'importFile':
                     transferTarFile = serverJsonObject['file']
                     transferTarPath = os.path.join(self.clientData['workDirectory'],transferTarFile)
                     try:
                        fp = open(transferTarPath,'wb')
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("Failed to to create transfer tarfile %s" % \
                                                                                              (transferTarPath)))
                     else:
                        self.clientData['importObjects'][fp]           = {}
                        self.clientData['importObjects'][fp]['path']   = transferTarFile
                        self.clientData['importObjects'][fp]['buffer'] = b""
                        clientMessage = {'messageType':'importFileReady',
                                         'file':transferTarFile}
                        self.serverConnection.postJsonMessage(clientMessage)
                     self.serverData['isBuildingOutputTar'] = False
                  elif serverJsonObject['messageType'] == 'importFileFailed':
                     transferTarFile = serverJsonObject['file']
                     fpDel = None
                     for fp in self.clientData['importObjects']:
                        if self.clientData['importObjects'][fp]['path'] == transferTarFile:
                           fpDel = fp
                           break
                     if fpDel:
                        del self.clientData['importObjects'][fpDel]
                     self.clientData['importFilesComplete'] = True
                     clientMessage = {'messageType':'importFilesComplete'}
                     self.serverConnection.postJsonMessage(clientMessage)
                     self.serverData['isBuildingOutputTar']  = False
                     self.clientData['isImportingOutputTar'] = False
                  elif serverJsonObject['messageType'] == 'serverExit':
                     self.serverData['exited'] = True
                     if self.clientData['disableJobMonitoring'] and not self.clientData['jobMonitoringComplete']:
                        self.clientData['jobMonitoringWait'] = True
                     else:
                        if self.clientData['disableJobMonitoring']:
                           if self.clientData['progressReport'] == 'text' or self.clientData['progressReport'] == 'submit':
                              self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % \
                                                            (os.path.join(os.getcwd(),self.clientData['runName'])))
                        exitCode = serverJsonObject['exitCode']
                        self.clientData['readyToExit'] = True
                        self.clientData['exitCode']    = exitCode
                        self.exit(exitCode)
                  elif serverJsonObject['messageType'] == 'exit':
                     exitCode = serverJsonObject['exitCode']
                     self.exit(exitCode)
                  elif serverJsonObject['messageType'] == 'wait':
                     self.wait()
                  elif serverJsonObject['messageType'] == 'null':
                     pass
                  else:
                     self.logger.log(logging.ERROR,getLogMessage("Discarded message type: %s" % \
                                                              (serverJsonObject['messageType'])))
                  if serverJsonObject['messageType'] != 'null':
                     tFinish = time.time()
                     self.logger.log(logging.DEBUG,getLogMessage("server request = %s took %f secs" % \
                                                     (serverJsonObject['messageType'],tFinish-tStart)))
            else:
               self.serverConnection.pushMessage(message + '\n')
               break
         else:
            self.logger.log(logging.ERROR,getLogMessage("Discarded server message: %s" % (message)))

         message = self.serverConnection.pullMessage(0)


   def run(self):
      result = {}
      result['jobId']              = 0
      result['runName']            = ""
      result['exitCode']           = 0
      result['serverDisconnected'] = False

      while not self.clientData['runCompleted']:
         if   self.clientData['jobMonitoringWait']:
            minSelectTimeout = 2*self.clientData['selectTimeout']* \
                               (1.-self.clientData['selectTimeoutSpread']/2.)
            maxSelectTimeout = 2*self.clientData['selectTimeout']* \
                               (1.+self.clientData['selectTimeoutSpread']/2.)
            selectTimeout = random.uniform(minSelectTimeout,maxSelectTimeout)
         elif self.serverData['childHasExited'] or self.clientData['readyToExit']:
            selectTimeout = 1.
         elif self.jobScanner:
            selectTimeout = 10.
         elif not self.clientData['isServerReadyForIO']:
            selectTimeout = 1.
         else:
            minSelectTimeout = self.clientData['selectTimeout']* \
                               (1.-self.clientData['selectTimeoutSpread']/2.)
            maxSelectTimeout = self.clientData['selectTimeout']* \
                               (1.+self.clientData['selectTimeoutSpread']/2.)
            selectTimeout = random.uniform(minSelectTimeout,maxSelectTimeout)

         if self.serverConnection and self.serverConnection.isConnected():
            if not self.serverConnection.isMessagePending():
               timeSinceLastWrite = time.time() - self.serverConnection.getConnectionWriteTime()
               if timeSinceLastWrite >= 0.25*selectTimeout:
                  if   self.clientData['isImportingOutputTar']:
                     clientMessage = {'messageType':'null'}
                     self.serverConnection.postJsonMessage(clientMessage)
                  elif self.clientData['isExportingInputTar'] or self.clientData['isExportingSubmitTar']:
                     clientMessage = {'messageType':'null'}
                     self.serverConnection.postJsonMessage(clientMessage)

         tStart = time.time()
         selectTimedOut = False
         try:
            serverReader,inputObjects,exportObjects,childProcessInputObjects,inputSignalPipes = self.getInputObjects()
            serverWriter,importObjects                                                        = self.getOutputObjects()
            readers = serverReader+inputObjects+exportObjects+childProcessInputObjects+inputSignalPipes
            writers = serverWriter+importObjects
            readyReaders,readyWriters,readyExceptions = select.select(readers,writers,[],selectTimeout)
            if not (readyReaders or readyWriters or readyExceptions):
               selectTimedOut = True
         except select.error:
            selectTimedOut = True
            readyReaders = []
            readyWriters = []

         tFinish = time.time()
#        self.logger.log(logging.DEBUG,getLogMessage("select time(%f): %d readers, %d writers, %d readyReaders, %d readyWriters" % \
#                                                   (tFinish-tStart,len(readers),len(writers),len(readyReaders),len(readyWriters))))
#
# send keepalive to notify server client is still running
#
         if self.serverConnection and self.serverConnection.isConnected():
            if not self.clientData['readyToExit']:
               timeSinceLastWrite = time.time() - self.serverConnection.getConnectionWriteTime()
               if selectTimedOut or (timeSinceLastWrite >= self.clientData['selectTimeout']):
                  clientMessage = {'messageType':'null'}
                  self.serverConnection.postJsonMessage(clientMessage)
#
# If the timeout has occurred (nothing to read/write)
#
         if readyReaders == [] and readyWriters == []:
            if   self.jobScanner:
               wantToQuit,jobsFinished = self.jobScanner.processInput()
               if wantToQuit:
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],self.clientData['runName'])
                  self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (resultsPath))
                  if not jobsFinished:
                     os.kill(os.getpid(),signal.SIGINT)
            elif self.clientData['readyToExit']:
               self.exit()

         for readyReader in readyReaders:
            if   readyReader in serverReader:
               self.serverConnection.receiveMessage()
            elif readyReader in childProcessInputObjects:
               self.scanLocalOutputFile(readyReader)
            elif self.jobScanner and readyReader == sys.stdin:
               wantToQuit,jobsFinished = self.jobScanner.processInput()
               if wantToQuit:
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],self.clientData['runName'])
                  self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (resultsPath))
                  if not jobsFinished:
                     os.kill(os.getpid(),signal.SIGINT)
            elif readyReader in inputSignalPipes:
               self.readSignalPipe(readyReader)
            else:
               self.readFile(readyReader)

         self.processServerRequests()

         for readyWriter in readyWriters:
            if   readyWriter in serverWriter:
               self.serverConnection.sendMessage()
            elif readyWriter in importObjects:
               self.writeFile(readyWriter)
         self.checkClosePendingObjects()

         serverDisconnected = False
         if not self.clientData['localExecution']      and not self.clientData['detach'] and \
            not self.clientData['jobMonitoringWait']   and not self.serverData['isBuildingOutputTar'] and \
            not self.clientData['isExportingInputTar'] and not self.clientData['isExportingSubmitTar']:
            if   self.serverConnection and self.serverConnection.isConnected():
               lastIOTimes = []
               lastIOTimes.append(self.serverConnection.getConnectionReadTime())
#              lastIOTimes.append(self.serverConnection.getConnectionWriteTime())
               if lastIOTimes:
                  timeSinceLastIO = time.time() - max(lastIOTimes)
                  if self.serverConnection and self.serverConnection.isConnected():
                     if timeSinceLastIO > self.clientData['disconnectMax']:
                        if not self.clientData['suspended']:
                           self.logger.log(logging.DEBUG,getLogMessage("server connected - no read for %f sec" % (timeSinceLastIO)))
                           serverDisconnected = True
            elif self.serverConnection:
               if not self.serverData['exited']:
                  self.logger.log(logging.DEBUG,getLogMessage("server not connected and not exited"))
                  serverDisconnected = True
         result['serverDisconnected'] = serverDisconnected

         if serverDisconnected:
            self.logger.log(logging.DEBUG,getLogMessage("server disconnected"))
            # equivalent to 'readyToDetach'
            if self.jobScanner:
               self.jobScanner.finish()
               self.jobScanner = None
            if not self.clientData['readyToExit']:
               self.clientData['readyToExit'] = True
               self.clientData['exitCode']    = 0
               if self.clientData['runName'] and self.serverData['jobId']:
                  notificationMessage = \
                               "Detaching from run %s.\n" % (self.clientData['runName']) + \
                               "    Check run status with the command: submit --status %d\n" % (self.serverData['jobId']) + \
                               "       Terminate run with the command: submit --kill %d\n" % (self.serverData['jobId']) + \
                               "   Re-connect to run with the command: submit --attach %d\n" % (self.serverData['jobId'])
                  self.__writeToStdout(notificationMessage)

         if   self.clientData['localWaiting'] and self.clientData['childPid']:
            pid = 0
            try:
               (pid,exitCode) = os.waitpid(self.clientData['childPid'],os.WNOHANG)
               if exitCode != 0:
                  if   os.WIFSIGNALED(exitCode):
                     exitCode = (1 << 7) | os.WTERMSIG(exitCode)
                  else:
                     if os.WIFEXITED(exitCode):
                        exitCode = os.WEXITSTATUS(exitCode)
            except:
               try:
                  os.kill(self.clientData['childPid'],0)
               except:
                  pid = self.clientData['childPid']
                  exitCode = 1

            if pid != 0:
               self.clientData['childPid']       = None
               self.clientData['exitCode']       = exitCode
               self.clientData['childHasExited'] = True

               rusage = resource.getrusage(resource.RUSAGE_CHILDREN)
               self.clientData['localFinishTime'] = time.time()
               self.clientData['realTime'] = self.clientData['localFinishTime']-self.clientData['localStartTime']
               self.clientData['cpuTime']  = rusage[0] + rusage[1]
               self.clientData['finishDate'] = time.strftime("%a %b %e %X %Z %Y")
               if self.clientData['isParametric']:
                  self.__updatePegasusWorkflowStatus()

               if self.clientData['reportMetrics']:
                  message =  "=SUBMIT-METRICS=>"
                  message += " job=%s" % (self.clientData['localJobId'])
                  message += " venue=local"
                  message += " status=%d" % (self.clientData['exitCode'])
                  message += " cpu=%f" % (self.clientData['cpuTime'])
                  message += " real=%f" % (self.clientData['realTime'])
                  self.__writeToStderr("%s\n" % (message))

               self.connectToServer('sendMetrics')
         elif self.clientData['jobMonitoringWait']:
            completePath = os.path.join(self.clientData['jobPath'],self.clientData['runName'],
                                        ".__remote_jobid_complete.%s_0" % (self.clientData['localJobId']))
            parameterCombinationsPath = os.path.join(self.clientData['jobPath'],self.clientData['runName'],
                                                     'parameterCombinations.csv')
            if os.path.exists(completePath):
               if self.clientData['progressReport'] == 'submit':
                  runProgressMessage = self.getWorkflowProgressMessage(parameterCombinationsPath)
                  self.clientData['statusModifiedTime'] = parameterCombinationsModificationTime
                  self.clientData['runProgressMessage'] = runProgressMessage
                  runProgressMessage += " timestamp=%.1f" % (time.time())
                  self.__writeToStdout("%s\n" % (runProgressMessage))

               self.clientData['jobMonitoringWait']     = False
               self.clientData['jobMonitoringComplete'] = True
               self.clientData['isServerReadyForIO']    = False
               self.clientData['importFilesComplete']   = False
               self.connectToServer('harvest')
            else:
               if self.clientData['progressReport'] == 'submit':
                  if os.path.exists(parameterCombinationsPath):
                     parameterCombinationsModificationTime = os.lstat(parameterCombinationsPath).st_mtime
                     if parameterCombinationsModificationTime > self.clientData['statusModifiedTime']:
                        runProgressMessage = self.getWorkflowProgressMessage(parameterCombinationsPath)
                        if runProgressMessage != self.clientData['runProgressMessage']:
                           self.clientData['statusModifiedTime'] = parameterCombinationsModificationTime
                           self.clientData['runProgressMessage'] = runProgressMessage
                           runProgressMessage += " timestamp=%.1f" % (time.time())
                           self.__writeToStdout("%s\n" % (runProgressMessage))

      result['jobId']    = self.serverData['jobId']
      result['runName']  = self.clientData['runName']
      result['exitCode'] = self.clientData['exitCode']

      return(result)


