#
# Copyright (c) 2004-2011 Purdue University All rights reserved.
#
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
#
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
#
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
#
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
import sys
import os
import re
import stat
import time
import datetime
import socket

from hubzero.submit.LogMessage             import logID as log
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore
from hubzero.submit.RemoteBatchCONDOR      import RemoteBatchCONDOR
from hubzero.submit.RemoteBatchLL          import RemoteBatchLL
from hubzero.submit.RemoteBatchLSF         import RemoteBatchLSF
from hubzero.submit.RemoteBatchPBS         import RemoteBatchPBS
from hubzero.submit.RemoteBatchPBS8        import RemoteBatchPBS8
from hubzero.submit.RemoteBatchSLURM       import RemoteBatchSLURM
from hubzero.submit.RemoteBatchAppScript   import RemoteBatchAppScript
from hubzero.submit.RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from hubzero.submit.RemoteInstantAppScript import RemoteInstantAppScript

class VenueMechanismGsiSsh(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                batchCommands,
                isParametric,
                localJobId,
                instanceId,
                destination,
                enteredCommand,
                x509ProxyPath,
                stageInTarFile,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                timeHistoryLogs):
      VenueMechanismCore.__init__(self,timeHistoryLogs,siteInfo,managerInfo,remoteMonitors,
                                       isMultiCoreRequest,nCpus,nNodes,ppn)

      self.venueMechanism        = 'gsissh'
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      remoteScriptsDirectory = siteInfo['remoteBinDirectory'].replace("$","\$")
      self.receiveInput          = os.path.join(remoteScriptsDirectory,batchCommands['receiveInput'])
      self.submitBatchJob        = os.path.join(remoteScriptsDirectory,batchCommands['submitBatchJob'])
      self.transmitResults       = os.path.join(remoteScriptsDirectory,batchCommands['transmitResults'])
      self.cleanupJob            = os.path.join(remoteScriptsDirectory,batchCommands['cleanupJob'])
      self.killBatchJob          = os.path.join(remoteScriptsDirectory,batchCommands['killBatchJob'])
      self.isParametric          = isParametric
      self.localJobId            = localJobId
      if instanceId.startswith('WF;'):
         self.instanceId         = ""
         self.nInstances         = int(instanceId.split(';')[1])
      else:
         self.instanceId         = instanceId
         self.nInstances         = 0
      self.remoteBatchSystem     = siteInfo['remoteBatchSystem']
      self.destination           = destination
      self.enteredCommand        = enteredCommand
      self.venue                 = siteInfo['venue']
      self.siteMonitorDesignator = siteInfo['siteMonitorDesignator']
      self.gsiHost               = siteInfo['gsiHost']
      if self.gsiHost != socket.gethostname():
         self.addVenueToBatchCommand = True
         self.sharedUserSpace    = False
      else:
         self.addVenueToBatchCommand = False
         self.sharedUserSpace    = siteInfo['sharedUserSpace']
      self.x509ProxyPath         = x509ProxyPath
      self.remoteUser            = siteInfo['remoteUser']
      self.logUserRemotely       = siteInfo['logUserRemotely']
      self.stageFiles            = siteInfo['stageFiles']
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile     = stageInTarFile
      else:
         self.stageInTarFile     = stageInTarFile + '.gz'
      self.transferExecutable    = transferExecutable
      self.executable            = executable
      self.stdinput              = stdinput
      self.arguments             = arguments
      self.useEnvironment        = useEnvironment
      self.environment           = environment
      self.remoteBatchQueue      = siteInfo['remoteBatchQueue']
      self.wallTime              = wallTime
      self.quotaLimit            = quotaLimit
      self.batchLogPath          = ""
      self.appScriptPath         = ""
      self.batchScriptPath       = ""
      self.nodeFilePath          = ""
      self.remoteJobIdNumber     = ""
      self.sshBaseCommand        = ""
      self.sshIdentityPath       = ""
      self.removeIdentity        = False

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob = False
         self.instanceDirectory = os.getcwd()
      else:
         self.isBatchJob = True
         self.instanceDirectory = os.path.join(os.getcwd(),self.localJobId,self.instanceId)
         if not os.path.isdir(self.instanceDirectory):
            os.makedirs(self.instanceDirectory)

      self.createSSHIdentityPath()

      if not self.sharedUserSpace and self.stageFiles:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.workingDirectory   = os.path.join(siteInfo['remoteScratchDirectory'], \
                                                "%s_%s_%s" % (str(epoch),localJobId,self.instanceId))
      else:
         self.workingDirectory   = os.getcwd()


   def createSSHIdentityPath(self):
      exitStatus = 0
      if not self.sshIdentityPath:
         updateKnownHosts = False
         if   self.remoteUser.startswith('USER:'):
            keyfamily = self.remoteUser.split(':')[1]
            self.remoteUser = self.hubUserName
            command = 'genuserpki ' + keyfamily
            log("command = " + command)
            exitStatus,stdOutput,stdError = self.executeCommand(command)
            if not exitStatus:
               self.sshIdentityPath = stdOutput.strip()
               log("IDENTITY = " + self.sshIdentityPath)
               updateKnownHosts = True
         elif self.remoteUser.startswith('USER'):
            self.remoteUser = self.hubUserName

         if not exitStatus:
            if updateKnownHosts:
               if self.addVenueToBatchCommand:
                  command = 'update-known-hosts' + " " + self.gsiHost
               else:
                  command = 'update-known-hosts' + " " + self.venue
               log("command = " + command)
               exitStatus,stdOutput,stdError = self.executeCommand(command)
               if not exitStatus:
                  self.sshBaseCommand = "ssh -T -x -a -i " + self.sshIdentityPath + " " + \
                                         self.remoteUser + "@" + self.venue
            else:
               self.sshBaseCommand = "ssh -T -x -a " + self.remoteUser + "@" + self.venue

         if not exitStatus:
            if self.remoteBatchSystem == 'SCRIPT':
               self.sshBaseCommand = "gsissh -T -x -a " + self.venue
            elif self.addVenueToBatchCommand:
               self.sshBaseCommand = "ssh -T -x -a -i " + self.sshIdentityPath + " " + \
                                      self.remoteUser + "@" + self.gsiHost
            else:
               self.sshBaseCommand = "gsissh -T -x -a " + self.venue


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      log("workingDirectory " + self.workingDirectory)
      if   self.remoteBatchSystem == 'PBS':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS(self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,
                                      self.wallTime,self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'PBS8':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS8(self.localJobId,self.instanceId,appScriptName,self.environment,
                                       self.transferExecutable,
                                       self.executable,self.arguments,self.stdinput,
                                       self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                       self.nNodes,self.ppn,
                                       self.wallTime,self.quotaLimit,
                                       self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'LSF':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLSF(self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,self.wallTime,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'LL':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLL(self.localJobId,self.instanceId,appScriptName,self.environment,
                                     self.transferExecutable,
                                     self.executable,self.arguments,self.stdinput,
                                     self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                     self.nNodes,self.ppn,
                                     self.wallTime,self.quotaLimit,
                                     self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SLURM':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchSLURM(self.localJobId,self.instanceId,appScriptName,self.environment,
                                        self.transferExecutable,
                                        self.executable,self.arguments,self.stdinput,
                                        self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                        self.nNodes,self.ppn,
                                        self.wallTime,self.quotaLimit,
                                        self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'CONDOR':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchCONDOR(self.localJobId,self.instanceId,self.instanceDirectory,
                                         appScriptName,self.environment,
                                         self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                         self.wallTime)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SCRIPT':
         remoteAppScript = RemoteInstantAppScript(self.localJobId,self.instanceId,self.isParametric,
                                                  self.workingDirectory,
                                                  self.transferExecutable,self.executable,
                                                  self.stdinput,self.arguments,
                                                  self.useEnvironment,self.environment,
                                                  self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                  self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteInstantSCRIPT(self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                           self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                           self.nNodes,self.ppn,
                                           self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()

      if batchLogName != "":
         self.batchLogPath    = os.path.join(self.instanceDirectory,batchLogName)
      if appScriptName != "":
         self.appScriptPath   = os.path.join(self.instanceDirectory,appScriptName)
      if batchScriptName != "":
         self.batchScriptPath = os.path.join(self.instanceDirectory,batchScriptName)
      if nodeFileName != "":
         self.nodeFilePath    = os.path.join(self.instanceDirectory,nodeFileName)

      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = 1
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

      if batchScript != "":
         fpBatchScript = open(self.batchScriptPath,'w')
         if fpBatchScript:
            fpBatchScript.write(batchScript)
            fpBatchScript.close()
            scriptFiles.append(self.batchScriptPath)
            if self.remoteBatchSystem == 'SCRIPT':
               os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         else:
            log("could not open %s for writing" % (self.batchScriptPath))
            sys.stderr.write("could not open %s for writing\n" % (self.batchScriptPath))
            sys.stderr.flush()
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if appScript != "":
         fpAppScript = open(self.appScriptPath,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(self.appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            scriptFiles.append(self.appScriptPath)
         else:
            log("could not open %s for writing" % (self.appScriptPath))
            sys.stderr.write("could not open %s for writing\n" % (self.appScriptPath))
            sys.stderr.flush()
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if len(nodeList) > 0:
         fpNodes = open(self.nodeFilePath,'w')
         if fpNodes:
            fpNodes.write('\n'.join(nodeList)+'\n')
            fpNodes.close()
         else:
            log("could not open %s for writing" % (self.nodeFilePath))
            sys.stderr.write("could not open %s for writing\n" % (self.nodeFilePath))
            sys.stderr.flush()
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      exitStatus = 0

      if self.remoteBatchQueue != "":
         self.jobStatistics[self.jobIndex]['venue'] = self.remoteBatchQueue + '@' + self.venue
      else:
         self.jobStatistics[self.jobIndex]['venue'] = self.venue
      if self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         stageInTarPath = os.path.join(self.instanceDirectory,self.stageInTarFile)

         if self.sshBaseCommand != "":
            if self.addVenueToBatchCommand:
               command = "cat " + stageInTarPath + " | " + self.sshBaseCommand + \
                                                   " \"" + self.receiveInput + " " + \
                                                                  self.venue + " " + \
                                                      remoteWorkingDirectory + " " + \
                                self.timeHistoryLogs['timestampTransferred'] + "\""
            else:
               command = "cat " + stageInTarPath + " | " + self.sshBaseCommand + \
                                                   " \"" + self.receiveInput + " " + \
                                                      remoteWorkingDirectory + " " + \
                                self.timeHistoryLogs['timestampTransferred'] + "\""
            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"")
            log(stdOutput)
         else:
            exitStatus = 1
            stdOutput  = ""
            stdError   = ""

      if exitStatus:
         self.filesSent = False
         self.jobStatistics[self.jobIndex]['exitCode'] = 11
#        if stdOutput != "":
#           stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.localJobId,self.instanceId))
#           fpStd = open(stdFile,'a')
#           if fpStd:
#              fpStd.write(stdOutput)
#              fpStd.close()
         if stdError != "":
            stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      if not self.stageFiles:
         if self.remoteBatchSystem == 'SCRIPT':
            self.sshBaseCommand = "gsissh -T -x -a " + self.venue
            if self.batchScriptPath != "":
               command = self.sshBaseCommand + " " + self.batchScriptPath
            else:
               command = self.sshBaseCommand + " " + self.appScriptPath
            log("command = " + command)
            exitStatus,stdOutput,stdError = self.executeCommand(command,True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         if self.remoteBatchSystem == 'SCRIPT':
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            if self.addVenueToBatchCommand:
               commandArgs.append(self.venue)
            commandArgs.append(remoteWorkingDirectory)
            if self.batchScriptPath != "":
               commandArgs.append(self.batchScriptPath)
            else:
               commandArgs.append(self.appScriptPath)
            if self.logUserRemotely:
               commandArgs.append(self.hubUserName)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""

            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"",True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            if self.addVenueToBatchCommand:
               commandArgs.append(self.venue)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            if self.logUserRemotely:
               commandArgs.append(self.hubUserName)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,remoteJobId,stdError = self.executeSSHCommand(command,"")
            if not exitStatus:
               remoteJobId = remoteJobId.strip()
               log("remoteJobId = " + remoteJobId)
               try:
                  if   self.remoteBatchSystem == 'PBS':
                     self.remoteJobIdNumber = remoteJobId.split('.')[0]
                  elif self.remoteBatchSystem == 'PBS8':
                     self.remoteJobIdNumber = remoteJobId.split('.')[0]
                  elif self.remoteBatchSystem == 'LSF':
#                 Job <851126> is submitted to default queue <normal>.
                     self.remoteJobIdNumber = remoteJobId.split('<')[1].split('>')[0]
                  elif self.remoteBatchSystem == 'LL':
                     self.remoteJobIdNumber = remoteJobId
                  elif self.remoteBatchSystem == 'SLURM':
                     self.remoteJobIdNumber = remoteJobId.split()[-1]
                  elif self.remoteBatchSystem == 'CONDOR':
#                    Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
                     self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"

                  self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               except:
                  exitStatus = 248

               if not exitStatus:
                  self.jobSubmitted = True
                  self.remoteJobMonitor.postNewJobSubmission(self.siteMonitorDesignator,self.remoteJobIdNumber,
                                                             str(self.hubUserId),
                                                             "%s_%s" % (self.localJobId,self.instanceId),self.destination)
               else:
                  self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
            else:
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
#        if stdOutput != "":
#           stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.localJobId,self.instanceId))
#           fpStd = open(stdFile,'a')
#           if fpStd:
#              fpStd.write(stdOutput)
#              fpStd.close()
         if stdError != "":
            stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['siteMonitorDesignator'] = self.siteMonitorDesignator
      waitForJobInfo['remoteJobId']           = self.remoteJobIdNumber
      waitForJobInfo['knownSite']             = self.destination

      return(waitForJobInfo)


   def waitForJob(self):
      if (self.siteMonitorDesignator != "") and \
         (self.remoteJobIdNumber != "") and \
         (self.destination != ""):
         self.remoteJobMonitor.waitForBatchJob(self.siteMonitorDesignator,self.remoteJobIdNumber,self.destination)


   def retrieveFiles(self):
      exitStatus = 0

      if self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         if self.jobSubmitted:
            if self.addVenueToBatchCommand:
               command = self.sshBaseCommand + " \"" + self.transmitResults + " " + \
                                                                 self.venue + " " + \
                                                    remoteWorkingDirectory + "\"" + \
                                               " | tar xzmf - --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
            else:
               command = self.sshBaseCommand + " \"" + self.transmitResults + " " + \
                                                    remoteWorkingDirectory + "\"" + \
                                               " | tar xzmf - --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"")
            log(stdOutput)
            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.localJobId,self.instanceId))
#                 fpStd = open(stdFile,'a')
#                 if fpStd:
#                    fpStd.write(stdOutput)
#                    fpStd.close()
               if stdError != "":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
                  fpStd = open(stdFile,'a')
                  if fpStd:
                     fpStd.write(stdError)
                     fpStd.close()

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self):
      if not self.filesCleanedup:
         if self.stageFiles:
            remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
            if self.jobSubmitted:
               if self.addVenueToBatchCommand:
                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + self.venue + " " + remoteWorkingDirectory + "\""
               else:
                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + remoteWorkingDirectory + "\""
               log("command = " + command)
               os.environ['X509_USER_PROXY'] = self.x509ProxyPath
               log(self.executeSSHCommand(command,"")[1])

         if self.batchScriptPath:
            if os.path.isfile(self.batchScriptPath):
               os.remove(self.batchScriptPath)
         if self.appScriptPath:
            if os.path.isfile(self.appScriptPath):
               os.remove(self.appScriptPath)
         if self.nodeFilePath:
            if os.path.isfile(self.nodeFilePath):
               os.remove(self.nodeFilePath)
         if self.batchLogPath:
            if os.path.isfile(self.batchLogPath):
               os.remove(self.batchLogPath)

         self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      if   self.remoteJobIdNumber:
         if self.addVenueToBatchCommand:
            command = self.sshBaseCommand + " \"" + self.killBatchJob + " " + \
                                                           self.venue + " " + \
                                               self.remoteJobIdNumber + " " + \
                                               self.remoteBatchSystem + "\""
         else:
            command = self.sshBaseCommand + " \"" + self.killBatchJob + " " + \
                                               self.remoteJobIdNumber + " " + \
                                               self.remoteBatchSystem + "\""
         log("command = " + command)
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         log(self.executeSSHCommand(command,"")[1])
      elif self.remoteBatchSystem == 'SCRIPT' and self.childPid != 0:
#        log("Send TERM to child script process %d" % (self.childPid))
#        os.kill(childPid,signalNumber.SIGTERM)
         if self.batchScriptPath != "":
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.batchScriptPath)
         else:
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.appScriptPath)
         log("command = " + command)
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         log(self.executeSSHCommand(command,"")[1])

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 8 | signalNumber

      self.scriptsKilled = True


