# @package      hubzero-submit-common
# @file         RemoteBatchPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
from Pegasus.DAX3 import *

class RemoteBatchPEGASUS:
   def __init__(self,
                hubUserName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                executable,
                arguments,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                x509SubmitProxy,
                gridsite,
                timeHistoryLogs):

      self.hubUserName        = hubUserName
      self.localJobId         = localJobId
      self.instanceId         = instanceId
      self.instanceDirectory  = instanceDirectory
      self.appScriptName      = appScriptName
      self.timestampStart     = timeHistoryLogs['timestampStart']
      self.timestampFinish    = timeHistoryLogs['timestampFinish']
      self.timeResults        = timeHistoryLogs['timeResults']

      basename = os.path.basename(executable)
      if basename.startswith('pegasus-'):
         self.daxPath         = ""
         self.executable      = executable
         userArguments = arguments.split()
         scriptArguments = []
         for userArgument in userArguments:
            if os.path.isfile(userArgument):
               absolutePath = os.path.abspath(userArgument)
               scriptArguments.append(absolutePath)
            else:
               scriptArguments.append(userArgument)
         self.arguments       = ' '.join(scriptArguments)
      else:
         self.daxPath         = os.path.join(self.instanceDirectory,"%s_%s.dax" % (self.localJobId,self.instanceId))
         self.executable      = 'pegasus-plan'
         self.arguments       = "--dax %s" % (self.daxPath)

      self.isMultiCoreRequest = isMultiCoreRequest
      self.computationMode    = managerInfo['computationMode']
      self.pegasusTemplates   = siteInfo['pegasusTemplates']
      self.x509SubmitProxy    = x509SubmitProxy
      self.gridsite           = gridsite

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# RemoteBatchPEGASUS:makeSerialTemplate
#
mkdir -p INSTANCEDIRECTORY/work
mkdir -p INSTANCEDIRECTORY/InProcessResults

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS \\
             EXECUTABLE --conf INSTANCEDIRECTORY/JOBID.pegasusrc \\
             --sites GRIDSITE \\
             --dir INSTANCEDIRECTORY/work \\
             --relative-dir pegasus \\
             --relative-submit-dir pegasus \\
             --output local \\
             ARGUMENTS \\
             --submit < /dev/null

date +"%s" > TS_FINISH
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_executable        = re.compile("EXECUTABLE")
      re_gridsite          = re.compile("GRIDSITE")
      re_arguments         = re.compile("ARGUMENTS")
      re_tsStart           = re.compile("TS_START")
      re_tsFinish          = re.compile("TS_FINISH")
      re_timeResults       = re.compile("TIME_RESULTS")

      template = self.__makeSerialTemplate()

      template = re_jobid.sub("%s_%s" % (self.localJobId,self.instanceId),template)
      template = re_instanceDirectory.sub(self.instanceDirectory,template)
      template = re_executable.sub(self.executable,template)
      template = re_gridsite.sub(self.gridsite,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      auxiliaryArguments = []
      if os.path.exists(self.pegasusTemplates['rc']):
         fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
         if fpRCTemplate:
            searchString = os.path.basename(self.executable) + '.arguments'
            settings = fpRCTemplate.readlines()
            for setting in settings:
#pegasus-plan.arguments = --nocleanup
               if setting.count(searchString) > 0:
                  try:
                     parameter,value = setting.split('=')
                     auxiliaryArguments.append(value.strip())
                  except:
                     pass
            fpRCTemplate.close()

      userArguments = self.arguments.split()
      location = 0
      while location < len(userArguments):
         if userArguments[location].startswith('-'):
            if   userArguments[location].startswith('-D'):
               keepArgument = False
            elif userArguments[location] == '-d' or userArguments[location] == '--dax':
               keepArgument = True
            elif userArguments[location] == '-b' or userArguments[location] == '--basename':
               keepArgument = True
            elif userArguments[location] == '-c' or userArguments[location] == '--cache':
               keepArgument = False
            elif userArguments[location] == '-C' or userArguments[location] == '--cluster':
               keepArgument = True
            elif userArguments[location] == '--conf':
               keepArgument = False
            elif userArguments[location] == '--dir':
               keepArgument = False
            elif userArguments[location] == '-f' or userArguments[location] == '--force':
               keepArgument = True
            elif userArguments[location] == '--force-replan':
               keepArgument = True
            elif userArguments[location] == '-g' or userArguments[location] == '--group':
               keepArgument = True
            elif userArguments[location] == '-h' or userArguments[location] == '--help':
               keepArgument = False
            elif userArguments[location] == '--inherited-rc-files':
               keepArgument = True
            elif userArguments[location] == '-j' or userArguments[location] == '--j':
               keepArgument = True
            elif userArguments[location] == '-n' or userArguments[location] == '--nocleanup':
               keepArgument = False
            elif userArguments[location] == '-o' or userArguments[location] == '--output':
               keepArgument = False
            elif userArguments[location] == '-q' or userArguments[location] == '--quiet':
               keepArgument = True
            elif userArguments[location] == '--relative-submit-dir':
               keepArgument = False
            elif userArguments[location] == '-s' or userArguments[location] == '--sites':
               keepArgument = False
            elif userArguments[location] == '-S' or userArguments[location] == '--submit':
               keepArgument = False
            elif userArguments[location].startswith('-v') or userArguments[location] == '--verbose':
               keepArgument = True
            elif userArguments[location] == '-V' or userArguments[location] == '--version':
               keepArgument = False

            if keepArgument:
               auxiliaryArguments.append(userArguments[location])
            location += 1
            while location < len(userArguments) and not userArguments[location].startswith('-'):
               if keepArgument:
                  auxiliaryArguments.append(userArguments[location])
               location += 1
         else:
            location += 1

      arguments = ' '.join(auxiliaryArguments)
      template = re_arguments.sub(arguments.strip(),template)

      return(template)


   def __buildCatalogs(self):
      # setup regex's for the template
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_x509SubmitProxy   = re.compile("X509SUBMITPROXY")

      for templateType in self.pegasusTemplates:
         pegasusTemplate = self.pegasusTemplates[templateType]
         if pegasusTemplate != "":
            fpTemplate = open(pegasusTemplate,'r')
            if fpTemplate:
               template = ''.join(fpTemplate.readlines())
               fpTemplate.close()
               template = re_jobid.sub("%s_%s" % (self.localJobId,self.instanceId),template)
               template = re_instanceDirectory.sub(self.instanceDirectory,template)
               if self.x509SubmitProxy != "":
                  template = re_x509SubmitProxy.sub(self.x509SubmitProxy,template)
               else:
                  template = re_x509SubmitProxy.sub(os.path.join(os.sep,'tmp','hub-proxy.%s' % (self.hubUserName)),template)
               if   templateType == 'rc':
                  pegasusFile = "%s_%s.pegasusrc" % (self.localJobId,self.instanceId)
               elif templateType == 'sites':
                  pegasusFile = "%s_%s_sites.xml" % (self.localJobId,self.instanceId)
               elif templateType == 'tc':
                  pegasusFile = "%s_%s_tc.txt" % (self.localJobId,self.instanceId)
               pegasusPath = os.path.join(self.instanceDirectory,pegasusFile)
               fpPegasusFile = open(pegasusPath,'w')
               if fpPegasusFile:
                  fpPegasusFile.write(template)
                  fpPegasusFile.close()


   def __buildDAX(self):
      if self.daxPath != "":
         dax = ADAG("%s_%s" % (self.localJobId,self.instanceId))

         appScriptPath = "file://" + os.path.join(self.instanceDirectory,self.appScriptName)
         appScript = Executable(name=self.appScriptName,arch=Arch.X86_64,os=OS.LINUX,installed=False)
         appScript.addPFN(PFN(appScriptPath,"local"))
         dax.addExecutable(appScript)

         stageInTarFile = "%s_%s_input.tar.gz" % (self.localJobId,self.instanceId)
         inputTarPath = "file://" + os.path.join(self.instanceDirectory,stageInTarFile)
         inputTar = File(stageInTarFile)
         inputTar.addPFN(PFN(inputTarPath,"local"))
         dax.addFile(inputTar)

         stageOutTarFile = stageInTarFile.replace('input','output')
         outputTar = File(stageOutTarFile)

         jobId = "%s_0" % (self.localJobId)
         job = Job(name=self.appScriptName,id=jobId)
         job.uses(inputTar,link=Link.INPUT,executable=False)
         job.uses(outputTar,link=Link.OUTPUT,register=True,transfer=True)
         dax.addJob(job)

         fpDAX = open(self.daxPath,'w')
         if fpDAX:
            dax.writeXML(fpDAX)
            fpDAX.close()


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.pegasus" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         batchScript = ""
      else:
         batchScript = self.__buildSerialFile()

      self.__buildCatalogs()
      self.__buildDAX()

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


