# @package      hubzero-submit-common
# @file         CommandParser.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

from optparse import OptionParser,SUPPRESS_HELP
#import sys
import os.path
import re
import math
import csv
import copy
import fnmatch

from hubzero.submit.LogMessage    import logID as log
from hubzero.submit.CommentedFile import CommentedFile

class CommandParser:
   def __grabMultipleArguments(self,option,opt_str,value,parser):
      assert value is None
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      while rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if  (arg == "--"):
            del rargs[0]
            break
         elif((arg[:2] == "--" and len(arg) > 2) or
              (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            break
         else:
            if not arg in value:
               value.append(arg)
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)

   def __grabStringArgument(self,option,opt_str,value,parser):
      assert value is None
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      if rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if((arg[:2] == "--" and len(arg) > 2) or
            (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            pass
         else:
            if not arg in value:
               value.append(arg)
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)

   def __grabMultipleParameterArguments(self,option,opt_str,value,parser):
      assert value is None
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      while rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if  (arg == "--"):
            del rargs[0]
            break
         elif((arg[:2] == "--" and len(arg) > 2) or
              (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            break
         else:
            if not arg in value:
               sep = getattr(parser.values,'separator')
               value.append((sep,arg))
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)

   def __grabParameterArgument(self,option,opt_str,value,parser):
      assert value is None
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      if rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if((arg[:2] == "--" and len(arg) > 2) or
            (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            pass
         else:
            if not arg in value:
               sep = getattr(parser.values,'separator')
               value.append((sep,arg))
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)

   def __grabMultipleHelpArguments(self,option,opt_str,value,parser,allowedHelpOptions):
      assert value is None
      value = []
      rargs = parser.rargs
      while rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if  (arg == "--"):
            del rargs[0]
            break
         elif((arg[:2] == "--" and len(arg) > 2) or
            (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            break
         elif arg in allowedHelpOptions:
            if not arg in value:
               value.append(arg)
               del rargs[0]
         else:
            break

      if len(value) == 0:
         setattr(parser.values,'help',True)
      else:
         if 'managers' in value:
            setattr(parser.values,'helpManagers',True)
         if 'venues' in value:
            setattr(parser.values,'helpVenues',True)
         if 'tools' in value:
            setattr(parser.values,'helpTools',True)


   def __grabHelpArgument(self,option,opt_str,value,parser,allowedHelpOptions):
      assert value is None
      value = []
      rargs = parser.rargs
      if rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if((arg[:2] == "--" and len(arg) > 2) or
            (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            pass
         else:
            if arg in allowedHelpOptions:
               value.append(arg)
               del rargs[0]

      if len(value) == 0:
         setattr(parser.values,'help',True)
      elif 'managers' in value:
         setattr(parser.values,'helpManagers',True)
      elif 'venues' in value:
         setattr(parser.values,'helpVenues',True)
      elif 'tools' in value:
         setattr(parser.values,'helpTools',True)


   def __init__(self,
                doubleDashTerminator=False):
      self.commandOptions       = None
      self.commandArguments     = []
      self.sweepParameters      = {}
      self.csvDataParameters    = []
      self.csvDataHasHeader     = False
      self.nCSVDataCombinations = 0

      self.parser = OptionParser(prog="submit",add_help_option=False)
      self.parser.disable_interspersed_args()
      self.parser.add_option("-j","--jobid",action="store",type="long",dest="jobId", \
                                            help=SUPPRESS_HELP)

      self.parser.set_defaults(localExecution=False)
      self.parser.add_option("-l","--local",action="store_true",dest="localExecution", \
                                            help="Execute command locally")

      self.parser.set_defaults(help=False)
      self.parser.set_defaults(helpManagers=False)
      self.parser.set_defaults(helpVenues=False)
      self.parser.set_defaults(helpTools=False)
      allowedHelpOptions = ['managers','venues','tools']

      if doubleDashTerminator:
         self.parser.add_option("-v","--venue",action="callback", \
                                               callback=self.__grabMultipleArguments,dest="destinations", \
                                               help="Remote job destination list terminated by --")
         self.parser.add_option("-i","--inputfile",action="callback", \
                                                   callback=self.__grabMultipleArguments,dest="inputFiles", \
                                                   help="Input file list terminated by --")
         self.parser.add_option("-o","--outputfile",action="callback", \
                                                    callback=self.__grabMultipleArguments,dest="outputFiles", \
                                                    help=SUPPRESS_HELP)
         self.parser.add_option("-p","--parameters",action="callback", \
                                                    callback=self.__grabMultipleParameterArguments,dest="parameters", \
                                                    help="Parameter sweep variables terminated by --")
         self.parser.add_option("-h","--help",action="callback", \
                                              callback=self.__grabMultipleHelpArguments,dest="help", \
                                              callback_kwargs={'allowedHelpOptions':allowedHelpOptions}, \
                                              help="Report command usage")
      else:
         self.parser.add_option("-v","--venue",action="callback", \
                                               callback=self.__grabStringArgument,dest="destinations", \
                                               help="Remote job destination")
         self.parser.add_option("-i","--inputfile",action="callback", \
                                                   callback=self.__grabStringArgument,dest="inputFiles", \
                                                   help="Input file")
         self.parser.add_option("-o","--outputfile",action="callback", \
                                                    callback=self.__grabStringArgument,dest="outputFiles", \
                                                    help=SUPPRESS_HELP)
         self.parser.add_option("-p","--parameters",action="callback", \
                                                    callback=self.__grabParameterArgument,dest="parameters", \
                                                    help="Parameter sweep variables")
         self.parser.add_option("-h","--help",action="callback", \
                                              callback=self.__grabHelpArgument,dest="help", \
                                              callback_kwargs={'allowedHelpOptions':allowedHelpOptions}, \
                                              help="Report command usage")

      self.parser.add_option("-d","--data",action="callback", \
                                           callback=self.__grabStringArgument,dest="csvData", \
                                           help="Parametric variable data - csv format")

      self.parser.set_defaults(separator=',')
      self.parser.add_option("-s","--separator",action="store",type="string",dest="separator", \
                                                help="Parameter sweep variable list separator")
      self.parser.add_option("-n","--nCpus",action="store",type="int",dest="nCpus", \
                                            help="Number of processors for MPI execution")
      self.parser.add_option("-N","--ppn",action="store",type="int",dest="ppn", \
                                          help="Number of processors/node for MPI execution")
      self.parser.add_option("-w","--wallTime",action="store",type="string",dest="wallTime", \
                                               help="Estimated walltime hh:mm:ss or minutes")
      self.parser.add_option("-e","--env",action="callback", \
                                          callback=self.__grabStringArgument,dest="environment", \
                                          help="Variable=value")
      self.parser.add_option("-m","--manager",action="callback", \
                                              callback=self.__grabStringArgument,dest="manager", \
                                              help="Multiprocessor job manager")
      self.parser.add_option("-r","--redundancy",action="store",type="int",dest="nRedundant", \
                                            help="Number of indentical simulations to execute in parallel")
      self.parser.set_defaults(metrics=False)
      self.parser.add_option("-M","--metrics",action="store_true",dest="metrics", \
                                              help="Report resource usage on exit")
      self.parser.set_defaults(waitSwitch=False)
      self.parser.add_option("-W","--wait",action="store_true",dest="waitSwitch", \
                                           help="Wait for reduced job load before submission")
      self.parser.set_defaults(debug=False)
      self.parser.add_option("--debug",action="store_true",dest="debug", \
                                       help=SUPPRESS_HELP)
      self.parser.set_defaults(quotaCheck=True)
      self.parser.add_option("-Q","--quota",action="store_true",dest="quotaCheck", \
                                            help="Enforce local user quota on remote execution host")
      self.parser.add_option("-q","--noquota",action="store_false",dest="quotaCheck", \
                                              help="Do not enforce local user quota on remote execution host")


   def parseArguments(self,
                      args):
      options,arguments = self.parser.parse_args(args)

      self.commandOptions   = options
      self.commandArguments = []
      for argument in arguments:
         self.commandArguments.append(argument.replace(' ','\ '))


   def getOption(self,
                 option):
      value = getattr(self.commandOptions,option)

      return(value)


   def getEnteredExecutable(self):
      enteredExecutable = self.commandArguments[0]

      return(enteredExecutable)


   def getEnteredCommand(self):
      enteredCommand = " ".join(self.commandArguments)

      return(enteredCommand)


   def getEnteredCommandArguments(self):

      return(self.commandArguments)


   def getParametricArgumentCount(self):
      parametricArgumentCount = 0
      if self.commandArguments:
         if len(self.commandArguments) > 1:
            for argument in self.commandArguments[1:]:
               if   '@:' in argument:
                  parametricArgumentCount += 1
               elif '@@' in argument:
                  parametricArgumentCount += 1

      return(parametricArgumentCount)


   def getParametricInputCount(self):
      parametricInputCount = 0
      if self.getOption('inputFiles'):
         for inputFile in self.getOption('inputFiles'):
            if   '@:' in inputFile:
               parametricInputCount += 1
            elif '@@' in inputFile:
               parametricInputCount += 1

      return(parametricInputCount)


   def showUsage(self):
      self.parser.print_help()


   def validateArguments(self):
      validArguments = True
      if self.getOption('localExecution') and self.getOption('destinations'):
         errorMessage = "Local execution can not be combined with remote destinations"
         log(errorMessage)
#        sys.stderr.write(errorMessage + "\n")
#        sys.stderr.flush()
         validArguments = False
      
      return(validArguments)


   def getParameterVector(self,
                          first,
                          last,
                          inc=1):
      parameterVector = []
      if isinstance(first,str):
         try:
            first = int(first)
         except ValueError:
            first = float(first)
      if isinstance(last,str):
         try:
            last = int(last)
         except ValueError:
            last = float(last)
      if isinstance(inc,str):
         try:
            inc = int(inc)
         except ValueError:
            inc = float(inc)

      if float(last-first)*inc > 0.:
         nPoints = int(math.floor((last-first)/float(inc))) + 1

         for iPoint in xrange(0,nPoints):
            value = first + inc*iPoint
            parameterVector.append(value)
      else:
         parameterVector.append(first)

      return(parameterVector)


   def parseParameter(self,
                      parameter,
                      separator=""):
      intExpr      = "[0-9]+"
      numExpr      = "[-+]?(?:[0-9]+\.?[0-9]*|\.[0-9]+)(?:[eEdD]\+?[0-9]+)?"
      matlabExpr   = "^(%s)[ \t]*:[ \t]*(%s)[ \t]*(?::[ \t]*(%s))?$" % (numExpr,numExpr,numExpr)
      rangeExpr    = "^(%s)[ \t]*-[ \t]*(%s)$" % (numExpr,numExpr)
      rangeByExpr  = "^(%s)[ \t]*[-:][ \t]*(%s)[ \t]+by[ \t]+(%s)$" % (numExpr,numExpr,numExpr)
      rangePtsExpr = "^(%s)[ \t]*[-:][ \t]*(%s)[ \t]+in[ \t]+(%s)(?:[ \t]*po?i?n?ts)?(?:[ \t]+(line?a?r?|log))?$" % \
                                                                      (numExpr,numExpr,intExpr)
      globExpr     = "^glob:(.+)"

      reIntExpr      = re.compile(intExpr)
      reNumExpr      = re.compile(numExpr)
      reMatlabExpr   = re.compile(matlabExpr)
      reRangeExpr    = re.compile(rangeExpr)
      reRangeByExpr  = re.compile(rangeByExpr)
      reRangePtsExpr = re.compile(rangePtsExpr)
      reGlobExpr     = re.compile(globExpr)

      if   reMatlabExpr.match(parameter):
      # matched: v1:v2 or v1:v2:v3
         v1,v2,v3 = reMatlabExpr.match(parameter).groups()
         if v3:
            parameterVector = self.getParameterVector(v1,v3,v2)
         else:
            parameterVector = self.getParameterVector(v1,v2)
      elif reRangeExpr.match(parameter):
      # matched: v1-v2
         v1,v2 = reRangeExpr.match(parameter).groups()
         parameterVector = self.getParameterVector(v1,v2)
      elif reRangeByExpr.match(parameter):
      # matched: v1-v2 by inc
         v1,v2,v3 = reRangeByExpr.match(parameter).groups()
         parameterVector = self.getParameterVector(v1,v2,v3)
      elif reRangePtsExpr.match(parameter):
      # matched: v1-v2/v3 log
         v1,v2,nPoints,how = reRangePtsExpr.match(parameter).groups()
         if int(nPoints) > 1:
            if how == 'log':
               if float(v1) > 0.:
                  first = math.log10(float(v1))
               else:
                  first = 0.
               if float(v2) > 0.:
                  last = math.log10(float(v2))
               else:
                  last = 0.
               inc = (last-first)/(float(nPoints)-1)
               parameterVector = []
               for value in self.getParameterVector(first,last,inc):
                  parameterVector.append(pow(10.0,value))
            else:
               v3 = (float(v2)-float(v1))/(float(nPoints)-1)
               parameterVector = self.getParameterVector(v1,v2,v3)
         else:
            parameterVector = []
      elif reGlobExpr.match(parameter):
      # look for files matching the glob pattern
         pattern = reGlobExpr.match(parameter).group(1)
         dirname = os.path.dirname(pattern)
         if dirname == "":
            dirname = os.getcwd()
         basename = os.path.basename(pattern)
         dirFiles = os.listdir(dirname)
         matchingFiles = fnmatch.filter(dirFiles,basename)
         matchingFiles.sort()
         parameterVector = []
         for matchingFile in matchingFiles:
            parameterVector.append(os.path.join(dirname,matchingFile))
      elif separator == "":
      # there is no separator char -- keep whole value string as one element
         parameterVector = []
         parameterVector.append(parameter)
      else:
         parameterVector = parameter.split(separator)

      return(parameterVector)


   def setSweepParameters(self):
      exitCode = 0
      if self.commandOptions.parameters:
         parameterPattern = re.compile('\s*parameter\s+(.*[^\s$])')
         separatorPattern = re.compile('\s*separator\s+(.*[^\s$])')
         commentPattern   = re.compile('\s*#.*')
         parameterExpr    = "^@@([a-zA-Z0-9_]+) *\= *(.+)"
         reParameterExpr = re.compile(parameterExpr)
         for separator,sweepParameters in self.commandOptions.parameters:
            for sweepParameter in sweepParameters.split(';'):
               if   reParameterExpr.match(sweepParameter):
                  name,parameter = reParameterExpr.match(sweepParameter).groups()
                  if not name in self.sweepParameters:
                     self.sweepParameters[name] = self.parseParameter(parameter,separator)
                  else:
                     errorMessage = "Parameter @@%s defined more than once" % (name)
                     log(errorMessage)
#                    sys.stderr.write(errorMessage + "\n")
#                    sys.stderr.flush()
                     exitCode = 1
               elif os.path.isfile(sweepParameter):
                  sep = ','
                  fpParams = open(sweepParameter,'r')
                  if fpParams:
                     eof = False
                     while not eof:
                        record = fpParams.readline()
                        if record != "":
                           record = commentPattern.sub("",record)
                           if   parameterPattern.match(record):
                              fileSweepParameters = parameterPattern.match(record).group(1)
                              for sweepParameter in fileSweepParameters.split(';'):
                                 if reParameterExpr.match(sweepParameter):
                                    name,parameter = reParameterExpr.match(sweepParameter).groups()
                                    if not name in self.sweepParameters:
                                       self.sweepParameters[name] = self.parseParameter(parameter,sep)
                                    else:
                                       errorMessage = "Parameter @@%s defined more than once" % (name)
                                       log(errorMessage)
#                                      sys.stderr.write(errorMessage + "\n")
#                                      sys.stderr.flush()
                                       exitCode = 1
                                 else:
                                    errorMessage = "Bad parameter definition \"%s\" -- should be @@NAME=expr" % (sweepParameter)
                                    log(errorMessage)
#                                   sys.stderr.write(errorMessage + "\n")
#                                   sys.stderr.flush()
                                    exitCode = 1
                           elif separatorPattern.match(record):
                              sep = separatorPattern.match(record).group(1)
                        else:
                           eof = True
                     fpParams.close()
               else:
                  errorMessage = "Bad parameter definition \"%s\" -- should be @@NAME=expr or a file" % (sweepParameter)
                  log(errorMessage)
#                 sys.stderr.write(errorMessage + "\n")
#                 sys.stderr.flush()
                  exitCode = 1

      return(exitCode)


   def setCSVDataParameters(self):
      exitCode = 0
      self.nCSVDataCombinations = 0
      if self.commandOptions.csvData:
         csvDataFile = self.commandOptions.csvData[0]
         headerExpr   = "^@@([a-zA-Z0-9_]+)"
         reHeaderExpr = re.compile(headerExpr)
         if os.path.isfile(csvDataFile):
            fpCSVData = open(csvDataFile,'rb')
            if fpCSVData:
               csvReader = csv.reader(fpCSVData)
               try:
                  firstRow = csvReader.next()
                  nHeaderLabels = 0
                  for headerLabel in firstRow:
                     if reHeaderExpr.match(headerLabel):
                        nHeaderLabels += 1
                  if   nHeaderLabels == 0:
                     nHeaderIdDigits = int(math.log10(len(firstRow))+1)
                     for headerId in xrange(1,len(firstRow)+1):
                        headerLabel = str(headerId).zfill(nHeaderIdDigits)
                        self.csvDataParameters.append(headerLabel)
                     self.nCSVDataCombinations += 1
                  elif nHeaderLabels == len(firstRow):
                     for headerLabel in firstRow:
                        name = reHeaderExpr.match(headerLabel).group(1)
                        self.csvDataParameters.append(name)
                     self.csvDataHasHeader  = True
                  else:
                     errorMessage = "Parameter data file %s: header not properly specified" % (csvDataFile)
                     log(errorMessage)
#                    sys.stderr.write(errorMessage + "\n")
#                    sys.stderr.flush()
                     exitCode = 1

                  for row in csvReader:
                     self.nCSVDataCombinations += 1

               except csv.Error,e:
                  errorMessage = "Error in parameter data file %s, line %d: %s" % (csvDataFile,csvReader.line_num,e)
                  log(errorMessage)
#                 sys.stderr.write(errorMessage + "\n")
#                 sys.stderr.flush()
                  exitCode = 1

               fpCSVData.close()
         else:
            errorMessage = "%s is not a file" % (csvDataFile)
            log(errorMessage)
#           sys.stderr.write(errorMessage + "\n")
#           sys.stderr.flush()
            exitCode = 1

      return(exitCode)


   def __getNextCSVDataCombination(self):
      csvDataFile = self.commandOptions.csvData[0]
      fpCSV = open(csvDataFile,'rb')
      if fpCSV:
         csvReader = csv.reader(fpCSV)
         if self.csvDataHasHeader:
            parameterNames = csvReader.next()
         try:
            for csvDataCombination in csvReader:
               yield(tuple(csvDataCombination))
         except StopIteration:
            fpCSV.close()
         except csv.Error,e:
            errorMessage = "Error in parameter data file %s, line %d: %s" % (csvDataFile,csvReader.line_num,e)
            log(errorMessage)
#           sys.stderr.write(errorMessage + "\n")
#           sys.stderr.flush()


   def __getCSVDataCombinations(self):
      csvDataCombinations = []
      if len(self.csvDataParameters) > 0:
         csvDataCombinations = list(self.__getNextCSVDataCombination())

      return(self.csvDataParameters,csvDataCombinations)


   # http://stackoverflow.com/questions/1681269/how-code-a-function-similar-to-itertools-product-in-python-2-5
   def __product(self,
                 *iterables):
      """ Equivalent of itertools.product for versions < 2.6,
          which does NOT build intermediate results.
          Omitted 'repeat' option.
          product('ABCD', 'xy') --> Ax Ay Bx By Cx Cy Dx Dy
      """
      nIters = len(iterables)
      lstLenths = []
      lstRemaining = [1]
      for i in xrange(nIters-1,-1,-1):
         m = len(iterables[i])
         lstLenths.insert(0, m)
         lstRemaining.insert(0, m * lstRemaining[0])
      nProducts = lstRemaining.pop(0)

      for p in xrange(nProducts):
         lstVals = []

         for i in xrange(nIters):
            j = p/lstRemaining[i]%lstLenths[i]
            lstVals.append(iterables[i][j])
         yield tuple(lstVals)


   def getParameterCombinationCount(self):
      parameterNames = self.sweepParameters.keys()
      if len(parameterNames) > 0:
         nSweepCombinations = 1
         for parameterName in parameterNames:
            nSweepCombinations *= len(self.sweepParameters[parameterName])
      else:
         nSweepCombinations = 0

      if (nSweepCombinations > 0) and (self.nCSVDataCombinations > 0):
         nCombinations = nSweepCombinations*self.nCSVDataCombinations
      else:
         nCombinations = max(nSweepCombinations,self.nCSVDataCombinations)

      return(nCombinations)


   def __getSweepParameterCombinations(self):
      sweepParameterNames        = self.sweepParameters.keys()
      sweepParameterCombinations = None

      if len(sweepParameterNames) > 0:
         sweepParameterValues = self.sweepParameters.values()
         sweepParameterCombinations = list(self.__product(*sweepParameterValues))

      return((sweepParameterNames,sweepParameterCombinations))


   def writeParameterCombinations(self,
                                  csvPath):
      csvDataParameters,csvDataCombinations          = self.__getCSVDataCombinations()
      sweepParameterNames,sweepParameterCombinations = self.__getSweepParameterCombinations()

      fpCSV = open(csvPath,'wb')
      if fpCSV:
         csvWriter = csv.writer(fpCSV)
         parameterCombinationCount = self.getParameterCombinationCount()
         nInstanceIdDigits = int(math.log10(parameterCombinationCount)+1)
         csvWriter.writerow(('# completed: 0/%d jobs' % (parameterCombinationCount),))
         if   len(csvDataParameters) > 0 and len(sweepParameterNames) > 0:
            parameters = ['Id','Status'] + csvDataParameters + sweepParameterNames
            csvWriter.writerow(parameters)
            instanceId = 0
            for csvDataCombination in csvDataCombinations:
               for sweepParameterCombination in sweepParameterCombinations:
                  instanceId += 1
                  instanceLabel = str(instanceId).zfill(nInstanceIdDigits)
                  combination = (instanceLabel,'setting up') + csvDataCombination + sweepParameterCombination
                  csvWriter.writerow(combination)
         elif len(csvDataParameters) > 0:
            csvDataParameters.insert(0,'Status')
            csvDataParameters.insert(0,'Id')
            csvWriter.writerow(csvDataParameters)
            instanceId = 0
            for csvDataCombination in csvDataCombinations:
               instanceId += 1
               instanceLabel = str(instanceId).zfill(nInstanceIdDigits)
               csvWriter.writerow((instanceLabel,'setting up') + csvDataCombination)
         elif len(sweepParameterNames) > 0:
            sweepParameterNames.insert(0,'Status')
            sweepParameterNames.insert(0,'Id')
            csvWriter.writerow(sweepParameterNames)
            instanceId = 0
            for sweepParameterCombination in sweepParameterCombinations:
               instanceId += 1
               instanceLabel = str(instanceId).zfill(nInstanceIdDigits)
               csvWriter.writerow((instanceLabel,'setting up') + sweepParameterCombination)

         fpCSV.close()


   def getNextParameterCombination(self):
      csvDataParameters,csvDataCombinations          = self.__getCSVDataCombinations()
      sweepParameterNames,sweepParameterCombinations = self.__getSweepParameterCombinations()

      if   len(csvDataParameters) > 0 and len(sweepParameterNames) > 0:
         parameters = csvDataParameters + sweepParameterNames
         for csvDataCombination in csvDataCombinations:
            for sweepParameterCombination in sweepParameterCombinations:
               combination = csvDataCombination + sweepParameterCombination
               substitutions = {}
               for parameterName,parameterValue in zip(parameters,combination):
                  substitutions[parameterName] = parameterValue
               yield(substitutions)
      elif len(csvDataParameters) > 0:
         parameters = csvDataParameters
         for csvDataCombination in csvDataCombinations:
            combination = csvDataCombination
            substitutions = {}
            for parameterName,parameterValue in zip(parameters,combination):
               substitutions[parameterName] = parameterValue
            yield(substitutions)
      elif len(sweepParameterNames) > 0:
         parameters = sweepParameterNames
         for sweepParameterCombination in sweepParameterCombinations:
            combination = sweepParameterCombination
            substitutions = {}
            for parameterName,parameterValue in zip(parameters,combination):
               substitutions[parameterName] = parameterValue
            yield(substitutions)


   def getNextParameterCombinationFromCSV(self,
                                          csvPath):
      fpCSV = open(csvPath,'rb')
      if fpCSV:
         csvReader = csv.reader(CommentedFile(fpCSV))
         parameterNames = csvReader.next()
         del parameterNames[0]
         del parameterNames[0]
         try:
            for parameterCombination in csvReader:
               del parameterCombination[0]
               del parameterCombination[0]
               substitutions = {}
               for parameterName,parameterValue in zip(parameterNames,parameterCombination):
                  substitutions[parameterName] = parameterValue
               yield(substitutions)
         except StopIteration:
            fpCSV.close()
         except csv.Error,e:
            errorMessage = "Error in parameter data file %s, line %d: %s" % (csvPath,csvReader.line_num,e)
            log(errorMessage)
#           sys.stderr.write(errorMessage + "\n")
#           sys.stderr.flush()


