# @package      hubzero-submit-common
# @file         RemoteWorkflowPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re

from Pegasus.DAX3 import *

class RemoteWorkflowPEGASUS:
   def __init__(self,
                hubUserName,
                localJobId,
                pegasusVersion,
                pegasusHome,
                pegasusTemplates,
                x509SubmitProxy,
                gridsite,
                timeHistoryLogs):

      self.hubUserName      = hubUserName
      self.localJobId       = localJobId
      self.jobDirectory     = os.path.join(os.getcwd(),self.localJobId)

      self.daxPath          = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable       = 'pegasus-plan'
      self.arguments        = "--dax %s" % (self.daxPath)

      self.pegasusVersion   = pegasusVersion
      self.pegasusHome      = pegasusHome
      self.pegasusTemplates = pegasusTemplates
      self.x509SubmitProxy  = x509SubmitProxy
      self.gridsite         = gridsite
      self.timestampStart   = timeHistoryLogs['timestampStart']
      self.timestampFinish  = timeHistoryLogs['timestampFinish']
      self.timeResults      = timeHistoryLogs['timeResults']


   def __makeWorkflowTemplate(self):
      return """#!/bin/sh
# RemoteWorkflowPEGASUS:makeWorkflowTemplate
#
mkdir -p INSTANCEDIRECTORY/work
mkdir -p INSTANCEDIRECTORY/InProcessResults

PEGASUSSETUP

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS \\
             EXECUTABLE --conf INSTANCEDIRECTORY/JOBID.pegasusrc \\
             --sites GRIDSITE \\
             --dir INSTANCEDIRECTORY/work \\
             --relative-dir pegasus \\
             --relative-submit-dir pegasus \\
             --output local \\
             ARGUMENTS \\
             --submit < /dev/null

date +"%s" > TS_FINISH

"""


   def __buildWorkflowScript(self):
      # setup regex's for the template
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_pegasusSetup      = re.compile("PEGASUSSETUP")
      re_executable        = re.compile("EXECUTABLE")
      re_gridsite          = re.compile("GRIDSITE")
      re_arguments         = re.compile("ARGUMENTS")
      re_tsStart           = re.compile("TS_START")
      re_tsFinish          = re.compile("TS_FINISH")
      re_timeResults       = re.compile("TIME_RESULTS")

      template = self.__makeWorkflowTemplate()

      try:
         pegasusSetupCommand = ". /apps/environ/.setup.sh\nuse -e -r pegasus-%s" % (self.pegasusVersion)
         pegasusSetupCommandPath = "INSTANCEDIRECTORY/work/pegasusSetup.sh"
         template = re_pegasusSetup.sub("cat > %s << EOF\n%s\nEOF\n. %s\n" % (pegasusSetupCommandPath,pegasusSetupCommand,
                                                                              pegasusSetupCommandPath),template)
      except:
         template = re_pegasusSetup.sub("",template)

      template = re_jobid.sub(self.localJobId,template)
      template = re_instanceDirectory.sub(self.jobDirectory,template)
      template = re_executable.sub(self.executable,template)
      template = re_gridsite.sub(self.gridsite,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      auxiliaryArguments = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if os.path.exists(self.pegasusTemplates['rc']):
         fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
         if fpRCTemplate:
            searchString = os.path.basename(self.executable) + '.arguments'
            settings = fpRCTemplate.readlines()
            for setting in settings:
#pegasus-plan.arguments = --nocleanup
               if setting.count(searchString) > 0:
                  try:
                     parameter,value = setting.split('=')
                     auxiliaryArguments.append(value.strip())
                  except:
                     pass
            fpRCTemplate.close()

      arguments = ' '.join(auxiliaryArguments)
      template = re_arguments.sub(arguments.strip(),template)

      return(template)


   def __buildCatalogs(self):
      # setup regex's for the template
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_x509SubmitProxy   = re.compile("X509SUBMITPROXY")
      re_pegasusHome       = re.compile("PEGASUSHOME")

      for templateType in self.pegasusTemplates:
         pegasusTemplate = self.pegasusTemplates[templateType]
         if pegasusTemplate != "":
            fpTemplate = open(pegasusTemplate,'r')
            if fpTemplate:
               template = ''.join(fpTemplate.readlines())
               fpTemplate.close()
               template = re_jobid.sub(self.localJobId,template)
               template = re_instanceDirectory.sub(self.jobDirectory,template)
               if self.x509SubmitProxy != "":
                  template = re_x509SubmitProxy.sub(self.x509SubmitProxy,template)
               else:
                  template = re_x509SubmitProxy.sub(os.path.join(os.sep,'tmp','hub-proxy.%s' % (self.hubUserName)),template)
               template = re_pegasusHome.sub(self.pegasusHome,template)
               if   templateType == 'rc':
                  pegasusFile = "%s.pegasusrc" % (self.localJobId)
               elif templateType == 'sites':
                  pegasusFile = "%s_sites.xml" % (self.localJobId)
               elif templateType == 'tc':
                  pegasusFile = "%s_tc.txt" % (self.localJobId)
               pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
               fpPegasusFile = open(pegasusPath,'w')
               if fpPegasusFile:
                  fpPegasusFile.write(template)
                  fpPegasusFile.close()


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(os.path.join(os.getcwd(),self.localJobId))
      for dirFile in dirFiles:
         dirPath = os.path.join(os.getcwd(),self.localJobId,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      dax = ADAG(self.localJobId)

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      reStageInTarFile = re.compile("[0-9_]+_input.tar.gz")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(os.getcwd(),self.localJobId,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles  = filter(reAppScriptPath.search,dirFiles)
         stageInTarFiles = filter(reStageInTarFile.search,dirFiles)

         if len(appScriptFiles) == 1 and len(stageInTarFiles) == 1:
            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            appScriptPath = "file://" + appScriptPath
            appScript = Executable(name=appScriptName,arch=Arch.X86_64,os=OS.LINUX,installed=False)
            appScript.addPFN(PFN(appScriptPath,"local"))
            dax.addExecutable(appScript)

            stageInTarFile = stageInTarFiles[0]
            inputTarPath = "file://" + os.path.join(instanceDirectory,stageInTarFile)
            inputTar = File(stageInTarFile)
            inputTar.addPFN(PFN(inputTarPath,"local"))
            dax.addFile(inputTar)

            stageOutTarFile = stageInTarFile.replace('input','output')
            outputTar = File(stageOutTarFile)

            jobId = "%s_%s" % (self.localJobId,instanceId)
            job = Job(name=appScriptName,id=jobId)
            job.uses(inputTar,link=Link.INPUT,executable=False)
            job.uses(outputTar,link=Link.OUTPUT,register=True,transfer=True)
            dax.addJob(job)

      fpDAX = open(self.daxPath,'w')
      if fpDAX:
         dax.writeXML(fpDAX)
         fpDAX.close()


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()

      return(workflowScriptName,workflowScript)


