# @package      hubzero-submit-common
# @file         LocalWorkflowPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re

from Pegasus.DAX3 import *

class LocalWorkflowPEGASUS:
   def __init__(self,
                pegasusVersion,
                pegasusHome,
                localJobId,
                pegasusTemplates,
                timeHistoryLogs):

      self.pegasusVersion   = pegasusVersion
      self.pegasusHome      = pegasusHome
      self.localJobId       = localJobId
      self.jobDirectory     = os.path.join(os.getcwd(),self.localJobId)

      self.daxPath          = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable       = 'pegasus-plan'
      self.arguments        = "--dax %s" % (self.daxPath)

      self.pegasusTemplates = pegasusTemplates
      self.timestampStart   = timeHistoryLogs['timestampStart']
      self.timestampFinish  = timeHistoryLogs['timestampFinish']
      self.timeResults      = timeHistoryLogs['timeResults']


   def __makeWorkflowTemplate(self):
      return """#!/bin/sh
# LocalWorkflowPEGASUS:makeWorkflowTemplate
#
. /apps/environ/.setup.sh
use -e -r pegasus-PEGASUSVERSION

mkdir -p INSTANCEDIRECTORY/work

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS \\
             EXECUTABLE --conf INSTANCEDIRECTORY/JOBID.pegasusrc \\
             --sites local \\
             --dir INSTANCEDIRECTORY/work \\
             --relative-dir pegasus \\
             --relative-submit-dir pegasus \\
             --output local \\
             ARGUMENTS \\
             --submit < /dev/null

date +"%s" > TS_FINISH

"""


   def __buildWorkflowScript(self):
      # setup regex's for the template
      re_pegasusVersion    = re.compile("PEGASUSVERSION")
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_executable        = re.compile("EXECUTABLE")
      re_arguments         = re.compile("ARGUMENTS")
      re_tsStart           = re.compile("TS_START")
      re_tsFinish          = re.compile("TS_FINISH")
      re_timeResults       = re.compile("TIME_RESULTS")

      template = self.__makeWorkflowTemplate()

      template = re_pegasusVersion.sub(self.pegasusVersion,template)
      template = re_jobid.sub(self.localJobId,template)
      template = re_instanceDirectory.sub(self.jobDirectory,template)
      template = re_executable.sub(self.executable,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      auxiliaryArguments = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if 'rc' in self.pegasusTemplates:
         searchString = os.path.basename(self.executable) + '.arguments'
         settings = self.pegasusTemplates['rc'].split('\n')
         for setting in settings:
#pegasus-plan.arguments = --nocleanup
            if setting.count(searchString) > 0:
               try:
                  parameter,value = setting.split('=')
                  auxiliaryArguments.append(value.strip())
               except:
                  pass

      arguments = ' '.join(auxiliaryArguments)
      template = re_arguments.sub(arguments.strip(),template)

      return(template)


   def __buildCatalogs(self):
      # setup regex's for the template
      re_jobid             = re.compile("JOBID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_pegasusHome       = re.compile("PEGASUSHOME")

      for templateType in self.pegasusTemplates:
         template = self.pegasusTemplates[templateType]
         template = re_jobid.sub(self.localJobId,template)
         template = re_instanceDirectory.sub(self.jobDirectory,template)
         template = re_pegasusHome.sub(self.pegasusHome,template)
         if   templateType == 'rc':
            pegasusFile = "%s.pegasusrc" % (self.localJobId)
         elif templateType == 'sites':
            pegasusFile = "%s_sites.xml" % (self.localJobId)
         pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
         fpPegasusFile = open(pegasusPath,'w')
         if fpPegasusFile:
            fpPegasusFile.write(template)
            fpPegasusFile.close()


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(os.path.join(os.getcwd(),self.localJobId))
      for dirFile in dirFiles:
         dirPath = os.path.join(os.getcwd(),self.localJobId,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      dax = ADAG(self.localJobId)

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(os.getcwd(),self.localJobId,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles  = filter(reAppScriptPath.search,dirFiles)

         if len(appScriptFiles) == 1:
            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            appScriptPath = "file://" + appScriptPath
            appScript = Executable(name=appScriptName,arch=Arch.X86_64,os=OS.LINUX,installed=True)
            appScript.addPFN(PFN(appScriptPath,"local"))
            dax.addExecutable(appScript)

            jobId = "%s_%s" % (self.localJobId,instanceId)
            job = Job(name=appScriptName,id=jobId)
            dax.addJob(job)

      fpDAX = open(self.daxPath,'w')
      if fpDAX:
         dax.writeXML(fpDAX)
         fpDAX.close()


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()

      return(workflowScriptName,workflowScript)


