# @package      hubzero-submit-common
# @file         JobOutput.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os.path
import traceback
import csv
from xml.dom import minidom, Node

from hubzero.submit.LogMessage    import logID as log
from hubzero.submit.CommentedFile import CommentedFile

def getDaxExecutables(daxPath):
   daxExecutables = []
   if os.path.exists(daxPath):
      daxDoc = minidom.parse(daxPath)
      rootNode = daxDoc.documentElement
      children = rootNode.getElementsByTagName("executable")

      for child in children:
         installed = child.getAttribute('installed')
         if not installed.lower() in ('true','t'):
            for grandChild in child.childNodes:
               if grandChild.nodeType == Node.ELEMENT_NODE:
                  if grandChild.tagName == "pfn":
                     url = grandChild.getAttribute("url")
                     if url.startswith("file://"):
                        executablePath = url[len("file://"):]
                        daxExecutables.append(executablePath)
   else:
      log("dax %s file is missing" % (daxPath))

   return(daxExecutables)


def getJobOutputContent(parentNode,
                        elementTag,
                        elementId,
                        dataTag):
   jobOutputContent = None
   content = []
   children = parentNode.getElementsByTagName(elementTag)
   for child in children:
      childId = child.getAttribute('id')
      if childId == elementId:
         for grandChild in child.childNodes:
            if grandChild.nodeType == Node.ELEMENT_NODE:
               if grandChild.tagName == dataTag:
                  if grandChild.hasChildNodes():
                     for greatGrandChild in grandChild.childNodes:
                        content.append(greatGrandChild.nodeValue)
   if content:
      jobOutputContent = '\n'.join(content)
   del content

   return(jobOutputContent)


def getJobEnvironmentVariable(parentNode,
                              environmentVariable):
   value = ""
   children = parentNode.getElementsByTagName('env')
   for child in children:
      childKey = child.getAttribute('key')
      if childKey == environmentVariable:
         for grandChild in child.childNodes:
            if grandChild.nodeType == Node.TEXT_NODE:
               value = grandChild.data

   return(value)


def getJobExecutable(parentNode):
   jobExecutable = ""
   children = parentNode.getElementsByTagName('mainjob')
   if len(children) == 1:
      grandChildren = children[0].getElementsByTagName('statcall')
      if len(grandChildren) == 1:
         greatGrandChildren = grandChildren[0].getElementsByTagName('file')
         if len(grandChildren) == 1:
            jobExecutable = greatGrandChildren[0].getAttribute('name')

   return(jobExecutable)


def processPegasusFiles(instanceDirectory,
                        scratchDirectory,
                        timeResultsBase):
   pegasusFileDirectory = os.path.join(scratchDirectory,'work')
   jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if not os.path.exists(jobStateLogPath):
      pegasusFileDirectory = os.path.join(instanceDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if os.path.exists(jobStateLogPath):
      condorIds = {}
      fpJobStateLog = open(jobStateLogPath,'rb')
      if fpJobStateLog:
         while True:
            record = fpJobStateLog.readline()
            if not record:
               break
            record = record.strip()
            if ' SUBMIT ' in record:
               parts = record.split()
               jobName,condorId,remoteSite = parts[1],parts[3],parts[4]
               if remoteSite != 'local':
                  if not jobName in condorIds:
                     condorIds[jobName] = []
                  condorIds[jobName].append(condorId)
         fpJobStateLog.close()

         jobStatsPath = os.path.join(instanceDirectory,'pegasusjobstats.csv')
         if os.path.exists(jobStatsPath):
            fpJobStats = open(jobStatsPath,'rb')
            if fpJobStats:
               csvReader = csv.reader(CommentedFile(fpJobStats))
               headers = csvReader.next()

               try:
                  columnSite = headers.index('Site')
                  columnJob = headers.index('Job')
                  columnTry = headers.index('Try')
                  columnExitcode = headers.index('Exitcode')
                  columnCondorQTime = headers.index('CondorQTime')
                  columnCPUTime = headers.index('CPU-Time')
                  columnKickstart = headers.index('Kickstart')
                  columnRuntime = headers.index('Runtime')

                  jobFileIndex = 1
                  jobFiles = {}

#                 log("%s %s %s %s %s %s %s %s" % (headers[columnSite], \
#                                                  headers[columnJob], \
#                                                  headers[columnTry], \
#                                                  headers[columnExitcode], \
#                                                  headers[columnCondorQTime], \
#                                                  headers[columnCPUTime], \
#                                                  headers[columnKickstart], \
#                                                  headers[columnRuntime]))
                  for row in csvReader:
                     if row[columnSite] != 'local' and row[columnSite] != '-':
                        if not row[columnJob] in jobFiles:
                           jobFileIndex += 1
                           jobFiles[row[columnJob]] = os.path.join(instanceDirectory,"%s_%d" % (timeResultsBase, \
                                                                                                jobFileIndex))
                        jobTry = int(row[columnTry])-1
                        try:
                           condorId = condorIds[row[columnJob]][jobTry]
                        except:
                           log(traceback.format_exc())
                           condorId = None
                        event = ""

#                       log("%s %s %s %s %s %s %s %s" % (row[columnSite], \
#                                                        row[columnJob], \
#                                                        row[columnTry], \
#                                                        row[columnExitcode], \
#                                                        row[columnCondorQTime], \
#                                                        row[columnCPUTime], \
#                                                        row[columnKickstart], \
#                                                        row[columnRuntime]))

                        jobOutputFile = "%s.out.%s" % (row[columnJob],str(jobTry).zfill(3))
                        jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                        if os.path.exists(jobOutputPath):
                           jobOutputDocumentObject = minidom.parse(jobOutputPath)
                           rootNode = jobOutputDocumentObject.documentElement

                           jobStdout = getJobOutputContent(rootNode,'statcall','stdout','data')
                           if jobStdout:
                              jobStdoutFile = os.path.join(instanceDirectory,"%s.stdout" % (row[columnJob]))
                              fpStdout = open(jobStdoutFile,'w')
                              fpStdout.write(jobStdout)
                              fpStdout.close()
                           del jobStdout

                           jobStderr = getJobOutputContent(rootNode,'statcall','stderr','data')
                           if jobStderr:
                              jobStderrFile = os.path.join(instanceDirectory,"%s.stderr" % (row[columnJob]))
                              fpStderr = open(jobStderrFile,'w')
                              fpStderr.write(jobStderr)
                              fpStderr.close()
                           del jobStderr

                           glideinSite = getJobEnvironmentVariable(rootNode,"GLIDEIN_Site")
                           if glideinSite:
                              row[columnSite] = glideinSite

                           executable = getJobExecutable(rootNode)
                           if executable:
                              event = '/' + os.path.basename(executable)
      
                           jobOutputDocumentObject.unlink()
                        else:
                           log("processPegasusFiles:Job output file missing: %s" % (jobOutputPath))

                        fpTimeResults = open(jobFiles[row[columnJob]],'a')
                        if fpTimeResults:
                           if row[columnExitcode] != '0':
                              fpTimeResults.write('Command exited with non-zero status %s\n' % (row[columnExitcode]))
                           fpTimeResults.write('real %s\n' % (row[columnKickstart]))
                           fpTimeResults.write('user %s\n' % (row[columnCPUTime]))
                           fpTimeResults.write('wait %s\n' % (row[columnCondorQTime]))
                           fpTimeResults.write('elapsed %s\n' % (row[columnRuntime]))
                           fpTimeResults.write('site %s\n' % (row[columnSite]))
                           if condorId:
                              fpTimeResults.write('jobId %s\n' % (condorId))
                           if event:
                              fpTimeResults.write('event %s\n' % (event))
                           fpTimeResults.close()
               except:
                  log(traceback.format_exc())
               fpJobStats.close()
            else:
               log("processPegasusFiles:Failed to open Job Stats: %s" % (jobStatsPath))
         else:
            log("processPegasusFiles:Job Stats missing: %s" % (jobStatsPath))
      else:
         log("processPegasusFiles:Failed to open Job State Log: %s" % (jobStateLogPath))
   else:
      log("processPegasusFiles:Job State Log missing for post-processing: %s" % (jobStateLogPath))


def getInProgressPegasusJobExitCodes(instanceDirectory,
                                     scratchDirectory):
   exitCodes = {}
   pegasusFileDirectory = os.path.join(scratchDirectory,'work')
   jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if not os.path.exists(jobStateLogPath):
      pegasusFileDirectory = os.path.join(instanceDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if os.path.exists(jobStateLogPath):
      fpJobStateLog = open(jobStateLogPath,'rb')
      if fpJobStateLog:
         while True:
            record = fpJobStateLog.readline()
            if not record:
               break
            record = record.strip()

            if   'SUBMIT' in record:
               parts = record.split()
               jobName,remoteSite = parts[1],parts[4]
               if remoteSite != 'local' and remoteSite != '-':
                  jobId = jobName.split('.')[0]
                  instance = int(jobId.split('_')[1])
                  if instance in exitCodes:
                     del exitCodes[instance]
            elif 'JOB_TERMINATED' in record:
               parts = record.split()
               jobName,remoteSite = parts[1],parts[4]
               if remoteSite != 'local' and remoteSite != '-':
                  jobId = jobName.split('.')[0]
                  instance = int(jobId.split('_')[1])
                  exitCodes[instance] = 'D'
            elif 'JOB_SUCCESS' in record:
               parts = record.split()
               jobName,remoteSite = parts[1],parts[4]
               if remoteSite != 'local' and remoteSite != '-':
                  jobId = jobName.split('.')[0]
                  instance = int(jobId.split('_')[1])
                  if instance in exitCodes:
                     exitCodes[instance] = 'D'
            elif 'JOB_FAILURE' in record:
               parts = record.split()
               jobName,remoteSite = parts[1],parts[4]
               if remoteSite != 'local' and remoteSite != '-':
                  jobId = jobName.split('.')[0]
                  instance = int(jobId.split('_')[1])
                  if instance in exitCodes:
                     exitCodes[instance] = 'EF'
         fpJobStateLog.close()
      else:
         log("getInProgressPegasusJobExitCodes:Failed to open Job State Log: %s" % (jobStateLogPath))
#  else:
#     log("getInProgressPegasusJobExitCodes:Job State Log missing: %s" % (jobStateLogPath))

   return(exitCodes)


def getInProgressPegasusJobStdFiles(instanceDirectory,
                                    scratchDirectory):
   pegasusFileDirectory = os.path.join(scratchDirectory,'work')
   jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if not os.path.exists(jobStateLogPath):
      pegasusFileDirectory = os.path.join(instanceDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if os.path.exists(jobStateLogPath):
      fpJobStateLog = open(jobStateLogPath,'rb')
      if fpJobStateLog:
         while True:
            record = fpJobStateLog.readline()
            if not record:
               break
            record = record.strip()
            if 'JOB_TERMINATED' in record:
               parts = record.split()
               jobName,remoteSite = parts[1],parts[4]
               jobId = jobName.split('.')[0]
               runNameId = '_'.join(jobName.split('_')[-2:])
               jobTry = 0
               if remoteSite != 'local' and remoteSite != '-':
                  jobOutputFile = "%s.out.%s" % (jobName,str(jobTry).zfill(3))
                  jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                  if os.path.exists(jobOutputPath):
                     jobOutputDocumentObject = minidom.parse(jobOutputPath)
                     rootNode = jobOutputDocumentObject.documentElement
   
                     jobStdout = getJobOutputContent(rootNode,'statcall','stdout','data')
                     if jobStdout:
                        jobStdoutFile = os.path.join(instanceDirectory,'InProcessResults',"%s.stdout" % (runNameId))
                        fpStdout = open(jobStdoutFile,'w')
                        fpStdout.write(jobStdout)
                        fpStdout.close()
                     del jobStdout
   
                     jobStderr = getJobOutputContent(rootNode,'statcall','stderr','data')
                     if jobStderr:
                        jobStderrFile = os.path.join(instanceDirectory,'InProcessResults',"%s.stderr" % (runNameId))
                        fpStderr = open(jobStderrFile,'w')
                        fpStderr.write(jobStderr)
                        fpStderr.close()
                     del jobStderr
                     jobOutputDocumentObject.unlink()
                  else:
                     log("getInProgressPegasusJobStdFiles:Job output file missing: %s" % (jobOutputPath))
         fpJobStateLog.close()
      else:
         log("getInProgressPegasusJobStdFiles:Failed to open Job State Log: %s" % (jobStateLogPath))
   else:
      log("getInProgressPegasusJobStdFiles:Job State Log missing: %s" % (jobStateLogPath))


def getPegasusStdTimeFiles(instanceDirectory,
                           scratchDirectory,
                           dagTimeResultsFile):
   pegasusFileDirectory = os.path.join(scratchDirectory,'work')
   jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if not os.path.exists(jobStateLogPath):
      pegasusFileDirectory = os.path.join(instanceDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
   if os.path.exists(jobStateLogPath):
      condorIds = {}
      fpJobStateLog = open(jobStateLogPath,'rb')
      if fpJobStateLog:
         while True:
            record = fpJobStateLog.readline()
            if not record:
               break
            record = record.strip()
            if ' SUBMIT ' in record:
               parts = record.split()
               jobName,condorId,remoteSite = parts[1],parts[3],parts[4]
               if remoteSite != 'local':
                  jobId = jobName.split('.')[0]
                  if not jobId in condorIds:
                     condorIds[jobId] = []
                  condorIds[jobId].append(condorId)
         fpJobStateLog.close()

         jobStatsPath = os.path.join(instanceDirectory,'pegasusjobstats.csv')
         if os.path.exists(jobStatsPath):
            fpJobStats = open(jobStatsPath,'rb')
            if fpJobStats:
               csvReader = csv.reader(CommentedFile(fpJobStats))
               headers = csvReader.next()

               try:
                  columnSite        = headers.index('Site')
                  columnJob         = headers.index('Job')
                  columnTry         = headers.index('Try')
                  columnExitcode    = headers.index('Exitcode')
                  columnCondorQTime = headers.index('CondorQTime')
                  columnCPUTime     = headers.index('CPU-Time')
                  columnKickstart   = headers.index('Kickstart')
                  columnRuntime     = headers.index('Runtime')

                  if dagTimeResultsFile.endswith('_0'):
                     timeResultsBase = dagTimeResultsFile[:-2]
                  else:
                     timeResultsBase = dagTimeResultsFile
                  jobFiles = {}

#                 log("%s %s %s %s %s %s %s %s" % (headers[columnSite], \
#                                                  headers[columnJob], \
#                                                  headers[columnTry], \
#                                                  headers[columnExitcode], \
#                                                  headers[columnCondorQTime], \
#                                                  headers[columnCPUTime], \
#                                                  headers[columnKickstart], \
#                                                  headers[columnRuntime]))
                  for row in csvReader:
                     if row[columnSite] != 'local' and row[columnSite] != '-':
                        jobName = row[columnJob]
                        runNameId = '_'.join(jobName.split('_')[-2:])
                        jobId = jobName.split('.')[0]
                        jobFileIndex = jobId.split('_')[1]
                        if not jobId in jobFiles:
                           jobFiles[jobId] = os.path.join(instanceDirectory,jobFileIndex,"%s_%s" % (timeResultsBase,jobFileIndex))
                        jobTry = int(row[columnTry])-1
                        try:
                           condorId = condorIds[jobId][jobTry]
                        except:
                           log(traceback.format_exc())
                           condorId = None
                        event = ""
#                       log("%s %s %s %s %s %s %s %s" % (row[columnSite], \
#                                                        row[columnJob], \
#                                                        row[columnTry], \
#                                                        row[columnExitcode], \
#                                                        row[columnCondorQTime], \
#                                                        row[columnCPUTime], \
#                                                        row[columnKickstart], \
#                                                        row[columnRuntime]))

                        jobOutputFile = "%s.out.%s" % (row[columnJob],str(jobTry).zfill(3))
                        jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                        if os.path.exists(jobOutputPath):
                           jobOutputDocumentObject = minidom.parse(jobOutputPath)
                           rootNode = jobOutputDocumentObject.documentElement

                           jobStdout = getJobOutputContent(rootNode,'statcall','stdout','data')
                           if jobStdout:
                              jobStdoutFile = os.path.join(instanceDirectory,jobFileIndex,"%s.stdout" % (runNameId))
                              fpStdout = open(jobStdoutFile,'w')
                              fpStdout.write(jobStdout)
                              fpStdout.close()
                           del jobStdout

                           jobStderr = getJobOutputContent(rootNode,'statcall','stderr','data')
                           if jobStderr:
                              jobStderrFile = os.path.join(instanceDirectory,jobFileIndex,"%s.stderr" % (runNameId))
                              fpStderr = open(jobStderrFile,'w')
                              fpStderr.write(jobStderr)
                              fpStderr.close()
                           del jobStderr

                           glideinSite = getJobEnvironmentVariable(rootNode,"GLIDEIN_Site")
                           if glideinSite:
                              row[columnSite] = glideinSite

                           executable = getJobExecutable(rootNode)
                           if executable:
                              event = '/' + os.path.basename(executable)
      
                           jobOutputDocumentObject.unlink()
                        else:
                           log("getPegasusStdTimeFiles:Job output file missing: %s" % (jobOutputPath))

                        fpTimeResults = open(jobFiles[jobId],'a')
                        if fpTimeResults:
                           if row[columnExitcode] != '0':
                              fpTimeResults.write('Command exited with non-zero status %s\n' % (row[columnExitcode]))
                           fpTimeResults.write('real %s\n' % (row[columnKickstart]))
                           fpTimeResults.write('user %s\n' % (row[columnCPUTime]))
                           fpTimeResults.write('wait %s\n' % (row[columnCondorQTime]))
                           fpTimeResults.write('elapsed %s\n' % (row[columnRuntime]))
                           fpTimeResults.write('site %s\n' % (row[columnSite]))
                           if condorId:
                              fpTimeResults.write('jobId %s\n' % (condorId))
                           if event:
                              fpTimeResults.write('event %s\n' % (event))
                           fpTimeResults.close()
               except:
                  log(traceback.format_exc())
               fpJobStats.close()
            else:
               log("getPegasusStdTimeFiles:Failed to open Job Stats: %s" % (jobStatsPath))
         else:
            log("getPegasusStdTimeFiles:Job Stats missing: %s" % (jobStatsPath))
      else:
         log("getPegasusStdTimeFiles:Failed to open Job State Log: %s" % (jobStateLogPath))
   else:
      log("getPegasusStdTimeFiles:Job State Log missing for post-processing: %s" % (jobStateLogPath))


