# @package      hubzero-submit-common
# @file         JobOutput.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import glob
import traceback
import re
import csv
import logging
from xml.dom import minidom, Node
import yaml
from xml.parsers.expat import ExpatError

from hubzero.submit.LogMessage    import getLogIDMessage as getLogMessage
from hubzero.submit.CommentedFile import CommentedFile

class JobOutput:
   def __init__(self):
      self.logger = logging.getLogger(__name__)


   def getDaxExecutables(self,
                         daxPath):
      daxExecutables = []
      if os.path.exists(daxPath):
         if   daxPath.endswith('.xml') or daxPath.endswith('.dax'):
            try:
               daxDoc = minidom.parse(daxPath)
            except ExpatError:
               message = "getDaxExecutables: %s file could not be parsed" % (daxPath)
               self.logger.log(logging.ERROR,getLogMessage(message))
            else:
               rootNode = daxDoc.documentElement
               children = rootNode.getElementsByTagName("executable")

               for child in children:
                  installed = child.getAttribute('installed')
                  if not installed.lower() in ('true','t'):
                     for grandChild in child.childNodes:
                        if grandChild.nodeType == Node.ELEMENT_NODE:
                           if grandChild.tagName == "pfn":
                              url = grandChild.getAttribute("url")
                              if url.startswith("file://"):
                                 executablePath = url[len("file://"):]
                                 daxExecutables.append(executablePath)
         elif daxPath.endswith('.yml'):
            try:
               with open(daxPath,'r') as fpTransactions:
                  daxDoc = yaml.load(fpTransactions,Loader=yaml.CSafeLoader)
            except ExpatError:
               message = "getDaxExecutables: %s file could not be parsed" % (daxPath)
               self.logger.log(logging.ERROR,getLogMessage(message))
            else:
               try:
                  tc = daxDoc['transformationCatalog']
               except KeyError:
                  pass
               else:
                  try:
                     transformations = tc['transformations']
                  except KeyError:
                     pass
                  else:
                     for transformation in transformations:
                        try:
                           sites = transformation['sites']
                        except KeyError:
                           pass
                        else: 
                           for site in sites:
                              if site['name'] == 'local':
                                 executablePath = site['pfn']
                                 daxExecutables.append(executablePath)
         else:
            self.logger.log(logging.ERROR,getLogMessage("dax %s file has unrecognized format" % (daxPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("dax %s file is missing" % (daxPath)))

      return(daxExecutables)


   def getDaxExecutableSites(self,
                             daxPath):
      daxExecutableSites = []
      if os.path.exists(daxPath):
         try:
            daxDoc = minidom.parse(daxPath)
         except ExpatError:
            message = "getDaxExecutableSites: %s file could not be parsed" % (daxPath)
            self.logger.log(logging.ERROR,getLogMessage(message))
         else:
            rootNode = daxDoc.documentElement
            children = rootNode.getElementsByTagName("executable")

            for child in children:
               for grandChild in child.childNodes:
                  if grandChild.nodeType == Node.ELEMENT_NODE:
                     if grandChild.tagName == "pfn":
                        site = grandChild.getAttribute("site")
                        if site:
                           daxExecutableSites.append(site)
      else:
         self.logger.log(logging.ERROR,getLogMessage("dax %s file is missing" % (daxPath)))

      return(daxExecutableSites)


   @staticmethod
   def __getJobOutputContent(parentNode,
                             elementTag,
                             elementId,
                             dataTag):
      jobOutputContent = None
      content = []
      children = parentNode.getElementsByTagName(elementTag)
      for child in children:
         childId = child.getAttribute('id')
         if childId == elementId:
            for grandChild in child.childNodes:
               if grandChild.nodeType == Node.ELEMENT_NODE:
                  if grandChild.tagName == dataTag:
                     if grandChild.hasChildNodes():
                        for greatGrandChild in grandChild.childNodes:
                           content.append(greatGrandChild.nodeValue)
      if content:
         jobOutputContent = '\n'.join(content)
      del content

      return(jobOutputContent)


   @staticmethod
   def __getJobEnvironmentVariable(parentNode,
                                   environmentVariable):
      value = ""
      children = parentNode.getElementsByTagName('env')
      for child in children:
         childKey = child.getAttribute('key')
         if childKey == environmentVariable:
            for grandChild in child.childNodes:
               if grandChild.nodeType == Node.TEXT_NODE:
                  value = grandChild.data

      return(value)


   @staticmethod
   def __getJobExecutable(parentNode):
      jobExecutable = ""
      children = parentNode.getElementsByTagName('mainjob')
      if len(children) == 1:
         grandChildren = children[0].getElementsByTagName('statcall')
         if len(grandChildren) == 1:
            greatGrandChildren = grandChildren[0].getElementsByTagName('file')
            if len(grandChildren) == 1:
               jobExecutable = greatGrandChildren[0].getAttribute('name')

      return(jobExecutable)


   def processPegasusFiles(self,
                           instanceDirectory,
                           scratchDirectory,
                           timeResultsBase):
      pegasusFileDirectory = os.path.join(scratchDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if not os.path.exists(jobStateLogPath):
         pegasusFileDirectory = os.path.join(instanceDirectory,'work')
         jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if os.path.exists(jobStateLogPath):
         condorIds = {}
         try:
            fpJobStateLog = open(jobStateLogPath,'r')
            try:
               while True:
                  record = fpJobStateLog.readline()
                  if not record:
                     break
                  record = record.strip()
                  if ' SUBMIT ' in record:
                     parts = record.split()
                     jobName,condorId,remoteSite = parts[1],parts[3],parts[4]
                     if not jobName in condorIds:
                        condorIds[jobName] = []
                     condorIds[jobName].append(condorId)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (jobStateLogPath)))
            finally:
               fpJobStateLog.close()

            jobStatsPath = os.path.join(instanceDirectory,'pegasusjobstats.csv')
            if os.path.exists(jobStatsPath):
               try:
                  fpJobStats = open(jobStatsPath,'r')
                  try:
                     csvReader = csv.reader(CommentedFile(fpJobStats))
                     try:
                        headers = next(csvReader)
                        try:
                           columnDaxLabel    = headers.index('Dax_Label')
                           columnSite        = headers.index('Site')
                           columnJob         = headers.index('Job')
                           columnTry         = headers.index('Try')
                           columnExitcode    = headers.index('Exitcode')
                           columnCondorQTime = headers.index('CondorQTime')
                           columnCPUTime     = headers.index('CPU-Time')
                           columnKickstart   = headers.index('Kickstart')
                           columnRuntime     = headers.index('Runtime')

                           jobFileIndex = 1
                           jobFiles = {}

#                          message = "%s %s %s %s %s %s %s %s" % (headers[columnSite], \
#                                                                 headers[columnJob], \
#                                                                 headers[columnTry], \
#                                                                 headers[columnExitcode], \
#                                                                 headers[columnCondorQTime], \
#                                                                 headers[columnCPUTime], \
#                                                                 headers[columnKickstart], \
#                                                                 headers[columnRuntime])
#                          self.logger.log(logging.DEBUG,getLogMessage(message))

                           for row in csvReader:
                              try:
                                 exitStatus = int(float(row[columnExitcode]))
                              except ValueError:
                                 message = "processPegasusFiles:exitStatus = %s" % (row[columnExitcode])
                                 self.logger.log(logging.DEBUG,getLogMessage(message))
                                 exitStatus = 2

                              if row[columnSite] != 'local' and row[columnSite] != '-' and exitStatus >= 0:
                                 if not row[columnJob] in jobFiles:
                                    jobFileIndex += 1
                                    jobFiles[row[columnJob]] = os.path.join(instanceDirectory,"%s_%d" % (timeResultsBase, \
                                                                                                         jobFileIndex))
                                 jobTry = int(row[columnTry])-1
                                 try:
                                    condorId = condorIds[row[columnJob]][jobTry]
                                 except:
                                    self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                                    condorId = None
                                 event = ""

#                                message = "%s %s %s %s %s %s %s %s" % (row[columnSite], \
#                                                                       row[columnJob], \
#                                                                       row[columnTry], \
#                                                                       row[columnExitcode], \
#                                                                       row[columnCondorQTime], \
#                                                                       row[columnCPUTime], \
#                                                                       row[columnKickstart], \
#                                                                       row[columnRuntime])
#                                self.logger.log(logging.DEBUG,getLogMessage(message))

                                 jobOutputFile = "%s.out.%s" % (row[columnJob],str(jobTry).zfill(3))
                                 jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                                 if not os.path.exists(jobOutputPath):
                                    jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*',jobOutputFile)
                                    jobOutputPath = glob.glob(jobOutputSearchPath)
                                    if len(jobOutputPath) == 1:
                                       jobOutputPath = jobOutputPath[0]
                                    else:
                                       jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*','*',jobOutputFile)
                                       jobOutputPath = glob.glob(jobOutputSearchPath)
                                       if len(jobOutputPath) == 1:
                                          jobOutputPath = jobOutputPath[0]
                                       else:
                                          jobOutputPath = ""

                                 if os.path.exists(jobOutputPath):
                                    try:
                                       jobOutputDocumentObject = minidom.parse(jobOutputPath)
                                    except ExpatError:
                                       message = "processPegasusFiles: job output file %s could not be parsed" % (jobOutputPath)
                                       self.logger.log(logging.ERROR,getLogMessage(message))
                                    else:
                                       rootNode = jobOutputDocumentObject.documentElement

                                       jobStdout = self.__getJobOutputContent(rootNode,'statcall','stdout','data')
                                       if jobStdout:
                                          jobStdoutFile = os.path.join(instanceDirectory,"%s.stdout" % (row[columnJob]))
                                          try:
                                             fpStdout = open(jobStdoutFile,'w')
                                             try:
                                                fpStdout.write(jobStdout)
                                             except (IOError,OSError):
                                                self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % \
                                                                                                      (jobStdoutFile)))
                                             finally:
                                                fpStdout.close()
                                          except (IOError,OSError):
                                             self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % \
                                                                                                  (jobStdoutFile)))
                                       del jobStdout

                                       jobStderr = self.__getJobOutputContent(rootNode,'statcall','stderr','data')
                                       if jobStderr:
                                          jobStderrFile = os.path.join(instanceDirectory,"%s.stderr" % (row[columnJob]))
                                          try:
                                             fpStderr = open(jobStderrFile,'w')
                                             try:
                                                fpStderr.write(jobStderr)
                                             except (IOError,OSError):
                                                self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % \
                                                                                                      (jobStderrFile)))
                                             finally:
                                                fpStderr.close()
                                          except (IOError,OSError):
                                             self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % \
                                                                                                  (jobStderrFile)))
                                       del jobStderr

                                       glideinSite = self.__getJobEnvironmentVariable(rootNode,"GLIDEIN_Site")
                                       if glideinSite:
                                          row[columnSite] = glideinSite

                                       executable = self.__getJobExecutable(rootNode)
                                       if executable:
                                          program = os.path.basename(executable)
                                          if not program.startswith(row[columnDaxLabel]):
                                             if '/' in executable:
                                                event = '/' + program
                                             else:
                                                event = program

                                       jobOutputDocumentObject.unlink()
                                 else:
                                    message = "processPegasusFiles:Job output file missing: %s" % (jobOutputPath)
                                    self.logger.log(logging.WARNING,getLogMessage(message))

                                 try:
                                    timeResultsPath = jobFiles[row[columnJob]]
                                    fpTimeResults = open(timeResultsPath,'a')
                                    try:
                                       if exitStatus != 0:
                                          fpTimeResults.write('Command exited with non-zero status %s\n' % (exitStatus))
                                       fpTimeResults.write('real %s\n' % (row[columnKickstart]))
                                       fpTimeResults.write('user %s\n' % (row[columnCPUTime]))
                                       fpTimeResults.write('wait %s\n' % (row[columnCondorQTime]))
                                       fpTimeResults.write('elapsed %s\n' % (row[columnRuntime]))
                                       fpTimeResults.write('site %s\n' % (row[columnSite]))
                                       if condorId:
                                          fpTimeResults.write('jobId %s\n' % (condorId))
                                       if event:
                                          fpTimeResults.write('event %s\n' % (event))
                                    except (IOError,OSError):
                                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (timeResultsPath)))
                                    finally:
                                       fpTimeResults.close()
                                 except (IOError,OSError):
                                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (timeResultsPath)))
                        except:
                           self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                     except StopIteration:
                        self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (jobStatsPath)))
                  except csv.Error:
                     self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (jobStatsPath)))
                  finally:
                     fpJobStats.close()
               except (IOError,OSError):
                  message = "processPegasusFiles:Failed to open Job Stats: %s" % (jobStatsPath)
                  self.logger.log(logging.ERROR,getLogMessage(message))
            else:
               message = "processPegasusFiles:Job Stats missing: %s" % (jobStatsPath)
               self.logger.log(logging.ERROR,getLogMessage(message))
         except (IOError,OSError):
            message = "processPegasusFiles:Failed to open Job State Log: %s" % (jobStateLogPath)
            self.logger.log(logging.ERROR,getLogMessage(message))
      else:
         message = "processPegasusFiles:Job State Log missing for post-processing: %s" % (jobStateLogPath)
         self.logger.log(logging.ERROR,getLogMessage(message))


   def getInProgressPegasusJobExitCodes(self,
                                        instanceDirectory,
                                        scratchDirectory):
      exitCodes = {}
      pegasusFileDirectory = os.path.join(scratchDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if not os.path.exists(jobStateLogPath):
         pegasusFileDirectory = os.path.join(instanceDirectory,'work')
         jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if os.path.exists(jobStateLogPath):
         try:
            fpJobStateLog = open(jobStateLogPath,'r')
            try:
               while True:
                  record = fpJobStateLog.readline()
                  if not record:
                     break
                  record = record.strip()

                  if   'SUBMIT' in record:
                     parts = record.split()
                     jobName,remoteSite = parts[1],parts[4]
                     if remoteSite != 'local' and remoteSite != '-':
                        if '.' in jobName:
                           jobId = jobName.split('.')[0]
                        else:
                           jobId = jobName.split('_sh_')[0]
                        instance = int(jobId.split('_')[1])
                        if instance in exitCodes:
                           del exitCodes[instance]
                  elif 'JOB_TERMINATED' in record:
                     parts = record.split()
                     jobName,remoteSite = parts[1],parts[4]
                     if remoteSite != 'local' and remoteSite != '-':
                        if '.' in jobName:
                           jobId = jobName.split('.')[0]
                        else:
                           jobId = jobName.split('_sh_')[0]
                        instance = int(jobId.split('_')[1])
                        exitCodes[instance] = 'D'
                  elif 'JOB_SUCCESS' in record:
                     parts = record.split()
                     jobName,remoteSite = parts[1],parts[4]
                     if remoteSite != 'local' and remoteSite != '-':
                        if '.' in jobName:
                           jobId = jobName.split('.')[0]
                        else:
                           jobId = jobName.split('_sh_')[0]
                        instance = int(jobId.split('_')[1])
                        if instance in exitCodes:
                           exitCodes[instance] = 'D'
                  elif 'JOB_FAILURE' in record:
                     parts = record.split()
                     jobName,remoteSite = parts[1],parts[4]
                     if remoteSite != 'local' and remoteSite != '-':
                        if '.' in jobName:
                           jobId = jobName.split('.')[0]
                        else:
                           jobId = jobName.split('_sh_')[0]
                        instance = int(jobId.split('_')[1])
                        if instance in exitCodes:
                           exitCodes[instance] = 'EF'
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (jobStateLogPath,)))
            finally:
               fpJobStateLog.close()
         except (IOError,OSError):
            message = "getInProgressPegasusJobExitCodes:Failed to open Job State Log: %s" % (jobStateLogPath)
            self.logger.log(logging.ERROR,getLogMessage(message))

      return(exitCodes)


   def getInProgressPegasusJobStdFiles(self,
                                       instanceDirectory,
                                       scratchDirectory):
      pegasusFileDirectory = os.path.join(scratchDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if not os.path.exists(jobStateLogPath):
         pegasusFileDirectory = os.path.join(instanceDirectory,'work')
         jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if os.path.exists(jobStateLogPath):
         try:
            fpJobStateLog = open(jobStateLogPath,'r')
            try:
               while True:
                  record = fpJobStateLog.readline()
                  if not record:
                     break
                  record = record.strip()
                  if 'JOB_TERMINATED' in record:
                     parts = record.split()
                     jobName,remoteSite = parts[1],parts[4]
                     if '.' in jobName:
                        jobId = jobName.split('.')[0]
                     else:
                        jobId = jobName.split('_sh_')[0]
                     runNameId = '_'.join(jobName.split('_')[-2:])
                     jobTry = 0
                     if remoteSite != 'local' and remoteSite != '-':
                        jobOutputFile = "%s.out.%s" % (jobName,str(jobTry).zfill(3))
                        jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                        if not os.path.exists(jobOutputPath):
                           jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*',jobOutputFile)
                           jobOutputPath = glob.glob(jobOutputSearchPath)
                           if len(jobOutputPath) == 1:
                              jobOutputPath = jobOutputPath[0]
                           else:
                              jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*','*',jobOutputFile)
                              jobOutputPath = glob.glob(jobOutputSearchPath)
                              if len(jobOutputPath) == 1:
                                 jobOutputPath = jobOutputPath[0]
                              else:
                                 jobOutputPath = ""

                        if os.path.exists(jobOutputPath):
                           try:
                              jobOutputDocumentObject = minidom.parse(jobOutputPath)
                           except ExpatError:
                              message = "getInProgressPegasusJobStdFiles: job output file %s could not be parsed" % (jobOutputPath)
                              self.logger.log(logging.ERROR,getLogMessage(message))
                           else:
                              rootNode = jobOutputDocumentObject.documentElement

                              jobStdout = self.__getJobOutputContent(rootNode,'statcall','stdout','data')
                              if jobStdout:
                                 jobStdoutFile = os.path.join(instanceDirectory,'InProcessResults',"%s.stdout" % (runNameId))
                                 try:
                                    fpStdout = open(jobStdoutFile,'w')
                                    try:
                                       fpStdout.write(jobStdout)
                                    except (IOError,OSError):
                                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (jobStdoutFile)))
                                    finally:
                                       fpStdout.close()
                                 except (IOError,OSError):
                                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (jobStdoutFile)))
                              del jobStdout

                              jobStderr = self.__getJobOutputContent(rootNode,'statcall','stderr','data')
                              if jobStderr:
                                 jobStderrFile = os.path.join(instanceDirectory,'InProcessResults',"%s.stderr" % (runNameId))
                                 try:
                                    fpStderr = open(jobStderrFile,'w')
                                    try:
                                       fpStderr.write(jobStderr)
                                    except (IOError,OSError):
                                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (jobStderrFile)))
                                    finally:
                                       fpStderr.close()
                                 except (IOError,OSError):
                                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (jobStderrFile)))
                              del jobStderr
                              jobOutputDocumentObject.unlink()
                        else:
                           message = "getInProgressPegasusJobStdFiles:Job output file missing: %s: %s" % (jobOutputPath,record)
                           self.logger.log(logging.ERROR,getLogMessage(message))
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (jobStateLogPath)))
            finally:
               fpJobStateLog.close()
         except (IOError,OSError):
            message = "getInProgressPegasusJobStdFiles:Failed to open Job State Log: %s" % (jobStateLogPath)
            self.logger.log(logging.ERROR,getLogMessage(message))
      else:
         message = "getInProgressPegasusJobStdFiles:Job State Log missing: %s" % (jobStateLogPath)
         self.logger.log(logging.ERROR,getLogMessage(message))


   def getPegasusStdTimeFiles(self,
                              instanceDirectory,
                              scratchDirectory,
                              dagTimeResultsFile):
      pegasusFileDirectory = os.path.join(scratchDirectory,'work')
      jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if not os.path.exists(jobStateLogPath):
         pegasusFileDirectory = os.path.join(instanceDirectory,'work')
         jobStateLogPath = os.path.join(pegasusFileDirectory,'jobstate.log')
      if os.path.exists(jobStateLogPath):
         condorIds = {}
         try:
            fpJobStateLog = open(jobStateLogPath,'r')
            try:
               while True:
                  record = fpJobStateLog.readline()
                  if not record:
                     break
                  record = record.strip()
                  if ' SUBMIT ' in record:
                     parts = record.split()
                     jobName,condorId,remoteSite = parts[1],parts[3],parts[4]
                     if '.' in jobName:
                        jobId = jobName.split('.')[0]
                     else:
                        jobId = jobName.split('_sh_')[0]
                     if not jobId in condorIds:
                        condorIds[jobId] = []
                     condorIds[jobId].append(condorId)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (jobStateLogPath)))
            finally:
               fpJobStateLog.close()

            jobStatsPath = os.path.join(instanceDirectory,'pegasusjobstats.csv')
            if os.path.exists(jobStatsPath):
               try:
                  fpJobStats = open(jobStatsPath,'r')
                  try:
                     csvReader = csv.reader(CommentedFile(fpJobStats))
                     try:
                        headers = next(csvReader)
                        try:
                           columnDaxLabel    = headers.index('Dax_Label')
                           columnSite        = headers.index('Site')
                           columnJob         = headers.index('Job')
                           columnTry         = headers.index('Try')
                           columnExitcode    = headers.index('Exitcode')
                           columnCondorQTime = headers.index('CondorQTime')
                           columnCPUTime     = headers.index('CPU-Time')
                           columnKickstart   = headers.index('Kickstart')
                           columnRuntime     = headers.index('Runtime')

                           if dagTimeResultsFile.endswith('_0'):
                              timeResultsBase = dagTimeResultsFile[:-2]
                           else:
                              timeResultsBase = dagTimeResultsFile
                           jobFiles = {}

#                          message = "%s %s %s %s %s %s %s %s" % (headers[columnSite], \
#                                                                 headers[columnJob], \
#                                                                 headers[columnTry], \
#                                                                 headers[columnExitcode], \
#                                                                 headers[columnCondorQTime], \
#                                                                 headers[columnCPUTime], \
#                                                                 headers[columnKickstart], \
#                                                                 headers[columnRuntime])
#                          self.logger.log(logging.DEBUG,getLogMessage(message))

                           jobPattern = re.compile('(([^_]*)_)?([0-9]*)_([0-9]*)[\.|_]sh_([^_]*)_([0-9]*)(.*)')
                           for row in csvReader:
                              try:
                                 exitStatus = int(float(row[columnExitcode]))
                              except ValueError:
                                 message = "getPegasusStdTimeFiles:exitStatus = %s" % (row[columnExitcode])
                                 self.logger.log(logging.DEBUG,getLogMessage(message))
                                 exitStatus = 2

                              if row[columnSite] != 'local' and row[columnSite] != '-' and exitStatus >= 0:
                                 jobName = row[columnJob]
                                 jobInstanceLabel = jobPattern.match(jobName)
                                 if jobInstanceLabel:
                                    runNameId    = jobInstanceLabel.group(5)
                                    jobId        = '_'.join([jobInstanceLabel.group(3),jobInstanceLabel.group(4)])
                                    jobFileIndex = jobInstanceLabel.group(4)
                                    if not jobId in jobFiles:
                                       jobFiles[jobId] = os.path.join(instanceDirectory,jobFileIndex,"%s_%s" % (timeResultsBase,
                                                                                                                jobFileIndex))
                                 else:
                                    self.logger.log(logging.ERROR,getLogMessage("Indecipherable jobname: %s" % (jobName)))
                                    continue

                                 jobTry = int(row[columnTry])-1
                                 try:
                                    condorId = condorIds[jobId][jobTry]
                                 except:
                                    self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                                    condorId = None
                                 event = ""

#                                message = "%s %s %s %s %s %s %s %s" % (row[columnSite], \
#                                                                       row[columnJob], \
#                                                                       row[columnTry], \
#                                                                       row[columnExitcode], \
#                                                                       row[columnCondorQTime], \
#                                                                       row[columnCPUTime], \
#                                                                       row[columnKickstart], \
#                                                                       row[columnRuntime])
#                                self.logger.log(logging.DEBUG,getLogMessage(message))

                                 jobOutputFile = "%s.out.%s" % (row[columnJob],str(jobTry).zfill(3))
                                 jobOutputPath = os.path.join(pegasusFileDirectory,jobOutputFile)
                                 if not os.path.exists(jobOutputPath):
                                    jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*',jobOutputFile)
                                    jobOutputPath = glob.glob(jobOutputSearchPath)
                                    if len(jobOutputPath) == 1:
                                       jobOutputPath = jobOutputPath[0]
                                    else:
                                       jobOutputSearchPath = os.path.join(pegasusFileDirectory,'*','*',jobOutputFile)
                                       jobOutputPath = glob.glob(jobOutputSearchPath)
                                       if len(jobOutputPath) == 1:
                                          jobOutputPath = jobOutputPath[0]
                                       else:
                                          jobOutputPath = ""

                                 if os.path.exists(jobOutputPath):
                                    try:
                                       jobOutputDocumentObject = minidom.parse(jobOutputPath)
                                    except ExpatError:
                                       message = "getPegasusStdTimeFiles: job output file %s could not be parsed" % (jobOutputPath)
                                       self.logger.log(logging.ERROR,getLogMessage(message))
                                    else:
                                       rootNode = jobOutputDocumentObject.documentElement

                                       jobStdout = self.__getJobOutputContent(rootNode,'statcall','stdout','data')
                                       if jobStdout:
                                          jobStdoutFile = os.path.join(instanceDirectory,jobFileIndex,"%s_%s.stdout" % \
                                                                                             (runNameId,jobFileIndex))
                                          try:
                                             fpStdout = open(jobStdoutFile,'w')
                                             try:
                                                fpStdout.write(jobStdout)
                                             except (IOError,OSError):
                                                self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % \
                                                                                                      (jobStdoutFile)))
                                             finally:
                                                fpStdout.close()
                                          except (IOError,OSError):
                                             self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % \
                                                                                                  (jobStdoutFile)))
                                       del jobStdout

                                       jobStderr = self.__getJobOutputContent(rootNode,'statcall','stderr','data')
                                       if jobStderr:
                                          jobStderrFile = os.path.join(instanceDirectory,jobFileIndex,"%s_%s.stderr" % \
                                                                                             (runNameId,jobFileIndex))
                                          try:
                                             fpStderr = open(jobStderrFile,'w')
                                             try:
                                                fpStderr.write(jobStderr)
                                             except (IOError,OSError):
                                                self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % \
                                                                                                      (jobStderrFile)))
                                             finally:
                                                fpStderr.close()
                                          except (IOError,OSError):
                                             self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % \
                                                                                                  (jobStderrFile)))
                                       del jobStderr

                                       glideinSite = self.__getJobEnvironmentVariable(rootNode,"GLIDEIN_Site")
                                       if glideinSite:
                                          row[columnSite] = glideinSite

                                       executable = self.__getJobExecutable(rootNode)
                                       if executable:
                                          program = os.path.basename(executable)
                                          if not program.startswith(row[columnDaxLabel]):
                                             if '/' in executable:
                                                event = '/' + program
                                             else:
                                                event = program

                                       jobOutputDocumentObject.unlink()
                                 else:
                                    message = "getPegasusStdTimeFiles:Job output file missing: %s" % (jobOutputPath)
                                    self.logger.log(logging.ERROR,getLogMessage(message))

                                 try:
                                    fpTimeResults = open(jobFiles[jobId],'a')
                                    try:
                                       if exitStatus != 0:
                                          fpTimeResults.write('Command exited with non-zero status %s\n' % (exitStatus))
                                       fpTimeResults.write('real %s\n' % (row[columnKickstart]))
                                       fpTimeResults.write('user %s\n' % (row[columnCPUTime]))
                                       fpTimeResults.write('wait %s\n' % (row[columnCondorQTime]))
                                       fpTimeResults.write('elapsed %s\n' % (row[columnRuntime]))
                                       fpTimeResults.write('site %s\n' % (row[columnSite]))
                                       if condorId:
                                          fpTimeResults.write('jobId %s\n' % (condorId))
                                       if event:
                                          fpTimeResults.write('event %s\n' % (event))
                                    except (IOError,OSError):
                                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (jobFiles[jobId])))
                                    finally:
                                       fpTimeResults.close()
                                 except (IOError,OSError):
                                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (jobFiles[jobId])))
                        except:
                           self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                     except StopIteration:
                        self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (jobStatsPath)))
                  except csv.Error:
                     self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (jobStatsPath)))
                  finally:
                     fpJobStats.close()
               except (IOError,OSError):
                  message = "getPegasusStdTimeFiles:Failed to open Job Stats: %s" % (jobStatsPath)
                  self.logger.log(logging.ERROR,getLogMessage(message))
            else:
               message = "getPegasusStdTimeFiles:Job Stats missing: %s" % (jobStatsPath)
               self.logger.log(logging.ERROR,getLogMessage(message))
         except (IOError,OSError):
            message = "getPegasusStdTimeFiles:Failed to open Job State Log: %s" % (jobStateLogPath)
            self.logger.log(logging.ERROR,getLogMessage(message))
      else:
         message = "getPegasusStdTimeFiles:Job State Log missing for post-processing: %s" % (jobStateLogPath)
         self.logger.log(logging.ERROR,getLogMessage(message))


