#
# @package      hubzero-submit-distributor
# @file         MessageCore.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import sys
import socket
import time
import traceback

from LogMessage import logID as log

class MessageCore:
   def __init__(self,
                bindHost="",
                bindPort=0,
                listenerHost="",
                listenerPort=0,
                repeatDelay=5):
      self.bindHost     = bindHost
      self.bindPort     = bindPort
      self.bindSocket   = None
      self.listenerHost = listenerHost
      self.listenerPort = listenerPort
      self.repeatDelay  = repeatDelay

      if bindPort > 0:
         bound = False
         nTry = 0
         self.bindSocket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
         while not bound and nTry < 10:
            try:
               nTry += 1
               self.bindSocket.bind((bindHost,bindPort))
               self.bindSocket.listen(512)
               bound = True
            except:
               time.sleep(repeatDelay)

         if not bound:
            log("Can't bind to port %d: %s %s" % (bindPort,sys.exc_info()[0],sys.exc_info()[1]))


   def isBound(self):
      return(self.bindSocket != None)


   def boundFileDescriptor(self):
      return(self.bindSocket.fileno())


   def acceptConnection(self):
      channel,details = self.bindSocket.accept()

      return(channel)


   def __receiveMessage(self,
                        channel,
                        messageLength,
                        bufferSize):
      bytesRemaining = messageLength
      message = ""

      try:
         while bytesRemaining:
            messageChunk = channel.recv(bufferSize)
            message += messageChunk
            bytesRemaining -= len(messageChunk)
            if messageChunk == "":
               if message != "":
                  log("socket connection broken in receiveMessage()")
               message = ""
               break
      except:
         log("Unexpected error in receiveMessage() " + message)
         message = ""
         log(traceback.format_exc())

      return(message)


   def receiveMessage(self,
                      channel,
                      messageLength,
                      bufferSize=128):
      return(self.__receiveMessage(channel,messageLength,bufferSize))


   def __sendMessage(self,
                     channel,
                     message,
                     fixedBufferSize):
      try:
         if fixedBufferSize > 0:
            fixedBufferMessage = "%-*s" % (fixedBufferSize,message)
            bytesRemaining = fixedBufferSize
            while bytesRemaining:
               transmittedLength = channel.send(fixedBufferMessage[fixedBufferSize-bytesRemaining:])
               bytesRemaining -= transmittedLength
               if transmittedLength == 0:
                  log("socket connection broken in sendMessage()")
                  transmittedLength = -1
                  break
         else:
            bytesRemaining = len(message)
            while bytesRemaining:
               transmittedLength = channel.send(message[len(message)-bytesRemaining:])
               bytesRemaining -= transmittedLength
               if transmittedLength == 0:
                  log("socket connection broken in sendMessage()")
                  transmittedLength = -1
                  break
      except:
         log("Unexpected error in sendMessage(%s)" % (message))
         log(traceback.format_exc())
         transmittedLength = -1

      return(transmittedLength)


   def sendMessage(self,
                   channel,
                   message,
                   fixedBufferSize=0):
      return(self.__sendMessage(channel,message,fixedBufferSize))


   def requestMessageResponse(self,
                              message,
                              messageBufferSize,
                              responseBufferSize,
                              recordTraceback=False):
      nTry = 0
      delay = 0
      posted = False
      while not posted:
         time.sleep(delay)
         try:
            nTry += 1
            messageResponseSocket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            messageResponseSocket.connect((self.listenerHost,self.listenerPort))
            if self.__sendMessage(messageResponseSocket,message,messageBufferSize) > 0:
               response = self.__receiveMessage(messageResponseSocket,responseBufferSize,128)
               if response != "":
                  posted = True
         except:
            if recordTraceback:
               log(traceback.format_exc())

         messageResponseSocket.close()
         delay = self.repeatDelay

      return(nTry,response)


   def requestMessageVariableResponse(self,
                                      message,
                                      messageBufferSize,
                                      responseBufferSize,
                                      recordTraceback=False):
      response = ""
      nTry = 0
      delay = 0
      posted = False
      while not posted:
         time.sleep(delay)
         try:
            nTry += 1
            messageResponseSocket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            messageResponseSocket.connect((self.listenerHost,self.listenerPort))
            if self.__sendMessage(messageResponseSocket,message,messageBufferSize) > 0:
               responseHeader = self.__receiveMessage(messageResponseSocket,responseBufferSize,responseBufferSize)
               if responseHeader != "":
                  responseLength = responseHeader.strip()
                  if int(responseLength) > 0:
                     response = self.__receiveMessage(messageResponseSocket,int(responseLength),int(responseLength))
                     if response != "":
                        posted = True
                  else:
                     posted = True
         except:
            if recordTraceback:
               log(traceback.format_exc())

         messageResponseSocket.close()
         delay = self.repeatDelay

      return(nTry,response)


   def requestMessageTimestampResponse(self,
                                       message,
                                       messageBufferSize,
                                       responseBufferSize,
                                       recordTraceback=False):
      response = ""
      nTry = 0
      delay = 0
      posted = False
      while not posted:
         time.sleep(delay)
         try:
            nTry += 1
            messageResponseSocket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            messageResponseSocket.connect((self.listenerHost,self.listenerPort))
            if self.__sendMessage(messageResponseSocket,message,messageBufferSize) > 0:
               responseHeader = self.__receiveMessage(messageResponseSocket,responseBufferSize,responseBufferSize)
               if responseHeader != "":
                  responseLength,responseTimestamp = responseHeader.strip().split()
                  if int(responseLength) > 0:
                     response = self.__receiveMessage(messageResponseSocket,int(responseLength),int(responseLength))
                     if response != "":
                        posted = True
                  else:
                     posted = True
         except:
            if recordTraceback:
               log(traceback.format_exc())

         messageResponseSocket.close()
         delay = self.repeatDelay

      return(nTry,response,responseTimestamp)

