#!/usr/bin/env python
#
# @package      hubzero-submit-distributor
# @file         monitorTunnel.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2004-2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  monitorTunnel.py
#
#  script which monitors ssh tunnels for use with distributor and related processes.
#
import sys
import os
import select
import signal
import socket
import time
from errno import EINTR

from hubzero.submit.LogMessage    import getLogMessageFileNo, openLog, logPID as log
from hubzero.submit.TunnelsInfo   import TunnelsInfo
from hubzero.submit.TunnelMonitor import TunnelMonitor

TUNNELMONITORHOST   = ""
TUNNELMONITORPORT   = 5729
MONITORROOT         = os.path.join(os.sep,'opt','submit')
MONITORLOGLOCATION  = os.path.join(os.sep,'var','log','submit','monitors')
MONITORLOGFILENAME  = "monitorTunnel.log"
LOGPATH             = os.path.join(MONITORLOGLOCATION,MONITORLOGFILENAME)
TUNNELSINFOLOCATION = MONITORROOT
TUNNELSINFOFILENAME = "tunnels.dat"
SSHIDENTITY         = os.path.join(MONITORROOT,".ssh","tunnel_rsa")


class MonitorTunnel:
   def __init__(self,
                installedDirectory,
                tunnelMonitorHost,
                tunnelMonitorPort,
                tunnelsInfoLocation,
                tunnelsInfoFilename,
                sshIdentity):
      self.binDirectory        = os.path.join(installedDirectory,'bin')
      self.tunnelMonitorHost   = tunnelMonitorHost
      self.tunnelMonitorPort   = tunnelMonitorPort
      self.tunnelsInfoLocation = tunnelsInfoLocation
      self.tunnelsInfoFilename = tunnelsInfoFilename
      self.sshIdentity         = sshIdentity

      self.tunnelMonitor = None
      self.activeTunnels = {}
      self.tunnelsInfo   = None

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def terminate(self):
      for activeTunnel in self.activeTunnels:
         tunnelPid,useCount = self.activeTunnels[activeTunnel]
         log("Send TERM to tunnel %s process" % (activeTunnel))
         os.kill(int(tunnelPid),signal.SIGTERM)

      log("*********************************")
      log("* tunnel server monitor stopped *")
      log("*********************************")


   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      self.terminate()
      sys.exit(1)


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGINT!")
      del self.tunnelsInfo

      self.tunnelsInfo = TunnelsInfo(self.tunnelsInfoLocation,self.tunnelsInfoFilename)
      log("Tunnel Info Reloaded!")


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGHUP!")
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGQUIT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGABRT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGTERM!")
      self.sigGEN_handler(signalNumber,frame)


   def monitor(self):
      os.environ['PATH'] = self.binDirectory + ':' + os.environ['PATH']

      self.tunnelMonitor = TunnelMonitor(self.tunnelMonitorHost,self.tunnelMonitorPort)
      if not self.tunnelMonitor.isBound():
         sys.exit(1)

      self.tunnelsInfo = TunnelsInfo(self.tunnelsInfoLocation,self.tunnelsInfoFilename)

      log("*********************************")
      log("* tunnel server monitor started *")
      log("*********************************")

      inputDescriptors  = []
      clientConnections = []

      monitorSocketFd = self.tunnelMonitor.boundFileDescriptor()
      inputDescriptors.append(monitorSocketFd)

      while 1:
         try:
            readyInputFds = select.select(inputDescriptors,[],[])[0]
         except select.error,err:
            if err[0] == EINTR:
               readyInputFds = []
            else:
               for inputDescriptor in inputDescriptors:
                  if isinstance(inputDescriptor,socket.socket):
                     try:
                        os.fstat(inputDescriptor.fileno())
                     except:
                        log(inputDescriptor)
                  else:
                     try:
                        os.fstat(inputDescriptor)
                     except:
                        log(inputDescriptor)
               self.terminate()
               raise

         if monitorSocketFd in readyInputFds:
            channel = self.tunnelMonitor.acceptConnection()
            clientConnections.append(channel)
            inputDescriptors.append(channel)

         for channel in readyInputFds:
            if channel in clientConnections:
               channelClosed = self.tunnelMonitor.processRequest(channel,self.tunnelsInfo,self.sshIdentity,self.activeTunnels)
               if channelClosed:
                  clientConnections.remove(channel)
                  inputDescriptors.remove(channel)


def daemonize():
   logMessageFileNo = getLogMessageFileNo()
   if logMessageFileNo != sys.stdout.fileno():
      os.close(sys.stdin.fileno())
      os.close(sys.stdout.fileno())
      os.close(sys.stderr.fileno())
      os.dup2(logMessageFileNo,1)
      os.dup2(logMessageFileNo,2)
      devnull = open("/dev/null","rw")
      os.dup2(sys.stdin.fileno(),devnull.fileno())

   if os.fork() != 0:
      os.wait()
      sys.exit(0)
   else:
      os.setsid()
      pid = os.fork()
      if pid != 0:
         sys.exit(0)

   time.sleep(2)


if __name__ == '__main__':

   openLog(LOGPATH)
   daemonize()

   __monitorTunnel__ = MonitorTunnel(MONITORROOT,TUNNELMONITORHOST,TUNNELMONITORPORT, \
                                                 TUNNELSINFOLOCATION,TUNNELSINFOFILENAME, \
                                                 SSHIDENTITY)
   __monitorTunnel__.monitor()


