# @package      hubzero-submit-distributor
# @file         JobDistributor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

__version__ = '2.5.1'

import sys
import os
import re
import signal
import select
import time
import subprocess
import shlex
import random
import traceback
import math
import copy
import shutil
from errno import EPIPE

from hubzero.submit.LogMessage               import openLog, logJobId as log, logSetJobId
from hubzero.submit.SitesInfo                import SitesInfo
from hubzero.submit.ToolsInfo                import ToolsInfo
from hubzero.submit.AppsAccessInfo           import AppsAccessInfo
from hubzero.submit.ManagersInfo             import ManagersInfo
from hubzero.submit.TunnelsInfo              import TunnelsInfo
from hubzero.submit.EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from hubzero.submit.CommandParser            import CommandParser
from hubzero.submit.VenueMechanismLocal      import VenueMechanismLocal
from hubzero.submit.VenueMechanismSsh        import VenueMechanismSsh
from hubzero.submit.VenueMechanismGsiSsh     import VenueMechanismGsiSsh
from hubzero.submit.RemoteJobMonitor         import RemoteJobMonitor
from hubzero.submit.RemoteProbeMonitor       import RemoteProbeMonitor
from hubzero.submit.RemoteTunnelMonitor      import RemoteTunnelMonitor
from hubzero.submit.TimeConversion           import hhmmssTomin
from hubzero.submit.TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand
from hubzero.submit.JobStatistic             import JobStatistic
from hubzero.submit.ParameterTemplate        import ParameterTemplate

HUBLOGFILE               = "distributor.log"
HUBGRIDJOBIDLOGFILE      = "gridjobid.log"
HUBGRIDJOBHISTORYLOGFILE = "gridjobhistory.log"

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
POSTPROCESSJOBCOMMAND  = 'postprocessjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

LOCALJOBID = ".__local_jobid"

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"

JOBGRIDRESOURCE = ".__grid_resource"
JOBGRIDHISTORY  = ".__grid_history"
JOBGRIDJOBID    = ".__grid_jobid"

class JobDistributor:
   def __init__(self,
                configFilePath,
                logDirectory,
                jobMonitorHost,
                jobMonitorPort,
                probeMonitorHost,
                probeMonitorPort,
                tunnelMonitorHost,
                tunnelMonitorPort,
                cloudMonitorHost,
                cloudMonitorPort):

      self.configData     = {}
      self.configFilePath = configFilePath

      self.version                  = __version__
      self.operationMode            = 0
      self.session                  = 0
      self.distributorPid           = os.getpid()
      self.jobId                    = 0
      self.jobs                     = {}
      self.jobStatistics            = {}
      self.fpUserLogPath            = None
      self.abortAttempted           = False
      self.disableStateCheck        = False
      self.disableProbeCheck        = True
      self.quotaLimit               = -1
      self.childPid                 = 0
      self.bufferSize               = 4096
      self.hubLogPath               = os.path.join(logDirectory,HUBLOGFILE)
      self.hubGridJobIdLogPath      = os.path.join(logDirectory,HUBGRIDJOBIDLOGFILE)
      self.hubGridJobHistoryLogPath = os.path.join(logDirectory,HUBGRIDJOBHISTORYLOGFILE)
      self.dataDirectory            = ''
      self.binDirectory             = ''
      self.jobMonitorHost           = jobMonitorHost
      self.jobMonitorPort           = jobMonitorPort
      self.probeMonitorHost         = probeMonitorHost
      self.probeMonitorPort         = probeMonitorPort
      self.tunnelMonitorHost        = tunnelMonitorHost
      self.tunnelMonitorPort        = tunnelMonitorPort
      self.cloudMonitorHost         = cloudMonitorHost
      self.cloudMonitorPort         = cloudMonitorPort
      self.runjobExists             = False

      self.batchCommands = {}
      self.batchCommands['receiveInput']    = RECEIVEINPUTCOMMAND
      self.batchCommands['submitBatchJob']  = SUBMITBATCHJOBCOMMAND
      self.batchCommands['postProcessJob']  = POSTPROCESSJOBCOMMAND
      self.batchCommands['transmitResults'] = TRANSMITRESULTSCOMMAND
      self.batchCommands['cleanupJob']      = CLEANUPJOBCOMMAND
      self.batchCommands['killBatchJob']    = KILLBATCHJOBCOMMAND

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.waitForJobsInfo = {}

      self.remoteMonitors = {}
      self.remoteMonitors['job']    = None
      self.remoteMonitors['probe']  = None
      self.remoteMonitors['tunnel'] = None
      self.remoteMonitors['cloud']  = None

      self.sitesInfo                = None
      self.cloudsInfo               = None
      self.tunnelsInfo              = None
      self.toolsInfo                = None
      self.appsAccessInfo           = None
      self.managersInfo             = None
      self.environmentWhitelistInfo = None

      self.successfulInstance    = None
      self.maximumSelectedSites  = 1
      self.userDestinations      = []
      self.inputFiles            = []
      self.outputFiles           = []
      self.isMultiCoreRequest    = False
      self.nCpus                 = 1
      self.ppn                   = ""
      self.wallTime              = 60
      self.environment           = ""
      self.managerSpecified      = False
      self.managerInfo           = {}
      self.x509SubmitProxy       = ""
      self.stdinput              = ""
      self.isParametric          = False
      self.parameterNames        = []
      self.parameterCombinations = None

      self.commandParser           = None
      self.enteredCommandArguments = []

      self.submitVenues        = None
      self.runName             = ""
      self.localJobId          = None
      self.hubUserId           = None
      self.hubUserName         = None
      self.isClientTTY         = True
      self.useSetup            = None
      self.pegasusVersion      = None
      self.pegasusHome         = None
      self.clientWorkDirectory = None
      self.clientTransferPath  = None

      openLog(self.hubLogPath)

      self.abortGlobal = {}
      self.abortGlobal['abortAttempted'] = self.abortAttempted
      self.abortGlobal['abortSignal']    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stdout: %s" % (message))


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stderr: %s" % (message))


   def setEnvironment(self):
      if self.configData['pbsRoot'] != "" and self.configData['pbsRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['pbsRoot'],'bin') + os.pathsep + os.environ['PATH']

      if self.configData['condorRoot'] != "" and self.configData['condorRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['condorRoot'],'bin') + os.pathsep + \
                              os.path.join(self.configData['condorRoot'],'sbin') + os.pathsep + os.environ['PATH']
         if self.configData['condorConfig'] != "":
            os.environ['CONDOR_CONFIG'] = self.configData['condorConfig']

      if self.configData['runjobRoot'] != "":
         if self.configData['runjobRoot'] == "REMOTE":
            self.runjobExists = True
         else:
            runjobScriptPath = os.path.join(self.configData['runjobRoot'],'bin','runjob.sh')
            if os.path.exists(runjobScriptPath):
               os.environ['PATH'] = os.path.join(self.configData['runjobRoot'],'bin') + os.pathsep + os.environ['PATH']
               self.runjobExists = True

      os.environ['PATH'] = self.binDirectory + os.pathsep + os.environ['PATH']


   def getEnvironment(self):
      self.hubUserName         = os.getenv("USER")
      self.hubUserId           = os.getuid()
      self.session             = int(os.getenv("SESSION",'0'))
      self.submitVenues        = os.getenv("SUBMITVENUES")
      self.isClientTTY         = bool(int(os.getenv("SUBMIT_ISCLIENTTTY",'0')))
      self.pegasusVersion      = os.getenv("PEGASUS_VERSION")
      self.pegasusHome         = os.getenv("PEGASUS_HOME")
      self.useSetup            = os.getenv("USE_SETUP_SCRIPT")
      self.clientWorkDirectory = os.getenv("CLIENT_WORK_DIRECTORY")
      self.clientTransferPath  = os.getenv("CLIENT_TRANSFER_PATH")


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inDistributorSection = False

      fpConfig = open(self.configFilePath,'r')
      if fpConfig:
         eof = False
         while not eof:
            record = fpConfig.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   sectionPattern.match(record):
                  sectionName = sectionPattern.match(record).group(2)
                  inDistributorSection = (sectionName == 'distributor')
                  if inDistributorSection:
                     self.configData = {'probeMonitoringInstalled':False,
                                        'maximumSelectedSites':1,
                                        'allowedVenueMechanisms':['local','ssh'],
                                        'dataDirectory':os.path.join(os.sep,'opt','submit'),
                                        'binDirectory':os.path.join(os.sep,'opt','submit','bin'),
                                        'runjobRoot':'',
                                        'condorRoot':'',
                                        'condorConfig':'',
                                        'pbsRoot':''
                                       }
               elif inDistributorSection:
                  if keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.configData:
                        if   isinstance(self.configData[key],list):
                           self.configData[key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.configData[key],bool):
                           self.configData[key] = bool(value.lower() == 'true')
                        elif isinstance(self.configData[key],float):
                           self.configData[key] = float(value)
                        elif isinstance(self.configData[key],int):
                           self.configData[key] = int(value)
                        else:
                           self.configData[key] = value
                     else:
                        log("Undefined key = value pair %s = %s" % (key,value))
            else:
               eof = True
         fpConfig.close()

      configured = True
      if os.path.isdir(self.configData['dataDirectory']):
         self.dataDirectory = self.configData['dataDirectory']
      else:
         log("Specified dataDirectory does not exist: %s" % (self.configData['dataDirectory']))
         configured = False

      if os.path.isdir(self.configData['binDirectory']):
         self.binDirectory = self.configData['binDirectory']
      else:
         log("Specified binDirectory does not exist: %s" % (self.configData['binDirectory']))
         configured = False

      if configured:
         self.setEnvironment()
         self.getEnvironment()
         self.maximumSelectedSites = self.configData['maximumSelectedSites']

      return(configured)


   def doRun(self):
      return(self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID)


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      exitCode = 0
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR:
         self.__writeToStdout("Submit distributor version: %s\n" % (self.version))

      reportVersion = self.commandParser.getOption('version') or \
                      self.commandParser.getOption('versionServer') or \
                      self.commandParser.getOption('versionDistributor')

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         self.disableStateCheck = self.commandParser.getOption('disableStateCheck')
         if self.configData['probeMonitoringInstalled']:
            self.disableProbeCheck = self.commandParser.getOption('disableProbeCheck')

         exitCode = self.commandParser.setSweepParameters()
         if exitCode == 0:
            exitCode = self.commandParser.setCSVDataParameters()
            if exitCode == 0:
               if self.commandParser.getParameterCombinationCount() > 0:
                  self.isParametric = True

               if not reportVersion:
                  enteredCommand = self.commandParser.getEnteredCommand()
                  if enteredCommand == "":
                     log("Command must be supplied")
                     self.__writeToStderr("Command must be supplied\n")
                     if not exitCode:
                        exitCode = 1
                  else:
                     if self.commandParser.getOption('runName'):
                        self.runName = self.commandParser.getOption('runName')

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      def getJobId(dataDirectory):
         nextLocalJobId = 0

         exitCode = 0
         localJobIdFileName = os.path.join(dataDirectory,"localJobId.dat")
         if not os.path.isfile(localJobIdFileName):
            fpLocalJobId = open(localJobIdFileName,'w')
            if fpLocalJobId:
               fpLocalJobId.write('0')
               fpLocalJobId.close()
            else:
               log("could not open %s for creating" % (localJobIdFileName))
               self.__writeToStderr("could not open %s for creating\n" % (localJobIdFileName))
               exitCode = 1

         if not exitCode:
            fpLocalJobId = open(localJobIdFileName,'r')
            if fpLocalJobId:
               previousLocalJobId = int(fpLocalJobId.readline().strip())
               fpLocalJobId.close()
               nextLocalJobId = previousLocalJobId+1
               fpLocalJobId = open(localJobIdFileName,'w')
               if fpLocalJobId:
                  fpLocalJobId.write('%d' % (nextLocalJobId))
                  fpLocalJobId.close()
               else:
                  log("could not open %s for writing" % (localJobIdFileName))
                  self.__writeToStderr("could not open %s for writing\n" % (localJobIdFileName))
            else:
               log("could not open %s for reading" % (localJobIdFileName))
               self.__writeToStderr("could not open %s for reading\n" % (localJobIdFileName))

         return(nextLocalJobId)

      exitCode = 0

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.commandParser.getOption('jobId'):
            self.jobId = self.commandParser.getOption('jobId')
         if self.jobId == 0:
            self.jobId = getJobId(self.dataDirectory)

      self.localJobId = "%08d" % (self.jobId)
      logSetJobId(self.jobId)

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.jobId == 0:
            exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setInfo(self):
      self.sitesInfo = SitesInfo(self.dataDirectory,"sites.dat",self.configData['allowedVenueMechanisms'])
      if not self.sitesInfo.siteExists('grid'):
         self.runjobExists = False

      self.cloudsInfo = None

      self.tunnelsInfo = TunnelsInfo(self.dataDirectory,"tunnels.dat")

      self.toolsInfo = ToolsInfo(self.dataDirectory,"tools.dat")
      self.toolsInfo.applyUserRestriction(self.hubUserName)
      self.toolsInfo.applyGroupRestriction()

      self.appsAccessInfo = AppsAccessInfo(self.dataDirectory,"appaccess.dat")

      self.managersInfo = ManagersInfo(self.dataDirectory,"managers.dat")

      self.environmentWhitelistInfo = EnvironmentWhitelistInfo(self.dataDirectory,"environmentwhitelist.dat")


   def showUsage(self):
      if self.operationMode & self.commandParser.OPERATIONMODEHELPUSAGE:
         self.commandParser.showUsage()

      if self.operationMode & self.commandParser.OPERATIONMODEHELPVENUES:
         venues = []
         for site in self.sitesInfo.getEnabledSites():
            venues.append(site)

         if len(venues) > 0:
            venues.sort()
            self.__writeToStdout("\nCurrently available VENUES are:\n")
            for venue in venues:
               self.__writeToStdout("   %s\n" % (venue))
         del venues

      if self.operationMode & self.commandParser.OPERATIONMODEHELPTOOLS:
         executableToolNames = self.toolsInfo.getToolNames()
         if len(executableToolNames) > 0:
            executableToolNames.sort()
            self.__writeToStdout("\nCurrently available TOOLs are:\n")
            for executableToolName in executableToolNames:
               self.__writeToStdout("   %s\n" % (executableToolName))
         del executableToolNames

      if self.operationMode & self.commandParser.OPERATIONMODEHELPMANAGERS:
         validManagers = self.managersInfo.getManagerNames()
         if len(validManagers) > 0:
            validManagers.sort()
            self.__writeToStdout("\nCurrently available MANAGERs are:\n")
            for validManager in validManagers:
               self.__writeToStdout("   %s\n" % (validManager))
         del validManagers


   def reportExitCondition(self):
      if self.doRun():
         if len(self.jobs) == 0:
            jobIndices = self.jobStatistics.keys()
            maximumJobIndex = max(jobIndices)

            try:
               mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
               remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
               remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
               exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
               cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
               elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
               waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
               nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']
               event          = self.jobStatistics[maximumJobIndex]['event']

               message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                         (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
               if event:
                  message += " event=%s" % (event)
               log(message)
               message += "\n"
               if self.fpUserLogPath:
                  self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                  self.fpUserLogPath.flush()
               os.write(3,message)
            except:
               pass
         else:
            instanceIndex = 0
            for instance in self.jobs:
               if (not self.successfulInstance) or \
                  (self.successfulInstance and (instance != self.successfulInstance)):
                  jobIndices = self.jobs[instance].jobStatistics.keys()
                  jobIndices.sort()
                  maximumJobIndex = max(jobIndices)

                  try:
                     for jobIndex in jobIndices:
                        if jobIndex != 0 or maximumJobIndex == 0:
                           mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                           remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                           remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                           exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                           cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                            self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                           elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                           waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                           nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                           event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                           instanceIndex += 1
                           message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                 (instanceIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                           if event:
                              message += " event=%s" % (event)
                           log(message)
                           message += "\n"
                           if self.fpUserLogPath:
                              self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                              self.fpUserLogPath.flush()
                           os.write(3,message)
                  except:
                     pass

            if self.successfulInstance:
               instance = self.successfulInstance
               jobIndices = self.jobs[instance].jobStatistics.keys()
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                        if self.jobs[instance].remoteBatchSystem == 'PEGASUS':
                           if mechanism == "" or mechanism == 'Unknown':
                              mechanism = 'PEGASUS'
                        else:
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                        cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                        event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                        instanceIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                                     (instanceIndex,mechanism,remoteId,remoteLocation, \
                                                                      exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        if event:
                           message += " event=%s" % (event)
                        log(message)
                        message += "\n"
                        if self.fpUserLogPath:
                           self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                           self.fpUserLogPath.flush()
                        os.write(3,message)
               except:
                  pass

      if self.fpUserLogPath:
         self.fpUserLogPath.close()
         self.fpUserLogPath = None


# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortGlobal['abortAttempted'] = self.abortAttempted
         self.abortGlobal['abortSignal']    = signalNumber

         for instance in self.waitForJobsInfo:
            if 'isBatchJob' in self.waitForJobsInfo[instance]:
               if not self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'R':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGINT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGHUP!")
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGQUIT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGABRT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGTERM!")
      self.sigGEN_handler(signalNumber,frame)


   def getUserQuota(self):
      quotaCommand = "quota -w"
      log("quotaCommand = " + quotaCommand)
      exitStatus,stdOutput,stdError = self.executeCommand(quotaCommand)
      if exitStatus == 0:
         try:
            quotaResults = stdOutput.strip().split('\n')[-1].split()
            if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
               currentUsage = int(quotaResults[1].strip("*"))
               self.quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
         except:
            log(stdOutput)
            log(stdError)
            log(traceback.format_exc())


   def getStdInputFile(self,
                       inputFiles):
      stdInputFile = '/dev/null'
      toCheck = []
      stdinFd = sys.stdin.fileno()
      toCheck.append(stdinFd)
      try:
         ready = select.select(toCheck,[],[],0.) # wait for input
      except select.error,err:
         ready = {}
         ready[0] = []
      if stdinFd in ready[0]:
         content = sys.stdin.read()
         if content != "":
            stdInputFile = ".__%s.stdin" % (self.localJobId)
            stdinFile = open(stdInputFile,'w')
            stdinFile.write(content)
            stdinFile.close()
            inputFiles.append(stdInputFile)
         del content

      return(stdInputFile)


   def executeCommand(self,
                      command,
                      streamOutput=False):
      if isinstance(command,list):
         child = subprocess.Popen(command,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      else:
         commandArgs = shlex.split(command)
         child = subprocess.Popen(commandArgs,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            log("%s failed w/ signal %d" % (command,os.WTERMSIG(err)))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            log("%s failed w/ exit code %d" % (command,err))
         if not streamOutput:
            log("%s" % ("".join(errData)))

      return(err,"".join(outData),"".join(errData))


   def buildJobDescription(self):
      exitCode = 0

      log("session = %d" % (self.session))
      if self.commandParser.getOption('quotaCheck'):
         self.getUserQuota()

      userDestinations = []
      inputFiles       = []
      nCpus            = 1
      ppn              = ""
      wallTime         = 60
      environment      = ""

      if self.submitVenues == None:
         if self.commandParser.getOption('destinations'):
            userDestinations = self.commandParser.getOption('destinations')
      else:
         userDestinations = self.submitVenues.split(',')
      if len(userDestinations) > 0:
         reasonsDenied = self.sitesInfo.purgeDisabledSites(userDestinations)
         if len(userDestinations) == 0:
            errorMessage = "Access to all sites listed as possible execution hosts has been denied.\n"
            for userDestination in reasonsDenied:
               errorMessage += "   %20s %s\n" % (userDestination,reasonsDenied[userDestination])
            errorMessage += "Please select another site or attempt execution at a later time."
            log(errorMessage)
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 7
         del reasonsDenied

      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            inputFiles.append(inputFile)
      self.stdinput = self.getStdInputFile(inputFiles)

      if self.commandParser.getOption('outputFiles'):
         for outputFile in self.commandParser.getOption('outputFiles'):
            self.outputFiles.append(outputFile)

      if self.commandParser.getOption('manager'):
         manager = self.commandParser.getOption('manager')[0]
         if self.managersInfo.managerExists(manager):
            self.managerSpecified = True
            self.managerInfo      = self.managersInfo.getManagerInfo(manager)
         else:
            validManagers = self.managersInfo.getManagerNames()
            log("Invalid manager %s specified. Valid managers are %s" % (manager,validManagers))
            self.__writeToStderr("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
            del validManagers
            if not exitCode:
               exitCode = 1
      else:
         self.managerInfo = self.managersInfo.getDefaultManagerInfo('serial')

      if self.commandParser.getOption('nCpus'):
         self.isMultiCoreRequest = True
         nCpus = int(self.commandParser.getOption('nCpus'))

      if self.commandParser.getOption('ppn'):
         if self.isMultiCoreRequest:
            ppn = str(self.commandParser.getOption('ppn'))

      if self.isParametric:
         self.maximumSelectedSites = 1
      else:
         if self.commandParser.getOption('nRedundant'):
            self.maximumSelectedSites = self.commandParser.getOption('nRedundant')

      if self.commandParser.getOption('wallTime'):
         if ":" in self.commandParser.getOption('wallTime'):
            wallTime = hhmmssTomin(self.commandParser.getOption('wallTime'))
            if wallTime < 0.:
               log("wallTime must be > 0")
               self.__writeToStderr("wallTime must be > 0\n")
               if not exitCode:
                  exitCode = 1
         else:
            try:
               wallTime = float(self.commandParser.getOption('wallTime'))
            except ValueError:
               log("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes")
               self.__writeToStderr("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes\n")
               if not exitCode:
                  exitCode = 1

      if self.commandParser.getOption('userLogPath'):
         userLogPath = self.commandParser.getOption('userLogPath')
         try:
            self.fpUserLogPath = open(userLogPath,'a')
         except:
            log("User log %s could not be opened." % (userLogPath))
            self.__writeToStderr("User log %s could not be opened.\n" % (userLogPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('userJobidPath'):
         userJobidPath = self.commandParser.getOption('userJobidPath')
         try:
            fpUserJobId = open(userJobidPath,'w')
            if fpUserJobId:
               fpUserJobId.write("%d\n" % (self.jobId))
               fpUserJobId.close()
         except:
            log("User jobId file %s could not be created." % (userJobidPath))
            self.__writeToStderr("User jobId file %s could not be created.\n" % (userJobidPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('x509SubmitProxy'):
         self.x509SubmitProxy = self.commandParser.getOption('x509SubmitProxy')

      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               log("Invalid environment variable %s specified." % (environmentVariableValue))
               self.__writeToStderr("Invalid environment variable %s specified.\n" % (environmentVariableValue))
               if not exitCode:
                  exitCode = 1
            else:
               if self.environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
                  environment += environmentVariable + "=" + value + " "
               else:
                  log("Environment variable %s could not be set." % (environmentVariable))
                  self.__writeToStderr("Environment variable %s could not be set.\n" % (environmentVariable))
         environment = environment.strip()

      if self.isMultiCoreRequest:
         if not self.managerSpecified:
            self.managerInfo = self.managersInfo.getDefaultManagerInfo('mpi')

      if not self.disableProbeCheck:
         ignoreProbeSites = self.sitesInfo.getIgnoreProbeSites()
         self.remoteMonitors['probe'] = RemoteProbeMonitor(self.probeMonitorHost,self.probeMonitorPort,
                                                           ignoreProbeSites=ignoreProbeSites)

      if len(userDestinations) > 0:
         goToGrid = self.sitesInfo.purgeOfflineSites(userDestinations,self.remoteMonitors['probe'])
         if len(userDestinations) == 0 and not goToGrid:
            errorMessage = "All sites listed as possible execution hosts are out of service.\n" + \
                           "Please select another site or attempt execution at a later time."
            log(errorMessage)
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 8

      nUserDestinations = len(userDestinations)
      self.sitesInfo.purgeResourceLimitedSites(userDestinations,nCpus,wallTime)
      if nUserDestinations > 0 and len(userDestinations) == 0:
         errorMessage = "All sites listed as possible execution hosts are unable to meet\n" + \
                        "the specified resource requirements.  Please select another site."
         log(errorMessage)
         self.__writeToStderr(errorMessage + "\n")
         exitCode = 10

      self.userDestinations = copy.copy(userDestinations)
      self.inputFiles       = copy.copy(inputFiles)
      self.nCpus            = copy.copy(nCpus)
      self.ppn              = copy.copy(ppn)
      self.wallTime         = copy.copy(wallTime)
      self.environment      = copy.copy(environment)

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def findInputFileOrDirectory(self,
                                inputFileOrDirectory):
      fileOrDirectory = ""
      stripedInputFileOrDirectory = inputFileOrDirectory.replace('@:','')
      if stripedInputFileOrDirectory.startswith(os.sep):
         relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
      else:
         relativeInputFileOrDirectory = stripedInputFileOrDirectory
      if self.clientWorkDirectory.startswith(os.sep):
         relativeClientWorkDirectory = self.clientWorkDirectory[1:]
      else:
         relativeClientWorkDirectory = self.clientWorkDirectory
      workDirectory = os.getcwd()

      if self.clientTransferPath:
         candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeClientWorkDirectory,relativeInputFileOrDirectory)
         if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
            fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory

      if not fileOrDirectory:
         if self.clientWorkDirectory == workDirectory:
            candidateFileOrDirectory = stripedInputFileOrDirectory
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientWorkDirectory,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
            else:
               candidateFileOrDirectory = os.path.join(workDirectory,relativeInputFileOrDirectory)
               if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
                  fileOrDirectory = candidateFileOrDirectory

      if stripedInputFileOrDirectory != inputFileOrDirectory:
         fileOrDirectory = '@:' + fileOrDirectory

      return(fileOrDirectory)


   class CommandError(Exception):
      pass

   class InstancesError(Exception):
      pass

   class InstanceError(Exception):
      pass


   def getInstanceDestinations(self):
      exitCode = 0
      userDestinations     = []
      instanceDestinations = []
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()

      try:
         executable = self.enteredCommandArguments[0]
         if "/" in executable:
            candidateFileOrDirectory = self.findInputFileOrDirectory(executable)
            if candidateFileOrDirectory:
               self.enteredCommandArguments[0] = candidateFileOrDirectory

         executable         = self.enteredCommandArguments[0]
         transferExecutable = True
         if "/" in executable:
            if   not os.path.isfile(executable):
               log("Missing executable: %s" % (executable))
               self.__writeToStderr("Missing executable: %s\n" % (executable))
               if not exitCode:
                  exitCode = 1
                  raise JobDistributor.CommandError
            elif not os.access(executable,os.X_OK):
               log("Permission denied on executable: %s" % (executable))
               self.__writeToStderr("Permission denied on executable: %s\n" % (executable))
               if not exitCode:
                  exitCode = 1
                  raise JobDistributor.CommandError
         else:
            transferExecutable = False

         if transferExecutable:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
            if not submissionAllowed:
               log("Access to %s denied for %s" % (executable,self.hubUserName))
               self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
               if not exitCode:
                  exitCode = 9
                  raise JobDistributor.CommandError
         else:
            applicationIsStaged = False
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  applicationIsStaged      = True
                  executableClassification = 'staged'
            if not applicationIsStaged:
               submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
               if not submissionAllowed:
                  log("Access to %s denied for %s" % (executable,self.hubUserName))
                  self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
                  if not exitCode:
                     exitCode = 9
                     raise JobDistributor.CommandError

         enabledSites = self.sitesInfo.getEnabledSites()
         self.toolsInfo.purgeDisabledSites(enabledSites)
         if not self.isParametric:
            pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
            self.toolsInfo.purgePegasusSites(pegasusSites)
            del pegasusSites

         userMappedDestinations = []
         userGridDestinations   = []
         for userDestination in self.userDestinations:
            if self.sitesInfo.siteExists(userDestination):
               userMappedDestinations.append(userDestination)
            else:
               userGridDestinations.append(userDestination)
               if not 'grid' in userMappedDestinations:
                  userMappedDestinations.append('grid')

         userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                executableClassification,userMappedDestinations) + \
                            self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                '*',userMappedDestinations)

         if 'grid' in userDestinations and len(userGridDestinations) > 0:
            userDestinations.remove('grid')
            for userGridDestination in userGridDestinations:
               userDestinations.append(userGridDestination)

         if len(self.userDestinations) > 0 and len(userDestinations) == 0:
            errorMessage = "All sites listed as possible execution hosts are unable to meet\n" + \
                           "the specified access requirements.  Please select another site."
            log(errorMessage)
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 10
            raise JobDistributor.CommandError

      except JobDistributor.CommandError:
         self.jobStatistics[0]['exitCode'] = exitCode
         raise

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination  = ""
      destinations = []

      try:
         if not transferExecutable:
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,self.remoteMonitors['probe'])
                  toolInfo = self.toolsInfo.selectTool(executable,expandedUserDestinations,self.remoteMonitors['probe'])
                  del expandedUserDestinations
                  if len(toolInfo['destinations']) == 0:
                     log("The combination of destination and executable is not properly specified.")
                     self.__writeToStderr("The combination of destination and executable is not properly specified.\n")
                     exitCode = 1
                     raise JobDistributor.InstancesError

                  if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
                     if self.managersInfo.managerExists(toolInfo['remoteManager']):
                        self.managerSpecified = True
                        self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable))
                        self.__writeToStderr("Invalid manager %s specified for tool %s.\n" % (toolInfo['remoteManager'],executable))
                        exitCode = 1
                        raise JobDistributor.InstancesError
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'
               else:
                  log("Access to %s is restricted by user or group" % (executable))
                  self.__writeToStderr("Access to %s is restricted by user or group\n" % (executable))
                  exitCode = 1
                  raise JobDistributor.InstancesError

         if   len(toolInfo['destinations']) == 1:
# if tool is staged and only one site is available
            destination = toolInfo['destinations'][0]
            instanceDestinations.append(destination)
         elif len(toolInfo['destinations']) == 0:
# if tool is not staged and only one user requested site is available.
            destinations = self.sitesInfo.selectSites(userDestinations)
            if len(destinations) == 1:
               destination = destinations[0]
               instanceDestinations.append(destination)

         if destination == "":
            if len(toolInfo['destinations']) > 1:
# if tool is staged pick site(s)
               if not self.isParametric:
                  destinations = self.sitesInfo.selectSites(toolInfo['destinations'])
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
               else:
                  pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                     toolInfo['destinations'])
                  destinations = self.sitesInfo.selectSites(pegasusSites)
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
                  else:
                     nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                              toolInfo['destinations'])
                     destinations = self.sitesInfo.selectSites(nonPegasusSites)
                     if len(destinations) > 0:
                        for instanceDestination in destinations:
                           instanceDestinations.append(instanceDestination)
                        destination = 'Sweep'

         if destination == "":
            if len(userDestinations) > 1:
# if user requested site is available
               if not self.isParametric:
                  nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','RUNJOB',
                                                                          userDestinations)
                  destinations = self.sitesInfo.selectSites(nonRunjobSites)
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
               else:
                  pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                     userDestinations)
                  destinations = self.sitesInfo.selectSites(pegasusSites)
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
                  else:
                     nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                              userDestinations)
                     destinations = self.sitesInfo.selectSites(nonPegasusSites)
                     if len(destinations) > 0:
                        for instanceDestination in destinations:
                           instanceDestinations.append(instanceDestination)
                        destination = 'Sweep'

         if not self.isParametric:
            if   self.sitesInfo.siteExists(destination):
# selected an existing site
               pass
            elif self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
# defer to tool or user specified runjob grid site selection
               if len(instanceDestinations) == 0:
                  instanceDestinations.append('2BeDetermined')
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteMonitors['probe']:
                  self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
               destinations = self.sitesInfo.selectUndeclaredSites(enabledSites,
                                                                   executableClassification,
                                                                   self.maximumSelectedSites)
               if len(destinations) > 0:
                  for instanceDestination in destinations:
                     instanceDestinations.append(instanceDestination)
               else:
                  log("Currently no sites are available")
                  self.__writeToStderr("Currently no sites are available\n")
                  exitCode = 1
                  raise JobDistributor.InstancesError
         else:
            if self.sitesInfo.siteExists(destination):
# selected an existing site
               pass
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteMonitors['probe']:
                  self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
               pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
               destinations = self.sitesInfo.selectUndeclaredSites(pegasusSites,
                                                                   executableClassification,1)
               if len(destinations) > 0:
                  destination = destinations[0]
                  instanceDestinations.append(destination)
               else:
                  nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                           enabledSites)
                  destinations = self.sitesInfo.selectUndeclaredSites(nonPegasusSites,
                                                                      executableClassification,
                                                                      self.maximumSelectedSites)
                  if len(destinations) > 0:
                     destination = 'Sweep'
                     for instanceDestination in destinations:
                        instanceDestinations.append(instanceDestination)
                  else:
                     log("Currently no sites are available")
                     self.__writeToStderr("Currently no sites are available\n")
                     exitCode = 1
                     raise JobDistributor.InstancesError

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode
         raise

      return(userDestinations,instanceDestinations,
             executable,transferExecutable,executableClassification,toolInfo)


   def buildRedundantJobInstances(self,
                                  userDestinations,
                                  instanceDestinations,
                                  executable,
                                  transferExecutable,
                                  executableClassification,
                                  toolInfo):
      enteredCommand = self.commandParser.getEnteredCommand()
      userInputFiles = []
      for inputFile in self.inputFiles:
         candidateFileOrDirectory = self.findInputFileOrDirectory(inputFile)
         if candidateFileOrDirectory:
            if not candidateFileOrDirectory in userInputFiles:
               userInputFiles.append(candidateFileOrDirectory)

      if len(self.enteredCommandArguments) > 1:
         for arg in self.enteredCommandArguments[1:]:
            arglets = re.split('( |=)',arg)
            for arglet in arglets:
               if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                  candidateFileOrDirectory = self.findInputFileOrDirectory(arglet)
                  if candidateFileOrDirectory:
                     if not candidateFileOrDirectory in userInputFiles:
                        userInputFiles.append(candidateFileOrDirectory)

      filesToRemove      = []
      emptyFilesToRemove = []
      runDirectoryChecked = False
      nInstanceIdDigits = max(2,int(math.log10(max(1,len(instanceDestinations)))+1))
      instance = 0
      for destination in instanceDestinations:
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)
         exitCode         = 0
         inputFiles       = copy.copy(userInputFiles)
         nCpus            = copy.copy(self.nCpus)
         ppn              = copy.copy(self.ppn)
         wallTime         = copy.copy(self.wallTime)
         environment      = copy.copy(self.environment)
         managerSpecified = copy.copy(self.managerSpecified)
         managerInfo      = copy.copy(self.managerInfo)

         siteInfo = self.sitesInfo.getDefaultSiteInfo()

         gridsite        = "Unknown"
         nNodes          = "1"
         arguments       = []
         cloudInstanceId = ""

         try:
            if   (self.sitesInfo.siteExists(destination)) and \
                  self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
               siteInfo = self.sitesInfo.getSiteInfo(destination)
               if self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                  gridsite = self.sitesInfo.getSitePegasusSite(destination)
               if not managerSpecified and siteInfo['remoteManager'] != "":
                  if self.managersInfo.managerExists(siteInfo['remoteManager']):
                     managerSpecified = True
                     managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                  else:
                     log("Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination))
                     self.__writeToStderr("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],
                                                                                           destination))
                     exitCode = 1
                     raise JobDistributor.InstanceError
               if ppn == "":
                  ppn = siteInfo['remotePpn']
               nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
               if int(nNodes) == 1:
                  if int(nCpus) < int(ppn):
                     ppn = str(nCpus)
            elif (self.sitesInfo.siteExists(destination)) and \
                  self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
               siteInfo = self.sitesInfo.getSiteInfo(destination)
               runjobVenues = self.sitesInfo.getSiteMembers(destination)
               gridsite     = " ".join(runjobVenues)
            else:
               if   self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
                  siteInfo = self.sitesInfo.getSiteInfo('grid')
                  if   len(toolInfo['destinations']) > 0:
                     gridsite = " ".join(toolInfo['destinations'])
                  elif len(userDestinations) > 0:
                     gridsite = " ".join(userDestinations)
                  else:
                     gridsite = destination
               elif self.runjobExists or transferExecutable:
                  log("Invalid destination selection")
                  self.__writeToStderr("Invalid destination selection\n")
                  exitCode = 1
                  raise JobDistributor.InstanceError

            if siteInfo['remoteBatchSystem'] != 'SCRIPT':
               runDirectory = os.path.join(os.getcwd(),self.runName)
               if os.path.exists(runDirectory):
                  if not runDirectoryChecked:
                     log("Run directory %s exists" % (runDirectory))
                     self.__writeToStderr("Run directory %s exists\n" % (runDirectory))
                     runDirectoryChecked = True
                  exitCode = 1
                  raise JobDistributor.InstanceError

            if transferExecutable:
               if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                  (not executableClassification in siteInfo['executableClassificationsAllowed']):
                  errorMessage = "The selected site does not meet the specified\n" + \
                                 "application access requirements.  Please select another site."
                  log(errorMessage)
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

            localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
            self.executeCommand("touch " + localJobIdFile)
            inputFiles.append(localJobIdFile)
            filesToRemove.append(localJobIdFile)

            if transferExecutable:
               inputFiles.append(executable)

            if siteInfo['stageFiles']:
               stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
               tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
               log("command = " + str(tarCommand))
               tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
               if tarExitStatus:
                  log(tarStdError)
                  log(tarStdOutput)
                  self.__writeToStderr(tarStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError

               filesToRemove.append(stageInTarFile)
               remoteArgs = []
               remoteArgs.append(self.enteredCommandArguments[0])
               if len(self.enteredCommandArguments) > 1:
                  tarFiles = tarStdOutput.strip().split('\n')
                  for arg in self.enteredCommandArguments[1:]:
                     rarg = ''
                     arglets = re.split('( |=)',arg)
                     for arglet in arglets:
                        if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                           argletIsFile = False
                           for tarFile in tarFiles:
                              if re.match(".*" + re.escape(tarFile) + "$",arglet):
                                 rarg += os.path.join('.',tarFile)
                                 argletIsFile = True
                                 break
                           if not argletIsFile:
                              rarg += arglet
                        else:
                           rarg += arglet
                     remoteArgs.append(rarg)

               remoteCommand = " ".join(remoteArgs)
               log("remoteCommand " + remoteCommand)

               arguments = remoteArgs[1:]
            else:
# if not shared work directory?
               stageInTarFile = ""
               remoteCommand = " ".join(self.enteredCommandArguments)
               log("remoteCommand " + remoteCommand)

               arguments = self.enteredCommandArguments[1:]

            useEnvironment = ""
            if siteInfo['passUseEnvironment']:
               reChoice = re.compile(".*_CHOICE$")
               environmentVars = filter(reChoice.search,os.environ.keys())
               if "PROMPT_CHOICE" in environmentVars:
                  environmentVars.remove("PROMPT_CHOICE")
               if "EDITOR_CHOICE" in environmentVars:
                  environmentVars.remove("EDITOR_CHOICE")
               if len(environmentVars) > 0:
                  useEnvironment = ". %s\n" % (self.useSetup)
                  if "ENVIRON_CONFIG_DIRS" in os.environ:
                     useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
                  for environmentVar in environmentVars:
                     useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

            managedEnvironment = managerInfo['environment']
            if len(managedEnvironment) > 0:
               for environmentVariableValue in managedEnvironment:
                  environmentVariable = ""
                  value               = ""
                  nParts = len(environmentVariableValue.split('='))
                  if   nParts == 1:
                     environmentVariable = environmentVariableValue.strip()
                     if environmentVariable in os.environ:
                        value = os.environ[environmentVariable]
                  elif nParts == 2:
                     environmentVariable,value = environmentVariableValue.split('=')
                     environmentVariable = environmentVariable.strip()
                     value               = value.strip()
                     if value == "":
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]

                  if environmentVariable != "" and value != "":
                     if environment.find(environmentVariable + '=') < 0:
                        environment += environmentVariable + "=" + value + " "
               environment = environment.strip()

            if len(toolInfo['environment']) > 0:
               for environmentVariableValue in toolInfo['environment']:
                  environmentVariable = ""
                  value               = ""
                  nParts = len(environmentVariableValue.split('='))
                  if nParts == 2:
                     environmentVariable,value = environmentVariableValue.split('=')
                     environmentVariable = environmentVariable.strip()
                     value               = value.strip()
                     if value == "":
                        value = " "
                  else:
                     log("Tool environment variable must be of the form Variable = Value.")
                     self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                  if environmentVariable != "" and value != "":
                     environment += environmentVariable + "=" + value + " "
               environment = environment.strip()

            if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
               if nCpus == 1:
                  self.isMultiCoreRequest = False
               else:
                  errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                  log(errorMessage)
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

            timeHistoryLogs = {}
            timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
            timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
            timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
            timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
            timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
            timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
            timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
            timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
            timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

            if   siteInfo['venueMechanism'] == 'local':
               self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                         self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                         self.batchCommands,self.isParametric,
                                                         self.runName,self.localJobId,instanceId,destination,
                                                         enteredCommand,gridsite,
                                                         stageInTarFile,
                                                         transferExecutable,executable,self.appsAccessInfo,
                                                         self.stdinput,arguments,useEnvironment,environment,
                                                         self.isMultiCoreRequest,siteInfo,managerInfo,
                                                         nCpus,nNodes,ppn,wallTime,
                                                         self.quotaLimit,
                                                         self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                         timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
            elif siteInfo['venueMechanism'] == 'ssh':
               self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                       self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                       self.batchCommands,self.isParametric,
                                                       self.runName,self.localJobId,instanceId,destination,
                                                       enteredCommand,gridsite,
                                                       self.tunnelsInfo,
                                                       cloudInstanceId,
                                                       stageInTarFile,
                                                       transferExecutable,executable,self.appsAccessInfo,
                                                       self.stdinput,arguments,useEnvironment,environment,
                                                       self.isMultiCoreRequest,siteInfo,managerInfo,
                                                       nCpus,nNodes,ppn,wallTime,
                                                       self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                       self.quotaLimit,
                                                       timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
            elif siteInfo['venueMechanism'] == 'gsissh':
               self.jobs[instance] = VenueMechanismGsiSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          enteredCommand,self.x509SubmitProxy,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,managerInfo,
                                                          nCpus,nNodes,ppn,wallTime,
                                                          self.quotaLimit,
                                                          timeHistoryLogs)

            exitCode,scriptFiles = self.jobs[instance].createScripts()
            if exitCode:
               raise JobDistributor.InstanceError

            if siteInfo['stageFiles']:
               if len(scriptFiles) > 0:
                  tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                  log("command = " + str(tarCommand))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

               niceCommand = "nice -n 19 gzip " + stageInTarFile
               log("command = " + niceCommand)
               gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
               if gzipExitStatus:
                  log(gzipStdError)
                  log(gzipStdOutput)
                  self.__writeToStderr(gzipStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               stageInTarFile += '.gz'
               filesToRemove.append(stageInTarFile)

            self.jobs[instance].stageFilesToInstanceDirectory()

            self.waitForJobsInfo[instance] = {}
            waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
            self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
            self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
            self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
            self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
            self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory

         except JobDistributor.InstanceError:
            if instance in self.jobs:
               self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
               self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
               self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
            else:
               self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
               self.jobStatistics[0]['venue']                  = siteInfo['venue']
               self.jobStatistics[0]['exitCode']               = exitCode

         stdFile = "%s.stderr" % (self.runName)
         emptyFilesToRemove.append(stdFile)
         for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory':
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueue,self.runName,instanceId,outFile)
               emptyFilesToRemove.append(batchQueueOutputFile)

      nonBatchJobRan = False
      markedForDeletion = []
      for instance in self.waitForJobsInfo:
         if not self.waitForJobsInfo[instance]['isBatchJob']:
            if not nonBatchJobRan:
               if self.jobs[instance].sendFiles():
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'R'
                  if self.jobs[instance].executeJob():
                     self.jobs[instance].postProcess()
                     self.jobs[instance].retrieveFiles()
                     self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               self.successfulInstance = instance
               nonBatchJobRan = True
            else:
               markedForDeletion.append(instance)

      if nonBatchJobRan:
         for instance in markedForDeletion:
            del self.jobs[instance]
      del markedForDeletion

      jobsRunning = False
      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['isBatchJob']:
            if self.jobs[instance].sendFiles():
               if self.jobs[instance].executeJob():
                  waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                  self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                  self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                  self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                  jobsRunning = True
               else:
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
            else:
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

      while jobsRunning and (not self.successfulInstance) and (not self.abortGlobal['abortAttempted']):
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForBatchJobs(self.waitForJobsInfo,self.abortGlobal)
         for instance in self.waitForJobsInfo:
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()
            if self.jobs[instance].wasJobSuccessful():
               self.successfulInstance = instance
         if len(completeRemoteJobIndexes) == 0:
            jobsRunning = False

      if self.abortGlobal['abortAttempted']:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
         for instance in self.waitForJobsInfo:
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
               self.jobs[instance].cleanupFiles()
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

      if self.successfulInstance:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               self.jobs[instance].killScripts(signal.SIGUSR1)
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
         for instance in self.waitForJobsInfo:
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
         for instance in self.waitForJobsInfo:
            if   self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
               self.jobs[instance].cleanupFiles()
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
            elif instance != self.successfulInstance:
               self.jobs[instance].cleanupFiles()

      if self.successfulInstance:
         self.jobs[self.successfulInstance].recoverFiles(False)
         fileToMove = "%s_%02d.stdout" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stdout" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.stderr" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stderr" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.SUCCESS" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.SUCCESS" % (self.runName)
            os.rename(fileToMove,fileNewName)
      else:
         for instance in self.waitForJobsInfo:
            self.jobs[instance].recoverStdFiles()
         if len(self.jobs) > 0:
            recoverInstance = random.choice(self.jobs.keys())
            self.jobs[recoverInstance].recoverFiles(True)

      for instance in self.waitForJobsInfo:
         self.jobs[instance].removeInstanceDirectory()

      for fileToRemove in filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildPegasusJobInstances(self,
                                destination,
                                executable,
                                transferExecutable,
                                executableClassification,
                                toolInfo):
      try:
         filesToRemove      = []
         emptyFilesToRemove = []
         jobPath = os.path.join(os.getcwd(),self.runName)
         if os.path.exists(jobPath):
            log("Run directory %s exists" % (jobPath))
            self.__writeToStderr("Run directory %s exists\n" % (jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(jobPath):
            os.mkdir(jobPath)
         if self.isClientTTY:
            parameterCombinationsPath = os.path.join(jobPath,'remoteParameterCombinations.csv')
         else:
            parameterCombinationsPath = os.path.join(jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(parameterCombinationsPath)

         instanceError = False
         inputsPath        = os.path.join(jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError,err:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = []
            for inputFile in self.inputFiles:
               template = ParameterTemplate(inputFile)
               try:
                  inputFile = template.substitute(substitutions)
               except KeyError,err:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
               candidateFileOrDirectory = self.findInputFileOrDirectory(inputFile)
               if candidateFileOrDirectory:
                  if not candidateFileOrDirectory in inputFiles:
                     inputFiles.append(candidateFileOrDirectory)

            if len(self.enteredCommandArguments) > 1:
               for arg in self.enteredCommandArguments[1:]:
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  arglets = re.split('( |=)',arg)
                  for arglet in arglets:
                     if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                        candidateFileOrDirectory = self.findInputFileOrDirectory(arglet)
                        if candidateFileOrDirectory:
                           if not candidateFileOrDirectory in inputFiles:
                              inputFiles.append(candidateFileOrDirectory)

            if len(inputFiles) > 1:
               filteredInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile.replace('@:','')
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(filteredInputFile)
               for inputFile in inputFiles:
                  if not '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(inputFile)
               del inputFiles
               inputFiles = filteredInputFiles

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  gridsite = self.sitesInfo.getSitePegasusSite(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],
                                                                           destination))
                        self.__writeToStderr("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],
                                                                                              destination))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               else:
                  errorMessage = "Invalid destination selection: %s" % (destination)
                  log(errorMessage)
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected site does not meet the specified\n" + \
                                    "application access requirements.  Please select another site."
                     log(errorMessage)
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     log(errorMessage)
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     fpInputFile = open(inputFile,'r')
                     if fpInputFile:
                        inputText = fpInputFile.readlines()
                        fpInputFile.close()
                        template = ParameterTemplate(''.join(inputText))
                        try:
                           inputText = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        instanceInputsPath = os.path.join(inputsPath,instanceId)
                        if not os.path.isdir(instanceInputsPath):
                           os.makedirs(instanceInputsPath)
                        inputName = os.path.basename(inputFile)
                        inputPath = os.path.join(instanceInputsPath,inputName)
                        fpInputPath = open(inputPath,'w')
                        if fpInputPath:
                           fpInputPath.writelines(inputText)
                           fpInputPath.close()
                           substitutedInputFiles.append(inputPath)
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  log("command = " + str(tarCommand))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  arg = self.enteredCommandArguments[0]
                  template = ParameterTemplate(arg)
                  try:
                     rarg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  remoteArgs.append(rarg)
                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        template = ParameterTemplate(arg)
                        try:
                           arg = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        rarg = ''
                        arglets = re.split('( |=)',arg)
                        for arglet in arglets:
                           if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                              argletIsFile = False
                              for tarFile in tarFiles:
                                 if re.match(".*" + re.escape(tarFile) + "$",arglet):
                                    rarg += os.path.join('.',tarFile)
                                    argletIsFile = True
                                    break
                              if not argletIsFile:
                                 rarg += arglet
                           else:
                              rarg += arglet
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = remoteArgs[1:]
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  remoteArgs = []
                  for arg in self.enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        rarg = template.substitute(substitutions)
                     except KeyError,err:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = remoteArgs[1:]

               useEnvironment = ""
               if siteInfo['passUseEnvironment']:
                  reChoice = re.compile(".*_CHOICE$")
                  environmentVars = filter(reChoice.search,os.environ.keys())
                  if "PROMPT_CHOICE" in environmentVars:
                     environmentVars.remove("PROMPT_CHOICE")
                  if "EDITOR_CHOICE" in environmentVars:
                     environmentVars.remove("EDITOR_CHOICE")
                  if len(environmentVars) > 0:
                     useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
                     useEnvironment += ". %s\n" % (self.useSetup)
                     if "ENVIRON_CONFIG_DIRS" in os.environ:
                        useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
                     for environmentVar in environmentVars:
                        useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        log("Tool environment variable must be of the form Variable = Value.")
                        self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,self.appsAccessInfo,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,managerInfo,
                                                            nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,managerInfo,
                                                          nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     log("command = " + str(tarCommand))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        log(tarStdError)
                        log(tarStdOutput)
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  log("command = " + niceCommand)
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     log(gzipStdError)
                     log(gzipStdOutput)
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               self.waitForJobsInfo[instance]['recentJobStatus'] = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         nInstances      = instance
         executeInstance = nInstances+1

         timeHistoryLogs = {}
         timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
         timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
         timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,"0")
         timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,"0")
         timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,"0")
         timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,"0")
         timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,"0")
         timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,"0")
         timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,"0")

         enteredCommand = self.commandParser.getEnteredCommand()

         if   siteInfo['venueMechanism'] == 'local':
            self.jobs[executeInstance] = VenueMechanismLocal(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,"WF;%d" % (nInstances),destination,
                                                             enteredCommand,gridsite,
                                                             stageInTarFile,
                                                             transferExecutable,executable,self.appsAccessInfo,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,managerInfo,
                                                             nCpus,nNodes,ppn,wallTime,
                                                             self.quotaLimit,
                                                             self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                             timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
         elif siteInfo['venueMechanism'] == 'ssh':
            self.jobs[executeInstance] = VenueMechanismSsh(self.remoteMonitors,
                                                           self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                           self.batchCommands,self.isParametric,
                                                           self.runName,self.localJobId,"WF;%d" % (nInstances),destination,
                                                           enteredCommand,gridsite,
                                                           self.tunnelsInfo,
                                                           cloudInstanceId,
                                                           stageInTarFile,
                                                           transferExecutable,executable,self.appsAccessInfo,
                                                           self.stdinput,arguments,useEnvironment,environment,
                                                           self.isMultiCoreRequest,siteInfo,managerInfo,
                                                           nCpus,nNodes,ppn,wallTime,
                                                           self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                           self.quotaLimit,
                                                           timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

         exitCode,scriptFiles = self.jobs[executeInstance].createScripts()
         if exitCode:
            raise JobDistributor.InstancesError

         self.waitForJobsInfo[executeInstance] = {}
         waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
         self.waitForJobsInfo[executeInstance]['isBatchJob']            = waitForJobInfo['isBatchJob']
         self.waitForJobsInfo[executeInstance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
         self.waitForJobsInfo[executeInstance]['remoteBatchSystem']     = self.jobs[executeInstance].remoteBatchSystem
         self.waitForJobsInfo[executeInstance]['instanceDirectory']     = self.jobs[executeInstance].instanceDirectory
         self.waitForJobsInfo[executeInstance]['scratchDirectory']      = self.jobs[executeInstance].scratchDirectory

         stdFile = "%s.stderr" % (self.runName)
         emptyFilesToRemove.append(stdFile)
         for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory':
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s.%s" % (batchQueue,self.runName,outFile)
               emptyFilesToRemove.append(batchQueueOutputFile)

         jobsRunning = False
         if self.waitForJobsInfo[executeInstance]['isBatchJob']:
            if self.jobs[executeInstance].sendFiles():
               if self.jobs[executeInstance].executeJob():
                  waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
                  self.waitForJobsInfo[executeInstance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                  self.waitForJobsInfo[executeInstance]['knownSite']       = waitForJobInfo['knownSite']
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = '?'
                  jobsRunning = True
               else:
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'
            else:
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

         while jobsRunning and (not self.successfulInstance) and (not self.abortGlobal['abortAttempted']):
            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForPegasusWorkflowJobs(self.waitForJobsInfo,
                                                                                             nInstances,parameterCombinationsPath,
                                                                                             self.isClientTTY,self.abortGlobal)
            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance
            if len(completeRemoteJobIndexes) == 0:
               jobsRunning = False

         if self.abortGlobal['abortAttempted']:
            if self.waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
               self.jobs[executeInstance].killScripts(self.abortGlobal['abortSignal'])
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            if 'recentJobSite' in self.waitForJobsInfo[executeInstance]:
               executionVenue = self.waitForJobsInfo[executeInstance]['recentJobSite']
               self.jobs[executeInstance].updateVenue(executionVenue)
            if self.waitForJobsInfo[executeInstance]['recentJobStatus'] == 'KD':
               self.jobs[executeInstance].cleanupFiles()
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

         if self.successfulInstance:
            for instance in xrange(1,nInstances+1):
               self.jobs[instance].jobSubmitted = True

         for instance in xrange(1,nInstances+1):
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()

         for fileToRemove in filesToRemove:
            if os.path.exists(fileToRemove):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if os.path.exists(fileToRemove):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

         if os.path.isdir(inputsPath):
            shutil.rmtree(inputsPath,True)

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildSweepJobInstances(self,
                              instanceDestinations,
                              executable,
                              transferExecutable,
                              executableClassification,
                              toolInfo):
      try:
         filesToRemove      = []
         emptyFilesToRemove = []
         jobPath = os.path.join(os.getcwd(),self.runName)
         if os.path.exists(jobPath):
            log("Run directory %s exists" % (jobPath))
            self.__writeToStderr("Run directory %s exists\n" % (jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(jobPath):
            os.mkdir(jobPath)
         if self.isClientTTY:
            parameterCombinationsPath = os.path.join(jobPath,'remoteParameterCombinations.csv')
         else:
            parameterCombinationsPath = os.path.join(jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(parameterCombinationsPath)

         instanceError = False
         inputsPath        = os.path.join(jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError,err:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = []
            for inputFile in self.inputFiles:
               template = ParameterTemplate(inputFile)
               try:
                  inputFile = template.substitute(substitutions)
               except KeyError,err:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
               candidateFileOrDirectory = self.findInputFileOrDirectory(inputFile)
               if candidateFileOrDirectory:
                  if not candidateFileOrDirectory in inputFiles:
                     inputFiles.append(candidateFileOrDirectory)

            if len(self.enteredCommandArguments) > 1:
               for arg in self.enteredCommandArguments[1:]:
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  arglets = re.split('( |=)',arg)
                  for arglet in arglets:
                     if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                        candidateFileOrDirectory = self.findInputFileOrDirectory(arglet)
                        if candidateFileOrDirectory:
                           if not candidateFileOrDirectory in inputFiles:
                              inputFiles.append(candidateFileOrDirectory)

            if len(inputFiles) > 1:
               filteredInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile.replace('@:','')
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(filteredInputFile)
               for inputFile in inputFiles:
                  if not '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(inputFile)
               del inputFiles
               inputFiles = filteredInputFiles

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               destination = random.choice(instanceDestinations)
               if   (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination))
                        self.__writeToStderr("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],
                                                                                              destination))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               elif (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  runjobVenues = self.sitesInfo.getSiteMembers(destination)
                  gridsite     = " ".join(runjobVenues)
               else:
                  if   self.runjobExists:
                     siteInfo = self.sitesInfo.getSiteInfo('grid')
                     gridsite = destination
                  elif transferExecutable:
                     errorMessage = "Invalid destination selection: %s" % (destination)
                     log(errorMessage)
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected site does not meet the specified\n" + \
                                    "application access requirements.  Please select another site."
                     log(errorMessage)
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     fpInputFile = open(inputFile,'r')
                     if fpInputFile:
                        inputText = fpInputFile.readlines()
                        fpInputFile.close()
                        template = ParameterTemplate(''.join(inputText))
                        try:
                           inputText = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        instanceInputsPath = os.path.join(inputsPath,instanceId)
                        if not os.path.isdir(instanceInputsPath):
                           os.makedirs(instanceInputsPath)
                        inputName = os.path.basename(inputFile)
                        inputPath = os.path.join(instanceInputsPath,inputName)
                        fpInputPath = open(inputPath,'w')
                        if fpInputPath:
                           fpInputPath.writelines(inputText)
                           fpInputPath.close()
                           substitutedInputFiles.append(inputPath)
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  log("command = " + str(tarCommand))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  arg = self.enteredCommandArguments[0]
                  template = ParameterTemplate(arg)
                  try:
                     rarg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  remoteArgs.append(rarg)
                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        template = ParameterTemplate(arg)
                        try:
                           arg = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        rarg = ''
                        arglets = re.split('( |=)',arg)
                        for arglet in arglets:
                           if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                              argletIsFile = False
                              for tarFile in tarFiles:
                                 if re.match(".*" + re.escape(tarFile) + "$",arglet):
                                    rarg += os.path.join('.',tarFile)
                                    argletIsFile = True
                                    break
                              if not argletIsFile:
                                 rarg += arglet
                           else:
                              rarg += arglet
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = remoteArgs[1:]
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  remoteArgs = []
                  for arg in self.enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        rarg = template.substitute(substitutions)
                     except KeyError,err:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = remoteArgs[1:]

               useEnvironment = ""
               if siteInfo['passUseEnvironment']:
                  reChoice = re.compile(".*_CHOICE$")
                  environmentVars = filter(reChoice.search,os.environ.keys())
                  if "PROMPT_CHOICE" in environmentVars:
                     environmentVars.remove("PROMPT_CHOICE")
                  if "EDITOR_CHOICE" in environmentVars:
                     environmentVars.remove("EDITOR_CHOICE")
                  if len(environmentVars) > 0:
                     useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
                     useEnvironment += ". %s\n" % (self.useSetup)
                     if "ENVIRON_CONFIG_DIRS" in os.environ:
                        useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
                     for environmentVar in environmentVars:
                        useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        log("Tool environment variable must be of the form Variable = Value.")
                        self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     log(errorMessage)
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,self.appsAccessInfo,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,managerInfo,
                                                            nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,managerInfo,
                                                          nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'gsissh':
                  self.jobs[instance] = VenueMechanismGsiSsh(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,instanceId,destination,
                                                             enteredCommand,self.x509SubmitProxy,
                                                             stageInTarFile,
                                                             transferExecutable,executable,self.appsAccessInfo,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,managerInfo,
                                                             nCpus,nNodes,ppn,wallTime,
                                                             self.quotaLimit,
                                                             timeHistoryLogs)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     log("command = " + str(tarCommand))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        log(tarStdError)
                        log(tarStdOutput)
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  log("command = " + niceCommand)
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     log(gzipStdError)
                     log(gzipStdOutput)
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stageInTarFile))

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
               self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
               self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
               self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
               self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s_%s.stderr" % (self.runName,instanceId)
            emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stdFile))
            for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory':
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueue,self.runName,instanceId,outFile)
                  emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,batchQueueOutputFile))

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         jobsRunning = False
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['isBatchJob']:
               if self.jobs[instance].sendFiles():
                  if self.jobs[instance].executeJob():
                     waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                     self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                     self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                     self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                     jobsRunning = True
                  else:
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

         while jobsRunning and (not self.abortGlobal['abortAttempted']):
            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForSweepJobs(self.waitForJobsInfo,
                                                                                   parameterCombinationsPath,
                                                                                   self.isClientTTY,self.abortGlobal)
            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
            if len(completeRemoteJobIndexes) == 0:
               jobsRunning = False

         for fileToRemove in filesToRemove:
            if os.path.exists(fileToRemove):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if os.path.exists(fileToRemove):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

         if os.path.isdir(inputsPath):
            shutil.rmtree(inputsPath,True)

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildJobInstances(self):
      try:
         exitCode = 0
         userDestinations,instanceDestinations, \
             executable,transferExecutable,executableClassification,toolInfo = self.getInstanceDestinations()
      except JobDistributor.CommandError:
         exitCode = self.jobStatistics[0]['exitCode']

      except JobDistributor.InstancesError:
         exitCode = self.jobStatistics[0]['exitCode']

      if not exitCode:
         self.remoteMonitors['job']    = RemoteJobMonitor(self.jobMonitorHost,self.jobMonitorPort)
         self.remoteMonitors['tunnel'] = RemoteTunnelMonitor(self.tunnelMonitorHost,self.tunnelMonitorPort)
         self.remoteMonitors['cloud']  = None

         if self.isParametric:
            pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                               instanceDestinations)
            if len(pegasusSites) > 0:
               destination = instanceDestinations[0]
               exitCode = self.buildPegasusJobInstances(destination,
                                                        executable,transferExecutable,executableClassification,toolInfo)
            else:
               exitCode = self.buildSweepJobInstances(instanceDestinations,
                                                      executable,transferExecutable,executableClassification,toolInfo)
               self.jobStatistics[0]['exitCode'] = exitCode
         else:
            exitCode = self.buildRedundantJobInstances(userDestinations,instanceDestinations,
                                                       executable,transferExecutable,executableClassification,toolInfo)

      return(exitCode)


   def run(self):
      exitCode = self.parseCommandArguments()
      if not exitCode:
         exitCode = self.setJobId()
         if not exitCode:
            if not self.runName:
               self.runName = self.localJobId
            self.setInfo()
            self.showUsage()
            if self.doRun():
               exitCode =  self.buildJobDescription()
               if not exitCode:
                  exitCode = self.buildJobInstances()

      self.reportExitCondition()

      return(exitCode)


