# @package      hubzero-submit-distributor
# @file         RemoteBatchSGE.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os.path
import re

from hubzero.submit.TimeConversion import minTohhmmss

class RemoteBatchSGE:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                localJobId,
                instanceId,
                appScriptName,
                environment,
                transferExecutable,
                executable,
                arguments,
                stdinput,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                timeHistoryLogs):
      self.hubUserName         = hubUserName
      self.hubUserId           = hubUserId
      self.runName             = runName
      self.localJobId          = localJobId
      self.instanceId          = instanceId
      self.appScriptName       = appScriptName
      self.environment         = environment
      self.stageFiles          = siteInfo['stageFiles']
      self.transferExecutable  = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable    = os.path.join(".",os.path.basename(executable))
         else:
            self.executable    = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable    = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable    = executable
      self.arguments           = arguments
      self.stdinput            = stdinput
      self.isMultiCoreRequest  = isMultiCoreRequest
      self.computationMode     = managerInfo['computationMode']
      self.preManagerCommands  = managerInfo['preManagerCommands']
      self.managerCommand      = managerInfo['managerCommand']
      self.postManagerCommands = managerInfo['postManagerCommands']
      self.nNodes              = nNodes
      self.ppn                 = ppn
      self.hostAttributes      = siteInfo['hostAttributes']
      self.remoteBatchQueue    = siteInfo['remoteBatchQueue']
      self.parallelEnvironment = siteInfo['parallelEnvironment']
      self.wallTime            = minTohhmmss(wallTime)
      if siteInfo['remoteBatchAccount'] != "":
         self.account          = '#$ -A ' + siteInfo['remoteBatchAccount']
      else:
         self.account          = ''
      self.quotaLimit          = quotaLimit
      self.timestampStart      = timeHistoryLogs['timestampStart']
      self.timestampFinish     = timeHistoryLogs['timestampFinish']
      self.timeResults         = timeHistoryLogs['timeResults']

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeSerialTemplate(self):
      return """#!/bin/sh
#$ -S /bin/sh
#$ -cwd
#$ -l h_rt=WALLTIME
#$ -N JOB_JOBID_INSTANCEID
#$ -o sge_RUNNAME_INSTANCEID.stdout
#$ -e sge_RUNNAME_INSTANCEID.stderr
#$ -m n
REMOTEBATCHACCOUNT
QUEUE

ENVIRONMENT

# RemoteBatchSGE:makeSerialTemplate

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   if [ ! -s TIME_RESULTS ] ; then
      cat > TIME_RESULTS << EOF
Command terminated by signal 3
real 0.00
user 0.00
sys 0.00
EOF
   fi
   exit 1
}

export PATH=${SGE_O_WORKDIR}:${PATH}
# Change to directory where job was submitted.
cd ${SGE_O_WORKDIR}

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

QUOTA

date +"%s" > TS_START

PREMANAGERCOMMANDS
${TIMEPATH} -p -o TIME_RESULTS EXECUTABLE ARGUMENTS STDIN > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_stdinput            = re.compile("STDIN")
      re_wallTime            = re.compile("WALLTIME")
      re_queue               = re.compile("QUEUE")
      re_environment         = re.compile("ENVIRONMENT")
      re_executable          = re.compile("EXECUTABLE")
      re_arguments           = re.compile("ARGUMENTS")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_runName             = re.compile("RUNNAME")
      re_jobId               = re.compile("JOBID")
      re_instanceId          = re.compile("INSTANCEID")
      re_nnodes              = re.compile("NNODES")
      re_attributes          = re.compile(":ATTRIBUTES")
      re_ppn                 = re.compile("PPN")
      re_processors          = re.compile("NPROCESSORS")
      re_account             = re.compile("REMOTEBATCHACCOUNT")
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_hubUserName         = re.compile("HUBUSERNAME")
      re_hubUserId           = re.compile("HUBUSERID")
      re_quota               = re.compile("QUOTA")

      template = self.__makeSerialTemplate()

      queueSGE = ""
      if self.remoteBatchQueue != "":
         queueSGE = "#$ -q " + self.remoteBatchQueue

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      if self.stdinput == "":
         template = re_stdinput.sub(self.stdinput,template)
      else:
         template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_wallTime.sub(self.wallTime,template)
      template = re_queue.sub(queueSGE,template)
      template = re_environment.sub(environmentExport,template)
      template = re_executable.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_nnodes.sub(self.nNodes,template)
      template = re_attributes.sub(self.hostAttributes,template)
      template = re_ppn.sub(self.ppn,template)
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      template = re_processors.sub(nProcessors,template)
      template = re_account.sub(self.account,template)
      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)
      if self.quotaLimit > 0:
         template = re_quota.sub("ulimit -f " + str(self.quotaLimit),template)
      else:
         template = re_quota.sub("",template)

      return(template)


   def __makeParallelTemplate(self):
      return """#!/bin/sh
#$ -S /bin/sh
#$ -cwd
#$ -l nodes=NNODES:ATTRIBUTES:ppn=PPN
#$ -l h_rt=WALLTIME
#$ -N JOB_JOBID_INSTANCEID
#$ -o sge_RUNNAME_INSTANCEID.stdout
#$ -e sge_RUNNAME_INSTANCEID.stderr
#$ -m n
REMOTEBATCHACCOUNT
QUEUE

ENVIRONMENT

# RemoteBatchSGE:makeParallelTemplate

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   if [ ! -s TIME_RESULTS ] ; then
      cat > TIME_RESULTS << EOF
Command terminated by signal 3
real 0.00
user 0.00
sys 0.00
EOF
   fi
   exit 1
}

export PATH=${SGE_O_WORKDIR}:${PATH}
# Change to directory where job was submitted.
cd ${SGE_O_WORKDIR}

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

APPLICATION=EXECUTABLE
timeMeasured=0

QUOTA

date +"%s" > TS_START

PREMANAGERCOMMANDS
${TIMEPATH} -p -o TIME_RESULTS MANAGERCOMMAND ${APPLICATION} ARGUMENTS STDIN > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

if [ ${timeMeasured} -eq 0 ] ; then
   startTime=`cat TS_START`
   finishTime=`cat TS_FINISH`
   let wallTime=finishTime-startTime
   cnt=0
   while [ "${cnt}" != "NPROCESSORS" ] ; do
      cat >> TIME_RESULTS << EOF
real ${wallTime}.00
user ${wallTime}.00
sys 0.00
EOF
      cnt=`expr ${cnt} + 1`
   done
fi
"""


   def __buildParallelFile(self):
      # setup regex's for the template
      re_stdinput            = re.compile("STDIN")
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_wallTime            = re.compile("WALLTIME")
      re_nnodes              = re.compile("NNODES")
      re_attributes          = re.compile(":ATTRIBUTES")
      re_ppn                 = re.compile("PPN")
      re_queue               = re.compile("QUEUE")
      re_environment         = re.compile("ENVIRONMENT")
      re_executable          = re.compile("EXECUTABLE")
      re_arguments           = re.compile("ARGUMENTS")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_runName             = re.compile("RUNNAME")
      re_jobId               = re.compile("JOBID")
      re_instanceId          = re.compile("INSTANCEID")
      re_processors          = re.compile("NPROCESSORS")
      re_account             = re.compile("REMOTEBATCHACCOUNT")
      re_hubUserName         = re.compile("HUBUSERNAME")
      re_hubUserId           = re.compile("HUBUSERID")
      re_quota               = re.compile("QUOTA")

      template = self.__makeParallelTemplate()

      queueSGE = ""
      if self.remoteBatchQueue != "":
         queueSGE = "#$ -q " + self.remoteBatchQueue

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      if self.stdinput == "":
         template = re_stdinput.sub(self.stdinput,template)
      else:
         template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_wallTime.sub(self.wallTime,template)
      template = re_nnodes.sub(self.nNodes,template)
      template = re_attributes.sub(self.hostAttributes,template)
      template = re_ppn.sub(self.ppn,template)
      template = re_queue.sub(queueSGE,template)
      template = re_environment.sub(environmentExport,template)
      template = re_executable.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      template = re_processors.sub(nProcessors,template)
      template = re_account.sub(self.account,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)
      if self.quotaLimit > 0:
         template = re_quota.sub("ulimit -f " + str(self.quotaLimit),template)
      else:
         template = re_quota.sub("",template)

      return(template)


   def __makeMPITemplate(self):
      return """#!/bin/sh
#$ -S /bin/sh
#$ -cwd
#$ -pe PARALLELENVIRONMENT NPROCESSORS
#$ -l h_rt=WALLTIME
#$ -N JOB_JOBID_INSTANCEID
#$ -o sge_RUNNAME_INSTANCEID.stdout
#$ -e sge_RUNNAME_INSTANCEID.stderr
#$ -m n
REMOTEBATCHACCOUNT
QUEUE

ENVIRONMENT

# RemoteBatchSGE:makeMPITemplate

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   if [ ! -s TIME_RESULTS ] ; then
      cat > TIME_RESULTS << EOF
Command terminated by signal 3
real 0.00
user 0.00
sys 0.00
EOF
   fi
   exit 1
}

export PATH=${SGE_O_WORKDIR}:${PATH}
# Change to directory where job was submitted.
cd ${SGE_O_WORKDIR}

QUOTA

chmod +x APPSCRIPTNAME
date +"%s" > TS_START

PREMANAGERCOMMANDS
MANAGERCOMMAND `pwd`/APPSCRIPTNAME > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

touch TIME_RESULTS-0
cat TIME_RESULTS-[0-9]* >> TIME_RESULTS
rm -f TIME_RESULTS-[0-9]*
"""


   def __buildMPIFile(self):
      # setup regex's for the template
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_parallelEnvironment = re.compile("PARALLELENVIRONMENT")
      re_wallTime            = re.compile("WALLTIME")
      re_nnodes              = re.compile("NNODES")
      re_attributes          = re.compile(":ATTRIBUTES")
      re_ppn                 = re.compile("PPN")
      re_processors          = re.compile("NPROCESSORS")
      re_queue               = re.compile("QUEUE")
      re_environment         = re.compile("ENVIRONMENT")
      re_appScriptName       = re.compile("APPSCRIPTNAME")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_runName             = re.compile("RUNNAME")
      re_jobId               = re.compile("JOBID")
      re_instanceId          = re.compile("INSTANCEID")
      re_account             = re.compile("REMOTEBATCHACCOUNT")
      re_hubUserName         = re.compile("HUBUSERNAME")
      re_hubUserId           = re.compile("HUBUSERID")
      re_quota               = re.compile("QUOTA")

      template = self.__makeMPITemplate()

      queueSGE = ""
      if self.remoteBatchQueue != "":
         queueSGE = "#$ -q " + self.remoteBatchQueue

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      template = re_parallelEnvironment.sub(self.parallelEnvironment,template)
      template = re_wallTime.sub(self.wallTime,template)
      template = re_nnodes.sub(self.nNodes,template)
      template = re_attributes.sub(self.hostAttributes,template)
      template = re_ppn.sub(self.ppn,template)
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      template = re_processors.sub(nProcessors,template)
      template = re_queue.sub(queueSGE,template)
      template = re_environment.sub(environmentExport,template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_account.sub(self.account,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)
      if self.quotaLimit > 0:
         template = re_quota.sub("ulimit -f " + str(self.quotaLimit),template)
      else:
         template = re_quota.sub("",template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.sge" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildParallelFile()
      else:
         batchScript = self.__buildSerialFile()

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


