# @package      hubzero-submit-distributor
# @file         VenueMechanismGsiSsh.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import sys
import os
import re
import stat
import time
import datetime
import socket
import logging

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore
from hubzero.submit.RemoteBatchCONDOR      import RemoteBatchCONDOR
from hubzero.submit.RemoteBatchLL          import RemoteBatchLL
from hubzero.submit.RemoteBatchLSF         import RemoteBatchLSF
from hubzero.submit.RemoteBatchPBS         import RemoteBatchPBS
from hubzero.submit.RemoteBatchPBS8        import RemoteBatchPBS8
from hubzero.submit.RemoteBatchSGE         import RemoteBatchSGE
from hubzero.submit.RemoteBatchSLURM       import RemoteBatchSLURM
from hubzero.submit.RemoteBatchAppScript   import RemoteBatchAppScript
from hubzero.submit.RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from hubzero.submit.RemoteInstantAppScript import RemoteInstantAppScript

class VenueMechanismGsiSsh(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                session,
                distributorPid,
                batchCommands,
                isParametric,
                runName,
                localJobId,
                instanceId,
                destination,
                enteredCommand,
                x509ProxyPath,
                stageInTarFile,
                transferExecutable,
                executable,
                appsAccessInfo,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                timeHistoryLogs):
      VenueMechanismCore.__init__(self,timeHistoryLogs,siteInfo,managerInfo,remoteMonitors,
                                       isMultiCoreRequest,nCpus,nNodes,ppn)

      self.logger                = logging.getLogger(__name__)
      self.venueMechanism        = 'gsissh'
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.session               = session
      self.distributorPid        = distributorPid
      remoteScriptsDirectory = siteInfo['remoteBinDirectory'].replace('$','\$')
      self.receiveInput          = os.path.join(remoteScriptsDirectory,batchCommands['receiveInput'])
      self.submitBatchJob        = os.path.join(remoteScriptsDirectory,batchCommands['submitBatchJob'])
      self.postProcessJob        = os.path.join(remoteScriptsDirectory,batchCommands['postProcessJob'])
      self.transmitResults       = os.path.join(remoteScriptsDirectory,batchCommands['transmitResults'])
      self.cleanupJob            = os.path.join(remoteScriptsDirectory,batchCommands['cleanupJob'])
      self.killBatchJob          = os.path.join(remoteScriptsDirectory,batchCommands['killBatchJob'])
      self.isParametric          = isParametric
      self.runName               = runName
      self.localJobId            = localJobId
      if instanceId.startswith('WF;'):
         self.instanceId         = ""
         self.nInstances         = int(instanceId.split(';')[1])
      else:
         self.instanceId         = instanceId
         self.nInstances         = 0
      self.remoteBatchSystem     = siteInfo['remoteBatchSystem']
      self.destination           = destination
      self.enteredCommand        = enteredCommand
      self.venue                 = siteInfo['venue']
      self.siteMonitorDesignator = siteInfo['siteMonitorDesignator']
      self.gsiHost               = siteInfo['gsiHost']
      if self.gsiHost != socket.gethostname():
         self.addVenueToBatchCommand = True
         self.sharedUserSpace    = False
      else:
         self.addVenueToBatchCommand = False
         self.sharedUserSpace    = siteInfo['sharedUserSpace']
      self.x509ProxyPath         = x509ProxyPath
      self.remoteUser            = siteInfo['remoteUser']
      self.logUserRemotely       = siteInfo['logUserRemotely']
      self.stageFiles            = siteInfo['stageFiles']
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile     = stageInTarFile
      else:
         self.stageInTarFile     = stageInTarFile + '.gz'
      self.transferExecutable    = transferExecutable
      self.executable            = executable
      self.appsAccessInfo        = appsAccessInfo
      self.stdinput              = stdinput
      if isinstance(arguments,list):
         self.arguments          = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments   += '"%s" ' % (argument)
            else:
               self.arguments   += "%s " % (argument)
      else:
         self.arguments          = arguments
      self.useEnvironment        = useEnvironment
      self.environment           = environment
      self.remoteBatchQueue      = siteInfo['remoteBatchQueue']
      self.wallTime              = wallTime
      self.quotaLimit            = quotaLimit
      self.batchLogPath          = ""
      self.appScriptPath         = ""
      self.batchScriptPath       = ""
      self.nodeFilePath          = ""
      self.remoteJobIdNumber     = ""
      self.sshBaseCommand        = ""
      self.sshIdentityPath       = ""
      self.removeIdentity        = False

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob           = False
         self.instanceDirectory    = os.getcwd()
      else:
         self.isBatchJob           = True
         if self.instanceId:
            self.instanceDirectory = os.path.join(os.getcwd(),self.runName,self.instanceId)
         else:
            self.instanceDirectory = os.path.join(os.getcwd(),self.runName)
         if not os.path.isdir(self.instanceDirectory):
            os.makedirs(self.instanceDirectory)

      self.createSSHIdentityPath()

      if not self.sharedUserSpace and self.stageFiles:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.workingDirectory = os.path.join(siteInfo['remoteScratchDirectory'], \
                                              "%s_%s_%s" % (str(epoch),localJobId,self.instanceId))
         self.scratchDirectory = self.workingDirectory
      else:
         self.workingDirectory = os.getcwd()
         if siteInfo['remoteScratchDirectory']:
            epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
            self.scratchDirectory = os.path.join(siteInfo['remoteScratchDirectory'], \
                                                 "%s_%s_%s" % (str(epoch),localJobId,self.instanceId))
         else:
            self.scratchDirectory = self.instanceDirectory


   def createSSHIdentityPath(self):
      exitStatus = 0
      if not self.sshIdentityPath:
         updateKnownHosts = False
         if   self.remoteUser.startswith('USER:'):
            keyfamily = self.remoteUser.split(':')[1]
            self.remoteUser = self.hubUserName
            command = 'genuserpki ' + keyfamily
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeCommand(command)
            if not exitStatus:
               self.sshIdentityPath = stdOutput.strip()
               self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
               updateKnownHosts = True
         elif self.remoteUser.startswith('USER'):
            self.remoteUser = self.hubUserName

         if not exitStatus:
            if updateKnownHosts:
               if self.addVenueToBatchCommand:
                  command = 'update-known-hosts' + " " + self.gsiHost
               else:
                  command = 'update-known-hosts' + " " + self.venue
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               exitStatus,stdOutput,stdError = self.executeCommand(command)
               if not exitStatus:
                  self.sshBaseCommand = "ssh -T -x -a -i " + self.sshIdentityPath + " " + \
                                         self.remoteUser + "@" + self.venue
            else:
               self.sshBaseCommand = "ssh -T -x -a " + self.remoteUser + "@" + self.venue

         if not exitStatus:
            if self.remoteBatchSystem == 'SCRIPT':
               self.sshBaseCommand = "gsissh -T -x -a " + self.venue
            elif self.addVenueToBatchCommand:
               self.sshBaseCommand = "ssh -T -x -a -i " + self.sshIdentityPath + " " + \
                                      self.remoteUser + "@" + self.gsiHost
            else:
               self.sshBaseCommand = "gsissh -T -x -a " + self.venue


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      self.logger.log(logging.INFO,getLogMessage("workingDirectory " + self.workingDirectory))
      if   self.remoteBatchSystem == 'PBS':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS(self.hubUserName,self.hubUserId,
                                      self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,
                                      self.wallTime,self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'PBS8':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS8(self.hubUserName,self.hubUserId,
                                       self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                       self.transferExecutable,
                                       self.executable,self.arguments,self.stdinput,
                                       self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                       self.nNodes,self.ppn,
                                       self.wallTime,self.quotaLimit,
                                       self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SGE':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchSGE(self.hubUserName,self.hubUserId,
                                      self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,
                                      self.wallTime,self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'LSF':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLSF(self.hubUserName,self.hubUserId,
                                      self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,
                                      self.wallTime,self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'LL':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLL(self.hubUserName,self.hubUserId,
                                     self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                     self.transferExecutable,
                                     self.executable,self.arguments,self.stdinput,
                                     self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                     self.nNodes,self.ppn,
                                     self.wallTime,self.quotaLimit,
                                     self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SLURM':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchSLURM(self.hubUserName,self.hubUserId,
                                        self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                        self.transferExecutable,
                                        self.executable,self.arguments,self.stdinput,
                                        self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                        self.nNodes,self.ppn,
                                        self.wallTime,self.quotaLimit,
                                        self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'CONDOR':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchCONDOR(self.hubUserName,self.hubUserId,
                                         self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                         appScriptName,self.environment,
                                         self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                         self.wallTime)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SCRIPT':
         remoteAppScript = RemoteInstantAppScript(self.hubUserName,self.hubUserId,
                                                  self.runName,self.localJobId,self.instanceId,False,
                                                  self.workingDirectory,
                                                  self.transferExecutable,self.executable,
                                                  self.stdinput,self.arguments,
                                                  self.useEnvironment,self.environment,
                                                  self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                  self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,
                                           self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                           self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                           self.nNodes,self.ppn,
                                           self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()

      if batchLogName != "":
         self.batchLogPath    = os.path.join(self.instanceDirectory,batchLogName)
      if appScriptName != "":
         self.appScriptPath   = os.path.join(self.instanceDirectory,appScriptName)
      if batchScriptName != "":
         self.batchScriptPath = os.path.join(self.instanceDirectory,batchScriptName)
      if nodeFileName != "":
         self.nodeFilePath    = os.path.join(self.instanceDirectory,nodeFileName)

      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = 1
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

      if batchScript != "":
         fpBatchScript = open(self.batchScriptPath,'w')
         if fpBatchScript:
            fpBatchScript.write(batchScript)
            fpBatchScript.close()
            scriptFiles.append(self.batchScriptPath)
            if self.remoteBatchSystem == 'SCRIPT':
               os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.batchScriptPath)))
            self._writeToStderr("could not open %s for writing\n" % (self.batchScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if appScript != "":
         fpAppScript = open(self.appScriptPath,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(self.appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            scriptFiles.append(self.appScriptPath)
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.appScriptPath)))
            self._writeToStderr("could not open %s for writing\n" % (self.appScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if len(nodeList) > 0:
         fpNodes = open(self.nodeFilePath,'w')
         if fpNodes:
            fpNodes.write('\n'.join(nodeList)+'\n')
            fpNodes.close()
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.nodeFilePath)))
            self._writeToStderr("could not open %s for writing\n" % (self.nodeFilePath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      exitStatus = 0

      if self.remoteBatchQueue != "":
         self.jobStatistics[self.jobIndex]['venue'] = self.remoteBatchQueue + '@' + self.venue
      else:
         self.jobStatistics[self.jobIndex]['venue'] = self.venue
      if self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
         stageInTarPath = os.path.join(self.instanceDirectory,self.stageInTarFile)

         if self.sshBaseCommand != "":
            if self.addVenueToBatchCommand:
               command = "cat " + stageInTarPath + " | " + self.sshBaseCommand + \
                                                   " \"" + self.receiveInput + " " + \
                                                                  self.venue + " " + \
                                                      remoteWorkingDirectory + " " + \
                                                       self.scratchDirectory + " " + \
                                self.timeHistoryLogs['timestampTransferred'] + "\""
            else:
               command = "cat " + stageInTarPath + " | " + self.sshBaseCommand + \
                                                   " \"" + self.receiveInput + " " + \
                                                      remoteWorkingDirectory + " " + \
                                                       self.scratchDirectory + " " + \
                                self.timeHistoryLogs['timestampTransferred'] + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"")
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
         else:
            exitStatus = 1
            stdOutput  = ""
            stdError   = ""

      if exitStatus:
         self.filesSent = False
         self.jobStatistics[self.jobIndex]['exitCode'] = 11
#        if stdOutput != "":
#           if self.instanceId:
#              stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#           else:
#              stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#           fpStd = open(stdFile,'a')
#           if fpStd:
#              fpStd.write(stdOutput)
#              fpStd.close()
         if stdError != "":
            if self.instanceId:
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      if not self.stageFiles:
         if self.remoteBatchSystem == 'SCRIPT':
            self.sshBaseCommand = "gsissh -T -x -a " + self.venue
            if self.batchScriptPath != "":
               command = self.sshBaseCommand + " " + self.batchScriptPath
            else:
               command = self.sshBaseCommand + " " + self.appScriptPath
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeCommand(command,True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
         if self.remoteBatchSystem == 'SCRIPT':
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            if self.addVenueToBatchCommand:
               commandArgs.append(self.venue)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(self.scratchDirectory)
            if self.batchScriptPath != "":
               commandArgs.append(self.batchScriptPath)
            else:
               commandArgs.append(self.appScriptPath)
            if self.logUserRemotely:
               commandArgs.append(self.hubUserName)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""

            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"",True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            if self.addVenueToBatchCommand:
               commandArgs.append(self.venue)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(self.scratchDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            if self.logUserRemotely:
               commandArgs.append(self.hubUserName)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"")
            if not exitStatus:
               remoteJobId = stdOutput.strip()
               self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
               try:
                  if   self.remoteBatchSystem == 'PBS':
                     self.remoteJobIdNumber = remoteJobId.split('.')[0]
                  elif self.remoteBatchSystem == 'PBS8':
                     self.remoteJobIdNumber = remoteJobId.split('.')[0]
                  elif self.remoteBatchSystem == 'SGE':
#                    Your job 16 ("JOB_00146752_01") has been submitted
                     self.remoteJobIdNumber = remoteJobId.split()[2]
                  elif self.remoteBatchSystem == 'LSF':
#                    Job <851126> is submitted to default queue <normal>.
                     self.remoteJobIdNumber = remoteJobId.split('<')[1].split('>')[0]
                  elif self.remoteBatchSystem == 'LL':
                     self.remoteJobIdNumber = remoteJobId
                  elif self.remoteBatchSystem == 'SLURM':
                     self.remoteJobIdNumber = remoteJobId.split()[-1]
                  elif self.remoteBatchSystem == 'CONDOR':
#                    Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
                     self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"

                  self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               except:
                  exitStatus = 248

               if not exitStatus:
                  self.jobSubmitted = True
                  self.remoteJobMonitor.postJobSubmission(self.siteMonitorDesignator,self.remoteJobIdNumber,
                                                          str(self.hubUserId),self.enteredCommand,
                                                          self.localJobId,self.instanceId,self.destination,
                                                          self.runName,self.nCpus,self.distributorPid)
               else:
                  self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
            else:
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
         if stdOutput != "":
            if self.instanceId:
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdOutput)
               fpStd.close()
         if stdError != "":
            if self.instanceId:
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['siteMonitorDesignator'] = self.siteMonitorDesignator
      waitForJobInfo['remoteJobId']           = self.remoteJobIdNumber
      waitForJobInfo['knownSite']             = self.destination

      return(waitForJobInfo)


   def postProcess(self):
      if not self.jobPostProcessed:

         self.jobPostProcessed = True


   def retrieveFiles(self):
      exitStatus = 0

      if self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
         if self.jobSubmitted:
            if self.addVenueToBatchCommand:
               command = self.sshBaseCommand + " \"" + self.transmitResults + " " + \
                                                                 self.venue + " " + \
                                                    remoteWorkingDirectory + "\"" + \
                                               " | tar xzmf - --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
            else:
               command = self.sshBaseCommand + " \"" + self.transmitResults + " " + \
                                                    remoteWorkingDirectory + "\"" + \
                                               " | tar xzmf - --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,"")
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 if self.instanceId:
#                    stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#                 else:
#                    stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#                 fpStd = open(stdFile,'a')
#                 if fpStd:
#                    fpStd.write(stdOutput)
#                    fpStd.close()
               if stdError != "":
                  if self.instanceId:
                     stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                  else:
                     stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                  fpStd = open(stdFile,'a')
                  if fpStd:
                     fpStd.write(stdError)
                     fpStd.close()

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self):
      if not self.filesCleanedup:
         if self.stageFiles:
            remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
            if self.jobSubmitted:
               if self.addVenueToBatchCommand:
                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + \
                                                               self.venue + " " + \
                                                   remoteWorkingDirectory + " " + \
                                                    self.scratchDirectory + " " + \
                                                   self.remoteBatchSystem + "\""
               else:
                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + \
                                                   remoteWorkingDirectory + " " + \
                                                    self.scratchDirectory + " " + \
                                                   self.remoteBatchSystem + "\""
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               os.environ['X509_USER_PROXY'] = self.x509ProxyPath
               self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,"")[1]))

         if self.batchScriptPath:
            if os.path.isfile(self.batchScriptPath):
               os.remove(self.batchScriptPath)
         if self.appScriptPath:
            if os.path.isfile(self.appScriptPath):
               os.remove(self.appScriptPath)
         if self.nodeFilePath:
            if os.path.isfile(self.nodeFilePath):
               os.remove(self.nodeFilePath)
         if self.batchLogPath:
            if os.path.isfile(self.batchLogPath):
               os.remove(self.batchLogPath)

         self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      if   self.remoteJobIdNumber:
         if self.addVenueToBatchCommand:
            command = self.sshBaseCommand + " \"" + self.killBatchJob + " " + \
                                                           self.venue + " " + \
                                               self.remoteJobIdNumber + " " + \
                                               self.remoteBatchSystem + "\""
         else:
            command = self.sshBaseCommand + " \"" + self.killBatchJob + " " + \
                                               self.remoteJobIdNumber + " " + \
                                               self.remoteBatchSystem + "\""
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,"")[1]))
      elif self.remoteBatchSystem == 'SCRIPT' and self.childPid != 0:
#        log("Send TERM to child script process %d" % (self.childPid))
#        os.kill(childPid,signalNumber.SIGTERM)
         if self.batchScriptPath != "":
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.batchScriptPath)
         else:
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.appScriptPath)
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,"")[1]))

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 7 | signalNumber

      self.scriptsKilled = True


