#!/usr/bin/env python
#
# @package      hubzero-submit-distributor
# @file         distributor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2014 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
"""Distribute computation jobs to remote resources

   Generate a pair of script files for remote job submission.
   The first file is the batch submission script, the second is a shell script
   that wraps the application and generates time usage data.

   RUN AS FOLLOWS:
     distributor -j <jobId> -v <venue> -i <inputfile> -o <outputfile>
                 -n <nCpus> -N <ppn> -w <wallTime> -e <environment variable>
                 -m <manager>

   <jobId>                 is the job identifier
   <executable>            is the MPI enabled executable to be run.
   <inputfile>             is a file to be sent to the remote site
   <outputfile>            is a file to be retrieved from the remote site
   <nCpus>                 the number of processors to use
   <ppn>                   the processors/node to use
   <wallTime>              wall time limit for remote process
   <environment variable>  variable=value
   <venue>                 remote venue
   <manager>               multi-processor control manager

                                                  | cpu time    |
                                                  | from time() |
                                                  |             |
                                      |  waiting  | real(wall)  |
                                      |   time    |    time     |
                                      |           |             |
     |------------|---------|---------|-----------|-------------|----------|----
  simulate     condor     site      files     simulation    simulation   files
   button        job    selected    staged     started        ended      staged
  pressed    submitted               in                                   out

   """
import os
import sys
import logging
from logging.handlers import SocketHandler,SysLogHandler

from hubzero.submit.JobDistributor import JobDistributor

LOGDIRECTORY      = os.path.join(os.sep,'var','log','submit','distributor')
HUBLOGFILE        = "distributor.log"
APPLICATIONLOGGER = logging.getLogger('')

CONFIGURATIONDIRECTORY = os.path.join(os.sep,'etc','submit')
CONFIGURATIONFILE      = 'distributor.conf'
CONFIGFILEPATH         = os.path.join(CONFIGURATIONDIRECTORY,CONFIGURATIONFILE)

JOBMONITORHOST    = ""
JOBMONITORPORT    = 5727
PROBEMONITORHOST  = ""
PROBEMONITORPORT  = 5728
TUNNELMONITORHOST = ""
TUNNELMONITORPORT = 5729
CLOUDMONITORHOST  = ""
CLOUDMONITORPORT  = 5726

def openLogger(logDirectory,
               hubLogFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logSyslogHandler = SysLogHandler(address="/dev/log",facility=SysLogHandler.LOG_LOCAL6)
   emptyFilter = EmptyFilter()
   logSyslogHandler.addFilter(emptyFilter)
   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logSyslogHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logSyslogHandler)


def distributor(configFilePath,
                logDirectory,
                jobMonitorHost,jobMonitorPort,
                probeMonitorHost,probeMonitorPort,
                tunnelMonitorHost,tunnelMonitorPort,
                cloudMonitorHost,cloudMonitorPort):
   jobDistributor = JobDistributor(configFilePath,logDirectory,
                                   jobMonitorHost,jobMonitorPort,
                                   probeMonitorHost,probeMonitorPort,
                                   tunnelMonitorHost,tunnelMonitorPort,
                                   cloudMonitorHost,cloudMonitorPort)
   if jobDistributor.configure():
      exitCode = jobDistributor.run()
   else:
      exitCode = 1

   sys.exit(exitCode)


if __name__ == '__main__':
   openLogger(LOGDIRECTORY,HUBLOGFILE)

   distributor(CONFIGFILEPATH,LOGDIRECTORY,
               JOBMONITORHOST,JOBMONITORPORT,
               PROBEMONITORHOST,PROBEMONITORPORT,
               TUNNELMONITORHOST,TUNNELMONITORPORT,
               CLOUDMONITORHOST,CLOUDMONITORPORT)


