# @package      hubzero-submit-distributor
# @file         VenueMechanismLocal.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import sys
import os
import re
import stat
import shutil
import time
import datetime
import logging

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.JobOutput              import JobOutput
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore
from hubzero.submit.RemoteBatchCONDOR      import RemoteBatchCONDOR
from hubzero.submit.RemoteBatchRUNJOB      import RemoteBatchRUNJOB
from hubzero.submit.RemoteBatchFACTORY     import RemoteBatchFACTORY
from hubzero.submit.RemoteBatchPEGASUS     import RemoteBatchPEGASUS
from hubzero.submit.RemoteBatchAppScript   import RemoteBatchAppScript
from hubzero.submit.RemoteBatchPBS         import RemoteBatchPBS
from hubzero.submit.RemoteBatchSGE         import RemoteBatchSGE
from hubzero.submit.RemoteInstantAppScript import RemoteInstantAppScript
from hubzero.submit.RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from hubzero.submit.RemoteRunjobAppScript  import RemoteRunjobAppScript
from hubzero.submit.RemoteWorkflowPEGASUS  import RemoteWorkflowPEGASUS

class VenueMechanismLocal(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                session,
                distributorPid,
                batchCommands,
                isParametric,
                runName,
                localJobId,
                instanceId,
                destination,
                enteredCommand,
                gridsite,
                stageInTarFile,
                transferExecutable,
                executable,
                appsAccessInfo,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                x509SubmitProxy,
                disableProbeCheck,
                disableStateCheck,
                timeHistoryLogs,
                useSetup,
                pegasusVersion,
                pegasusHome):
      VenueMechanismCore.__init__(self,timeHistoryLogs,siteInfo,managerInfo,remoteMonitors,
                                       isMultiCoreRequest,nCpus,nNodes,ppn)

      self.logger                           = logging.getLogger(__name__)
      self.jobOutput                        = JobOutput()
      self.venueMechanism                   = 'local'
      self.hubUserName                      = hubUserName
      self.hubUserId                        = hubUserId
      self.session                          = session
      self.distributorPid                   = distributorPid
      remoteScriptsDirectory = siteInfo['remoteBinDirectory'].replace('$','\$')
      self.receiveInput                     = os.path.join(remoteScriptsDirectory,batchCommands['receiveInput'])
      self.submitBatchJob                   = os.path.join(remoteScriptsDirectory,batchCommands['submitBatchJob'])
      self.postProcessJob                   = os.path.join(remoteScriptsDirectory,batchCommands['postProcessJob'])
      self.transmitResults                  = os.path.join(remoteScriptsDirectory,batchCommands['transmitResults'])
      self.cleanupJob                       = os.path.join(remoteScriptsDirectory,batchCommands['cleanupJob'])
      self.killBatchJob                     = os.path.join(remoteScriptsDirectory,batchCommands['killBatchJob'])
      self.isParametric                     = isParametric
      self.runName                          = runName
      self.localJobId                       = localJobId
      if instanceId.startswith('WF;'):
         self.instanceId                    = ""
         self.nInstances                    = int(instanceId.split(';')[1])
      else:
         self.instanceId                    = instanceId
         self.nInstances                    = 0
      self.remoteBatchSystem                = siteInfo['remoteBatchSystem']
      if self.remoteBatchSystem == 'RUNJOB':
         self.destination                   = 'grid'
      else:
         self.destination                   = destination
      self.venue                            = siteInfo['venue']
      self.enteredCommand                   = enteredCommand
      self.gridsite                         = gridsite
      self.siteMonitorDesignator            = siteInfo['siteMonitorDesignator']
      self.logUserRemotely                  = siteInfo['logUserRemotely']
      self.sharedUserSpace                  = siteInfo['sharedUserSpace']
      self.stageFiles                       = siteInfo['stageFiles']
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile                = stageInTarFile
      else:
         self.stageInTarFile                = stageInTarFile + '.gz'
      self.transferExecutable               = transferExecutable
      self.executable                       = executable
      self.appsAccessInfo                   = appsAccessInfo
      self.stdinput                         = stdinput
      if isinstance(arguments,list):
         self.arguments                     = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments              += '"%s" ' % (argument)
            else:
               self.arguments              += "%s " % (argument)
      else:
         self.arguments                     = arguments
      self.useEnvironment                   = useEnvironment
      self.environment                      = environment
      self.remoteBatchQueue                 = siteInfo['remoteBatchQueue']
      self.pegasusTemplates                 = siteInfo['pegasusTemplates']
      self.executableClassificationsAllowed = siteInfo['executableClassificationsAllowed']
      self.wallTime                         = wallTime
      self.quotaLimit                       = quotaLimit
      self.useSetup                         = useSetup
      self.pegasusVersion                   = pegasusVersion
      self.pegasusHome                      = pegasusHome
      self.x509SubmitProxy                  = x509SubmitProxy
      self.disableProbeCheck                = disableProbeCheck
      self.disableStateCheck                = disableStateCheck
      self.batchLogPath                     = ""
      self.appScriptPath                    = ""
      self.batchScriptPath                  = ""
      self.nodeFilePath                     = ""
      self.remoteJobIdNumber                = ""

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob           = False
         self.instanceDirectory    = os.getcwd()
      else:
         self.isBatchJob           = True
         if self.instanceId:
            self.instanceDirectory = os.path.join(os.getcwd(),self.runName,self.instanceId)
         else:
            self.instanceDirectory = os.path.join(os.getcwd(),self.runName)
         if not os.path.isdir(self.instanceDirectory):
            os.makedirs(self.instanceDirectory)

      if self.remoteBatchSystem == 'RUNJOB':
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.workingDirectory        = "%s_%s_%s" % (str(epoch),localJobId,self.instanceId)
         self.scratchDirectory        = self.workingDirectory
      else:
         self.workingDirectory        = os.getcwd()
         if siteInfo['remoteScratchDirectory']:
            epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
            self.scratchDirectory     = os.path.join(siteInfo['remoteScratchDirectory'],"%s_%s" % (str(epoch),localJobId))
            if self.instanceId:
               self.scratchDirectory += "_%s" % (self.instanceId)
         else:
            self.scratchDirectory     = self.instanceDirectory


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      self.logger.log(logging.INFO,getLogMessage("workingDirectory " + self.workingDirectory))
      if   self.remoteBatchSystem == 'PBS':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS(self.hubUserName,self.hubUserId,
                                      self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,self.wallTime,
                                      self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'SGE':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchSGE(self.hubUserName,self.hubUserId,
                                      self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                      self.transferExecutable,
                                      self.executable,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                      self.nNodes,self.ppn,self.wallTime,
                                      self.quotaLimit,
                                      self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'CONDOR':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchCONDOR(self.hubUserName,self.hubUserId,
                                         self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                         appScriptName,self.environment,
                                         self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                         self.wallTime)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'RUNJOB':
         remoteAppScript = RemoteRunjobAppScript(self.hubUserName,self.hubUserId,
                                                 self.runName,self.localJobId,self.instanceId,self.isParametric,
                                                 self.transferExecutable,self.executable,self.stdinput,
                                                 self.arguments,self.environment,
                                                 self.isMultiCoreRequest,self.managerInfo,
                                                 self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchRUNJOB(self.hubUserName,self.hubUserId,
                                         self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                         appScriptName,self.environment,self.transferExecutable,self.executable,self.arguments,
                                         self.isMultiCoreRequest,self.managerInfo,
                                         self.nCpus,self.ppn,self.wallTime,
                                         self.gridsite,
                                         self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                         self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'FACTORY':
         remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,False,
                                                self.transferExecutable,self.executable,
                                                self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchFACTORY(self.hubUserName,self.hubUserId,
                                          self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                          appScriptName,self.environment,
                                          self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                          self.wallTime)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
      elif self.remoteBatchSystem == 'PEGASUS':
         if self.isParametric:
            if self.instanceId != "":
               remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                      self.runName,self.localJobId,self.instanceId,True,
                                                      self.transferExecutable,self.executable,
                                                      self.stdinput,self.arguments,
                                                      self.useEnvironment,self.environment,
                                                      self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                      self.timeHistoryLogs)
               appScriptName,appScript = remoteAppScript.buildAppScript()
               batchLogName    = ""
               batchScriptName = ""
               batchScript     = ""
               nodeFileName    = ""
               nodeList        = []
            else:
               appScriptName = ""
               appScript     = ""
               remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
               workflow = RemoteWorkflowPEGASUS(self.hubUserName,self.hubUserId,
                                                self.runName,self.localJobId,self.instanceId,
                                                self.instanceDirectory,remoteScratchDirectory,
                                                self.useSetup,self.pegasusVersion,self.pegasusHome,
                                                self.pegasusTemplates,self.x509SubmitProxy,self.gridsite,
                                                self.timeHistoryLogs)
               batchScriptName,batchScript = workflow.buildWorkflowScript()
               batchLogName = ""
               nodeFileName = ""
               nodeList     = []
         else:
            remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,
                                                   self.runName,self.localJobId,self.instanceId,False,
                                                   self.transferExecutable,self.executable,
                                                   self.stdinput,self.arguments,
                                                   self.useEnvironment,self.environment,
                                                   self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                   self.timeHistoryLogs)
            appScriptName,appScript = remoteAppScript.buildAppScript()
            remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
            remoteBatch = RemoteBatchPEGASUS(self.hubUserName,self.hubUserId,
                                             self.runName,self.localJobId,self.instanceId,
                                             self.instanceDirectory,remoteScratchDirectory,
                                             self.useSetup,self.pegasusVersion,self.pegasusHome,
                                             appScriptName,
                                             self.executable,self.arguments,
                                             self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                             self.x509SubmitProxy,self.gridsite,self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
            nodeFileName,nodeList = remoteBatch.getBatchNodeList()
            userDaxExecutables = remoteBatch.getUserDaxExecutables()
            for userDaxExecutable in userDaxExecutables:
               submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(userDaxExecutable)
               if not submissionAllowed:
                  message = "Access to %s denied for %s" % (userDaxExecutable,self.hubUserName)
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self._writeToStderr(message + "\n")
                  if not exitCode:
                     exitCode = 9
               if (not '*' in self.executableClassificationsAllowed) and \
                  (not executableClassification in self.executableClassificationsAllowed):
                  message = "The selected site does not meet the specified\n" + \
                            "application access requirements.  Please select another site."
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self._writeToStderr(message + "\n")
                  if not exitCode:
                     exitCode = 10
      elif self.remoteBatchSystem == 'SCRIPT':
         remoteAppScript = RemoteInstantAppScript(self.hubUserName,self.hubUserId,
                                                  self.runName,self.localJobId,self.instanceId,False,
                                                  self.workingDirectory,
                                                  self.transferExecutable,self.executable,
                                                  self.stdinput,self.arguments,
                                                  self.useEnvironment,self.environment,
                                                  self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                                  self.timeHistoryLogs)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,
                                           self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                           self.isMultiCoreRequest,self.siteInfo,self.managerInfo,
                                           self.nNodes,self.ppn,
                                           self.timeHistoryLogs)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()

      if batchLogName != "":
         self.batchLogPath    = os.path.join(self.instanceDirectory,batchLogName)
      if appScriptName != "":
         self.appScriptPath   = os.path.join(self.instanceDirectory,appScriptName)
      if batchScriptName != "":
         self.batchScriptPath = os.path.join(self.instanceDirectory,batchScriptName)
      if nodeFileName != "":
         self.nodeFilePath    = os.path.join(self.instanceDirectory,nodeFileName)

      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = 1
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

      if batchScript != "":
         fpBatchScript = open(self.batchScriptPath,'w')
         if fpBatchScript:
            fpBatchScript.write(batchScript)
            fpBatchScript.close()
            if   self.remoteBatchSystem == 'SCRIPT':
               os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            elif self.remoteBatchSystem == 'RUNJOB':
               os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            elif self.remoteBatchSystem == 'PEGASUS':
               os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.batchScriptPath)))
            self._writeToStderr("could not open %s for writing\n" % (self.batchScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if appScript != "":
         fpAppScript = open(self.appScriptPath,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(self.appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            scriptFiles.append(self.appScriptPath)
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.appScriptPath)))
            self._writeToStderr("could not open %s for writing\n" % (self.appScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if len(nodeList) > 0:
         fpNodes = open(self.nodeFilePath,'w')
         if fpNodes:
            fpNodes.write('\n'.join(nodeList)+'\n')
            fpNodes.close()
         else:
            self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (self.nodeFilePath)))
            self._writeToStderr("could not open %s for writing\n" % (self.nodeFilePath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      if self.remoteBatchSystem != 'RUNJOB':
         if self.remoteBatchQueue != "":
            self.jobStatistics[self.jobIndex]['venue'] = self.remoteBatchQueue + '@' + self.venue
         else:
            self.jobStatistics[self.jobIndex]['venue'] = self.venue

         remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
         command = self.receiveInput + " " + self.instanceDirectory + " " + \
                                             remoteScratchDirectory + " " + \
                                             os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampTransferred'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         self.logger.log(logging.INFO,getLogMessage(stdOutput))

         if exitStatus:
            self.filesSent = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId:
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              fpStd = open(stdFile,'a')
#              if fpStd:
#                 fpStd.write(stdOutput)
#                 fpStd.close()
            if stdError != "":
               if self.instanceId:
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               fpStd = open(stdFile,'a')
               if fpStd:
                  fpStd.write(stdError)
                  fpStd.close()

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      if self.remoteBatchSystem == 'SCRIPT':
         if self.batchScriptPath != "":
            command = self.batchScriptPath
         else:
            command = self.appScriptPath
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         exitStatus,stdOutput,stdError = self.executeCommand(command,True)
         self.jobSubmitted = True
         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         commandArgs = []
         remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
         if self.remoteBatchSystem == 'RUNJOB':
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(self.instanceDirectory)
            commandArgs.append(remoteScratchDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            command = ' '.join(commandArgs)
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeCommand(command)
         else:
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(self.instanceDirectory)
            commandArgs.append(remoteScratchDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            if self.logUserRemotely:
               commandArgs.append(self.hubUserName)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = ' '.join(commandArgs)
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeLaunchCommand(command)

         if not exitStatus:
            remoteJobId = stdOutput.strip()
            self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
            try:
               if   self.remoteBatchSystem == 'PBS':
                  self.remoteJobIdNumber = remoteJobId.split('.')[0]
               elif self.remoteBatchSystem == 'SGE':
#                 Your job 16 ("JOB_00146752_01") has been submitted
                  self.remoteJobIdNumber = remoteJobId.split()[2]
               elif self.remoteBatchSystem == 'CONDOR':
#                 Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
                  self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
               elif self.remoteBatchSystem == 'RUNJOB':
#                 Your job has been submitted with id 19.0
                  self.remoteJobIdNumber = re.search('with id [0-9]+',remoteJobId).group().split()[2] + ".0"
               elif self.remoteBatchSystem == 'FACTORY':
#                 Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 105.
                  self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
               elif self.remoteBatchSystem == 'PEGASUS':
#                 1 job(s) submitted to cluster 105.
                  self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"

               self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
            except:
               exitStatus = 248

            if not exitStatus:
               self.jobSubmitted = True
               if self.remoteBatchSystem == 'PEGASUS':
                  self.remoteJobMonitor.postPegasusWorkflowSubmission(self.siteMonitorDesignator,self.remoteJobIdNumber,
                                                                      str(self.hubUserId),self.enteredCommand,
                                                                      self.localJobId,self.nInstances,self.destination,
                                                                      self.runName,self.nCpus,self.distributorPid)
               else:
                  self.remoteJobMonitor.postJobSubmission(self.siteMonitorDesignator,self.remoteJobIdNumber,
                                                          str(self.hubUserId),self.enteredCommand,
                                                          self.localJobId,self.instanceId,self.destination,
                                                          self.runName,self.nCpus,self.distributorPid)
            else:
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
         if stdOutput != "":
            if self.instanceId:
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdOutput)
               fpStd.close()
         if stdError != "":
            if self.instanceId:
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['siteMonitorDesignator'] = self.siteMonitorDesignator
      waitForJobInfo['remoteJobId']           = self.remoteJobIdNumber
      waitForJobInfo['knownSite']             = self.destination

      return(waitForJobInfo)


   def postProcess(self):
      if not self.jobPostProcessed:
         if self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and not self.instanceId) or not self.isParametric:
                  if not self.sharedUserSpace:
                     remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
                  else:
                     remoteWorkingDirectory = self.instanceDirectory
                  remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
                  command = self.postProcessJob + " " + remoteWorkingDirectory + " " + \
                                                        remoteScratchDirectory + " " + \
                                                        "PEGASUS"
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

                  self.jobPostProcessed = True


   def retrieveFiles(self):
      exitStatus = 0

      if self.remoteBatchSystem != 'RUNJOB':
         if self.sharedUserSpace and self.stageFiles:
            outTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
            stageOutTarFiles = []
            stageOutTarFiles.append(os.path.join(self.instanceDirectory,"%s.gz" % (outTarFile)))
            stageOutTarFiles.append(os.path.join(self.instanceDirectory,outTarFile))
            stageOutTarFiles.append("%s.gz" % (outTarFile))
            stageOutTarFiles.append(outTarFile)
            stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),"%s.gz" % (outTarFile)))
            stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),outTarFile))
            for stageOutTarFile in stageOutTarFiles:
               if os.path.isfile(stageOutTarFile):
                  if stageOutTarFile.endswith('.gz'):
                     command = "tar xzmf " + stageOutTarFile + \
                                         " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
                  else:
                     command = "tar xmf " + stageOutTarFile + \
                                        " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  exitStatus,stdOutput,stdError = self.executeCommand(command)
                  if exitStatus == 0:
                     os.remove(stageOutTarFile)
                  else:
                     self.logger.log(logging.ERROR,getLogMessage(stdOutput))
                  break

            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 if self.instanceId:
#                    stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#                 else:
#                    stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#                 fpStd = open(stdFile,'a')
#                 if fpStd:
#                    fpStd.write(stdOutput)
#                    fpStd.close()
               if stdError != "":
                  if self.instanceId:
                     stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                  else:
                     stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                  fpStd = open(stdFile,'a')
                  if fpStd:
                     fpStd.write(stdError)
                     fpStd.close()

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self):
      if not self.filesCleanedup:
         if   self.remoteBatchSystem == 'RUNJOB':
            workDirectoryName = "runjob-workdir-%s_%s" % (self.localJobId,self.instanceId)
            workDirectoryName = os.path.join(self.instanceDirectory,workDirectoryName)
            if os.path.isdir(workDirectoryName):
               command = "cleanupkilled.sh " + self.remoteBatchSystem + " " + \
                                               workDirectoryName + " " + \
                                               self.timeHistoryLogs['jobGridJobId']
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))
            if self.jobSubmitted:
               self.recordGridResourceUse()
               self.logGridJobId()
               self.logGridHistories()
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
         elif self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and not self.instanceId) or not self.isParametric:
                  remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
                  if self.isParametric:
                     self.jobOutput.getPegasusStdTimeFiles(self.instanceDirectory,
                                                           remoteScratchDirectory,
                                                           self.timeHistoryLogs['timeResults'])
                  else:
                     self.jobOutput.processPegasusFiles(self.instanceDirectory,
                                                        remoteScratchDirectory,
                                                        self.timeHistoryLogs['timeResults'])

                  command = self.cleanupJob + " " + self.instanceDirectory + " " + \
                                                    remoteScratchDirectory + " " + \
                                                    self.remoteBatchSystem
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

            workDirectoryName = os.path.join(self.instanceDirectory,'work')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            inProcessDirectoryName = os.path.join(self.instanceDirectory,'InProcessResults')
            if os.path.isdir(inProcessDirectoryName):
               shutil.rmtree(inProcessDirectoryName,True)
            for ftype in '.dax','_sites.xml','.pegasusrc','_tc.txt':
               pegasusFile = os.path.join(self.instanceDirectory,"%s_%s%s" % (self.localJobId,self.instanceId,ftype))
               if os.path.isfile(pegasusFile):
                  os.remove(pegasusFile)
               else:
                  pegasusFile = os.path.join(self.instanceDirectory,"%s%s" % (self.localJobId,ftype))
                  if os.path.isfile(pegasusFile):
                     os.remove(pegasusFile)
            pegasusFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
            if os.path.exists(pegasusFile):
               if(os.path.getsize(pegasusFile) == 0):
                  os.remove(pegasusFile)
            stageInTarFile = os.path.join(self.instanceDirectory,self.stageInTarFile)
            if os.path.isfile(stageInTarFile):
               os.remove(stageInTarFile)

         if self.batchScriptPath:
            if os.path.isfile(self.batchScriptPath):
               os.remove(self.batchScriptPath)
         if self.appScriptPath:
            if os.path.isfile(self.appScriptPath):
               os.remove(self.appScriptPath)
         if self.nodeFilePath:
            if os.path.isfile(self.nodeFilePath):
               os.remove(self.nodeFilePath)
         if self.batchLogPath:
            if os.path.isfile(self.batchLogPath):
               os.remove(self.batchLogPath)
         if self.sharedUserSpace and self.stageFiles:
            stageOutTarFile = "%s_%s_output.tar.gz" % (self.localJobId,self.instanceId)
            if os.path.isfile(stageOutTarFile):
               os.remove(stageOutTarFile)
            else:
               stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
               if os.path.isfile(stageOutTarFile):
                  os.remove(stageOutTarFile)

         self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      if self.remoteJobIdNumber:
         if self.remoteBatchSystem == 'RUNJOB':
            self.logGridHistories()

         command = self.killBatchJob + " " + self.remoteJobIdNumber + " " + self.remoteBatchSystem
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 7 | signalNumber

      self.scriptsKilled = True


