# @package      hubzero-submit-distributor
# @file         InfosInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2014-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class InfosInfo:
   def __init__(self,
                configFilePath):
      self.logger = logging.getLogger(__name__)
      self.infos = {}

      infoPattern     = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      infoName        = ""

      if os.path.exists(configFilePath):
         try:
            fpInfo = open(configFilePath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   infoPattern.match(record):
                        infoName = infoPattern.match(record).group(2)
                        self.infos[infoName] = {'directory':os.path.join(os.sep,'etc','submit'),
                                                'filename':"",
                                                'topDirectory':""
                                               }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.infos[infoName]:
                           if   isinstance(self.infos[infoName][key],list):
                              self.infos[infoName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.infos[infoName][key],bool):
                              self.infos[infoName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.infos[infoName][key],float):
                              self.infos[infoName][key] = float(value)
                           elif isinstance(self.infos[infoName][key],int):
                              self.infos[infoName][key] = int(value)
                           elif isinstance(self.infos[infoName][key],dict):
                              try:
                                 sampleKey   = self.infos[infoName][key].keys()[0]
                                 sampleValue = self.infos[infoName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.infos[infoName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.infos[infoName][key][dictKey] = dictValue
                           else:
                              self.infos[infoName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for info %s" % (key,value,infoName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Info configuration file %s could not be read" % (configFilePath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Info configuration file %s could not be opened" % (configFilePath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Info configuration file %s is missing" % (configFilePath)))


   def getInfoPath(self,
                   infoName):
      infoPath = ""
      if infoName in self.infos:
         info = self.infos[infoName]
         if info['filename']:
            infoPath = os.path.join(info['directory'],info['filename'])
         else:
            infoPath = os.path.join(info['directory'],info['topDirectory'])

      return(infoPath)


