# @package      hubzero-submit-distributor
# @file         RemoteWorkflowPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

try:
   from Pegasus.DAX3 import *
except:
   from hubzero.submit.DAX3 import *

from hubzero.submit.LogMessage        import getLogIDMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteWorkflowPEGASUS:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                jobDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                pegasusTemplates,
                x509SubmitProxy,
                sshIdentityPath,
                gridsite,
                wallTime,
                submissionScriptsInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.jobDirectory          = jobDirectory
      self.scratchDirectory      = scratchDirectory

      self.daxPath               = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable            = 'pegasus-plan'
      self.arguments             = "--dax %s" % (self.daxPath)

      self.useSetup              = useSetup
      self.pegasusVersion        = pegasusVersion
      self.pegasusHome           = pegasusHome
      self.pegasusTemplates      = pegasusTemplates
      self.x509SubmitProxy       = x509SubmitProxy
      self.sshIdentityPath       = sshIdentityPath
      self.gridsite              = gridsite
      self.wallTime              = wallTime
      self.submissionScriptsInfo = submissionScriptsInfo
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timeResults           = timeHistoryLogs['timeResults']


   def __buildWorkflowScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Workflow','PEGASUS','plan')

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if os.path.exists(self.pegasusTemplates['rc']):
         try:
            fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
            try:
               searchString = os.path.basename(self.executable) + '.arguments'
               settings = fpRCTemplate.readlines()
               for setting in settings:
#pegasus-plan.arguments = --nocleanup
                  if setting.count(searchString) > 0:
                     try:
                        parameter,value = setting.split('=')
                        auxiliaryArguments.append(value.strip())
                     except:
                        pass
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (self.pegasusTemplates['rc'])))
            finally:
               fpRCTemplate.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.pegasusTemplates['rc'])))

      substitutions = {}
      substitutions["USESETUP"]          = self.useSetup
      substitutions["PEGASUSVERSION"]    = self.pegasusVersion
      substitutions["RUNNAME"]           = self.runName
      substitutions["JOBID"]             = self.localJobId
      substitutions["INSTANCEID"]        = self.instanceId
      substitutions["INSTANCEDIRECTORY"] = self.jobDirectory
      substitutions["SCRATCHDIRECTORY"]  = self.scratchDirectory
      substitutions["EXECUTABLE"]        = self.executable
      substitutions["GRIDSITE"]          = self.gridsite
      arguments = ' '.join(auxiliaryArguments)
      substitutions["ARGUMENTS"]         = arguments.strip()
      arguments = ' '.join(discardedArguments)
      substitutions["DISCARDED"]         = arguments.strip()
      arguments = ' '.join(notRecognizedArguments)
      substitutions["NOTRECOGNIZED"]     = arguments.strip()
      substitutions["TS_START"]          = self.timestampStart
      substitutions["TS_FINISH"]         = self.timestampFinish
      substitutions["TIME_RESULTS"]      = self.timeResults
      substitutions["HUBUSERNAME"]       = self.hubUserName
      substitutions["HUBUSERID"]         = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildCatalogs(self):
      re_instanceIdEmpty = re.compile("_@@{INSTANCEID}")
      substitutions = {}
      substitutions["SUBMITTERCLASS"]       = str(self.submitterClass)
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEDIRECTORY"]    = self.jobDirectory
      substitutions["BASESCRATCHDIRECTORY"] = os.path.basename(self.scratchDirectory)
      substitutions["SCRATCHDIRECTORY"]     = self.scratchDirectory
      if self.x509SubmitProxy != "":
         substitutions["X509SUBMITPROXY"]   = self.x509SubmitProxy
      else:
         substitutions["X509SUBMITPROXY"]   = os.path.join(os.sep,'tmp','hub-proxy.%s' % (self.hubUserName))
      substitutions["PEGASUSHOME"]          = self.pegasusHome
      substitutions["SSHPRIVATEKEYPATH"]    = self.sshIdentityPath

      for templateType in self.pegasusTemplates:
         pegasusTemplate = self.pegasusTemplates[templateType]
         if pegasusTemplate != "":
            try:
               fpTemplate = open(pegasusTemplate,'r')
               try:
                  template = ''.join(fpTemplate.readlines())
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (pegasusTemplate)))
               else:
                  if self.instanceId != "0":
                     substitutions["INSTANCEID"] = self.instanceId
                  else:
                     template = re_instanceIdEmpty.sub("",template)
                     substitutions["INSTANCEID"] = "00"

                  pegasusTemplate = ParameterTemplate(template)
                  try:
                     pegasusText = pegasusTemplate.substitute_recur(substitutions)
                  except KeyError,err:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
                  except TypeError,err:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pegasus template substitution failed:\n%s\n" % (template)))

                  if   templateType == 'rc':
                     pegasusFile = "%s.pegasusrc" % (self.localJobId)
                  elif templateType == 'sites':
                     pegasusFile = "%s_sites.xml" % (self.localJobId)
                  elif templateType == 'tc':
                     pegasusFile = "%s_tc.txt" % (self.localJobId)
                  pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
                  try:
                     fpPegasusFile = open(pegasusPath,'w')
                     try:
                        fpPegasusFile.write(pegasusText)
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (pegasusPath)))
                     finally:
                        fpPegasusFile.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusPath)))
               finally:
                  fpTemplate.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusTemplate)))


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(self.jobDirectory)
      for dirFile in dirFiles:
         dirPath = os.path.join(self.jobDirectory,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      dax = ADAG(self.localJobId)

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      reStageInTarFile = re.compile("[0-9_]+_input.tar.gz")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(self.jobDirectory,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles  = filter(reAppScriptPath.search,dirFiles)
         stageInTarFiles = filter(reStageInTarFile.search,dirFiles)

         if len(appScriptFiles) == 1 and len(stageInTarFiles) == 1:
            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            appScriptPath = "file://" + appScriptPath
            appScript = Executable(name=appScriptName,arch=Arch.X86_64,os=OS.LINUX,installed=False)
            appScript.addPFN(PFN(appScriptPath,"local"))
            dax.addExecutable(appScript)

            stageInTarFile = stageInTarFiles[0]
            inputTarPath = "file://" + os.path.join(instanceDirectory,stageInTarFile)
            inputTar = File(stageInTarFile)
            inputTar.addPFN(PFN(inputTarPath,"local"))
            dax.addFile(inputTar)

            stageOutTarFile = stageInTarFile.replace('input','output')
            outputTar = File(stageOutTarFile)

            jobId = "%s_%s" % (self.runName,instanceId)
            job = Job(name=appScriptName,id=jobId)
            job.addProfile(Profile(Namespace.HINTS, key='executionPool',value="%s" % (self.gridsite.split(',')[0])))
            job.addProfile(Profile(Namespace.GLOBUS,key='maxwalltime',value=self.wallTime))
            job.uses(inputTar,link=Link.INPUT,executable=False)
            job.uses(outputTar,link=Link.OUTPUT,register=True,transfer=True)
            dax.addJob(job)

      try:
         fpDAX = open(self.daxPath,'w')
         try:
            dax.writeXML(fpDAX)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.daxPath)))
         finally:
            fpDAX.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.daxPath)))


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()
      workflowScriptExecutable = True

      return(workflowScriptName,workflowScript,workflowScriptExecutable)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# 1 job(s) submitted to cluster 105.
         remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


