# @package      hubzero-submit-distributor
# @file         IdentitiesManager.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import hashlib
import signal
import subprocess
import select
import pwd
import time
import json
import logging

from hubzero.submit.LogMessage       import getLogIDMessage as getLogMessage
from hubzero.submit.BoundConnections import BoundConnections
from hubzero.submit.DaemonsInfo      import DaemonsInfo
from hubzero.submit.InfosInfo        import InfosInfo
from hubzero.submit.IdentitiesInfo   import IdentitiesInfo
from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

GENERATEGRIDPROXYFILE    = 'generate-grid-proxy.sh'
GENERATEVOMSPROXYFILE    = 'generate-voms-proxy.sh'
PERSONALIZECOPYPROXYFILE = 'personalize-copy-proxy.sh'
PERSONALIZESAMLPROXYFILE = 'personalize-saml-proxy.sh'
REFRESHCONDORPROXYFILE   = 'refresh-condor-proxy.sh'
REFRESHJOBMONPROXYFILE   = 'refresh-jobmon-proxy.sh'
GENUSERIDFILE            = 'genuserid.sh'
GENUSERPKIFILE           = 'genuserpki.sh'

class IdentitiesManager(BoundConnections):
   EVENTNONE               = 1 << 0
   EVENTGENERATEGRIDPROXY  = 1 << 1
   EVENTGENERATEVOMSPROXY  = 1 << 2
   EVENTREFRESHCONDORPROXY = 1 << 3
   EVENTREFRESHJOBMONPROXY = 1 << 4

   nextEventId = 0

   class Event:
      def __init__(self,
                   what,
                   identityName,
                   when):
         IdentitiesManager.nextEventId += 1

         self.eventId      = IdentitiesManager.nextEventId
         self.what         = what
         self.identityName = identityName
         self.when         = when


      def show(self):
         return("%d %ld %d %s" % (self.eventId,self.when,self.what,self.identityName))


      def __cmp__(self,
                  other): 
         if   self.when < other.when:
            comparison = -1
         elif self.when > other.when:
            comparison = +1
         else:
            comparison = 0

         return(comparison)


   class EventQueue:
      def __init__(self):
         self.items = []


      def purge(self):
         del self.items
         self.items = []


      def isEmpty(self):
         return(self.items == [])


      def add(self,
              item):
         self.items.append(item)


      def pop(self):
         locMin = 0
         for i1 in range(1,len(self.items)):
            if self.items[i1] < self.items[locMin]:
               locMin = i1
         item = self.items[locMin]
         self.items[locMin:locMin+1] = []

         return(item)


      def peek(self):
         locMin = 0
         for i1 in range(1,len(self.items)):
            if self.items[i1] < self.items[locMin]:
               locMin = i1
         item = self.items[locMin]

         return(item)


      def getEventId(self,
                     eventId):
         item = None
         for i1 in range(0,len(self.items)):
            if self.items[i1].eventId == eventId:
               item = self.items[i1]
               self.items[i1:i1+1] = []
               break

         return(item)


      def show(self):
         report = []
         for i1 in range(0,len(self.items)):
            report.append(self.items[i1].show())
         reportMessage = '\n'.join(report)
         del report

         return(reportMessage)


   def __init__(self,
                configurationDirectory,
                daemonsConfigurationFile,
                infosConfigurationFile,
                identityDirectory,
                proxyBinDirectory,
                listenURI):
      BoundConnections.__init__(self,listenURI,logConnection=False)

      self.logger = logging.getLogger(__name__)

      self.configurationDirectory   = configurationDirectory
      self.daemonsConfigurationFile = daemonsConfigurationFile
      self.infosConfigurationFile   = infosConfigurationFile
      self.identityDirectory        = identityDirectory
      self.proxyBinDirectory        = proxyBinDirectory

      self.generateGridProxyPath    = os.path.join(self.proxyBinDirectory,GENERATEGRIDPROXYFILE)
      self.generateVomsProxyPath    = os.path.join(self.proxyBinDirectory,GENERATEVOMSPROXYFILE)
      self.personalizeCopyProxyPath = os.path.join(self.proxyBinDirectory,PERSONALIZECOPYPROXYFILE)
      self.personalizeSAMLProxyPath = os.path.join(self.proxyBinDirectory,PERSONALIZESAMLPROXYFILE)
      self.refreshCondorproxyPath   = os.path.join(self.proxyBinDirectory,REFRESHCONDORPROXYFILE)
      self.refreshJobmonProxyPath   = os.path.join(self.proxyBinDirectory,REFRESHJOBMONPROXYFILE)
      self.genUserIDPath            = os.path.join(self.proxyBinDirectory,GENUSERIDFILE)
      self.genUserPKIPath           = os.path.join(self.proxyBinDirectory,GENUSERPKIFILE)

      self.infosInfo      = None
      self.identitiesInfo = None
      self.settingInfo    = False

      self.remoteJobMonitor = None
      self.activeIdentities = []
      self.eventQueue       = self.EventQueue()
      self.terminating      = False


   def setInfo(self):
      errorInSetInfo = False
      if not self.settingInfo:
         self.settingInfo = True

         configFilePath = os.path.join(self.configurationDirectory,self.infosConfigurationFile)
         if self.infosInfo:
            del self.infosInfo
         self.infosInfo = InfosInfo(configFilePath)

         if self.identitiesInfo:
            del self.identitiesInfo
         self.identitiesInfo = IdentitiesInfo(self.infosInfo.getInfoPath('identities'))

         enabledIdentityNames = self.identitiesInfo.getEnabledIdentityNames()
         for identityName in enabledIdentityNames:
            self.activeIdentities.append(identityName)

         configFilePath = os.path.join(self.configurationDirectory,self.daemonsConfigurationFile)
         daemonsInfo  = DaemonsInfo(configFilePath)
         jobListenURI = daemonsInfo.getDaemonListenURI('jobMonitor','tcp')
         self.remoteJobMonitor = RemoteJobMonitor(jobListenURI)

         self.settingInfo = False

      return(errorInSetInfo)


   def terminate(self):
      if not self.terminating:
         self.closeListeningConnection()
         self.terminating = True


   def executeCommand(self,
                      commandArgs):
      child = subprocess.Popen(commandArgs,bufsize=0,
                               stdin=subprocess.PIPE,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      childPid   = child.pid
      childout   = child.stdout
      childoutFd = childout.fileno()
      childerr   = child.stderr
      childerrFd = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         ready = select.select(toCheck,[],[]) # wait for input

         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(childPid,0)
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.INFO,getLogMessage("%s failed w/ signal %d" % (commandArgs,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.INFO,getLogMessage("%s failed w/ exit code %d" % (commandArgs,err)))
         self.logger.log(logging.INFO,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def isEmpty(self):
      return(self.eventQueue.isEmpty())


   def purge(self):
      return(self.eventQueue.purge())


   def getNextEventTime(self):
      nextEventTime = -1
      if not self.eventQueue.isEmpty():
         event = self.eventQueue.peek()
         nextEventTime = event.when

      return(nextEventTime)


   def generateCommunityGridProxy(self,
                                  identityName):
      now = time.time()
      identityGridProxyInfo = self.identitiesInfo.getIdentityGridProxyInfo(identityName)
      if identityGridProxyInfo:
         generateProxyCommand = [self.generateGridProxyPath,
                                 identityGridProxyInfo['certificateDirectory'],
                                 identityGridProxyInfo['certFile'],
                                 identityGridProxyInfo['keyFile'],
                                 identityGridProxyInfo['communityProxyFile']]
         exitStatus,stdOutput,stdError = self.executeCommand(generateProxyCommand)
         if exitStatus:
            self.logger.log(logging.ERROR,getLogMessage("Identity manager %s proxy generation failed" % (identityName)))
            if stdOutput:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
            if stdError:
               self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
         else:
            self.logger.log(logging.INFO,getLogMessage("Identity manager %s proxy generated" % (identityName)))
            interval = 60.*self.identitiesInfo.getIdentityKeyValue(identityName,'communityRefreshInterval')
            generateProxyEvent = self.Event(self.EVENTGENERATEGRIDPROXY,
                                            identityName,
                                            now+interval)
            self.eventQueue.add(generateProxyEvent)


   def generateCommunityGridProxies(self):
      identitiesWithGridProxy = self.identitiesInfo.getIdentitiesWithKeyValue('proxyGenerator','grid')
      for identityName in identitiesWithGridProxy:
         self.generateCommunityGridProxy(identityName)


   def generateCommunityVomsProxy(self,
                                  identityName):
      now = time.time()
      identityVomsProxyInfo = self.identitiesInfo.getIdentityVomsProxyInfo(identityName)
      if identityVomsProxyInfo:
         generateProxyCommand = [self.generateVomsProxyPath,
                                 identityVomsProxyInfo['certificateDirectory'],
                                 identityVomsProxyInfo['certFile'],
                                 identityVomsProxyInfo['keyFile'],
                                 identityVomsProxyInfo['communityProxyFile']]
         exitStatus,stdOutput,stdError = self.executeCommand(generateProxyCommand)
         if exitStatus:
            self.logger.log(logging.ERROR,getLogMessage("Identity manager %s proxy generation failed" % (identityName)))
            if stdOutput:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
            if stdError:
               self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
         else:
            self.logger.log(logging.INFO,getLogMessage("Identity manager %s proxy generated" % (identityName)))
            interval = 60.*self.identitiesInfo.getIdentityKeyValue(identityName,'communityRefreshInterval')
            generateProxyEvent = self.Event(self.EVENTGENERATEVOMSPROXY,
                                            identityName,
                                            now+interval)
            self.eventQueue.add(generateProxyEvent)


   def generateCommunityVomsProxies(self):
      identitiesWithVomsProxy = self.identitiesInfo.getIdentitiesWithKeyValue('proxyGenerator','voms')
      for identityName in identitiesWithVomsProxy:
         self.generateCommunityVomsProxy(identityName)


   def generatePersonalProxyWithSAML(self,
                                     identityName,
                                     hubUserName):
      identityPath = ""
      try:
         hubUserId = pwd.getpwnam(hubUserName).pw_uid
      except:
         self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%s'" % (hubUserName)))
      else:
         gridshibHome = self.identitiesInfo.getIdentityKeyValue(identityName,'gridshibHome')

         userProxyFileTemplate = self.identitiesInfo.getIdentityKeyValue(identityName,'userProxyFile')
         userProxyFileTemplate = userProxyFileTemplate.replace('@@HUBUSERID',str(hubUserId))
         userProxyFile         = 'hub-proxy.' + userProxyFileTemplate.replace('@@HUBUSERNAME',hubUserName)

         generateProxyCommand = ['sudo','-u',hubUserName,
                                 self.personalizeSAMLProxyPath,gridshibHome,userProxyFile]
         exitStatus,stdOutput,stdError = self.executeCommand(generateProxyCommand)
         if exitStatus:
            self.logger.log(logging.ERROR,getLogMessage("Identity manager %s, user %s proxy generation failed" % \
                                                                                    (identityName,hubUserName)))
            if stdOutput:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
            if stdError:
               self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
         else:
            self.logger.log(logging.INFO,getLogMessage("Identity manager %s, user %s proxy generated" % \
                                                                           (identityName,hubUserName)))
            identityPath = os.path.join(self.identityDirectory,userProxyFile)

      return(identityPath)


   def generatePersonalProxyWithoutSAML(self,
                                        identityName,
                                        hubUserName):
      identityPath = ""
      try:
         hubUserId = pwd.getpwnam(hubUserName).pw_uid
      except:
         self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%s'" % (hubUserName)))
      else:
         certificateDirectory = self.identitiesInfo.getIdentityKeyValue(identityName,'certificateDirectory')
         communityProxyFile   = self.identitiesInfo.getIdentityKeyValue(identityName,'communityProxyFile')

         userProxyFileTemplate = self.identitiesInfo.getIdentityKeyValue(identityName,'userProxyFile')
         userProxyFileTemplate = userProxyFileTemplate.replace('@@HUBUSERID',str(hubUserId))
         userProxyFile         = 'hub-proxy.' + userProxyFileTemplate.replace('@@HUBUSERNAME',hubUserName)

         generateProxyCommand = ['sudo','-u',hubUserName,
                                 self.personalizeCopyProxyPath,certificateDirectory,communityProxyFile,userProxyFile]
         exitStatus,stdOutput,stdError = self.executeCommand(generateProxyCommand)
         if exitStatus:
            self.logger.log(logging.ERROR,getLogMessage("Identity manager %s, user %s proxy generation failed" % \
                                                                                    (identityName,hubUserName)))
            if stdOutput:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
            if stdError:
               self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
         else:
            self.logger.log(logging.INFO,getLogMessage("Identity manager %s, user %s proxy generated" % \
                                                                           (identityName,hubUserName)))
            identityPath = os.path.join(self.identityDirectory,userProxyFile)

      return(identityPath)


   def generatePermanentUserProxies(self):
      identityNames = self.identitiesInfo.getIdentitiesWithKeyValue('identityType','x509')
      for identityName in identityNames:
         permanentUsers = self.identitiesInfo.getIdentityKeyValue(identityName,'permanentUsers')
         if permanentUsers:
            personalizeMethod = self.identitiesInfo.getIdentityKeyValue(identityName,'personalizeMethod')
            if   personalizeMethod == 'saml':
               for permanentUser in permanentUsers:
                  self.generatePersonalProxyWithSAML(identityName,permanentUser)
            elif personalizeMethod == 'copy':
               for permanentUser in permanentUsers:
                  self.generatePersonalProxyWithoutSAML(identityName,permanentUser)


   def scheduleRefreshProxies(self):
      now = time.time()
      identityNames = self.identitiesInfo.getIdentitiesWithKeyValue('identityType','x509')
      for identityName in identityNames:
         refreshMethod = self.identitiesInfo.getIdentityKeyValue(identityName,'refreshMethod')
         interval = 60.*self.identitiesInfo.getIdentityKeyValue(identityName,'refreshInterval')
         if   refreshMethod == 'condorQ':
            scheduleRefreshEvent = self.Event(self.EVENTREFRESHCONDORPROXY,
                                              identityName,
                                              now+interval)
         elif refreshMethod == 'jobMonitor':
            scheduleRefreshEvent = self.Event(self.EVENTREFRESHJOBMONPROXY,
                                              identityName,
                                              now+interval)
         self.eventQueue.add(scheduleRefreshEvent)


   def refreshUserCondorProxies(self,
                                identityName):
      now = time.time()
      identityRefreshProxyInfo = self.identitiesInfo.getIdentityRefreshProxyInfo(identityName)
      if   identityRefreshProxyInfo['personalizeMethod'] == 'copy':
         refreshProxyCommand = [self.refreshCondorproxyPath,
                                identityRefreshProxyInfo['condorConfig'],
                                identityRefreshProxyInfo['condorQconstraint'],
                                ' '.join(identityRefreshProxyInfo['permanentUsers']),
                                identityRefreshProxyInfo['personalizeMethod'],
                                identityRefreshProxyInfo['certificateDirectory'],
                                identityRefreshProxyInfo['communityProxyFile'],
                                'hub-proxy.' + identityRefreshProxyInfo['userProxyFile']]
      elif identityRefreshProxyInfo['personalizeMethod'] == 'saml':
         refreshProxyCommand = [self.refreshCondorproxyPath,
                                identityRefreshProxyInfo['condorConfig'],
                                identityRefreshProxyInfo['condorQconstraint'],
                                ' '.join(identityRefreshProxyInfo['permanentUsers']),
                                identityRefreshProxyInfo['personalizeMethod'],
                                identityRefreshProxyInfo['gridshibHome'],
                                'hub-proxy.' + identityRefreshProxyInfo['userProxyFile']]

      exitStatus,stdOutput,stdError = self.executeCommand(refreshProxyCommand)
      if exitStatus:
         self.logger.log(logging.ERROR,getLogMessage("Identity manager %s refresh user proxies failed" % (identityName)))
         if stdOutput:
            self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
         if stdError:
            self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
      else:
         self.logger.log(logging.INFO,getLogMessage("Identity manager %s user proxies refreshed" % (identityName)))
         interval = 60.*identityRefreshProxyInfo['refreshInterval']
         scheduleRefreshEvent = self.Event(self.EVENTREFRESHCONDORPROXY,
                                           identityName,
                                           now+interval)
         self.eventQueue.add(scheduleRefreshEvent)


   def refreshUserJobMonitorProxies(self,
                                    identityName):
      now = time.time()
      usersWithActiveJobWithIdentity = self.remoteJobMonitor.queryIdentityActiveJobUsers(identityName)
      identityRefreshProxyInfo = self.identitiesInfo.getIdentityRefreshProxyInfo(identityName)
      if   identityRefreshProxyInfo['personalizeMethod'] == 'copy':
         refreshProxyCommand = [self.refreshJobmonProxyPath,
                                ' '.join(usersWithActiveJobWithIdentity),
                                ' '.join(identityRefreshProxyInfo['permanentUsers']),
                                identityRefreshProxyInfo['personalizeMethod'],
                                identityRefreshProxyInfo['certificateDirectory'],
                                identityRefreshProxyInfo['communityProxyFile'],
                                'hub-proxy.' + identityRefreshProxyInfo['userProxyFile']]
      elif identityRefreshProxyInfo['personalizeMethod'] == 'saml':
         refreshProxyCommand = [self.refreshJobmonProxyPath,
                                ' '.join(usersWithActiveJobWithIdentity),
                                ' '.join(identityRefreshProxyInfo['permanentUsers']),
                                identityRefreshProxyInfo['personalizeMethod'],
                                identityRefreshProxyInfo['gridshibHome'],
                                'hub-proxy.' + identityRefreshProxyInfo['userProxyFile']]

      exitStatus,stdOutput,stdError = self.executeCommand(refreshProxyCommand)
      if exitStatus:
         self.logger.log(logging.ERROR,getLogMessage("Identity manager %s refresh user proxies failed" % (identityName)))
         if stdOutput:
            self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
         if stdError:
            self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
      else:
         self.logger.log(logging.INFO,getLogMessage("Identity manager %s user proxies refreshed" % (identityName)))
         interval = 60.*identityRefreshProxyInfo['refreshInterval']
         scheduleRefreshEvent = self.Event(self.EVENTREFRESHJOBMONPROXY,
                                           identityName,
                                           now+interval)
         self.eventQueue.add(scheduleRefreshEvent)


   def generatePersonalSSHKeyCopy(self,
                                  identityName,
                                  hubUserName):
      identityPath = ""
      try:
         hubUserId = pwd.getpwnam(hubUserName).pw_uid
      except:
         self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%s'" % (hubUserName)))
      else:
         communityPrivateKeyPath = self.identitiesInfo.getIdentityKeyValue(identityName,'communityPrivateKeyPath')

         userPrivateKeyFileTemplate = self.identitiesInfo.getIdentityKeyValue(identityName,'userPrivateKeyFile')
         userPrivateKeyFileTemplate = userPrivateKeyFileTemplate.replace('@@HUBUSERID',str(hubUserId))
         userPrivateKeyFile         = 'hub-ssh.' + userPrivateKeyFileTemplate.replace('@@HUBUSERNAME',hubUserName)

         userPrivateKeyPath = os.path.join(self.identityDirectory,userPrivateKeyFile)
         if os.path.exists(userPrivateKeyPath):
            identityPath = userPrivateKeyPath
         else:
            generatePrivateKeyCommand = ['sudo','-u',hubUserName,
                                         self.genUserIDPath,communityPrivateKeyPath,userPrivateKeyFile]
            exitStatus,stdOutput,stdError = self.executeCommand(generatePrivateKeyCommand)
            if exitStatus:
               self.logger.log(logging.ERROR,getLogMessage("Identity manager %s, user %s private key generation failed" % \
                                                                                             (identityName,hubUserName)))
               if stdOutput:
                  self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
               if stdError:
                  self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
            else:
               self.logger.log(logging.INFO,getLogMessage("Identity manager %s, user %s private key generated" % \
                                                                                    (identityName,hubUserName)))
               identityPath = userPrivateKeyPath

      return(identityPath)


   def generatePersonalSSHKeyPair(self,
                                  identityName,
                                  hubUserName):
      identityPath = ""
      try:
         hubUserId = pwd.getpwnam(hubUserName).pw_uid
      except:
         self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%s'" % (hubUserName)))
      else:
         jobKeyFile = self.identitiesInfo.getIdentityKeyValue(identityName,'jobKeyFile')

         generateSSHKeyPairCommand = ['sudo','-u',hubUserName,
                                      self.genUserPKIPath,jobKeyFile]
         exitStatus,stdOutput,stdError = self.executeCommand(generateSSHKeyPairCommand)
         if exitStatus:
            self.logger.log(logging.ERROR,getLogMessage("Identity manager %s, user %s key pair generation failed" % \
                                                                                       (identityName,hubUserName)))
            if stdOutput:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput.strip()))
            if stdError:
               self.logger.log(logging.ERROR,getLogMessage(stdError.strip()))
         else:
            self.logger.log(logging.INFO,getLogMessage("Identity manager %s, user %s key pair generated" % \
                                                                              (identityName,hubUserName)))
            identityPath = stdOutput.strip()

      return(identityPath)


   def generatePermanentUserSSHKey(self):
      identityNames = self.identitiesInfo.getIdentitiesWithKeyValue('identityType','communitySSH')
      for identityName in identityNames:
         permanentUsers = self.identitiesInfo.getIdentityKeyValue(identityName,'permanentUsers')
         if permanentUsers:
            for permanentUser in permanentUsers:
               self.generatePersonalSSHKeyCopy(identityName,permanentUser)


   def generatePermanentUserSSHKeyPair(self):
      identityNames = self.identitiesInfo.getIdentitiesWithKeyValue('identityType','personalPKI')
      for identityName in identityNames:
         permanentUsers = self.identitiesInfo.getIdentityKeyValue(identityName,'permanentUsers')
         if permanentUsers:
            for permanentUser in permanentUsers:
               self.generatePersonalSSHKeyPair(identityName,permanentUser)


   def generatePersonalIdentity(self,
                                identityName,
                                hubUserName):
      identityPath = ""
      identityType = self.identitiesInfo.getIdentityKeyValue(identityName,'identityType')
      if   identityType == 'x509':
         personalizeMethod = self.identitiesInfo.getIdentityKeyValue(identityName,'personalizeMethod')
         if   personalizeMethod == 'saml':
            identityPath = self.generatePersonalProxyWithSAML(identityName,hubUserName)
         elif personalizeMethod == 'copy':
            identityPath = self.generatePersonalProxyWithoutSAML(identityName,hubUserName)
      elif identityType == 'communitySSH':
         identityPath = self.generatePersonalSSHKeyCopy(identityName,hubUserName)
      elif identityType == 'personalPKI':
         identityPath = self.generatePersonalSSHKeyPair(identityName,hubUserName)

      return(identityType,identityPath)


   def generateUserHash(self,
                        identityName,
                        hubUserName,
                        hubUserId):
      userHash = hubUserName
      md5HashTokens = self.identitiesInfo.getIdentityKeyValue(identityName,'md5HashTokens')
      if md5HashTokens:
         hashToken = '#'.join(md5HashTokens)
         userId = ':'.join([hashToken,str(hubUserId)])
         userHash = hashlib.md5(userId+'\n').hexdigest()

      return(userHash)


   def processEvents(self):
      while not self.eventQueue.isEmpty():
         event = self.eventQueue.peek()
         if event.when <= time.time():
            event = self.eventQueue.pop()
            if   event.what == self.EVENTGENERATEGRIDPROXY:
               self.generateCommunityGridProxy(event.identityName)
            elif event.what == self.EVENTGENERATEVOMSPROXY:
               self.generateCommunityVomsProxy(event.identityName)
            elif event.what == self.EVENTREFRESHCONDORPROXY:
               self.refreshUserCondorProxies(event.identityName)
            elif event.what == self.EVENTREFRESHJOBMONPROXY:
               self.refreshUserJobMonitorProxies(event.identityName)
         else:
            break


   def processRequests(self):
      for channel in self.activeChannels:
         message = self.pullMessage(channel,0)
         while message:
            args = message.split()
            if args[0] == 'json':
               jsonMessageLength = int(args[1])
               jsonMessage = self.pullMessage(channel,jsonMessageLength)
               if len(jsonMessage) > 0:
                  try:
                     jsonObject = json.loads(jsonMessage)
                  except ValueError:
                     self.logger.log(logging.ERROR,getLogMessage("JSON object %s could not be decoded" % (jsonMessage)))
                  else:
                     if   jsonObject['messageType'] == 'getUserIdentities':
                        identityNames = jsonObject['identityNames']
                        hubUserName = jsonObject['hubUserName']
                        identityPaths = {}
                        for identityName in identityNames:
                           identityType,identityPath = self.generatePersonalIdentity(identityName,hubUserName)
                           identityPaths[identityType] = identityPath
                        returnMessage = {'messageType':'userIdentity',
                                         'hubUserName':hubUserName,
                                         'identityPaths':identityPaths}
                        self.postJsonMessage(channel,returnMessage)
                     elif jsonObject['messageType'] == 'getUserHashes':
                        identityNames = jsonObject['identityNames']
                        hubUserName = jsonObject['hubUserName']
                        hubUserId   = jsonObject['hubUserId']
                        identityHashes = []
                        for identityName in identityNames:
                           identityHash = self.generateUserHash(identityName,hubUserName,hubUserId)
                           identityHashes.append(identityHash)
                        returnMessage = {'messageType':'userHashes',
                                         'hubUserName':hubUserName,
                                         'hubUserId':hubUserId,
                                         'identityHashes':identityHashes}
                        self.postJsonMessage(channel,returnMessage)
                     else:
                        self.logger.log(logging.ERROR,getLogMessage("Discarded message type: %s" % (jsonObject['messageType'])))
               else:
                  self.pushMessage(channel,message + '\n')
                  break
            else:
               self.logger.log(logging.ERROR,getLogMessage("Discarded message: %s" % (message)))

            message = self.pullMessage(channel,0)


