# @package      hubzero-submit-distributor
# @file         RemoteInstantSCRIPT.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import re
import random
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteInstantSCRIPT:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                workingDirectory,
                appScriptName,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                nNodes,
                ppn,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.workingDirectory      = workingDirectory
      self.appScriptName         = appScriptName
      self.isMultiCoreRequest    = isMultiCoreRequest
      self.computationMode       = managerInfo['computationMode']
      self.preManagerCommands    = managerInfo['preManagerCommands']
      self.managerCommand        = managerInfo['managerCommand']
      self.postManagerCommands   = managerInfo['postManagerCommands']
      self.nNodes                = nNodes
      self.ppn                   = ppn
      self.venue                 = siteInfo['venue']
      self.venues                = siteInfo['venues']
      self.venueIndex            = siteInfo['venueIndex']
      self.submissionScriptsInfo = submissionScriptsInfo
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timestampTransferred  = timeHistoryLogs['timestampTransferred']
      self.timeResults           = timeHistoryLogs['timeResults']

      self.nodeFileName = ""
      self.nodeList     = []


   def __buildMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','SCRIPT','mpi')

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]  = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]      = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"] = "\n".join(self.postManagerCommands)
      substitutions["NNODES"]              = self.nNodes
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]         = nProcessors
      substitutions["APPSCRIPTNAME"]       = self.appScriptName
      substitutions["TS_TRANSFERRED"]      = self.timestampTransferred
      substitutions["TS_START"]            = self.timestampStart
      substitutions["TS_FINISH"]           = self.timestampFinish
      substitutions["TIME_RESULTS"]        = self.timeResults
      substitutions["RUNNAME"]             = self.runName
      substitutions["JOBID"]               = self.localJobId
      substitutions["INSTANCEID"]          = self.instanceId
      substitutions["WORKINGDIRECTORY"]    = self.workingDirectory
      substitutions["NODEFILE"]            = self.nodeFileName
      substitutions["HUBUSERNAME"]         = self.hubUserName
      substitutions["HUBUSERID"]           = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildNodeList(self):
      del self.nodeList
      self.nodeList = []
      self.nodeFileName = "%s_%s.machinelist" % (self.localJobId,self.instanceId)

      for core in xrange(int(self.ppn)):
         self.nodeList.append(self.venue)
      eligible = range(len(self.venues))
      eligible.remove(self.venueIndex)
      for node in xrange(int(self.nNodes) - 1):
         index = eligible[random.randint(0,len(eligible)-1)]
         enclosure = self.venues[index]
         for core in xrange(int(self.ppn)):
            self.nodeList.append(enclosure)
         eligible.remove(index)
         if len(eligible) == 0:
            eligible = range(len(self.venues))


   def __buildMatlabPCTFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','SCRIPT','matlabpct')

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]  = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"] = "\n".join(self.postManagerCommands)
      substitutions["APPSCRIPTNAME"]       = self.appScriptName
      substitutions["TS_TRANSFERRED"]      = self.timestampTransferred
      substitutions["TS_START"]            = self.timestampStart
      substitutions["TS_FINISH"]           = self.timestampFinish
      substitutions["TIME_RESULTS"]        = self.timeResults
      substitutions["WORKINGDIRECTORY"]    = self.workingDirectory
      substitutions["HUBUSERNAME"]         = self.hubUserName
      substitutions["HUBUSERID"]           = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            self.__buildNodeList()
            batchScript = self.__buildMPIFile()
            batchScriptName = "%s_%s.cl" % (self.localJobId,self.instanceId)
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
         elif self.computationMode == 'matlabpct':
            batchScript = self.__buildMatlabPCTFile()
            batchScriptName = "%s_%s.cl" % (self.localJobId,self.instanceId)
      else:
         batchScript = ""
      batchScriptExecutable = True

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
         remoteJobIdNumber = remoteJobId
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


