# @package      hubzero-submit-distributor
# @file         VenueMechanismCore.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import sys
import os
import select
import subprocess
import shlex
import time
import datetime
import re
import shutil
import copy
import logging
from errno import EPIPE

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic import JobStatistic

class VenueMechanismCore:
   def __init__(self,
                timeHistoryLogs,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                remoteMonitors,
                isMultiCoreRequest,
                nCpus,
                nNodes,
                ppn):
      self.logger                = logging.getLogger(__name__)
      self.timeHistoryLogs       = copy.copy(timeHistoryLogs)
      self.siteInfo              = copy.copy(siteInfo)
      self.submissionScriptsInfo = copy.copy(submissionScriptsInfo)
      self.managerInfo           = copy.copy(managerInfo)
      self.remoteJobMonitor      = remoteMonitors['job']
      self.remoteTunnelMonitor   = remoteMonitors['tunnel']
      self.remoteCloudMonitor    = remoteMonitors['cloud']
      self.remoteIdentityManager = remoteMonitors['identity']

      self.childPid   = 0
      self.bufferSize = 4096

      self.remoteAppScript = None
      self.remoteBatch     = None

      self.venueMechanism    = ''
      self.remoteBatchSystem = ''
      self.enteredCommand    = ""

      self.isMultiCoreRequest = isMultiCoreRequest
      self.nCpus              = nCpus
      self.nNodes             = nNodes
      self.ppn                = ppn

      self.jobSubmitted  = False
      self.jobStatistics = {}
      self.jobIndex      = 0
      if self.isMultiCoreRequest:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      else:
         self.jobStatistics[self.jobIndex] = JobStatistic(1)

      self.scriptsCreated    = False
      self.filesSent         = False
      self.jobPostProcessed  = False
      self.filesRetrieved    = False
      self.filesCleanedup    = False
      self.scriptsKilled     = False
      self.runName           = ""
      self.localJobId        = ""
      self.instanceId        = "-1"
      self.instanceDirectory = ""
      self.stageInTarFile    = ""


   def _writeToStdout(self,
                      message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def _writeToStderr(self,
                      message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def recordGridResourceUse(self):
      reFiles = re.compile(self.timeHistoryLogs['jobGridResource'] + "_[0-9]+$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = filter(reFiles.search,dirFiles)
      if len(matchingFiles) > 0:
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(self.timeHistoryLogs['jobGridResource']):
               jobIndex = int(matchingFile[len(self.timeHistoryLogs['jobGridResource'])+1:])
            else:
               jobIndex = 1

            if not jobIndex in self.jobStatistics:
               self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)

            try:
               gridResourcePath = os.path.join(self.instanceDirectory,matchingFile)
               fpGridResource = open(gridResourcePath,'r')
               try:
                  gridResource = fpGridResource.readline().strip()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (gridResourcePath)))
               else:
                  gridType = gridResource.split()[0]
                  if   gridType == 'gt2':
                     gateKeeper = gridResource.split()[1].split('/')[0]
                     jobManager = gridResource.split()[1].split('/')[1].split('-')[1].upper()
                  elif gridType == 'gt4':
                     gateKeeper = gridResource.split()[1]
                     jobManager = gridResource.split()[2]
                  elif gridType == 'gt5':
                     gateKeeper = gridResource.split()[1].split('/')[0]
                     jobManager = gridResource.split()[1].split('/')[1].split('-')[1].upper()

                  self.jobStatistics[jobIndex]['venue']                  = gateKeeper
                  self.jobStatistics[jobIndex]['jobSubmissionMechanism'] = self.venueMechanism + jobManager
               finally:
                  fpGridResource.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridResourcePath)))

            os.remove(os.path.join(self.instanceDirectory,matchingFile))
      else:
         message = self.timeHistoryLogs['jobGridResource'] + " not present in " + self.instanceDirectory
         self.logger.log(logging.ERROR,getLogMessage(message))


   def logGridHistories(self):

      gridHistoryLogPath = self.timeHistoryLogs['hubGridJobHistoryLogPath']
      try:
         fpGridHistoryLog = open(gridHistoryLogPath,'a')
      except (IOError,OSError):
         fpGridHistoryLog = None
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridHistoryLogPath)))

      reFiles = re.compile(self.timeHistoryLogs['jobGridHistory'] + "_[0-9]+$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = filter(reFiles.search,dirFiles)
      if len(matchingFiles) > 0:
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(self.timeHistoryLogs['jobGridHistory']):
               jobIndex = int(matchingFile[len(self.timeHistoryLogs['jobGridHistory'])+1:])
            else:
               jobIndex = 1

            if not jobIndex in self.jobStatistics:
               self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)

            gridHistoryPath = os.path.join(self.instanceDirectory,matchingFile)
            if fpGridHistoryLog:
               try:
                  fpGridHistory = open(gridHistoryPath,'r')
                  try:
                     gridHistory = fpGridHistory.readline()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (gridHistoryPath)))
                  else:
                     try:
                        fpGridHistoryLog.write("%s_%s\t%d\t%s\t%s\n" % (self.localJobId,self.instanceId,jobIndex, \
                                                                        gridHistory.strip(),self.enteredCommand))
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (gridHistoryLogPath)))
                  finally:
                     fpGridHistory.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridHistoryPath)))
            os.remove(gridHistoryPath)

      reFiles = re.compile(self.timeHistoryLogs['jobGridHistory'] + "_[0-9]+_$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = filter(reFiles.search,dirFiles)
      if len(matchingFiles) > 0:
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(self.timeHistoryLogs['jobGridHistory']):
               jobIndex = int(matchingFile[len(self.timeHistoryLogs['jobGridHistory'])+1:-1])
            else:
               jobIndex = 1

            if not jobIndex in self.jobStatistics:
               self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)

            gridHistoryPath = os.path.join(self.instanceDirectory,matchingFile)
            if fpGridHistoryLog:
               try:
                  fpGridHistory = open(gridHistoryPath,'r')
                  try:
                     jobInProgressSite,jobInProgressSetupCompleted = fpGridHistory.readline().rstrip().split()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (gridHistoryPath)))
                  else:
                     jobInProgressCompleted = int(time.mktime(datetime.datetime.utcnow().timetuple()))
                     jobInProgressExitStatus = -3
                     try:
                        fpGridHistoryLog.write("%s_%s\t%d\t%s\t%s\t%d\t%d\t%s\n" % (self.localJobId,self.instanceId,jobIndex, \
                                                                                    jobInProgressSite, \
                                                                                    jobInProgressSetupCompleted, \
                                                                                    jobInProgressCompleted,
                                                                                    jobInProgressExitStatus, \
                                                                                    self.enteredCommand))
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (gridHistoryLogPath)))
                  finally:
                     fpGridHistory.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridHistoryPath)))
            os.remove(gridHistoryPath)

      if fpGridHistoryLog:
         fpGridHistoryLog.close()


   def logGridJobId(self):
      reFiles = re.compile(self.timeHistoryLogs['jobGridJobId'] + "_[0-9]+$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = filter(reFiles.search,dirFiles)
      if len(matchingFiles) > 0:
         gridJobIdLogPath = self.timeHistoryLogs['hubGridJobIdLogPath']
         try:
            fpGridJobIdLog = open(gridJobIdLogPath,'a')
         except (IOError,OSError):
            fpGridJobIdLog = None
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridJobIdLogPath)))

         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(self.timeHistoryLogs['jobGridJobId']):
               jobIndex = int(matchingFile[len(self.timeHistoryLogs['jobGridJobId'])+1:])
            else:
               jobIndex = 1

            if not jobIndex in self.jobStatistics:
               self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)

            gridJobIdPath = os.path.join(self.instanceDirectory,matchingFile)
            if fpGridJobIdLog:
               try:
                  fpGridJobId = open(gridJobIdPath,'r')
                  try:
                     gridJobId = fpGridJobId.readline()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (gridJobIdPath)))
                  else:
                     try:
                        fpGridJobIdLog.write("%s_%s %s\n" % (self.localJobId.lstrip('0'),self.instanceId,gridJobId.strip()))
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (gridJobIdLogPath)))
                  finally:
                     fpGridJobId.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (gridJobIdPath)))
            os.remove(gridJobIdPath)
         if fpGridJobIdLog:
            fpGridJobIdLog.close()
      else:
         message = self.timeHistoryLogs['jobGridJobId'] + " not present in " + self.instanceDirectory
         self.logger.log(logging.ERROR,getLogMessage(message))


   def executeCommand(self,
                      command,
                      streamOutput=False):
      child = subprocess.Popen(command,shell=True,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self._writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self._writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def executeSSHCommand(self,
                         sshCommand,
                         tunnelDesignator,
                         streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.incrementTunnelUse(tunnelDesignator)
      exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

      while exitStatus and (stdError.count("You don't exist, go away!") > 0):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.decrementTunnelUse(tunnelDesignator)

      return(exitStatus,stdOutput,stdError)


   def executeLaunchCommand(self,
                            launchCommand,
                            streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

      while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      return(exitStatus,stdOutput,stdError)


   def updateVenue(self,
                   executionVenue):
      if self.remoteBatchSystem == 'FACTORY':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue
      if self.remoteBatchSystem == 'PEGASUS':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue


   def getVenue(self):
      executionVenue = self.jobStatistics[self.jobIndex]['venue']

      return(executionVenue)


   def recordJobStatisticTime(self,
                              statistic,
                              timeFileBasename):
      timePath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timePath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTime: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTime(statistic,timePath)
      else:
         self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))


   def recordJobStatisticTimer(self,
                               timeFileBasename):
      timerPath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timerPath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTimer: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTimer(timerPath)
      else:
         if self.jobSubmitted:
            self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))


   def recordJobStatistics(self):
      self.recordJobStatisticTimer(self.timeHistoryLogs['timeResults'])

      if self.jobSubmitted:
         self.recordJobStatisticTime('transferCompleteTime',self.timeHistoryLogs['timestampTransferred'])
         self.recordJobStatisticTime('jobStartedTime',self.timeHistoryLogs['timestampStart'])
         self.recordJobStatisticTime('jobFinshedTime',self.timeHistoryLogs['timestampFinish'])

      for jobIndex in self.jobStatistics:
         self.jobStatistics[jobIndex].setWaitingTime()
         self.jobStatistics[jobIndex].setElapsedRunTime()


   def wasJobSuccessful(self):
      success = False
      for jobIndex in self.jobStatistics:
         if self.jobStatistics[jobIndex]['exitCode'] == 0:
            success = True
            break

      return(success)


   def removeJobStatisticsFiles(self):
      for timeFileBasename in self.timeHistoryLogs['timeResults'], \
                              self.timeHistoryLogs['timestampTransferred'], \
                              self.timeHistoryLogs['timestampStart'], \
                              self.timeHistoryLogs['timestampFinish']:
         reFiles = re.compile(timeFileBasename + "(_[0-9]+)?$")
         dirFiles = os.listdir(self.instanceDirectory)
         matchingFiles = filter(reFiles.search,dirFiles)
         if len(matchingFiles) > 0:
            for matchingFile in matchingFiles:
               os.remove(os.path.join(self.instanceDirectory,matchingFile))


   def stageFilesToInstanceDirectory(self):
      if self.instanceDirectory != "":
         if self.instanceDirectory != os.getcwd():
            if os.path.isdir(self.instanceDirectory):
               if self.stageInTarFile != "":
                  srcPath = os.path.join(os.getcwd(),self.stageInTarFile)
                  if os.path.exists(srcPath):
                     dstPath = os.path.join(self.instanceDirectory,self.stageInTarFile)
                     os.rename(srcPath,dstPath)


   def registerJob(self):
      self.remoteJobMonitor.postJobRegistration(self.siteInfo['siteName'],
                                                self.localJobId,self.instanceId,
                                                str(self.hubUserId),self.submitterClass,
                                                self.distributorPid)


   def isJobReleased(self):
      jobReleased = self.remoteJobMonitor.isJobReleased(self.localJobId,self.instanceId)

      return(jobReleased)


   def removeJobRegistration(self):
      self.remoteJobMonitor.deleteJobRegistration(self.localJobId,self.instanceId)


   def removeInstanceDirectory(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("removeInstanceDirectory:os.getcwd(): No such file or directory"))
      else:
         if self.instanceDirectory != currentWorkingDirectory:
            shutil.rmtree(self.instanceDirectory,True)
            jobDirectory = os.path.dirname(self.instanceDirectory)
            if jobDirectory != currentWorkingDirectory:
               try:
                  os.rmdir(jobDirectory)
               except:
                  pass


   def moveTree(self,
                src,
                dst,
                symlinks=False):
      if os.path.isdir(src):
         if os.path.exists(dst):
            if not os.path.isdir(dst):
               self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (dst)))
               return
         else:
            os.mkdir(dst)
         names = os.listdir(src)
         for name in names:
            srcPath = os.path.join(src,name)
            dstPath = os.path.join(dst,name)
            try:
               if symlinks and os.path.islink(srcPath):
                  linkto = os.readlink(srcPath)
                  os.symlink(linkto,dstPath)
               elif os.path.isdir(srcPath):
                  self.moveTree(srcPath,dstPath,symlinks)
               else:
                  os.rename(srcPath,dstPath)
            except (IOError,os.error),why:
               self.logger.log(logging.ERROR,getLogMessage("moveTree: Can't move %s to %s: %s" % (srcPath,dstPath,str(why))))
      else:
         self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (src)))


   def recoverFiles(self,
                    ingoreInstanceFiles):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.exists(self.instanceDirectory):
            if os.path.isdir(self.instanceDirectory):
               if self.instanceDirectory != currentWorkingDirectory:
                  dirFiles = os.listdir(self.instanceDirectory)
                  if ingoreInstanceFiles:
                     reIgnoreFiles = re.compile(".*%s_%s.*" % (self.runName,self.instanceId))
                     ignoreFiles = filter(reIgnoreFiles.search,dirFiles)
                     for ignoreFile in ignoreFiles:
                        dirFiles.remove(ignoreFile)

                  for dirFile in dirFiles:
                     srcFile = os.path.join(self.instanceDirectory,dirFile)
                     dstFile = os.path.join(currentWorkingDirectory,dirFile)
                     try:
                        if os.path.isdir(srcFile):
                           self.moveTree(srcFile,dstFile)
                        else:
                           os.rename(srcFile,dstFile)
                     except:
                        self.logger.log(logging.ERROR,getLogMessage("%s recovery failed" % (srcFile)))


   def recoverStdFiles(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverStdFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.isdir(self.instanceDirectory):
            if self.instanceDirectory != currentWorkingDirectory:
               for fileType in 'stderr','stdout','FAILURE':
                  reFiles = re.compile(".*%s_%s\.%s" % (self.runName,self.instanceId,fileType))
                  dirFiles = os.listdir(self.instanceDirectory)
                  matchingFiles = filter(reFiles.search,dirFiles)
                  if len(matchingFiles) > 0:
                     matchingFiles.sort(reverse=True)
                     fileToAppend = "%s.%s" % (self.runName,fileType)
                     try:
                        fpAppend = open(fileToAppend,'a')
                        for matchingFile in matchingFiles:
                           fileToRecover = os.path.join(self.instanceDirectory,matchingFile)
                           try:
                              fpRecover = open(fileToRecover,'r')
                              try:
                                 recoverText = fpRecover.readlines()
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (fileToRecover)))
                              else:
                                 try:
                                    fpAppend.writelines(recoverText)
                                 except (IOError,OSError):
                                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (fileToAppend)))
                              finally:
                                 fpRecover.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToRecover)))
                        fpAppend.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToAppend)))


