# @package      hubzero-submit-distributor
# @file         JobDistributor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
__version__ = '3.0.8'

import sys
import os
import re
import signal
import select
import time
import subprocess
import shlex
import random
import traceback
import math
import copy
import shutil
import logging
from errno import EPIPE

from hubzero.submit.LogMessage               import getLogJobIdMessage as getLogMessage, logSetJobId
from hubzero.submit.DaemonsInfo              import DaemonsInfo
from hubzero.submit.InfosInfo                import InfosInfo
from hubzero.submit.SitesInfo                import SitesInfo
from hubzero.submit.ToolsInfo                import ToolsInfo
from hubzero.submit.AppsAccessInfo           import AppsAccessInfo
from hubzero.submit.ManagersInfo             import ManagersInfo
from hubzero.submit.TunnelsInfo              import TunnelsInfo
from hubzero.submit.EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from hubzero.submit.SubmissionScriptsInfo    import SubmissionScriptsInfo
from hubzero.submit.CommandParser            import CommandParser
from hubzero.submit.VenueMechanismLocal      import VenueMechanismLocal
from hubzero.submit.VenueMechanismSsh        import VenueMechanismSsh
from hubzero.submit.VenueMechanismGsiSsh     import VenueMechanismGsiSsh
from hubzero.submit.RemoteIdentityManager    import RemoteIdentityManager
from hubzero.submit.RemoteJobMonitor         import RemoteJobMonitor
from hubzero.submit.RemoteProbeMonitor       import RemoteProbeMonitor
from hubzero.submit.RemoteTunnelMonitor      import RemoteTunnelMonitor
from hubzero.submit.TimeConversion           import hhmmssTomin
from hubzero.submit.TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand
from hubzero.submit.JobStatistic             import JobStatistic
from hubzero.submit.ParameterTemplate        import ParameterTemplate

HUBGRIDJOBIDLOGFILE      = "gridjobid.log"
HUBGRIDJOBHISTORYLOGFILE = "gridjobhistory.log"

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
POSTPROCESSJOBCOMMAND  = 'postprocessjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

LOCALJOBID = ".__local_jobid"

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"

JOBGRIDRESOURCE = ".__grid_resource"
JOBGRIDHISTORY  = ".__grid_history"
JOBGRIDJOBID    = ".__grid_jobid"

class JobDistributor:
   def __init__(self,
                configurationDirectory,
                distributorConfigurationFile,
                daemonsConfigurationFile,
                infosConfigurationFile,
                logDirectory):
      self.logger = logging.getLogger(__name__)

      self.configData = {}
      self.configurationDirectory       = configurationDirectory
      self.distributorConfigurationFile = distributorConfigurationFile
      self.daemonsConfigurationFile     = daemonsConfigurationFile
      self.infosConfigurationFile       = infosConfigurationFile

      self.version                  = __version__
      self.operationMode            = 0
      self.session                  = 0
      self.submitterClass           = 1
      self.distributorPid           = os.getpid()
      self.jobId                    = 0
      self.jobs                     = {}
      self.jobStatistics            = {}
      self.reportMetrics            = False
      self.fpUserLogPath            = None
      self.abortAttempted           = False
      self.disableStateCheck        = False
      self.disableProbeCheck        = True
      self.quotaLimit               = -1
      self.childPid                 = 0
      self.bufferSize               = 4096
      self.hubGridJobIdLogPath      = os.path.join(logDirectory,HUBGRIDJOBIDLOGFILE)
      self.hubGridJobHistoryLogPath = os.path.join(logDirectory,HUBGRIDJOBHISTORYLOGFILE)
      self.dataDirectory            = ''
      self.binDirectory             = ''

      configFilePath = os.path.join(self.configurationDirectory,self.daemonsConfigurationFile)
      self.daemonsInfo       = DaemonsInfo(configFilePath)
      self.jobListenURI      = self.daemonsInfo.getDaemonListenURI('jobMonitor','tcp')
      self.identityListenURI = self.daemonsInfo.getDaemonListenURI('identitiesManager','tcp')
      self.probeListenURI    = self.daemonsInfo.getDaemonListenURI('probeMonitor','tcp')
      self.tunnelListenURI   = self.daemonsInfo.getDaemonListenURI('tunnelMonitor','tcp')
      self.cloudListenURI    = self.daemonsInfo.getDaemonListenURI('cloudMonitor','tcp')

      self.runjobExists = False

      self.batchCommands = {}
      self.batchCommands['receiveInput']    = RECEIVEINPUTCOMMAND
      self.batchCommands['submitBatchJob']  = SUBMITBATCHJOBCOMMAND
      self.batchCommands['postProcessJob']  = POSTPROCESSJOBCOMMAND
      self.batchCommands['transmitResults'] = TRANSMITRESULTSCOMMAND
      self.batchCommands['cleanupJob']      = CLEANUPJOBCOMMAND
      self.batchCommands['killBatchJob']    = KILLBATCHJOBCOMMAND

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.waitForJobsInfo = {}

      self.remoteMonitors = {}
      self.remoteMonitors['job']      = None
      self.remoteMonitors['identity'] = None
      self.remoteMonitors['probe']    = None
      self.remoteMonitors['tunnel']   = None
      self.remoteMonitors['cloud']    = None

      self.infosInfo                = None
      self.sitesInfo                = None
      self.cloudsInfo               = None
      self.tunnelsInfo              = None
      self.toolsInfo                = None
      self.appsAccessInfo           = None
      self.managersInfo             = None
      self.environmentWhitelistInfo = None
      self.submissionScriptsInfo    = None

      self.successfulInstance    = None
      self.maximumSelectedSites  = 1
      self.userDestinations      = []
      self.inputFiles            = []
      self.outputFiles           = []
      self.isMultiCoreRequest    = False
      self.nCpus                 = 1
      self.ppn                   = ""
      self.wallTime              = 60
      self.environment           = ""
      self.managerSpecified      = False
      self.managerInfo           = {}
      self.x509SubmitProxy       = ""
      self.stdinput              = ""
      self.isParametric          = False
      self.parameterNames        = []
      self.parameterCombinations = None

      self.commandParser           = None
      self.enteredCommandArguments = []
      self.tailFiles               = []

      self.submitVenues         = None
      self.runName              = ""
      self.localJobId           = None
      self.hubUserId            = None
      self.hubUserName          = None
      self.progressReport       = 'text'
      self.isClientTTY          = True
      self.useSetup             = None
      self.pegasusVersion       = None
      self.pegasusHome          = None
      self.clientWorkDirectory  = None
      self.clientTransferPath   = None
      self.doubleDashTerminator = False

      self.abortGlobal = {}
      self.abortGlobal['abortAttempted'] = self.abortAttempted
      self.abortGlobal['abortSignal']    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def setEnvironment(self):
      if self.configData['pbsRoot'] != "" and self.configData['pbsRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['pbsRoot'],'bin') + os.pathsep + os.environ['PATH']

      if self.configData['condorRoot'] != "" and self.configData['condorRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['condorRoot'],'bin') + os.pathsep + \
                              os.path.join(self.configData['condorRoot'],'sbin') + os.pathsep + os.environ['PATH']
         if self.configData['condorConfig'] != "":
            os.environ['CONDOR_CONFIG'] = self.configData['condorConfig']

      if self.configData['runjobRoot'] != "":
         if self.configData['runjobRoot'] == "REMOTE":
            self.runjobExists = True
         else:
            runjobScriptPath = os.path.join(self.configData['runjobRoot'],'bin','runjob.sh')
            if os.path.exists(runjobScriptPath):
               os.environ['PATH'] = os.path.join(self.configData['runjobRoot'],'bin') + os.pathsep + os.environ['PATH']
               self.runjobExists = True

      os.environ['PATH'] = self.binDirectory + os.pathsep + os.environ['PATH']


   def getEnvironment(self):
      self.hubUserName          = os.getenv("USER")
      self.hubUserId            = os.getuid()
      self.session              = int(os.getenv("SESSION",'0'))
      self.submitterClass       = int(os.getenv("SUBMITTER_CLASS",'1'))
      self.submitVenues         = os.getenv("SUBMITVENUES")
      self.isClientTTY          = bool(int(os.getenv("SUBMIT_ISCLIENTTTY",'0')))
      self.pegasusVersion       = os.getenv("PEGASUS_VERSION")
      self.pegasusHome          = os.getenv("PEGASUS_HOME")
      self.useSetup             = os.getenv("USE_SETUP_SCRIPT")
      self.clientWorkDirectory  = os.getenv("CLIENT_WORK_DIRECTORY")
      self.clientTransferPath   = os.getenv("CLIENT_TRANSFER_PATH")
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'False').lower() == 'true')


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inDistributorSection = False

      try:
         configFilePath = os.path.join(self.configurationDirectory,self.distributorConfigurationFile)
         fpConfig = open(configFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inDistributorSection = (sectionName == 'distributor')
                     if inDistributorSection:
                        self.configData = {'probeMonitoringInstalled':False,
                                           'maximumSelectedSites':1,
                                           'allowedVenueMechanisms':['local','ssh'],
                                           'dataDirectory':os.path.join(os.sep,'opt','submit'),
                                           'binDirectory':os.path.join(os.sep,'opt','submit','bin'),
                                           'runjobRoot':'',
                                           'condorRoot':'',
                                           'condorConfig':'',
                                           'pbsRoot':''
                                          }
                  elif inDistributorSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = self.configData[key].keys()[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.configData[key] = {} 
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
               else:
                  eof = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (configFilePath)))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (configFilePath)))

      configured = True
      if os.path.isdir(self.configData['dataDirectory']):
         self.dataDirectory = self.configData['dataDirectory']
      else:
         message = "Specified dataDirectory does not exist: %s" % (self.configData['dataDirectory'])
         self.logger.log(logging.ERROR,getLogMessage(message))
         configured = False

      if os.path.isdir(self.configData['binDirectory']):
         self.binDirectory = self.configData['binDirectory']
      else:
         message = "Specified binDirectory does not exist: %s" % (self.configData['binDirectory'])
         self.logger.log(logging.ERROR,getLogMessage(message))
         configured = False

      if configured:
         self.setEnvironment()
         self.getEnvironment()
         self.maximumSelectedSites = self.configData['maximumSelectedSites']

      return(configured)


   def doRun(self):
      return(self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID)


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.logger.log(logging.INFO,getLogMessage("Args are:" + str(sys.argv)))
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR:
         self.__writeToStdout("Submit distributor version: %s\n" % (self.version))

      reportVersion = self.commandParser.getOption('version') or \
                      self.commandParser.getOption('versionServer') or \
                      self.commandParser.getOption('versionDistributor')

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         self.disableStateCheck = self.commandParser.getOption('disableStateCheck')
         if self.configData['probeMonitoringInstalled']:
            self.disableProbeCheck = self.commandParser.getOption('disableProbeCheck')

         exitCode = self.commandParser.setSweepParameters()
         if exitCode == 0:
            exitCode = self.commandParser.setCSVDataParameters()
            if exitCode == 0:
               if self.commandParser.getParameterCombinationCount() > 0:
                  self.isParametric = True

               if not reportVersion:
                  enteredCommand = self.commandParser.getEnteredCommand()
                  if enteredCommand == "":
                     self.logger.log(logging.ERROR,getLogMessage("Command must be supplied"))
                     self.__writeToStderr("Command must be supplied\n")
                     if not exitCode:
                        exitCode = 1
                  else:
                     if self.commandParser.getOption('runName'):
                        self.runName = self.commandParser.getOption('runName')
                     self.reportMetrics = self.commandParser.getOption('metrics')

      if   not self.commandParser.getOption('progress'):
         self.progressReport = 'silent'
      elif self.commandParser.getOption('progressCurses'):
         self.progressReport = 'curses'
      elif self.commandParser.getOption('progressSubmit'):
         self.progressReport = 'submit'
      elif self.commandParser.getOption('progressText'):
         self.progressReport = 'text'
      elif self.commandParser.getOption('progressPegasus'):
         self.progressReport = 'pegasus'
      else:
         if self.isParametric and self.isClientTTY:
            self.progressReport = 'curses'
         else:
            self.progressReport = 'text'

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      def getJobId(dataDirectory):
         nextLocalJobId = 0

         exitCode = 0
         localJobIdFileName = os.path.join(dataDirectory,"localJobId.dat")
         if not os.path.isfile(localJobIdFileName):
            try:
               fpLocalJobId = open(localJobIdFileName,'w')
               try:
                  fpLocalJobId.write('0')
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("could not open %s for creating" % (localJobIdFileName)))
                  self.__writeToStderr("could not open %s for creating\n" % (localJobIdFileName))
                  exitCode = 1
               finally:
                  fpLocalJobId.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("could not open %s for creating" % (localJobIdFileName)))
               self.__writeToStderr("could not open %s for creating\n" % (localJobIdFileName))
               exitCode = 1

         if not exitCode:
            try:
               fpLocalJobId = open(localJobIdFileName,'r')
               try:
                  previousLocalJobId = int(fpLocalJobId.readline().strip())
                  nextLocalJobId = previousLocalJobId+1
                  try:
                     fpLocalJobId = open(localJobIdFileName,'w')
                     try:
                        fpLocalJobId.write('%d' % (nextLocalJobId))
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (localJobIdFileName)))
                     finally:
                        fpLocalJobId.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (localJobIdFileName)))
                     self.__writeToStderr("could not open %s for writing\n" % (localJobIdFileName))
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (localJobIdFileName)))
               finally:
                  fpLocalJobId.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("could not open %s for reading" % (localJobIdFileName)))
               self.__writeToStderr("could not open %s for reading\n" % (localJobIdFileName))

         return(nextLocalJobId)

      exitCode = 0

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.commandParser.getOption('jobId'):
            self.jobId = self.commandParser.getOption('jobId')
         if self.jobId == 0:
            self.jobId = getJobId(self.dataDirectory)

      self.localJobId = "%08d" % (self.jobId)
      logSetJobId(self.jobId)

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.jobId == 0:
            exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setInfo(self):
      configFilePath = os.path.join(self.configurationDirectory,self.infosConfigurationFile)
      self.infosInfo = InfosInfo(configFilePath)

      self.sitesInfo = SitesInfo(self.infosInfo.getInfoPath('sites'),
                                 restrictionUser=os.getenv("USER"),
                                 templateDirectory=self.dataDirectory,
                                 allowedVenueMechanisms=self.configData['allowedVenueMechanisms'],
                                 pegasusVersion=self.pegasusVersion)
      if not self.sitesInfo.siteExists('grid'):
         self.runjobExists = False

      self.tunnelsInfo = TunnelsInfo(self.infosInfo.getInfoPath('tunnels'))

      self.toolsInfo = ToolsInfo(self.infosInfo.getInfoPath('tools'))
      self.toolsInfo.applyUserRestriction(self.hubUserName)
      self.toolsInfo.applyGroupRestriction()

      self.appsAccessInfo = AppsAccessInfo(self.infosInfo.getInfoPath('access'))

      self.managersInfo = ManagersInfo(self.infosInfo.getInfoPath('managers'))

      self.environmentWhitelistInfo = EnvironmentWhitelistInfo(self.infosInfo.getInfoPath('environment'))

      self.submissionScriptsInfo = SubmissionScriptsInfo('Distributor',
                                                         submissionScriptRootPath=self.infosInfo.getInfoPath('submissionscripts'))


   def showUsage(self):
      if self.operationMode & self.commandParser.OPERATIONMODEHELPUSAGE:
         self.commandParser.showUsage()

      if self.operationMode & self.commandParser.OPERATIONMODEHELPVENUES:
         venues = []
         for site in self.sitesInfo.getEnabledSites():
            venues.append(site)

         if len(venues) > 0:
            venues.sort()
            self.__writeToStdout("\nCurrently available VENUES are:\n")
            for venue in venues:
               self.__writeToStdout("   %s\n" % (venue))
         del venues

      if self.operationMode & self.commandParser.OPERATIONMODEHELPTOOLS:
         executableToolNames = self.toolsInfo.getToolNames()
         if len(executableToolNames) > 0:
            executableToolNames.sort()
            self.__writeToStdout("\nCurrently available TOOLs are:\n")
            for executableToolName in executableToolNames:
               self.__writeToStdout("   %s\n" % (executableToolName))
         del executableToolNames

      if self.operationMode & self.commandParser.OPERATIONMODEHELPMANAGERS:
         validManagers = self.managersInfo.getManagerNames()
         if len(validManagers) > 0:
            validManagers.sort()
            self.__writeToStdout("\nCurrently available MANAGERs are:\n")
            for validManager in validManagers:
               self.__writeToStdout("   %s\n" % (validManager))
         del validManagers


   def reportExitCondition(self):
      if self.doRun():
         if len(self.jobs) == 0:
            jobIndices = self.jobStatistics.keys()
            maximumJobIndex = max(jobIndices)

            try:
               mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
               remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
               remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
               exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
               cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
               elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
               waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
               nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']
               event          = self.jobStatistics[maximumJobIndex]['event']

               message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                         (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
               if event:
                  message += " event=%s" % (event)
               self.logger.log(logging.INFO,getLogMessage(message))
               message += "\n"
               if self.fpUserLogPath:
                  self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                  self.fpUserLogPath.flush()
               os.write(3,message)
            except:
               pass
         else:
            instanceIndex = 0
            for instance in self.jobs:
               if (self.successfulInstance == None) or \
                  (self.successfulInstance != None and (instance != self.successfulInstance)):
                  jobIndices = self.jobs[instance].jobStatistics.keys()
                  jobIndices.sort()
                  maximumJobIndex = max(jobIndices)

                  try:
                     for jobIndex in jobIndices:
                        if jobIndex != 0 or maximumJobIndex == 0:
                           mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                           remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                           remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                           exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                           cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                            self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                           elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                           waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                           nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                           event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                           instanceIndex += 1
                           message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                 (instanceIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                           if event:
                              message += " event=%s" % (event)
                           self.logger.log(logging.INFO,getLogMessage(message))
                           message += "\n"
                           if self.fpUserLogPath:
                              self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                              self.fpUserLogPath.flush()
                           os.write(3,message)
                  except:
                     pass

            if self.successfulInstance != None:
               instance = self.successfulInstance
               jobIndices = self.jobs[instance].jobStatistics.keys()
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                        if self.jobs[instance].remoteBatchSystem == 'PEGASUS':
                           if mechanism == "" or mechanism == 'Unknown':
                              mechanism = 'PEGASUS'
                        else:
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                        cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                        event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                        instanceIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                                     (instanceIndex,mechanism,remoteId,remoteLocation, \
                                                                      exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        if event:
                           message += " event=%s" % (event)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        message += "\n"
                        if self.fpUserLogPath:
                           self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                           self.fpUserLogPath.flush()
                        os.write(3,message)
               except:
                  pass

      if self.fpUserLogPath:
         self.fpUserLogPath.close()
         self.fpUserLogPath = None


# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortGlobal['abortAttempted'] = self.abortAttempted
         self.abortGlobal['abortSignal']    = signalNumber

         for instance in self.waitForJobsInfo:
            if 'isBatchJob' in self.waitForJobsInfo[instance]:
               if not self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'R':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def getUserQuota(self):
      quotaCommand = "quota -w"
      self.logger.log(logging.INFO,getLogMessage("quotaCommand = " + quotaCommand))
      exitStatus,stdOutput,stdError = self.executeCommand(quotaCommand)
      if exitStatus == 0:
         try:
            quotaResults = stdOutput.strip().split('\n')[-1].split()
            if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
               currentUsage = int(quotaResults[1].strip("*"))
               self.quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
         except:
            self.logger.log(logging.ERROR,getLogMessage(stdOutput))
            self.logger.log(logging.ERROR,getLogMessage(stdError))
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))


   def getStdInputFile(self,
                       inputFiles):
      stdInputFile = '/dev/null'
      toCheck = []
      stdinFd = sys.stdin.fileno()
      toCheck.append(stdinFd)
      try:
         ready = select.select(toCheck,[],[],0.) # wait for input
      except select.error,err:
         ready = {}
         ready[0] = []

      if stdinFd in ready[0]:
         content = sys.stdin.read()
         if content != "":
            stdInputFile = ".__%s.stdin" % (self.localJobId)
            try:
               stdinFile = open(stdInputFile,'w')
               try:
                  stdinFile.write(content)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdInputFile)))
               else:
                  inputFiles.append(stdInputFile)
               finally:
                  stdinFile.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdInputFile)))
         del content

      return(stdInputFile)


   def executeCommand(self,
                      command,
                      streamOutput=False):
      if isinstance(command,list):
         child = subprocess.Popen(command,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      else:
         commandArgs = shlex.split(command)
         child = subprocess.Popen(commandArgs,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def buildJobDescription(self):
      exitCode = 0

      self.logger.log(logging.INFO,getLogMessage("session = %d" % (self.session)))
      if self.commandParser.getOption('quotaCheck'):
         self.getUserQuota()

      userDestinations = []
      inputFiles       = []
      nCpus            = 1
      ppn              = ""
      wallTime         = 60
      environment      = ""

      if self.submitVenues == None:
         if self.commandParser.getOption('destinations'):
            userDestinations = self.commandParser.getOption('destinations')
      else:
         userDestinations = self.submitVenues.split(',')
      if len(userDestinations) > 0:
         reasonsDenied = self.sitesInfo.purgeDisabledSites(userDestinations)
         if len(userDestinations) == 0:
            errorMessage = "Access to all sites listed as possible execution hosts has been denied.\n"
            for userDestination in reasonsDenied:
               errorMessage += "   %20s %s\n" % (userDestination,reasonsDenied[userDestination])
            errorMessage += "Please select another site or attempt execution at a later time."
            self.logger.log(logging.ERROR,getLogMessage(errorMessage))
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 7
         del reasonsDenied

      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            inputFiles.append(inputFile)
      self.stdinput = self.getStdInputFile(inputFiles)

      if self.commandParser.getOption('outputFiles'):
         for outputFile in self.commandParser.getOption('outputFiles'):
            self.outputFiles.append(outputFile)

      if self.commandParser.getOption('manager'):
         manager = self.commandParser.getOption('manager')[0]
         if self.managersInfo.managerExists(manager):
            self.managerSpecified = True
            self.managerInfo      = self.managersInfo.getManagerInfo(manager)
         else:
            validManagers = self.managersInfo.getManagerNames()
            message = "Invalid manager %s specified. Valid managers are %s" % (manager,validManagers)
            self.logger.log(logging.ERROR,getLogMessage(message))
            self.__writeToStderr("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
            del validManagers
            if not exitCode:
               exitCode = 1
      else:
         self.managerInfo = self.managersInfo.getDefaultManagerInfo('serial')

      if self.commandParser.getOption('nCpus') != None:
         self.isMultiCoreRequest = True
         nCpus = self.commandParser.getOption('nCpus')

      if self.commandParser.getOption('ppn'):
         if self.isMultiCoreRequest:
            ppn = str(self.commandParser.getOption('ppn'))

      if self.commandParser.getOption('nRedundant'):
         self.maximumSelectedSites = self.commandParser.getOption('nRedundant')

      if self.commandParser.getOption('wallTime'):
         if ":" in self.commandParser.getOption('wallTime'):
            wallTime = hhmmssTomin(self.commandParser.getOption('wallTime'))
            if wallTime < 0.:
               self.logger.log(logging.ERROR,getLogMessage("wallTime must be > 0"))
               self.__writeToStderr("wallTime must be > 0\n")
               if not exitCode:
                  exitCode = 1
         else:
            try:
               wallTime = float(self.commandParser.getOption('wallTime'))
            except ValueError:
               message = "Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes"
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes\n")
               if not exitCode:
                  exitCode = 1

      if self.commandParser.getOption('userLogPath'):
         userLogPath = self.commandParser.getOption('userLogPath')
         try:
            self.fpUserLogPath = open(userLogPath,'a')
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("User log %s could not be opened." % (userLogPath)))
            self.__writeToStderr("User log %s could not be opened.\n" % (userLogPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('userJobidPath'):
         userJobidPath = self.commandParser.getOption('userJobidPath')
         try:
            fpUserJobId = open(userJobidPath,'w')
            try:
               fpUserJobId.write("%d\n" % (self.jobId))
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (userJobidPath)))
               if not exitCode:
                  exitCode = 1
            finally:
               fpUserJobId.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("User jobId file %s could not be created." % (userJobidPath)))
            self.__writeToStderr("User jobId file %s could not be created.\n" % (userJobidPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('x509SubmitProxy'):
         self.x509SubmitProxy = self.commandParser.getOption('x509SubmitProxy')

      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               message = "Invalid environment variable %s specified." % (environmentVariableValue)
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("Invalid environment variable %s specified.\n" % (environmentVariableValue))
               if not exitCode:
                  exitCode = 1
            else:
               if self.environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
                  environment += environmentVariable + "=" + value + " "
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Environment variable %s could not be set." % (environmentVariable)))
                  self.__writeToStderr("Environment variable %s could not be set.\n" % (environmentVariable))
         environment = environment.strip()

      if self.isMultiCoreRequest:
         if not self.managerSpecified:
            self.managerInfo = self.managersInfo.getDefaultManagerInfo('mpi')

      if not self.disableProbeCheck:
         ignoreProbeSites = self.sitesInfo.getIgnoreProbeSites()
         self.remoteMonitors['probe'] = RemoteProbeMonitor(self.probeListenURI,
                                                           ignoreProbeSites=ignoreProbeSites)

      if len(userDestinations) > 0:
         goToGrid = self.sitesInfo.purgeOfflineSites(userDestinations,self.remoteMonitors['probe'])
         if len(userDestinations) == 0 and not goToGrid:
            errorMessage = "All sites listed as possible execution hosts are out of service.\n" + \
                           "Please select another site or attempt execution at a later time."
            self.logger.log(logging.ERROR,getLogMessage(errorMessage))
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 8

      nUserDestinations = len(userDestinations)
      self.sitesInfo.purgeResourceLimitedSites(userDestinations,nCpus,wallTime)
      if nUserDestinations > 0 and len(userDestinations) == 0:
         errorMessage = "All sites listed as possible execution hosts are unable to meet\n" + \
                        "the specified resource requirements.  Please select another site."
         self.logger.log(logging.ERROR,getLogMessage(errorMessage))
         self.__writeToStderr(errorMessage + "\n")
         exitCode = 10

      self.userDestinations = copy.copy(userDestinations)
      self.inputFiles       = copy.copy(inputFiles)
      self.nCpus            = copy.copy(nCpus)
      self.ppn              = copy.copy(ppn)
      self.wallTime         = copy.copy(wallTime)
      self.environment      = copy.copy(environment)

      self.jobStatistics[0]['exitCode'] = exitCode

      if self.commandParser.getOption('tailFiles'):
         tailFiles = self.commandParser.getOption('tailFiles')
         for tailFile in tailFiles:
            if ':' in tailFile:
               fileName,nLines = tailFile.split(':')
               try:
                  nLines = int(nLines)
               except:
                  nLines = 10
            else:
               fileName = tailFile
               nLines = 10
            self.tailFiles.append("%s:%d" % (fileName,nLines))

      nLines = self.commandParser.getOption('tailStdoutLength')
      if nLines:
         self.tailFiles.append("%s:%s" % ('#STDOUT#',nLines[-1]))
      nLines = self.commandParser.getOption('tailStderrLength')
      if nLines:
         self.tailFiles.append("%s:%s" % ('#STDERR#',nLines[-1]))

      return(exitCode)


   def findInputFileOrDirectory(self,
                                currentWorkingDirectory,
                                inputFileOrDirectory):
      fileOrDirectory = ""
      stripedInputFileOrDirectory = inputFileOrDirectory.replace('@:','')
      if stripedInputFileOrDirectory.startswith(os.sep):
         relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
      else:
         relativeInputFileOrDirectory = stripedInputFileOrDirectory
      if self.clientWorkDirectory.startswith(os.sep):
         relativeClientWorkDirectory = self.clientWorkDirectory[1:]
      else:
         relativeClientWorkDirectory = self.clientWorkDirectory

      if self.clientTransferPath:
         candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeClientWorkDirectory,relativeInputFileOrDirectory)
         if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
            fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory

      if not fileOrDirectory:
         if self.clientWorkDirectory == currentWorkingDirectory:
            candidateFileOrDirectory = stripedInputFileOrDirectory
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientWorkDirectory,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
            else:
               candidateFileOrDirectory = os.path.join(currentWorkingDirectory,relativeInputFileOrDirectory)
               if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
                  fileOrDirectory = candidateFileOrDirectory

      if fileOrDirectory:
         if stripedInputFileOrDirectory != inputFileOrDirectory:
            fileOrDirectory = '@:' + fileOrDirectory

      return(fileOrDirectory)


   def isClientTransferFileOrDirectory(self,
                                       inputFileOrDirectory):
      clientTransferFileorDirectory = False
      stripedInputFileOrDirectory = inputFileOrDirectory.replace('@:','')
      if stripedInputFileOrDirectory.startswith(os.sep):
         relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
      else:
         relativeInputFileOrDirectory = stripedInputFileOrDirectory
      if self.clientWorkDirectory.startswith(os.sep):
         relativeClientWorkDirectory = self.clientWorkDirectory[1:]
      else:
         relativeClientWorkDirectory = self.clientWorkDirectory

      if self.clientTransferPath:
         candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeClientWorkDirectory,relativeInputFileOrDirectory)
         if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
            clientTransferFileorDirectory = True
         else:
            candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               clientTransferFileorDirectory = True

      return(clientTransferFileorDirectory)


   class CommandError(Exception):
      pass

   class InstancesError(Exception):
      pass

   class InstanceError(Exception):
      pass


   def getUseEnvironment(self):
      useEnvironment = ""
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = filter(reChoice.search,os.environ.keys())
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         includedScripts = []
         useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
         useEnvironment += ". %s\n" % (self.useSetup)
         if "ENVIRON_CONFIG_DIRS" in os.environ:
            useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
         for environmentVar in environmentVars:
            includedScript = os.environ[environmentVar]
            if not includedScript in includedScripts:
               useEnvironment += "use -e -r " + includedScript + "\n"
               includedScripts.append(includedScript)
         del includedScripts

      return(useEnvironment)


   def getInstanceDestinations(self,
                               currentWorkingDirectory):
      exitCode                 = 0
      userDestinations         = []
      instanceDestinations     = []
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()

      try:
         executable = self.enteredCommandArguments[0]
         if "/" in executable:
            candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
            if candidateFileOrDirectory:
               self.enteredCommandArguments[0] = candidateFileOrDirectory

         executable         = self.enteredCommandArguments[0]
         transferExecutable = True
         if "/" in executable:
            if   not os.path.isfile(executable):
               self.logger.log(logging.ERROR,getLogMessage("Missing executable: %s" % (executable)))
               self.__writeToStderr("Missing executable: %s\n" % (executable))
               if not exitCode:
                  exitCode = 1
                  raise JobDistributor.CommandError
            elif not os.access(executable,os.X_OK):
               self.logger.log(logging.ERROR,getLogMessage("Permission denied on executable: %s" % (executable)))
               self.__writeToStderr("Permission denied on executable: %s\n" % (executable))
               if not exitCode:
                  exitCode = 1
                  raise JobDistributor.CommandError
         else:
            transferExecutable = False

         if transferExecutable:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
            if not submissionAllowed:
               self.logger.log(logging.ERROR,getLogMessage("Access to %s denied for %s" % (executable,self.hubUserName)))
               self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
               if not exitCode:
                  exitCode = 9
                  raise JobDistributor.CommandError
         else:
            applicationIsStaged = False
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  applicationIsStaged      = True
                  executableClassification = 'staged'
            if not applicationIsStaged:
               submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
               if not submissionAllowed:
                  self.logger.log(logging.ERROR,getLogMessage("Access to %s denied for %s" % (executable,self.hubUserName)))
                  self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
                  if not exitCode:
                     exitCode = 9
                     raise JobDistributor.CommandError

         enabledSites = self.sitesInfo.getEnabledSites()
         self.toolsInfo.purgeDisabledSites(enabledSites)
         if not self.isParametric:
            pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
            self.toolsInfo.purgePegasusSites(pegasusSites)
            del pegasusSites

         userMappedDestinations = []
         userGridDestinations   = []
         for userDestination in self.userDestinations:
            if   self.sitesInfo.siteExists(userDestination):
               userMappedDestinations.append(userDestination)
            elif self.sitesInfo.siteExists('grid'):
               if 'grid' in enabledSites:
                  userGridDestinations.append(userDestination)
                  if not 'grid' in userMappedDestinations:
                     userMappedDestinations.append('grid')
            else:
               errorMessage = "Requested execution venue does not exist.\n" + \
                              "Please select another site."
               self.logger.log(logging.ERROR,getLogMessage(errorMessage))
               self.__writeToStderr(errorMessage + "\n")
               exitCode = 10
               raise JobDistributor.CommandError

         userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                executableClassification,userMappedDestinations) + \
                            self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                '*',userMappedDestinations)

         if 'grid' in userDestinations and len(userGridDestinations) > 0:
            userDestinations.remove('grid')
            for userGridDestination in userGridDestinations:
               userDestinations.append(userGridDestination)

         if len(self.userDestinations) > 0 and len(userDestinations) == 0:
            errorMessage = "All sites listed as possible execution hosts are unable to meet\n" + \
                           "the specified access requirements.  Please select another site."
            self.logger.log(logging.ERROR,getLogMessage(errorMessage))
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 10
            raise JobDistributor.CommandError

      except JobDistributor.CommandError:
         self.jobStatistics[0]['exitCode'] = exitCode
         raise

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination  = ""
      destinations = []

      try:
         if not transferExecutable:
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,
                                                                                 remoteProbeMonitor=self.remoteMonitors['probe'])
                  toolInfo = self.toolsInfo.selectTool(executable,expandedUserDestinations,self.remoteMonitors['probe'])
                  del expandedUserDestinations
                  if len(toolInfo['destinations']) == 0:
                     message = "The combination of destination and executable is not properly specified."
                     self.logger.log(logging.ERROR,getLogMessage(message))
                     self.__writeToStderr("The combination of destination and executable is not properly specified.\n")
                     exitCode = 1
                     raise JobDistributor.InstancesError

                  if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
                     if self.managersInfo.managerExists(toolInfo['remoteManager']):
                        self.managerSpecified = True
                        self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("Invalid manager %s specified for tool %s.\n" % (toolInfo['remoteManager'],executable))
                        exitCode = 1
                        raise JobDistributor.InstancesError
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Access to %s is restricted by user or group" % (executable)))
                  self.__writeToStderr("Access to %s is restricted by user or group\n" % (executable))
                  exitCode = 1
                  raise JobDistributor.InstancesError

         if   len(toolInfo['destinations']) == 1:
# if tool is staged and only one site is available
            destination = toolInfo['destinations'][0]
            instanceDestinations.append(destination)
         elif len(toolInfo['destinations']) == 0:
# if tool is not staged and only one user requested site is available.
            destinations = self.sitesInfo.selectSites(userDestinations)
            if len(destinations) == 1:
               destination = destinations[0]
               instanceDestinations.append(destination)

         if destination == "":
            if len(toolInfo['destinations']) > 1:
# if tool is staged pick site(s)
               if not self.isParametric:
                  destinations = self.sitesInfo.selectSites(toolInfo['destinations'])
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
               else:
                  pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                     toolInfo['destinations'])
                  destinations = self.sitesInfo.selectSites(pegasusSites)
                  del pegasusSites
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
                  else:
                     nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                              toolInfo['destinations'])
                     destinations = self.sitesInfo.selectSites(nonPegasusSites)
                     del nonPegasusSites
                     if len(destinations) > 0:
                        for instanceDestination in destinations:
                           instanceDestinations.append(instanceDestination)
                        destination = 'Sweep'

         if destination == "":
            if len(userDestinations) > 1:
# if user requested site is available
               if not self.isParametric:
                  nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','RUNJOB',
                                                                          userDestinations)
                  destinations = self.sitesInfo.selectSites(nonRunjobSites)
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
               else:
                  pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                     userDestinations)
                  destinations = self.sitesInfo.selectSites(pegasusSites)
                  del pegasusSites
                  if len(destinations) > 0:
                     destinationIndex = random.randint(0,len(destinations)-1)
                     destination = destinations[destinationIndex]
                     instanceDestinations.append(destination)
                  else:
                     nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                              userDestinations)
                     destinations = self.sitesInfo.selectSites(nonPegasusSites)
                     del nonPegasusSites
                     if len(destinations) > 0:
                        for instanceDestination in destinations:
                           instanceDestinations.append(instanceDestination)
                        destination = 'Sweep'

         if not self.isParametric:
            if   self.sitesInfo.siteExists(destination):
# selected an existing site
               pass
            elif self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
# defer to tool or user specified runjob grid site selection
               if len(instanceDestinations) == 0:
                  instanceDestinations.append('2BeDetermined')
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteMonitors['probe']:
                  self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
               destinations = self.sitesInfo.selectUndeclaredSites(enabledSites,
                                                                   executableClassification,
                                                                   self.maximumSelectedSites)
               if len(destinations) > 0:
                  for instanceDestination in destinations:
                     instanceDestinations.append(instanceDestination)
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Currently no sites are available"))
                  self.__writeToStderr("Currently no sites are available\n")
                  exitCode = 1
                  raise JobDistributor.InstancesError
         else:
            if   self.sitesInfo.siteExists(destination):
# selected an existing site
               pass
            elif destination == 'Sweep':
# non pegausus sweep using one or more destinations
               pass
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteMonitors['probe']:
                  self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
               if self.pegasusVersion:
                  pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
               else:
                  pegasusSites = []
               destinations = self.sitesInfo.selectUndeclaredSites(pegasusSites,
                                                                   executableClassification,
                                                                   1)
               del pegasusSites
               if len(destinations) > 0:
                  destination = destinations[0]
                  instanceDestinations.append(destination)
               else:
                  nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                           enabledSites)
                  destinations = self.sitesInfo.selectUndeclaredSites(nonPegasusSites,
                                                                      executableClassification,
                                                                      self.maximumSelectedSites)
                  del nonPegasusSites
                  if len(destinations) > 0:
                     destination = 'Sweep'
                     for instanceDestination in destinations:
                        instanceDestinations.append(instanceDestination)
                  else:
                     self.logger.log(logging.ERROR,getLogMessage("Currently no sites are available"))
                     self.__writeToStderr("Currently no sites are available\n")
                     exitCode = 1
                     raise JobDistributor.InstancesError

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode
         raise

      return(userDestinations,instanceDestinations,
             executable,transferExecutable,executableClassification,toolInfo)


   def buildRedundantJobInstances(self,
                                  userDestinations,
                                  instanceDestinations,
                                  currentWorkingDirectory,
                                  executable,
                                  transferExecutable,
                                  executableClassification,
                                  toolInfo):
      enteredCommand = self.commandParser.getEnteredCommand()
      userInputFiles = []
      for inputFile in self.inputFiles:
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,inputFile)
         if candidateFileOrDirectory:
            if not candidateFileOrDirectory in userInputFiles:
               userInputFiles.append(candidateFileOrDirectory)

      if len(self.enteredCommandArguments) > 1:
         for arg in self.enteredCommandArguments[1:]:
            if not arg.startswith('-'):
               candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
               if candidateFileOrDirectory:
                  if not candidateFileOrDirectory in userInputFiles:
                     userInputFiles.append(candidateFileOrDirectory)
            else:
               lexArgs = shlex.split(arg)
               for lexArg in lexArgs:
                  if '=' in lexArg:
                     arglets = lexArg.split('=')
                     for arglet in arglets[1:]:
                        candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                        if candidateFileOrDirectory:
                           if not candidateFileOrDirectory in userInputFiles:
                              userInputFiles.append(candidateFileOrDirectory)
                  else:
                     candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                     if candidateFileOrDirectory:
                        if not candidateFileOrDirectory in userInputFiles:
                           userInputFiles.append(candidateFileOrDirectory)

      filesToRemove      = []
      emptyFilesToRemove = []
      runDirectoryChecked = False
      nInstanceIdDigits = max(2,int(math.log10(max(1,len(instanceDestinations)))+1))
      instance = 0
      for destination in instanceDestinations:
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)
         exitCode         = 0
         inputFiles       = copy.copy(userInputFiles)
         nCpus            = copy.copy(self.nCpus)
         ppn              = copy.copy(self.ppn)
         wallTime         = copy.copy(self.wallTime)
         environment      = copy.copy(self.environment)
         managerSpecified = copy.copy(self.managerSpecified)
         managerInfo      = copy.copy(self.managerInfo)

         siteInfo = self.sitesInfo.getDefaultSiteInfo()

         gridsite        = "Unknown"
         nNodes          = "1"
         arguments       = []
         cloudInstanceId = ""

         try:
            if   (self.sitesInfo.siteExists(destination)) and \
                  self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
               siteInfo = self.sitesInfo.getSiteInfo(destination)
               if self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                  gridsite = self.sitesInfo.getSitePegasusSites(destination)
               if not managerSpecified and siteInfo['remoteManager'] != "":
                  if self.managersInfo.managerExists(siteInfo['remoteManager']):
                     managerSpecified = True
                     managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                  else:
                     message = "Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination)
                     self.logger.log(logging.ERROR,getLogMessage(message))
                     self.__writeToStderr("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],
                                                                                           destination))
                     exitCode = 1
                     raise JobDistributor.InstanceError
               if nCpus == 0:
                  nCpus = int(siteInfo['remotePpn'])
               if ppn == "":
                  ppn = siteInfo['remotePpn']
               nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
               if int(nNodes) == 1:
                  if int(nCpus) < int(ppn):
                     ppn = str(nCpus)
            elif (self.sitesInfo.siteExists(destination)) and \
                  self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
               siteInfo = self.sitesInfo.getSiteInfo(destination)
               runjobVenues = self.sitesInfo.getSiteMembers(destination)
               gridsite     = " ".join(runjobVenues)
            else:
               if   self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
                  siteInfo = self.sitesInfo.getSiteInfo('grid')
                  if   len(toolInfo['destinations']) > 0:
                     gridsite = " ".join(toolInfo['destinations'])
                  elif len(userDestinations) > 0:
                     gridsite = " ".join(userDestinations)
                  else:
                     gridsite = destination
               elif self.runjobExists or transferExecutable:
                  self.logger.log(logging.ERROR,getLogMessage("Invalid destination selection"))
                  self.__writeToStderr("Invalid destination selection\n")
                  exitCode = 1
                  raise JobDistributor.InstanceError

            if siteInfo['remoteBatchSystem'] != 'SCRIPT':
               if not runDirectoryChecked:
                  runDirectory = os.path.join(currentWorkingDirectory,self.runName)
                  if os.path.exists(runDirectory):
                     self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (runDirectory)))
                     self.__writeToStderr("Run directory %s exists\n" % (runDirectory))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  runDirectoryChecked = True

            if transferExecutable:
               if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                  (not executableClassification in siteInfo['executableClassificationsAllowed']):
                  errorMessage = "The selected site does not meet the specified\n" + \
                                 "application access requirements.  Please select another site."
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

            localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
            self.executeCommand("touch " + localJobIdFile)
            inputFiles.append(localJobIdFile)
            filesToRemove.append(localJobIdFile)

            if transferExecutable:
               inputFiles.append(executable)

            if siteInfo['stageFiles']:
               stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
               tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
               self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
               tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
               if tarExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                  self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                  self.__writeToStderr(tarStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError

               filesToRemove.append(stageInTarFile)

               remoteArgs = []
               remoteArgs.append(self.enteredCommandArguments[0])

               if len(self.enteredCommandArguments) > 1:
                  tarFiles = tarStdOutput.strip().split('\n')
                  for arg in self.enteredCommandArguments[1:]:
                     if not arg.startswith('-'):
                        argIsFile = False
                        for tarFile in tarFiles:
                           if re.match(".*" + tarFile.replace(' ','\ ') + "$",arg):
                              rarg = os.path.join('.',tarFile.replace(' ','\ '))
                              argIsFile = True
                              isClientTransfer = self.isClientTransferFileOrDirectory(arg)
                              if isClientTransfer and siteInfo['sharedUserSpace']:
                                 clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
                                 if not os.path.isfile(rarg):
                                    shutil.copy2(clientTransferPath,rarg)
                              break
                        if not argIsFile:
                           rarg = arg
                     else:
                        lexArgs = shlex.split(arg)
                        rlexArgs = []
                        for lexArg in lexArgs:
                           if '=' in lexArg:
                              arglets = lexArg.split('=')
                              rarglets = [arglets[0]]
                              for arglet in arglets[1:]:
                                 argletIsFile = False
                                 for tarFile in tarFiles:
                                    if re.match(".*" + tarFile.replace(' ','\ ') + "$",arglet):
                                       rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                       rarglets.append(rarglet)
                                       argletIsFile = True
                                       isClientTransfer = self.isClientTransferFileOrDirectory(arglet)
                                       if isClientTransfer and siteInfo['sharedUserSpace']:
                                          clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                                          if not os.path.isfile(rarglet):
                                             shutil.copy2(clientTransferPath,rarglet)
                                       break
                                 if not argletIsFile:
                                    rarglets.append(arglet)
                              rlexArgs.append('='.join(rarglets))
                           else:
                              argletIsFile = False
                              for tarFile in tarFiles:
                                 if re.match(".*" + tarFile.replace(' ','\ ') + "$",lexArg):
                                    rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                    argletIsFile = True
                                    isClientTransfer = self.isClientTransferFileOrDirectory(lexArg)
                                    if isClientTransfer and siteInfo['sharedUserSpace']:
                                       clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                                       if not os.path.isfile(rarglet):
                                          shutil.copy2(clientTransferPath,rarglet)
                                    break
                              if argletIsFile:
                                 rlexArgs.append(rarglet)
                              else:
                                 rlexArgs.append(lexArg)
                        rarg = ' '.join(rlexArgs)
                     remoteArgs.append(rarg)

               remoteCommand = " ".join(remoteArgs)
               self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

               arguments = remoteArgs[1:]
            else:
# if not shared work directory?
               stageInTarFile = ""
               remoteCommand = " ".join(self.enteredCommandArguments)
               self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

               arguments = self.enteredCommandArguments[1:]

            if siteInfo['passUseEnvironment']:
               useEnvironment = self.getUseEnvironment()
            else:
               useEnvironment = ""

            managedEnvironment = managerInfo['environment']
            if len(managedEnvironment) > 0:
               for environmentVariableValue in managedEnvironment:
                  environmentVariable = ""
                  value               = ""
                  nParts = len(environmentVariableValue.split('='))
                  if   nParts == 1:
                     environmentVariable = environmentVariableValue.strip()
                     if environmentVariable in os.environ:
                        value = os.environ[environmentVariable]
                  elif nParts == 2:
                     environmentVariable,value = environmentVariableValue.split('=')
                     environmentVariable = environmentVariable.strip()
                     value               = value.strip()
                     if value == "":
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]

                  if environmentVariable != "" and value != "":
                     if environment.find(environmentVariable + '=') < 0:
                        environment += environmentVariable + "=" + value + " "
               environment = environment.strip()

            if len(toolInfo['environment']) > 0:
               for environmentVariableValue in toolInfo['environment']:
                  environmentVariable = ""
                  value               = ""
                  nParts = len(environmentVariableValue.split('='))
                  if nParts == 2:
                     environmentVariable,value = environmentVariableValue.split('=')
                     environmentVariable = environmentVariable.strip()
                     value               = value.strip()
                     if value == "":
                        value = " "
                  else:
                     self.logger.log(logging.ERROR,getLogMessage("Tool environment variable must be of the form Variable = Value."))
                     self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                  if environmentVariable != "" and value != "":
                     environment += environmentVariable + "=" + value + " "
               environment = environment.strip()

            if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
               if nCpus == 1:
                  self.isMultiCoreRequest = False
               else:
                  errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

            timeHistoryLogs = {}
            timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
            timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
            timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
            timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
            timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
            timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
            timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
            timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
            timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

            if   siteInfo['venueMechanism'] == 'local':
               self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                         self.hubUserName,self.hubUserId,self.submitterClass,
                                                         currentWorkingDirectory,self.session,self.distributorPid,
                                                         self.batchCommands,self.isParametric,
                                                         self.runName,self.localJobId,instanceId,destination,
                                                         self.tailFiles,enteredCommand,gridsite,
                                                         stageInTarFile,
                                                         transferExecutable,executable,self.appsAccessInfo,
                                                         self.stdinput,arguments,useEnvironment,environment,
                                                         self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                         managerInfo,nCpus,nNodes,ppn,wallTime,
                                                         self.quotaLimit,
                                                         self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                         timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
            elif siteInfo['venueMechanism'] == 'ssh':
               self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                       self.hubUserName,self.hubUserId,self.submitterClass,
                                                       currentWorkingDirectory,self.session,self.distributorPid,
                                                       self.batchCommands,self.isParametric,
                                                       self.runName,self.localJobId,instanceId,destination,
                                                       self.tailFiles,enteredCommand,gridsite,
                                                       self.tunnelsInfo,
                                                       cloudInstanceId,
                                                       stageInTarFile,
                                                       transferExecutable,executable,self.appsAccessInfo,
                                                       self.stdinput,arguments,useEnvironment,environment,
                                                       self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                       managerInfo,nCpus,nNodes,ppn,wallTime,
                                                       self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                       self.quotaLimit,
                                                       timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
            elif siteInfo['venueMechanism'] == 'gsissh':
               self.jobs[instance] = VenueMechanismGsiSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                          managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

            exitCode,scriptFiles = self.jobs[instance].createScripts()
            if exitCode:
               raise JobDistributor.InstanceError

            if siteInfo['stageFiles']:
               if len(scriptFiles) > 0:
                  tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

               niceCommand = "nice -n 19 gzip " + stageInTarFile
               self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
               gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
               if gzipExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                  self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                  self.__writeToStderr(gzipStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               stageInTarFile += '.gz'
               filesToRemove.append(stageInTarFile)

            self.jobs[instance].stageFilesToInstanceDirectory()

            self.waitForJobsInfo[instance] = {}
            waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
            self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
            self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
            self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
            self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
            self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory
            self.waitForJobsInfo[instance]['recentJobStatus']       = '?'

         except JobDistributor.InstanceError:
            if instance in self.jobs:
               self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
               self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
               self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
            else:
               self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
               self.jobStatistics[0]['venue']                  = siteInfo['venue']
               self.jobStatistics[0]['exitCode']               = exitCode

         stdFile = "%s.stderr" % (self.runName)
         emptyFilesToRemove.append(stdFile)
         for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory','boinc':
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueue,self.runName,instanceId,outFile)
               emptyFilesToRemove.append(batchQueueOutputFile)

      nInstances = instance
      nonBatchJobs = []
      for instance in self.waitForJobsInfo:
         if not self.waitForJobsInfo[instance]['isBatchJob']:
            nonBatchJobs.append(instance)
         self.jobs[instance].registerJob()
         self.waitForJobsInfo[instance]['state'] = 'held'
      if self.progressReport == 'text':
         if nInstances == 1:
            self.__writeToStdout("Run %d registered %d job instance. %s\n" % \
                                       (self.jobId,nInstances,time.ctime()))
         else:
            self.__writeToStdout("Run %d registered %d job instances. %s\n" % \
                                        (self.jobId,nInstances,time.ctime()))

      nBatchJobsHeld = nInstances-len(nonBatchJobs)
      nJobsReleased = 0
      while nonBatchJobs and (self.successfulInstance == None):
         instance = random.choice(nonBatchJobs)
         if self.waitForJobsInfo[instance]['state'] == 'held':
            if self.jobs[instance].isJobReleased():
               self.waitForJobsInfo[instance]['state'] = 'released'
               nJobsReleased += 1
               if self.progressReport == 'text':
                  self.__writeToStdout("Run %d instance %d released for submission %s.\n" % \
                                                        (self.jobId,instance,time.ctime()))
               if self.jobs[instance].sendFiles():
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'R'
                  if self.jobs[instance].executeJob():
                     self.jobs[instance].postProcess()
                     self.jobs[instance].retrieveFiles()
                     self.jobs[instance].cleanupFiles()
                     self.successfulInstance = instance
               self.jobs[instance].recordJobStatistics()
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               nonBatchJobs.remove(instance)

      jobsRunning = False
      if self.successfulInstance == None:
         while (jobsRunning or nBatchJobsHeld > 0) and \
               (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['state'] == 'held':
                     if self.jobs[instance].isJobReleased():
                        self.waitForJobsInfo[instance]['state'] = 'released'
                        nJobsReleased += 1
                        if self.progressReport == 'text':
                           self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                                                 (self.jobId,instance,time.ctime()))
                        if self.jobs[instance].sendFiles():
                           if self.jobs[instance].executeJob():
                              waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                              self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                              self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                              self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                           else:
                              self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                        else:
                           self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForBatchJobs(self.waitForJobsInfo,
                                                                                   self.progressReport,
                                                                                   self.abortGlobal)

            nBatchJobsHeld = 0
            jobsRunning = False
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
                     if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                        jobsRunning = True
               else:
                  nBatchJobsHeld += 1

            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance

      if self.abortGlobal['abortAttempted']:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                  self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

      if self.successfulInstance != None:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                  self.jobs[instance].killScripts(signal.SIGUSR1)
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if   self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               elif instance != self.successfulInstance:
                  self.jobs[instance].cleanupFiles()

      if self.successfulInstance != None:
         self.jobs[self.successfulInstance].recoverFiles(False)
         fileToMove = "%s_%02d.stdout" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stdout" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.stderr" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stderr" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.SUCCESS" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.SUCCESS" % (self.runName)
            os.rename(fileToMove,fileNewName)
      else:
         for instance in self.waitForJobsInfo:
            self.jobs[instance].recoverStdFiles()
         if len(self.jobs) > 0:
            recoverInstance = random.choice(self.jobs.keys())
            self.jobs[recoverInstance].recoverFiles(True)

      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['state'] == 'held':
            self.jobs[instance].removeJobRegistration()

      for instance in self.waitForJobsInfo:
         self.jobs[instance].removeInstanceDirectory()

      for fileToRemove in filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildPegasusJobInstances(self,
                                destination,
                                currentWorkingDirectory,
                                executable,
                                transferExecutable,
                                executableClassification,
                                toolInfo):
      try:
         filesToRemove      = []
         emptyFilesToRemove = []
         jobPath = os.path.join(currentWorkingDirectory,self.runName)
         if os.path.exists(jobPath):
            self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (jobPath)))
            self.__writeToStderr("Run directory %s exists\n" % (jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(jobPath):
            os.mkdir(jobPath)
         if self.progressReport == 'curses' or self.reportMetrics:
            parameterCombinationsPath = os.path.join(jobPath,'remoteParameterCombinations.csv')
         else:
            parameterCombinationsPath = os.path.join(jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(parameterCombinationsPath)

         instanceError = False
         inputsPath        = os.path.join(jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError,err:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = []
            for inputFile in self.inputFiles:
               template = ParameterTemplate(inputFile)
               try:
                  inputFile = template.substitute(substitutions)
               except KeyError,err:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
               candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,inputFile)
               if candidateFileOrDirectory:
                  if not candidateFileOrDirectory in inputFiles:
                     inputFiles.append(candidateFileOrDirectory)

            if len(self.enteredCommandArguments) > 1:
               for arg in self.enteredCommandArguments[1:]:
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  if not arg.startswith('-'):
                     candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
                     if candidateFileOrDirectory:
                        if not candidateFileOrDirectory in inputFiles:
                           inputFiles.append(candidateFileOrDirectory)
                  else:
                     lexArgs = shlex.split(arg)
                     for lexArg in lexArgs:
                        if '=' in lexArg:
                           arglets = lexArg.split('=')
                           for arglet in arglets[1:]:
                              candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                              if candidateFileOrDirectory:
                                 if not candidateFileOrDirectory in inputFiles:
                                    inputFiles.append(candidateFileOrDirectory)
                        else:
                           candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                           if candidateFileOrDirectory:
                              if not candidateFileOrDirectory in inputFiles:
                                 inputFiles.append(candidateFileOrDirectory)

            if len(inputFiles) > 1:
               filteredInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile.replace('@:','')
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(filteredInputFile)
               for inputFile in inputFiles:
                  if not '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(inputFile)
               del inputFiles
               inputFiles = filteredInputFiles

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  gridsite = self.sitesInfo.getSitePegasusSites(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("%s\n" % (message))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               else:
                  errorMessage = "Invalid destination selection: %s" % (destination)
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected site does not meet the specified\n" + \
                                    "application access requirements.  Please select another site."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     try:
                        fpInputFile = open(inputFile,'r')
                        try:
                           inputText = fpInputFile.readlines()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                        else:
                           template = ParameterTemplate(''.join(inputText))
                           try:
                              inputText = template.substitute(substitutions)
                           except KeyError,err:
                              self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                              exitCode = 1
                              raise JobDistributor.InstanceError
                           instanceInputsPath = os.path.join(inputsPath,instanceId)
                           if not os.path.isdir(instanceInputsPath):
                              os.makedirs(instanceInputsPath)
                           inputName = os.path.basename(inputFile)
                           inputPath = os.path.join(instanceInputsPath,inputName)
                           try:
                              fpInputPath = open(inputPath,'w')
                              try:
                                 fpInputPath.writelines(inputText)
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                              else:
                                 substitutedInputFiles.append(inputPath)
                              finally:
                                 fpInputPath.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                        finally:
                           fpInputFile.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  arg = self.enteredCommandArguments[0]
                  template = ParameterTemplate(arg)
                  try:
                     rarg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  remoteArgs.append(rarg)

                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        template = ParameterTemplate(arg)
                        try:
                           arg = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        if not arg.startswith('-'):
                           argIsFile = False
                           for tarFile in tarFiles:
                              if re.match(".*" + tarFile.replace(' ','\ ') + "$",arg):
                                 rarg = os.path.join('.',tarFile.replace(' ','\ '))
                                 argIsFile = True
                                 break
                           if not argIsFile:
                              rarg = arg
                        else:
                           lexArgs = shlex.split(arg)
                           rlexArgs = []
                           for lexArg in lexArgs:
                              if '=' in lexArg:
                                 arglets = lexArg.split('=')
                                 rarglets = [arglets[0]]
                                 for arglet in arglets[1:]:
                                    argletIsFile = False
                                    for tarFile in tarFiles:
                                       if re.match(".*" + tarFile.replace(' ','\ ') + "$",arglet):
                                          rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                          rarglets.append(rarglet)
                                          argletIsFile = True
                                          break
                                    if not argletIsFile:
                                       rarglets.append(arglet)
                                 rlexArgs.append('='.join(rarglets))
                              else:
                                 argletIsFile = False
                                 for tarFile in tarFiles:
                                    if re.match(".*" + tarFile.replace(' ','\ ') + "$",lexArg):
                                       rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                       argletIsFile = True
                                       break
                                 if argletIsFile:
                                    rlexArgs.append(rarglet)
                                 else:
                                    rlexArgs.append(lexArg)
                           rarg = ' '.join(rlexArgs)
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

                  arguments = remoteArgs[1:]
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  remoteArgs = []
                  for arg in self.enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        rarg = template.substitute(substitutions)
                     except KeyError,err:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

                  arguments = remoteArgs[1:]

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        message = "Tool environment variable must be of the form Variable = Value."
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,self.appsAccessInfo,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                            managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                          managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               self.waitForJobsInfo[instance]['recentJobStatus'] = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         nInstances      = instance
#        executeInstance = nInstances+1
         executeInstance = 0

         timeHistoryLogs = {}
         timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
         timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
         timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,"0")
         timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,"0")
         timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,"0")
         timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,"0")
         timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,"0")
         timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,"0")
         timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,"0")

         enteredCommand = self.commandParser.getEnteredCommand()

         if   siteInfo['venueMechanism'] == 'local':
            self.jobs[executeInstance] = VenueMechanismLocal(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.submitterClass,
                                                             currentWorkingDirectory,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,"WF;%d" % (nInstances),destination,
                                                             self.tailFiles,enteredCommand,gridsite,
                                                             stageInTarFile,
                                                             transferExecutable,executable,self.appsAccessInfo,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                             managerInfo,nCpus,nNodes,ppn,wallTime,
                                                             self.quotaLimit,
                                                             self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                             timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
         elif siteInfo['venueMechanism'] == 'ssh':
            self.jobs[executeInstance] = VenueMechanismSsh(self.remoteMonitors,
                                                           self.hubUserName,self.hubUserId,self.submitterClass,
                                                           currentWorkingDirectory,self.session,self.distributorPid,
                                                           self.batchCommands,self.isParametric,
                                                           self.runName,self.localJobId,"WF;%d" % (nInstances),destination,
                                                           self.tailFiles,enteredCommand,gridsite,
                                                           self.tunnelsInfo,
                                                           cloudInstanceId,
                                                           stageInTarFile,
                                                           transferExecutable,executable,self.appsAccessInfo,
                                                           self.stdinput,arguments,useEnvironment,environment,
                                                           self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                           managerInfo,nCpus,nNodes,ppn,wallTime,
                                                           self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                           self.quotaLimit,
                                                           timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

         exitCode,scriptFiles = self.jobs[executeInstance].createScripts()
         if exitCode:
            raise JobDistributor.InstancesError

         self.waitForJobsInfo[executeInstance] = {}
         waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
         self.waitForJobsInfo[executeInstance]['isBatchJob']            = waitForJobInfo['isBatchJob']
         self.waitForJobsInfo[executeInstance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
         self.waitForJobsInfo[executeInstance]['remoteBatchSystem']     = self.jobs[executeInstance].remoteBatchSystem
         self.waitForJobsInfo[executeInstance]['instanceDirectory']     = self.jobs[executeInstance].instanceDirectory
         self.waitForJobsInfo[executeInstance]['scratchDirectory']      = self.jobs[executeInstance].scratchDirectory
         self.waitForJobsInfo[executeInstance]['recentJobStatus']       = '?'

         stdFile = "%s.stderr" % (self.runName)
         emptyFilesToRemove.append(stdFile)
         for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory','boinc':
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s.%s" % (batchQueue,self.runName,outFile)
               emptyFilesToRemove.append(batchQueueOutputFile)

         nonBatchJobs = []
         if not self.waitForJobsInfo[executeInstance]['isBatchJob']:
            nonBatchJobs.append(executeInstance)
         self.jobs[executeInstance].registerJob()
         self.waitForJobsInfo[executeInstance]['state'] = 'held'
         if self.progressReport == 'text':
            self.__writeToStdout("Run %d registered 1 job instance. %s\n" % \
                                                 (self.jobId,time.ctime()))

         nBatchJobsHeld = 1-len(nonBatchJobs)
         nJobsReleased = 0
         jobsRunning = False
         while (jobsRunning or nBatchJobsHeld > 0) and \
               (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
            if self.waitForJobsInfo[executeInstance]['isBatchJob']:
               if self.waitForJobsInfo[executeInstance]['state'] == 'held':
                  if self.jobs[executeInstance].isJobReleased():
                     self.waitForJobsInfo[executeInstance]['state'] = 'released'
                     nJobsReleased += 1
                     if self.progressReport == 'text':
                        self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                                       (self.jobId,executeInstance,time.ctime()))
                     if self.jobs[executeInstance].sendFiles():
                        if self.jobs[executeInstance].executeJob():
                           waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
                           self.waitForJobsInfo[executeInstance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                           self.waitForJobsInfo[executeInstance]['knownSite']       = waitForJobInfo['knownSite']
                           self.waitForJobsInfo[executeInstance]['recentJobStatus'] = '?'
                        else:
                           self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'
                     else:
                        self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForPegasusWorkflowJobs(self.waitForJobsInfo,
                                                                                             nInstances,parameterCombinationsPath,
                                                                                             self.progressReport,self.abortGlobal)
            nBatchJobsHeld = 0
            jobsRunning = False
            if self.waitForJobsInfo[executeInstance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[executeInstance]:
                  executionVenue = self.waitForJobsInfo[executeInstance]['recentJobSite']
                  self.jobs[executeInstance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance

         if self.abortGlobal['abortAttempted']:
            if self.waitForJobsInfo[executeInstance]['state'] == 'released':
               if self.waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
                  self.jobs[executeInstance].killScripts(self.abortGlobal['abortSignal'])
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            if self.waitForJobsInfo[executeInstance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[executeInstance]:
                  executionVenue = self.waitForJobsInfo[executeInstance]['recentJobSite']
                  self.jobs[executeInstance].updateVenue(executionVenue)
            if self.waitForJobsInfo[executeInstance]['state'] == 'released':
               if self.waitForJobsInfo[executeInstance]['recentJobStatus'] == 'KD':
                  self.jobs[executeInstance].cleanupFiles()
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

         if self.successfulInstance != None:
            for instance in xrange(1,nInstances+1):
               self.jobs[instance].jobSubmitted = True

         for instance in xrange(1,nInstances+1):
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()

         if self.waitForJobsInfo[executeInstance]['state'] == 'held':
            self.jobs[executeInstance].removeJobRegistration()

         for fileToRemove in filesToRemove:
            if os.path.exists(fileToRemove):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if os.path.exists(fileToRemove):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

         if os.path.isdir(inputsPath):
            shutil.rmtree(inputsPath,True)

         if self.progressReport == 'text' or \
            self.progressReport == 'submit' or self.progressReport == 'pagasus':
            self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (jobPath))

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildSweepJobInstances(self,
                              instanceDestinations,
                              currentWorkingDirectory,
                              executable,
                              transferExecutable,
                              executableClassification,
                              toolInfo):
      try:
         filesToRemove      = []
         emptyFilesToRemove = []
         jobPath = os.path.join(currentWorkingDirectory,self.runName)
         if os.path.exists(jobPath):
            self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (jobPath)))
            self.__writeToStderr("Run directory %s exists\n" % (jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(jobPath):
            os.mkdir(jobPath)
         if self.progressReport == 'curses' or self.reportMetrics:
            parameterCombinationsPath = os.path.join(jobPath,'remoteParameterCombinations.csv')
         else:
            parameterCombinationsPath = os.path.join(jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(parameterCombinationsPath)

         instanceError = False
         inputsPath        = os.path.join(jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError,err:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = []
            for inputFile in self.inputFiles:
               template = ParameterTemplate(inputFile)
               try:
                  inputFile = template.substitute(substitutions)
               except KeyError,err:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
               candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,inputFile)
               if candidateFileOrDirectory:
                  if not candidateFileOrDirectory in inputFiles:
                     inputFiles.append(candidateFileOrDirectory)

            if len(self.enteredCommandArguments) > 1:
               for arg in self.enteredCommandArguments[1:]:
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  if not arg.startswith('-'):
                     candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
                     if candidateFileOrDirectory:
                        if not candidateFileOrDirectory in inputFiles:
                           inputFiles.append(candidateFileOrDirectory)
                  else:
                     lexArgs = shlex.split(arg)
                     for lexArg in lexArgs:
                        if '=' in lexArg:
                           arglets = lexArg.split('=')
                           for arglet in arglets[1:]:
                              candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                              if candidateFileOrDirectory:
                                 if not candidateFileOrDirectory in inputFiles:
                                    inputFiles.append(candidateFileOrDirectory)
                        else:
                           candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                           if candidateFileOrDirectory:
                              if not candidateFileOrDirectory in inputFiles:
                                 inputFiles.append(candidateFileOrDirectory)

            if len(inputFiles) > 1:
               filteredInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile.replace('@:','')
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(filteredInputFile)
               for inputFile in inputFiles:
                  if not '@:' in inputFile:
                     filteredInputFile = '@:' + inputFile
                     if not filteredInputFile in filteredInputFiles:
                        filteredInputFiles.append(inputFile)
               del inputFiles
               inputFiles = filteredInputFiles

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               destination = random.choice(instanceDestinations)
               if   (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],
                                                                                              destination))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               elif (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  runjobVenues = self.sitesInfo.getSiteMembers(destination)
                  gridsite     = " ".join(runjobVenues)
               else:
                  if   self.runjobExists:
                     siteInfo = self.sitesInfo.getSiteInfo('grid')
                     gridsite = destination
                  elif transferExecutable:
                     errorMessage = "Invalid destination selection: %s" % (destination)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected site does not meet the specified\n" + \
                                    "application access requirements.  Please select another site."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     try:
                        fpInputFile = open(inputFile,'r')
                        try:
                           inputText = fpInputFile.readlines()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                        else:
                           template = ParameterTemplate(''.join(inputText))
                           try:
                              inputText = template.substitute(substitutions)
                           except KeyError,err:
                              self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                              exitCode = 1
                              raise JobDistributor.InstanceError
                           instanceInputsPath = os.path.join(inputsPath,instanceId)
                           if not os.path.isdir(instanceInputsPath):
                              os.makedirs(instanceInputsPath)
                           inputName = os.path.basename(inputFile)
                           inputPath = os.path.join(instanceInputsPath,inputName)
                           try:
                              fpInputPath = open(inputPath,'w')
                              try:
                                 fpInputPath.writelines(inputText)
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                              else:
                                 substitutedInputFiles.append(inputPath)
                              finally:
                                 fpInputPath.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                        finally:
                           fpInputFile.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  arg = self.enteredCommandArguments[0]
                  template = ParameterTemplate(arg)
                  try:
                     rarg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  remoteArgs.append(rarg)

                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        template = ParameterTemplate(arg)
                        try:
                           arg = template.substitute(substitutions)
                        except KeyError,err:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        if not arg.startswith('-'):
                           argIsFile = False
                           for tarFile in tarFiles:
                              if re.match(".*" + tarFile.replace(' ','\ ') + "$",arg):
                                 rarg = os.path.join('.',tarFile.replace(' ','\ '))
                                 argIsFile = True
                                 break
                           if not argIsFile:
                              rarg = arg
                        else:
                           lexArgs = shlex.split(arg)
                           rlexArgs = []
                           for lexArg in lexArgs:
                              if '=' in lexArg:
                                 arglets = lexArg.split('=')
                                 rarglets = [arglets[0]]
                                 for arglet in arglets[1:]:
                                    argletIsFile = False
                                    for tarFile in tarFiles:
                                       if re.match(".*" + tarFile.replace(' ','\ ') + "$",arglet):
                                          rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                          rarglets.append(rarglet)
                                          argletIsFile = True
                                          break
                                    if not argletIsFile:
                                       rarglets.append(arglet)
                                 rlexArgs.append('='.join(rarglets))
                              else:
                                 argletIsFile = False
                                 for tarFile in tarFiles:
                                    if re.match(".*" + tarFile.replace(' ','\ ') + "$",lexArg):
                                       rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                       argletIsFile = True
                                       break
                                 if argletIsFile:
                                    rlexArgs.append(rarglet)
                                 else:
                                    rlexArgs.append(lexArg)
                           rarg = ' '.join(rlexArgs)
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

                  arguments = remoteArgs[1:]
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  remoteArgs = []
                  for arg in self.enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        rarg = template.substitute(substitutions)
                     except KeyError,err:
                        self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     if '@:' in rarg:
                        rarg = rarg.replace('@:','')
                     remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

                  arguments = remoteArgs[1:]

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        message = "Tool environment variable must be of the form Variable = Value."
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("Tool environment variable must be of the form Variable = Value.\n")

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,self.appsAccessInfo,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                            managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,self.appsAccessInfo,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                          managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'gsissh':
                  self.jobs[instance] = VenueMechanismGsiSsh(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.submitterClass,
                                                             currentWorkingDirectory,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,instanceId,destination,
                                                             self.tailFiles,enteredCommand,gridsite,
                                                             stageInTarFile,
                                                             transferExecutable,executable,self.appsAccessInfo,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,self.submissionScriptsInfo,
                                                             managerInfo,nCpus,nNodes,ppn,wallTime,
                                                             self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                             self.quotaLimit,
                                                             timeHistoryLogs,self.useSetup,self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stageInTarFile))

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
               self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
               self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
               self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
               self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory
               self.waitForJobsInfo[instance]['recentJobStatus']       = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s_%s.stderr" % (self.runName,instanceId)
            emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stdFile))
            for batchQueue in 'pbs','lsf','ll','sge','slurm','condor','factory','boinc':
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueue,self.runName,instanceId,outFile)
                  emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,batchQueueOutputFile))

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         nInstances = instance
         nonBatchJobs = []
         for instance in self.waitForJobsInfo:
            if not self.waitForJobsInfo[instance]['isBatchJob']:
               nonBatchJobs.append(instance)
            self.jobs[instance].registerJob()
            self.waitForJobsInfo[instance]['state'] = 'held'
         if self.progressReport == 'text':
            if nInstances == 1:
               self.__writeToStdout("Run %d registered %d job. %s\n" % \
                                 (self.jobId,nInstances,time.ctime()))
            else:
               self.__writeToStdout("Run %d registered %d jobs. %s\n" % \
                                  (self.jobId,nInstances,time.ctime()))

         nBatchJobsHeld = nInstances-len(nonBatchJobs)
         nJobsReleased = 0
         jobsRunning = False
         while (jobsRunning or nBatchJobsHeld > 0) and (not self.abortGlobal['abortAttempted']):
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['state'] == 'held':
                     if self.jobs[instance].isJobReleased():
                        self.waitForJobsInfo[instance]['state'] = 'released'
                        nJobsReleased += 1
                        if self.progressReport == 'text':
                           self.__writeToStdout("Run %d job %d released for submission. %s\n" % \
                                                            (self.jobId,instance,time.ctime()))
                        if self.jobs[instance].sendFiles():
                           if self.jobs[instance].executeJob():
                              waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                              self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                              self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                              self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                           else:
                              self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                        else:
                           self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForSweepJobs(self.waitForJobsInfo,
                                                                                   parameterCombinationsPath,
                                                                                   self.progressReport,self.abortGlobal)
            nBatchJobsHeld = 0
            jobsRunning = False
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
                     if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                        jobsRunning = True
               else:
                  nBatchJobsHeld += 1
            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()

         if self.abortGlobal['abortAttempted']:
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                     self.jobs[instance].cleanupFiles()
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'held':
               self.jobs[instance].removeJobRegistration()

         for fileToRemove in filesToRemove:
            if os.path.exists(fileToRemove):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if os.path.exists(fileToRemove):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

         if os.path.isdir(inputsPath):
            shutil.rmtree(inputsPath,True)

         if self.progressReport == 'text' or \
            self.progressReport == 'submit' or self.progressReport == 'pagasus':
            self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (jobPath))

      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildJobInstances(self):
      exitCode = 0
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("buildJobInstances:os.getcwd(): No such file or directory"))
         exitCode = 1
      else:
         try:
            userDestinations,instanceDestinations, \
                executable,transferExecutable,executableClassification,toolInfo = \
                                                                        self.getInstanceDestinations(currentWorkingDirectory)
         except JobDistributor.CommandError:
            exitCode = self.jobStatistics[0]['exitCode']
         except JobDistributor.InstancesError:
            exitCode = self.jobStatistics[0]['exitCode']

      if not exitCode:
         self.remoteMonitors['job']      = RemoteJobMonitor(self.jobListenURI)
         self.remoteMonitors['identity'] = RemoteIdentityManager(self.identityListenURI)
         self.remoteMonitors['tunnel']   = RemoteTunnelMonitor(self.tunnelListenURI)

         if self.isParametric:
            pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                               instanceDestinations)
            if len(pegasusSites) > 0:
               destination = instanceDestinations[0]
               exitCode = self.buildPegasusJobInstances(destination,currentWorkingDirectory,
                                                        executable,transferExecutable,executableClassification,toolInfo)
            else:
               exitCode = self.buildSweepJobInstances(instanceDestinations,currentWorkingDirectory,
                                                      executable,transferExecutable,executableClassification,toolInfo)
               self.jobStatistics[0]['exitCode'] = exitCode
            del pegasusSites
         else:
            exitCode = self.buildRedundantJobInstances(userDestinations,instanceDestinations,currentWorkingDirectory,
                                                       executable,transferExecutable,executableClassification,toolInfo)

      return(exitCode)


   def run(self):
      exitCode = self.parseCommandArguments()
      if not exitCode:
         exitCode = self.setJobId()
         if not exitCode:
            if not self.runName:
               self.runName = self.localJobId
            self.setInfo()
            self.showUsage()
            if self.doRun():
               exitCode =  self.buildJobDescription()
               if not exitCode:
                  exitCode = self.buildJobInstances()

      self.reportExitCondition()

      return(exitCode)


