# @package      hubzero-submit-distributor
# @file         CloudsInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import glob
import json
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class CloudsInfo:
   def __init__(self,
                cloudsPath):
      self.logger = logging.getLogger(__name__)

      self.clouds = {}

      if os.path.isdir(cloudsPath):
         for cloudsInfoPath in glob.iglob(os.path.join(cloudsPath,'*')):
            self.readCloudsInfoFile(cloudsInfoPath)
      else:
         for cloudsInfoPath in glob.iglob(cloudsPath):
            self.readCloudsInfoFile(cloudsInfoPath)

      markedForDeletion = []
      for cloudName in self.clouds:
         if self.clouds[cloudName]['state'] != 'enabled':
            markedForDeletion.append(cloudName)
      for cloudName in markedForDeletion:
         del self.clouds[cloudName]
      del markedForDeletion


   def readCloudsInfoFile(self,
                          cloudsInfoPath):
      cloudPattern    = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile(r'\s*#.*')
      cloudName       = ""

      if os.path.exists(cloudsInfoPath):
         try:
            fpInfo = open(cloudsInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   cloudPattern.match(record):
                        cloudName = cloudPattern.match(record).group(2)
                        self.clouds[cloudName] = {'host':'',
                                                  'port':8444,
                                                  'accessId':'',
                                                  'accessSecret':'',
                                                  'image':'',
                                                  'maximumInstances':2,
                                                  'maximumJobs':1,
                                                  'maximumIdleTime':30,
                                                  'state':'enabled'
                                                 }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.clouds[cloudName]:
                           if   isinstance(self.clouds[cloudName][key],list):
                              self.clouds[cloudName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.clouds[cloudName][key],bool):
                              self.clouds[cloudName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.clouds[cloudName][key],float):
                              self.clouds[cloudName][key] = float(value)
                           elif isinstance(self.clouds[cloudName][key],int):
                              self.clouds[cloudName][key] = int(value)
                           elif isinstance(self.clouds[cloudName][key],dict):
                              try:
                                 sampleKey   = list(self.clouds[cloudName][key].keys())[0]
                                 sampleValue = self.clouds[cloudName][key][sampleKey]
                              except:
                                 try:
                                    self.clouds[cloudName][key] = json.loads(value)
                                 except:
                                    self.clouds[cloudName][key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.clouds[cloudName][key] = {}
               
                              if not self.clouds[cloudName][key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.clouds[cloudName][key][dictKey] = dictValue
                           else:
                              self.clouds[cloudName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for cloud %s" % (key,value,cloudName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Clouds configuration file %s is could not be read" % \
                                                                                                (cloudsInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Clouds configuration file %s is could not be opened" % \
                                                                                               (cloudsInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Clouds configuration file %s is missing" % \
                                                                                (cloudsInfoPath)))


   def getCloudInfo(self,
                    cloudName):
      cloud = self.clouds[cloudName]

      host             = cloud['host']
      port             = cloud['port']
      accessId         = cloud['accessId']
      accessSecret     = cloud['accessSecret']
      image            = cloud['image']
      maximumInstances = cloud['maximumInstances']
      maximumJobs      = cloud['maximumJobs']
      maximumIdleTime  = cloud['maximumIdleTime']

      return(host,port,accessId,accessSecret,image,maximumInstances,maximumJobs,maximumIdleTime)


