# @package      hubzero-submit-distributor
# @file         JobKill.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
__version__ = '3.4.2'

import sys
import os
import re
import signal
import subprocess
import time
import logging
from errno import EPIPE

from hubzero.submit.CommandParser    import CommandParser
from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

class JobKill:
   def __init__(self,
                listenURI):
      self.logger           = logging.getLogger(__name__)
      self.version          = __version__
      self.operationMode    = 0
      self.remoteJobMonitor = RemoteJobMonitor(listenURI)

      self.commandParser        = None
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'0'))
      self.killIds              = []

      self.hubUserId = os.getuid()

      self.abortAttempted = False
      self.abortSignal    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         pass


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         pass


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODERUNKILL:
         self.killIds = [int(killId) for killId in self.commandParser.getOption('killIds')]
      else:
         exitCode = 1

      return(exitCode)

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortSignal    = signalNumber


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   @staticmethod
   def __killProcess(pid):
      try:
         os.kill(pid,signal.SIGINT)
         for retry in range(0,12):
            time.sleep(10)
            os.kill(pid,0)

         os.kill(pid,signal.SIGTERM)
         for retry in range(0,6):
            time.sleep(10)
            os.kill(pid,0)

         os.kill(pid,signal.SIGKILL)
      except:
         pass


   def killActiveJob(self):
      """Kill jobs"""

      for killId in self.killIds:
         activeJobPid = self.remoteJobMonitor.queryUserActiveJobPid(self.hubUserId,killId)
         if activeJobPid:
            self.__killProcess(activeJobPid)


