# @package      hubzero-submit-distributor
# @file         RemoteBatchBOINC.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import logging
import xml.etree.cElementTree as elementTree

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteBatchBOINC:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                environment,
                isMultiCoreRequest,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                wallTime):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.instanceDirectory        = instanceDirectory
      self.stageFiles               = siteInfo['stageFiles'] and not siteInfo['sharedUserSpace']
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.toolFilesInfo            = toolFilesInfo
      self.dockerImageInfo          = dockerImageInfo
      self.appScriptName            = appScriptName
      self.environment              = environment
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.computationMode          = managerInfo['computationMode']
      self.wallTime                 = str(int(wallTime))

      self.nodeFileName = ""
      self.nodeList     = []

      self.toolInputTemplateFileName  = ""
      self.toolInputTemplate          = ""
      self.toolOutputTemplateFileName = ""
      self.toolOutputTemplate         = ""


   def __buildSerialFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','BOINC','serial')

      commandsBOINC = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsBOINC = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsBOINC = commandSeparator.join(self.submissionScriptCommands)

      substitutions = {}
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["ENVIRONMENT"]              = self.environment
      substitutions["INSTANCEDIRECTORY"]        = self.instanceDirectory
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsBOINC

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = ""
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
      else:
         batchLogName = "boinc_%s_%s.log" % (self.runName,self.instanceId)
         if self.instanceId == "0":
            batchScript = ""
            batchScriptExecutable = False
         else:
            batchScript = self.__buildSerialFile()
            batchScriptExecutable = True
         batchScriptName = "%s_%s.boinc" % (self.localJobId,self.instanceId)

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


# borrowed from http://effbot.org/zone/element-lib.htm#prettyprint
   def __indentXML(self,
                   element,
                   level=0):
      padding = "\n" + level*"   "
      if len(element):
         if not element.text or not element.text.strip():
            element.text = padding + "   "
         if not element.tail or not element.tail.strip():
            element.tail = padding
         for element in element:
            self.__indentXML(element, level+1)
         if not element.tail or not element.tail.strip():
            element.tail = padding
      else:
         if level and (not element.tail or not element.tail.strip()):
            element.tail = padding


   def getBatchToolInputTemplate(self):
      if self.toolFilesInfo:
         toolInputTemplateFileName = "%s_%s.inputTemplate" % (self.localJobId,self.instanceId)

         toolInputTemplateRoot = elementTree.Element("input_template")
         fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
         elementTree.SubElement(fileinfo,"physical_name").text = "%s_%s_input.tar.gz" % (self.localJobId,self.instanceId)

         fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
         elementTree.SubElement(fileinfo,"physical_name").text = self.toolFilesInfo['vboxFile']
         for fileInfoAttribute in self.toolFilesInfo['fileInfoAttributes']:
            if fileInfoAttribute != 'gzip':
               elementTree.SubElement(fileinfo,fileInfoAttribute)

         fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
         elementTree.SubElement(fileinfo,"physical_name").text = self.toolFilesInfo['boincAppFile']
         for fileInfoAttribute in self.toolFilesInfo['fileInfoAttributes']:
            if fileInfoAttribute != 'gzip':
               elementTree.SubElement(fileinfo,fileInfoAttribute)

         for layerFile in self.dockerImageInfo['layerFiles']:
            fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
            elementTree.SubElement(fileinfo,"physical_name").text = layerFile
            for fileInfoAttribute in self.toolFilesInfo['fileInfoAttributes']:
               elementTree.SubElement(fileinfo,fileInfoAttribute)

         fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
         elementTree.SubElement(fileinfo,"physical_name").text = self.dockerImageInfo['imageFile']
         for fileInfoAttribute in self.toolFilesInfo['fileInfoAttributes']:
            elementTree.SubElement(fileinfo,fileInfoAttribute)

         for appsFile in self.toolFilesInfo['appsFiles']:
            fileinfo = elementTree.SubElement(toolInputTemplateRoot,"file_info")
            elementTree.SubElement(fileinfo,"physical_name").text = appsFile
            for fileInfoAttribute in self.toolFilesInfo['fileInfoAttributes']:
               elementTree.SubElement(fileinfo,fileInfoAttribute)

         workunit = elementTree.SubElement(toolInputTemplateRoot,"workunit")
         fileref = elementTree.SubElement(workunit,"file_ref")
         elementTree.SubElement(fileref,"open_name").text = os.path.join('shared',
                                                                         "%s_%s_input.tar.gz" % (self.localJobId,self.instanceId))
         for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
            elementTree.SubElement(fileref,fileRefAttribute)

         fileref = elementTree.SubElement(workunit,"file_ref")
         elementTree.SubElement(fileref,"open_name").text = 'vbox_job.xml'
         for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
            elementTree.SubElement(fileref,fileRefAttribute)

         fileref = elementTree.SubElement(workunit,"file_ref")
         elementTree.SubElement(fileref,"open_name").text = os.path.join('shared','boinc_app')
         for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
            elementTree.SubElement(fileref,fileRefAttribute)

         for layerFile in self.dockerImageInfo['layerFiles']:
            fileref = elementTree.SubElement(workunit,"file_ref")
            elementTree.SubElement(fileref,"open_name").text = os.path.join('shared','image',layerFile)
            for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
               elementTree.SubElement(fileref,fileRefAttribute)

         fileref = elementTree.SubElement(workunit,"file_ref")
         elementTree.SubElement(fileref,"open_name").text = os.path.join('shared','image',self.dockerImageInfo['imageFile'])
         for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
            elementTree.SubElement(fileref,fileRefAttribute)

         for appsFile in self.toolFilesInfo['appsFiles']:
            fileref = elementTree.SubElement(workunit,"file_ref")
            elementTree.SubElement(fileref,"open_name").text = os.path.join('shared','appsSource',appsFile)
            for fileRefAttribute in self.toolFilesInfo['fileRefAttributes']:
               elementTree.SubElement(fileref,fileRefAttribute)

         if 'fpopsEstimate' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"rsc_fpops_est").text = str(self.toolFilesInfo['fpopsEstimate'])
         if 'fpopsBound' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"rsc_fpops_bound").text = str(self.toolFilesInfo['fpopsBound'])
         if 'memoryRequirement' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"rsc_memory_bound").text = str(self.toolFilesInfo['memoryRequirement'])
         if 'diskRequirement' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"rsc_disk_bound").text = str(self.toolFilesInfo['diskRequirement'])
         if 'delayBound' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"delay_bound").text = str(self.toolFilesInfo['delayBound'])
         if 'minimumQuorum' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"min_quorum").text = str(self.toolFilesInfo['minimumQuorum'])
         if 'targetResults' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"target_nresults").text = str(self.toolFilesInfo['targetResults'])
         if 'maximumErrorResults' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"max_error_results").text = str(self.toolFilesInfo['maximumErrorResults'])
         if 'maximumTotalResults' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"max_total_results").text = str(self.toolFilesInfo['maximumTotalResults'])
         if 'maximumSuccessResults' in self.toolFilesInfo:
            elementTree.SubElement(workunit,"max_success_results").text = str(self.toolFilesInfo['maximumSuccessResults'])
#        if 'sizeClass' in self.toolFilesInfo:
#           elementTree.SubElement(workunit,"size_class").text = str(self.toolFilesInfo['sizeClass'])

         self.__indentXML(toolInputTemplateRoot)
         toolInputTemplate = elementTree.tostring(toolInputTemplateRoot)
      else:
         toolInputTemplateFileName = self.toolInputTemplateFileName
         toolInputTemplate         = self.toolInputTemplate

      return(toolInputTemplateFileName,toolInputTemplate)


   def getBatchToolOutputTemplate(self):
      if self.toolFilesInfo:
         toolOutputTemplateFileName = "%s_%s.outputTemplate" % (self.localJobId,self.instanceId)

         toolOutputTemplateRoot = elementTree.Element("output_template")

         fileinfo = elementTree.SubElement(toolOutputTemplateRoot,"file_info")
         name = elementTree.SubElement(fileinfo,"name")
         elementTree.SubElement(name,"OUTFILE_0")
#        elementTree.SubElement(fileinfo,"name").text = "%s_%s_output.tar.gz" % (self.localJobId,self.instanceId)
         elementTree.SubElement(fileinfo,"generated_locally")
         elementTree.SubElement(fileinfo,"upload_when_present")
         elementTree.SubElement(fileinfo,"max_nbytes").text = "209715200"
         url = elementTree.SubElement(fileinfo,"url")
         elementTree.SubElement(url,"UPLOAD_URL")

         result = elementTree.SubElement(toolOutputTemplateRoot,"result")
         fileref = elementTree.SubElement(result,"file_ref")
         name = elementTree.SubElement(fileref,"file_name")
         elementTree.SubElement(name,"OUTFILE_0")
#        elementTree.SubElement(fileref,"file_name").text = "%s_%s_output.tar.gz" % (self.localJobId,self.instanceId)
         elementTree.SubElement(fileref,"open_name").text = os.path.join('shared','results',
                                                                         "%s_%s_output.tar.gz" % (self.localJobId,self.instanceId))
         elementTree.SubElement(fileref,"copy_file").text = "1"

         elementTree.SubElement(toolOutputTemplateRoot,"report_immediately")

         self.__indentXML(toolOutputTemplateRoot)
         toolOutputTemplate = elementTree.tostring(toolOutputTemplateRoot)
# currently a formatting requirement of file_deleter.cpp (<name>file</name>)
         toolOutputTemplate = re.sub("\n *<OUTFILE_0 />\n *","<OUTFILE_0 />",toolOutputTemplate)
      else:
         toolOutputTemplateFileName = self.toolOutputTemplateFileName
         toolOutputTemplate         = self.toolOutputTemplate

      return(toolOutputTemplateFileName,toolOutputTemplate)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
         remoteJobIdNumber = remoteJobId
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


