# @package      hubzero-submit-distributor
# @file         VenueMechanismSsh.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import re
import stat
import time
import datetime
import shutil
import logging

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.JobOutput              import JobOutput
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore
from hubzero.submit.RemoteBatchBOINC       import RemoteBatchBOINC
from hubzero.submit.RemoteBatchCONDOR      import RemoteBatchCONDOR
from hubzero.submit.RemoteBatchFACTORY     import RemoteBatchFACTORY
from hubzero.submit.RemoteBatchPEGASUS     import RemoteBatchPEGASUS
from hubzero.submit.RemoteBatchLL          import RemoteBatchLL
from hubzero.submit.RemoteBatchLSF         import RemoteBatchLSF
from hubzero.submit.RemoteBatchPBS         import RemoteBatchPBS
from hubzero.submit.RemoteBatchSGE         import RemoteBatchSGE
from hubzero.submit.RemoteBatchSLURM       import RemoteBatchSLURM
from hubzero.submit.RemoteBatchAppScript   import RemoteBatchAppScript
from hubzero.submit.RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from hubzero.submit.RemoteInstantAppScript import RemoteInstantAppScript
from hubzero.submit.RemoteWorkflowPEGASUS  import RemoteWorkflowPEGASUS

class VenueMechanismSsh(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                submitterClass,
                currentWorkingDirectory,
                session,
                distributorPid,
                batchCommands,
                isParametric,
                runName,
                localJobId,
                instanceId,
                destination,
                tailFiles,
                enteredCommand,
                gridsite,
                pegasusSiteInfo,
                tunnelsInfo,
                cloudInstanceId,
                stageInTarFile,
                transferExecutable,
                executable,
                appsAccessInfo,
                event,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                disableJobMonitoring,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                wallTime,
                x509SubmitProxy,
                disableProbeCheck,
                quotaLimit,
                timeHistoryLogs,
                useSetup,
                pegasusVersion,
                pegasusHome):
      VenueMechanismCore.__init__(self,timeHistoryLogs,disableJobMonitoring,
                                       siteInfo,toolFilesInfo,dockerImageInfo,submissionScriptsInfo,managerInfo,
                                       remoteMonitors,isMultiCoreRequest,nCpus,nNodes,ppn,event)

      self.logger                           = logging.getLogger(__name__)
      self.jobOutput                        = JobOutput()
      self.venueMechanism                   = 'ssh'
      self.hubUserName                      = hubUserName
      self.hubUserId                        = hubUserId
      self.submitterClass                   = submitterClass
      self.session                          = session
      self.distributorPid                   = distributorPid
      remoteScriptsDirectory = siteInfo['remoteBinDirectory'].replace('$','\$')
      self.receiveInput                     = os.path.join(remoteScriptsDirectory,batchCommands['receiveInput'])
      self.submitBatchJob                   = os.path.join(remoteScriptsDirectory,batchCommands['submitBatchJob'])
      self.postProcessJob                   = os.path.join(remoteScriptsDirectory,batchCommands['postProcessJob'])
      self.transmitResults                  = os.path.join(remoteScriptsDirectory,batchCommands['transmitResults'])
      self.cleanupJob                       = os.path.join(remoteScriptsDirectory,batchCommands['cleanupJob'])
      self.killBatchJob                     = os.path.join(remoteScriptsDirectory,batchCommands['killBatchJob'])
      self.isParametric                     = isParametric
      self.runName                          = runName
      self.localJobId                       = localJobId
      if instanceId.startswith('WF;'):
         self.instanceId                    = "0"
         self.nInstances                    = int(instanceId.split(';')[1])
      else:
         self.instanceId                    = instanceId
         self.nInstances                    = 1
      self.remoteBatchSystem                = siteInfo['remoteBatchSystem']
      self.destination                      = destination
      self.venue                            = siteInfo['venue']
      self.venuePort                        = siteInfo['venuePort']
      self.sshOptions                       = siteInfo['sshOptions']
      self.tailFiles                        = tailFiles
      self.enteredCommand                   = enteredCommand
      self.gridsite                         = gridsite
      self.pegasusSiteInfo                  = pegasusSiteInfo
      self.siteMonitorDesignator            = siteInfo['siteMonitorDesignator']
      self.tunnelDesignator                 = siteInfo['tunnelDesignator']
      if self.tunnelDesignator != "":
         gatewayHost,localHost = tunnelsInfo.getSSHTunnelHosts(self.tunnelDesignator)
         self.gatewayHost                   = gatewayHost
         self.localHost                     = localHost
      else:
         self.gatewayHost                   = ""
         self.localHost                     = ""
      self.cloudDesignator                  = siteInfo['cloudDesignator']
      self.cloudInstanceId                  = cloudInstanceId
      self.remoteUser                       = siteInfo['remoteUser']
      self.sharedUserSpace                  = siteInfo['sharedUserSpace']
      self.logUserRemotely                  = siteInfo['logUserRemotely']
      self.stageFiles                       = siteInfo['stageFiles']
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile                = stageInTarFile
      else:
         self.stageInTarFile                = stageInTarFile + '.gz'
      self.transferExecutable               = transferExecutable
      self.executable                       = executable
      self.appsAccessInfo                   = appsAccessInfo
      self.stdinput                         = stdinput
      if isinstance(arguments,list):
         self.arguments                     = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments              += "%s " % (argument.replace('\\ ',' ').replace(' ','\ '))
            else:
               self.arguments              += "%s " % (argument)
      else:
         self.arguments                     = arguments
      self.useEnvironment                   = useEnvironment
      self.environment                      = environment
      self.remoteBatchQueue                 = siteInfo['remoteBatchQueue']
      self.pegasusTemplates                 = siteInfo['pegasusTemplates']
      self.executableClassificationsAllowed = siteInfo['executableClassificationsAllowed']
      self.wallTime                         = wallTime
      self.x509SubmitProxy                  = x509SubmitProxy
      self.disableProbeCheck                = disableProbeCheck
      self.quotaLimit                       = quotaLimit
      self.useSetup                         = useSetup
      self.pegasusVersion                   = pegasusVersion
      self.pegasusHome                      = pegasusHome
      self.batchLogPath                     = ""
      self.appScriptPath                    = ""
      self.batchScriptPath                  = ""
      self.nodeFilePath                     = ""
      self.toolInputTemplateFilePath        = ""
      self.toolOutputTemplateFilePath       = ""
      self.remoteJobIdNumber                = ""
      self.sshBaseCommand                   = ""
      self.sshIdentityPath                  = ""

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob           = False
         self.instanceDirectory    = currentWorkingDirectory
      else:
         self.isBatchJob           = True
         if self.instanceId != "0":
            self.instanceDirectory = os.path.join(currentWorkingDirectory,self.runName,self.instanceId)
         else:
            self.instanceDirectory = os.path.join(currentWorkingDirectory,self.runName)
         if not os.path.isdir(self.instanceDirectory):
            os.makedirs(self.instanceDirectory)

      self.createIdentityPaths()

      if not self.sharedUserSpace and self.stageFiles:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.workingDirectory        = os.path.join(siteInfo['remoteScratchDirectory'],"%s_%s" % (str(epoch),localJobId))
         if self.instanceId != "0":
            self.workingDirectory    += "_%s" % (self.instanceId)
         self.scratchDirectory        = self.workingDirectory
      else:
         self.workingDirectory = currentWorkingDirectory
         if siteInfo['remoteScratchDirectory']:
            epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
            self.scratchDirectory     = os.path.join(siteInfo['remoteScratchDirectory'],"%s_%s" % (str(epoch),localJobId))
            if self.instanceId != "0":
               self.scratchDirectory += "_%s" % (self.instanceId)
         else:
            self.scratchDirectory     = self.instanceDirectory


   def createIdentityPaths(self):
      exitStatus = 0
      if not self.sshIdentityPath:
         updateKnownHosts = False
         if   self.remoteUser.startswith('USER:'):
            self.remoteUser = self.hubUserName
            identityPaths = self.remoteIdentityManager.queryUserIdentities(self.siteInfo['identityManagers'],
                                                                           self.hubUserName)
            self.sshIdentityPath = identityPaths['personalPKI']
            self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
            updateKnownHosts = True
         elif self.remoteUser.startswith('USER'):
            self.remoteUser = self.hubUserName
         else:
            identityPaths = self.remoteIdentityManager.queryUserIdentities(self.siteInfo['identityManagers'],
                                                                           self.hubUserName)
            self.sshIdentityPath = identityPaths['communitySSH']
            self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
            updateKnownHosts = True

         if not exitStatus:
            if updateKnownHosts:
               if self.tunnelDesignator == "":
                  command = 'update-known-hosts' + " " + self.venue
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  exitStatus,stdOutput,stdError = self.executeCommand(command)
               else:
                  if self.gatewayHost != "" and self.localHost != "":
                     command = 'update-known-hosts' + " \'" + self.gatewayHost + "\'"
                     self.logger.log(logging.INFO,getLogMessage("command = " + command))
                     exitStatus,stdOutput,stdError = self.executeCommand(command)
                     if not exitStatus:
                        command = 'update-known-hosts' + " \'" + self.localHost + "\'"
                        self.logger.log(logging.INFO,getLogMessage("command = " + command))
                        exitStatus,stdOutput,stdError = self.executeCommand(command)
               if not exitStatus:
                  if self.tunnelDesignator == "":
                     self.sshBaseCommand = "ssh -T -x -a %s -i %s -p %d %s@%s" % (self.sshOptions,self.sshIdentityPath,
                                                                                  self.venuePort,
                                                                                  self.remoteUser,self.venue)
                  else:
                     tunnelAddress,tunnelPort = self.remoteTunnelMonitor.getTunnelAddressPort(self.tunnelDesignator)
                     self.sshBaseCommand = "ssh -T -x -a %s -i %s -p %s %s@%s" % (self.sshOptions,self.sshIdentityPath,
                                                                                  tunnelPort,
                                                                                  self.remoteUser,tunnelAddress)
            else:
               self.sshBaseCommand = "ssh -T -x -a %s -p %d %s@%s" % (self.sshOptions,self.venuePort,self.remoteUser,self.venue)


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      if self.sshBaseCommand != "":
         self.logger.log(logging.INFO,getLogMessage("workingDirectory " + self.workingDirectory))
         if   self.remoteBatchSystem == 'PBS':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchPBS(self.hubUserName,self.hubUserId,self.submitterClass,
                                              self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                              self.transferExecutable,
                                              self.executable,self.arguments,self.stdinput,
                                              self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                              self.submissionScriptsInfo,
                                              self.managerInfo,self.nNodes,self.ppn,
                                              self.wallTime,self.quotaLimit,
                                              self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'SGE':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchSGE(self.hubUserName,self.hubUserId,self.submitterClass,
                                              self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                              self.transferExecutable,
                                              self.executable,self.arguments,self.stdinput,
                                              self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                              self.submissionScriptsInfo,
                                              self.managerInfo,self.nNodes,self.ppn,
                                              self.wallTime,self.quotaLimit,
                                              self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'LSF':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchLSF(self.hubUserName,self.hubUserId,self.submitterClass,
                                              self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                              self.transferExecutable,
                                              self.executable,self.arguments,self.stdinput,
                                              self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                              self.submissionScriptsInfo,
                                              self.managerInfo,self.nNodes,self.ppn,
                                              self.wallTime,self.quotaLimit,
                                              self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'LL':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchLL(self.hubUserName,self.hubUserId,self.submitterClass,
                                             self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                             self.transferExecutable,
                                             self.executable,self.arguments,self.stdinput,
                                             self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                             self.submissionScriptsInfo,
                                             self.managerInfo,self.nNodes,self.ppn,
                                             self.wallTime,self.quotaLimit,
                                             self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'SLURM':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchSLURM(self.hubUserName,self.hubUserId,self.submitterClass,
                                                self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                                self.transferExecutable,
                                                self.executable,self.arguments,self.stdinput,
                                                self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                self.submissionScriptsInfo,
                                                self.managerInfo,self.nNodes,self.ppn,
                                                self.wallTime,self.quotaLimit,
                                                self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'CONDOR':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchCONDOR(self.hubUserName,self.hubUserId,self.submitterClass,
                                                 self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                 appScriptName,self.environment,
                                                 self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                 self.submissionScriptsInfo,
                                                 self.managerInfo,self.wallTime)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'BOINC':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchBOINC(self.hubUserName,self.hubUserId,self.submitterClass,
                                                self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                appScriptName,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                self.submissionScriptsInfo,
                                                self.managerInfo,self.wallTime)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'FACTORY':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchFACTORY(self.hubUserName,self.hubUserId,self.submitterClass,
                                                  self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                  appScriptName,self.environment,
                                                  self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                  self.submissionScriptsInfo,
                                                  self.managerInfo,self.x509SubmitProxy,self.wallTime)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'PEGASUS':
            if self.isParametric:
               if self.instanceId != "0":
                  self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                              self.runName,self.localJobId,self.instanceId,True,
                                                              self.transferExecutable,self.executable,
                                                              self.stdinput,self.arguments,
                                                              self.useEnvironment,self.environment,
                                                              self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                              self.managerInfo,self.timeHistoryLogs)
                  appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
                  batchLogName               = ""
                  batchScriptName            = ""
                  batchScript                = ""
                  nodeFileName               = ""
                  nodeList                   = []
                  toolInputTemplateFileName  = ""
                  toolInputTemplate          = ""
                  toolOutputTemplateFileName = ""
                  toolOutputTemplate         = ""
               else:
                  appScriptName = ""
                  appScript     = ""
                  remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
                  self.remoteBatch = RemoteWorkflowPEGASUS(self.hubUserName,self.hubUserId,self.submitterClass,
                                                           self.runName,self.localJobId,self.instanceId,
                                                           self.instanceDirectory,remoteScratchDirectory,
                                                           self.useSetup,self.pegasusVersion,self.pegasusHome,
                                                           self.pegasusTemplates,self.x509SubmitProxy,self.sshIdentityPath,
                                                           self.gridsite,self.pegasusSiteInfo,self.wallTime,
                                                           self.siteInfo,self.submissionScriptsInfo,self.timeHistoryLogs)
                  batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildWorkflowScript()
                  batchLogName               = ""
                  nodeFileName               = ""
                  nodeList                   = []
                  toolInputTemplateFileName  = ""
                  toolInputTemplate          = ""
                  toolOutputTemplateFileName = ""
                  toolOutputTemplate         = ""
            else:
               self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                           self.runName,self.localJobId,self.instanceId,False,
                                                           self.transferExecutable,self.executable,
                                                           self.stdinput,self.arguments,
                                                           self.useEnvironment,self.environment,
                                                           self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                           self.managerInfo,self.timeHistoryLogs)
               appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
               remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
               self.remoteBatch = RemoteBatchPEGASUS(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.runName,self.localJobId,self.instanceId,
                                                     self.instanceDirectory,remoteScratchDirectory,
                                                     self.useSetup,self.pegasusVersion,self.pegasusHome,
                                                     appScriptName,
                                                     self.executable,self.arguments,self.isMultiCoreRequest,
                                                     self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                     self.submissionScriptsInfo,self.managerInfo,
                                                     self.x509SubmitProxy,self.sshIdentityPath,self.gridsite,self.pegasusSiteInfo,
                                                     self.wallTime,self.timeHistoryLogs)
               batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
               nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
               toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
               toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
               userDaxExecutables                                             = self.remoteBatch.getUserDaxExecutables()
               for userDaxExecutable in userDaxExecutables:
                  submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(userDaxExecutable)
                  if not submissionAllowed:
                     message = "Access to %s denied for %s" % (userDaxExecutable,self.hubUserName)
                     self.logger.log(logging.ERROR,getLogMessage(message))
                     self._writeToStderr(message + "\n")
                     if not exitCode:
                        exitCode = 9
                  if (not '*' in self.executableClassificationsAllowed) and \
                     (not executableClassification in self.executableClassificationsAllowed):
                     message = "The selected site does not meet the specified\n" + \
                               "application access requirements.  Please select another site."
                     self.logger.log(logging.ERROR,getLogMessage(message))
                     self._writeToStderr(message + "\n")
                     if not exitCode:
                        exitCode = 10
         elif self.remoteBatchSystem == 'SCRIPT':
            self.remoteAppScript = RemoteInstantAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                          self.runName,self.localJobId,self.instanceId,False,
                                                          self.workingDirectory,
                                                          self.transferExecutable,self.executable,
                                                          self.stdinput,self.arguments,
                                                          self.useEnvironment,self.environment,
                                                          self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                          self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,self.submitterClass,
                                                   self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                                   self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                   self.submissionScriptsInfo,
                                                   self.managerInfo,self.nNodes,self.ppn,
                                                   self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()
         elif self.remoteBatchSystem == 'BGSCRIPT':
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,self.submitterClass,
                                                   self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                                   self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                   self.submissionScriptsInfo,
                                                   self.managerInfo,self.nNodes,self.ppn,
                                                   self.timeHistoryLogs)
            batchLogName,batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildBatchScript()
            nodeFileName,nodeList                                          = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate                    = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate                  = self.remoteBatch.getBatchToolOutputTemplate()

         if batchLogName != "":
            self.batchLogPath               = os.path.join(self.instanceDirectory,batchLogName)
         if appScriptName != "":
            self.appScriptPath              = os.path.join(self.instanceDirectory,appScriptName)
         if batchScriptName != "":
            self.batchScriptPath            = os.path.join(self.instanceDirectory,batchScriptName)
         if nodeFileName != "":
            self.nodeFilePath               = os.path.join(self.instanceDirectory,nodeFileName)
         if toolInputTemplateFileName != "":
            self.toolInputTemplateFilePath  = os.path.join(self.instanceDirectory,toolInputTemplateFileName)
         if toolOutputTemplateFileName != "":
            self.toolOutputTemplateFilePath = os.path.join(self.instanceDirectory,toolOutputTemplateFileName)

         jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
         self.jobIndex = int(self.instanceId)
         if not self.jobIndex in self.jobStatistics:
            self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
         self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism
         if self.event:
            self.jobStatistics[self.jobIndex]['event'] = self.event

         if batchScript != "":
            try:
               fpBatchScript = open(self.batchScriptPath,'w')
               try:
                  fpBatchScript.write(batchScript)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.batchScriptPath)))
                  self._writeToStderr("%s could not be written\n" % (self.batchScriptPath))
                  self.jobStatistics[0]['exitCode'] = 1
                  exitCode = 1
               else:
                  scriptFiles.append(self.batchScriptPath)
                  if batchScriptExecutable:
                     os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
               finally:
                  fpBatchScript.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.batchScriptPath)))
               self._writeToStderr("%s could not be opened\n" % (self.batchScriptPath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1

         if appScript != "":
            try:
               fpAppScript = open(self.appScriptPath,'w')
               try:
                  fpAppScript.write(appScript)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.appScriptPath)))
                  self._writeToStderr("%s could not be written\n" % (self.appScriptPath))
                  self.jobStatistics[0]['exitCode'] = 1
                  exitCode = 1
               else:
                  scriptFiles.append(self.appScriptPath)
                  if appScriptExecutable:
                     os.chmod(self.appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
               finally:
                  fpAppScript.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.appScriptPath)))
               self._writeToStderr("%s could not be opened\n" % (self.appScriptPath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1

         if len(nodeList) > 0:
            try:
               fpNodes = open(self.nodeFilePath,'w')
               try:
                  fpNodes.write('\n'.join(nodeList)+'\n')
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.nodeFilePath)))
                  self._writeToStderr("%s could not be written\n" % (self.nodeFilePath))
                  self.jobStatistics[0]['exitCode'] = 1
                  exitCode = 1
               else:
                  scriptFiles.append(self.nodeFilePath)
               finally:
                  fpNodes.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.nodeFilePath)))
               self._writeToStderr("%s could not be opened\n" % (self.nodeFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1

         if toolInputTemplate != "":
            try:
               fpToolInputTemplate = open(self.toolInputTemplateFilePath,'w')
               try:
                  fpToolInputTemplate.write(toolInputTemplate)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.toolInputTemplateFilePath)))
                  self._writeToStderr("%s could not be written\n" % (self.toolInputTemplateFilePath))
                  self.jobStatistics[0]['exitCode'] = 1
                  exitCode = 1
               finally:
                  fpToolInputTemplate.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.toolInputTemplateFilePath)))
               self._writeToStderr("%s could not be opened\n" % (self.toolInputTemplateFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1

         if toolOutputTemplate != "":
            try:
               fpToolOutputTemplate = open(self.toolOutputTemplateFilePath,'w')
               try:
                  fpToolOutputTemplate.write(toolOutputTemplate)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.toolOutputTemplateFilePath)))
                  self._writeToStderr("%s could not be written\n" % (self.toolOutputTemplateFilePath))
                  self.jobStatistics[0]['exitCode'] = 1
                  exitCode = 1
               finally:
                  fpToolOutputTemplate.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.toolOutputTemplateFilePath)))
               self._writeToStderr("%s could not be opened\n" % (self.toolOutputTemplateFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
      else:
         exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      exitStatus = 0
      self.jobStatistics[self.jobIndex]['venue'] = self.venue

      if   not self.sharedUserSpace and self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
         remoteScratchDirectory = self.scratchDirectory.replace('$','\$')

         if self.sshBaseCommand != "":
            stageInTarPath = os.path.join(self.instanceDirectory,self.stageInTarFile).replace(' ','\ ')
            command = "cat " + stageInTarPath + " | " + self.sshBaseCommand + \
                                                  " \"" + self.receiveInput + " " + \
                                                     remoteWorkingDirectory + " " + \
                                                     remoteScratchDirectory + " " + \
                               self.timeHistoryLogs['timestampTransferred'] + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
         else:
            exitStatus = 1
            stdOutput  = ""
            stdError   = ""

         if exitStatus:
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
      elif self.sharedUserSpace and self.stageFiles:
         remoteWorkingDirectory = self.instanceDirectory
         remoteScratchDirectory = self.scratchDirectory.replace('$','\$')

         if self.sshBaseCommand != "":
            command = self.sshBaseCommand + " \"" + self.receiveInput + " " + \
                                                    remoteWorkingDirectory + " " + \
                                                    remoteScratchDirectory + " " + \
                                                    self.timeHistoryLogs['timestampTransferred'] + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
         else:
            exitStatus = 1
            stdOutput  = ""
            stdError   = ""

         if exitStatus:
            self.jobStatistics[self.jobIndex]['exitCode'] = 11

      if exitStatus:
         self.filesSent = False
#        if stdOutput != "":
#           if self.instanceId != "0":
#              stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#           else:
#              stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#           try:
#              fpStd = open(stdFile,'a')
#              try:
#                 fpStd.write(stdOutput)
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#              finally:
#                 fpStd.close()
#           except (IOError,OSError):
#              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         if stdError != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdError)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
         self.removeJobRegistration()

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      if   self.remoteBatchSystem == 'SCRIPT':
         if self.stageFiles:
            remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
            remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(remoteScratchDirectory)
            if self.batchScriptPath != "":
               commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            else:
               commandArgs.append(os.path.join('.',os.path.basename(self.appScriptPath)))
            if self.logUserRemotely:
               hubUserHash = self.remoteIdentityManager.queryUserHash(self.siteInfo['identityManagers'],
                                                                      self.hubUserName,
                                                                      self.hubUserId)
               commandArgs.append(hubUserHash)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))

            self.removeJobRegistration()

            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator,True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            if self.batchScriptPath != "":
               command = self.sshBaseCommand + " " + self.batchScriptPath
            else:
               command = self.sshBaseCommand + " " + self.appScriptPath
            self.logger.log(logging.INFO,getLogMessage("command = " + command))

            self.removeJobRegistration()

            exitStatus,stdOutput,stdError = self.executeCommand(command,True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      elif self.remoteBatchSystem == 'BGSCRIPT':
         if self.stageFiles:
            remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
            remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(remoteScratchDirectory)
            if self.batchScriptPath != "":
               commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            else:
               commandArgs.append(os.path.join('.',os.path.basename(self.appScriptPath)))
            if self.logUserRemotely:
               hubUserHash = self.remoteIdentityManager.queryUserHash(self.siteInfo['identityManagers'],
                                                                      self.hubUserName,
                                                                      self.hubUserId)
               commandArgs.append(hubUserHash)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""

            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)
            if not exitStatus:
               remoteJobId = stdOutput.strip()
               self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
               self.jobSubmitted = True
               self.remoteJobIdNumber = self.remoteBatch.getRemoteJobIdNumber(remoteJobId)

               self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               self.remoteJobMonitor.postJobSubmission(self.siteInfo['siteName'],
                                                       self.siteInfo['identityManagers'],
                                                       self.siteMonitorDesignator,
                                                       self.remoteJobIdNumber,str(self.hubUserId),
                                                       self.tailFiles,self.enteredCommand,
                                                       self.scratchDirectory,
                                                       self.localJobId,self.instanceId,self.destination,
                                                       self.runName,self.nCpus,self.distributorPid)
            else:
               self.removeJobRegistration()
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         if self.stageFiles:
            if self.sharedUserSpace:
               remoteWorkingDirectory = self.instanceDirectory
            else:
               remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
            remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(remoteScratchDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            if self.logUserRemotely:
               hubUserHash = self.remoteIdentityManager.queryUserHash(self.siteInfo['identityManagers'],
                                                                      self.hubUserName,
                                                                      self.hubUserId)
               commandArgs.append(hubUserHash)
               commandArgs.append(str(self.hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)

            if not exitStatus:
               remoteJobId = stdOutput.strip()
               self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
               self.remoteJobIdNumber = self.remoteBatch.getRemoteJobIdNumber(remoteJobId)
               if self.remoteJobIdNumber != '-1':
                  self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               else:
                  exitStatus = 248

               if not exitStatus:
                  self.jobSubmitted = True
                  if self.remoteBatchSystem == 'PEGASUS':
                     self.remoteJobMonitor.postPegasusWorkflowSubmission(self.siteInfo['siteName'],
                                                                         self.siteInfo['identityManagers'],
                                                                         self.siteMonitorDesignator,
                                                                         self.remoteJobIdNumber,str(self.hubUserId),
                                                                         self.enteredCommand,
                                                                         self.localJobId,self.nInstances,self.destination,
                                                                         self.runName,self.nCpus,self.distributorPid)
                  else:
                     self.remoteJobMonitor.postJobSubmission(self.siteInfo['siteName'],
                                                             self.siteInfo['identityManagers'],
                                                             self.siteMonitorDesignator,
                                                             self.remoteJobIdNumber,str(self.hubUserId),
                                                             self.tailFiles,self.enteredCommand,
                                                             self.scratchDirectory,
                                                             self.localJobId,self.instanceId,self.destination,
                                                             self.runName,self.nCpus,self.distributorPid)
               else:
                  self.removeJobRegistration()
                  self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
            else:
               self.removeJobRegistration()
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            if self.sharedUserSpace:
               remoteWorkingDirectory = self.instanceDirectory
            else:
               remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
            remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(remoteScratchDirectory)
            commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)

            if not exitStatus:
               remoteJobId = stdOutput.strip()
               self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
               self.remoteJobIdNumber = self.remoteBatch.getRemoteJobIdNumber(remoteJobId)
               if self.remoteJobIdNumber != '-1':
                  self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               else:
                  exitStatus = 248

               if not exitStatus:
                  self.jobSubmitted = True
                  if self.remoteBatchSystem == 'PEGASUS':
                     self.remoteJobMonitor.postPegasusWorkflowSubmission(self.siteInfo['siteName'],
                                                                         self.siteInfo['identityManagers'],
                                                                         self.siteMonitorDesignator,
                                                                         self.remoteJobIdNumber,str(self.hubUserId),
                                                                         self.enteredCommand,
                                                                         self.localJobId,self.nInstances,self.destination,
                                                                         self.runName,self.nCpus,self.distributorPid)
                  else:
                     self.remoteJobMonitor.postJobSubmission(self.siteInfo['siteName'],
                                                             self.siteInfo['identityManagers'],
                                                             self.siteMonitorDesignator,
                                                             self.remoteJobIdNumber,str(self.hubUserId),
                                                             self.tailFiles,self.enteredCommand,
                                                             self.scratchDirectory,
                                                             self.localJobId,self.instanceId,self.destination,
                                                             self.runName,self.nCpus,self.distributorPid)
               else:
                  self.removeJobRegistration()
                  self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
            else:
               self.removeJobRegistration()
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
         if stdOutput != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdOutput)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         if stdError != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdError)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['siteMonitorDesignator'] = self.siteMonitorDesignator
      waitForJobInfo['remoteJobId']           = self.remoteJobIdNumber
      waitForJobInfo['knownSite']             = self.destination

      return(waitForJobInfo)


   def postProcess(self):
      if not self.jobPostProcessed:
         if self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and self.instanceId == "0") or not self.isParametric:
                  if not self.sharedUserSpace:
                     remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
                  else:
                     remoteWorkingDirectory = self.instanceDirectory
                  remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
                  command = self.sshBaseCommand + " \"" + self.postProcessJob + " " + \
                                                          remoteWorkingDirectory + " " + \
                                                          remoteScratchDirectory + " " + \
                                                          "PEGASUS" + "\""
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)
                  self.logger.log(logging.INFO,getLogMessage(stdOutput))
                  if exitStatus:
                     self.logger.log(logging.INFO,getLogMessage(stdError))

                  self.jobPostProcessed = True


   def retrieveFiles(self):
      exitStatus = 0

      if   not self.sharedUserSpace and self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
         if self.jobSubmitted:
            if self.instanceId != "0":
               stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
            else:
               stageOutTarFile = "%s_output.tar" % (self.localJobId)
            command = self.sshBaseCommand + " \"" + self.transmitResults + " " + \
                                                  remoteWorkingDirectory + " " + \
                                                         stageOutTarFile + "\"" + \
                                            " | tar xzmf - --ignore-case --exclude '*hub-proxy.*' -C " + \
                                                  self.instanceDirectory.replace(' ','\ ')
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,self.tunnelDesignator)
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 if self.instanceId != "0":
#                    stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#                 else:
#                    stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#                 try:
#                    fpStd = open(stdFile,'a')
#                    try:
#                       fpStd.write(stdOutput)
#                    except (IOError,OSError):
#                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                    finally:
#                       fpStd.close()
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

               if stdError != "":
                  if self.instanceId != "0":
                     stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                  else:
                     stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                  try:
                     fpStd = open(stdFile,'a')
                     try:
                        fpStd.write(stdError)
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                     finally:
                        fpStd.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
      elif self.sharedUserSpace and self.stageFiles:
         outTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
         stageOutTarFiles = []
         stageOutTarFiles.append(os.path.join(self.instanceDirectory,"%s.gz" % (outTarFile)))
         stageOutTarFiles.append(os.path.join(self.instanceDirectory,outTarFile))
         stageOutTarFiles.append("%s.gz" % (outTarFile))
         stageOutTarFiles.append(outTarFile)
         stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),"%s.gz" % (outTarFile)))
         stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),outTarFile))
         for stageOutTarFile in stageOutTarFiles:
            if os.path.isfile(stageOutTarFile):
               if stageOutTarFile.endswith('.gz'):
                  command = "tar xzmf " + stageOutTarFile + \
                                      " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory.replace(' ','\ ')
               else:
                  command = "tar xmf " + stageOutTarFile + \
                                     " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory.replace(' ','\ ')
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               exitStatus,stdOutput,stdError = self.executeCommand(command)
               if exitStatus == 0:
                  try:
                     os.remove(stageOutTarFile)
                  except:
                     pass
               else:
                  self.jobStatistics[self.jobIndex]['exitCode'] = 12
                  if stdOutput != "":
                     if self.instanceId != "0":
                        stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
                     else:
                        stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
                     try:
                        fpStd = open(stdFile,'a')
                        try:
                           fpStd.write(stdOutput)
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                        finally:
                           fpStd.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

                  if stdError != "":
                     if self.instanceId != "0":
                        stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                     else:
                        stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                     try:
                        fpStd = open(stdFile,'a')
                        try:
                           fpStd.write(stdError)
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                        finally:
                           fpStd.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
               break

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self):
      if not self.filesCleanedup:
         if self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and self.instanceId == "0") or not self.isParametric:
                  if not self.sharedUserSpace:
                     remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
                  else:
                     remoteWorkingDirectory = self.instanceDirectory
                  remoteScratchDirectory = self.scratchDirectory.replace('$','\$')

                  if self.isParametric:
                     self.jobOutput.getPegasusStdTimeFiles(self.instanceDirectory,
                                                           self.scratchDirectory,
                                                           self.timeHistoryLogs['timeResults'])
                  else:
                     self.jobOutput.processPegasusFiles(self.instanceDirectory,
                                                        self.scratchDirectory,
                                                        self.timeHistoryLogs['timeResults'])

                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + \
                                                          remoteWorkingDirectory + " " + \
                                                          remoteScratchDirectory + " " + \
                                                          self.remoteBatchSystem + "\""
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,self.tunnelDesignator)[1]))

            workDirectoryName = os.path.join(self.instanceDirectory,'work')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            inProcessDirectoryName = os.path.join(self.instanceDirectory,'InProcessResults')
            if os.path.isdir(inProcessDirectoryName):
               shutil.rmtree(inProcessDirectoryName,True)

            self.cleanupPegasusFiles()
         else:
            if   not self.sharedUserSpace and self.stageFiles:
               remoteWorkingDirectory = self.workingDirectory.replace('$','\$')
               remoteScratchDirectory = self.scratchDirectory.replace('$','\$')
               if self.jobSubmitted:
                  command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + \
                                                   remoteWorkingDirectory + " " + \
                                                   remoteScratchDirectory + " " + \
                                                   self.remoteBatchSystem + "\""
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,self.tunnelDesignator)[1]))
            elif self.sharedUserSpace and self.stageFiles:
               self.cleanupStageOutTarFile()

         self.cleanupStageInTarFile()
         self.cleanupScriptTemplateLogFiles()

         self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      if   self.remoteJobIdNumber:
         command = self.sshBaseCommand + " \"" + self.killBatchJob + " " + \
                                            self.remoteJobIdNumber + " " + \
                                            self.remoteBatchSystem + "\""
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,self.tunnelDesignator)[1]))
      elif self.remoteBatchSystem == 'SCRIPT' and self.childPid != 0:
#        log("Send TERM to child script process %d" % (self.childPid))
#        os.kill(childPid,signalNumber.SIGTERM)
         if self.batchScriptPath != "":
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.batchScriptPath)
         else:
            command = self.sshBaseCommand + " pkill -TERM -f " + os.path.basename(self.appScriptPath)
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         self.logger.log(logging.INFO,getLogMessage(self.executeSSHCommand(command,self.tunnelDesignator)[1]))

         if self.batchScriptPath != "":
            command = self.sshBaseCommand + " pkill -0 -f " + os.path.basename(self.batchScriptPath)
         else:
            command = self.sshBaseCommand + " pkill -0 -f " + os.path.basename(self.appScriptPath)
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         statusCode = self.executeSSHCommand(command,self.tunnelDesignator)[0]
         while statusCode == 0:
            time.sleep(2)
            statusCode = self.executeSSHCommand(command,self.tunnelDesignator)[0]

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 7 | signalNumber

      self.scriptsKilled = True


