# @package      hubzero-submit-distributor
# @file         JobDistributor.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
__version__ = '3.7.1'

import sys
import os
import re
import signal
import select
import datetime
import time
import pwd
import subprocess
import shlex
import random
import requests
import traceback
import math
import copy
import shutil
import logging
from errno import EPIPE

from hubzero.submit.LogMessage               import getLogJobIdMessage as getLogMessage, logSetJobId
from hubzero.submit.DaemonsInfo              import DaemonsInfo
from hubzero.submit.InfosInfo                import InfosInfo
from hubzero.submit.SitesInfo                import SitesInfo
from hubzero.submit.TapisSitesInfo           import TapisSitesInfo
from hubzero.submit.FileMoversInfo           import FileMoversInfo
from hubzero.submit.ToolsInfo                import ToolsInfo
from hubzero.submit.AppsAccessInfo           import AppsAccessInfo
from hubzero.submit.ManagersInfo             import ManagersInfo
from hubzero.submit.ToolFilesInfo            import ToolFilesInfo
from hubzero.submit.DockerImagesInfo         import DockerImagesInfo
from hubzero.submit.TunnelsInfo              import TunnelsInfo
from hubzero.submit.EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from hubzero.submit.SubmissionScriptsInfo    import SubmissionScriptsInfo
from hubzero.submit.CommandParser            import CommandParser
from hubzero.submit.VenueMechanismCache      import VenueMechanismCache
from hubzero.submit.VenueMechanismLocal      import VenueMechanismLocal
from hubzero.submit.VenueMechanismSsh        import VenueMechanismSsh
from hubzero.submit.RemoteIdentityManager    import RemoteIdentityManager
from hubzero.submit.RemoteJobMonitor         import RemoteJobMonitor
from hubzero.submit.RemoteProbeMonitor       import RemoteProbeMonitor
from hubzero.submit.RemoteTunnelMonitor      import RemoteTunnelMonitor
from hubzero.submit.TimeConversion           import hhmmssTomin
from hubzero.submit.TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand
from hubzero.submit.JobStatistic             import JobStatistic
from hubzero.submit.ParameterTemplate        import ParameterTemplate

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
POSTPROCESSJOBCOMMAND  = 'postprocessjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

PRESTAGEINPUTCOMMAND = 'prestagejobinputs.sh'
COPYINPUTCOMMAND     = 'copyjobinputs.sh'
COPYOUTPUTCOMMAND    = 'copyjoboutputs.sh'

LOCALJOBID          = ".__local_jobid"
REMOTEJOBID         = ".__remote_jobid"
REMOTEJOBIDCOMPLETE = ".__remote_jobid_complete"

TIMESTAMPINPUTBUILT  = ".__timestamp_inputbuilt"
TIMESTAMPINPUTSTAGED = ".__timestamp_inputstaged"
TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"

BATCHQUEUETYPES = ['pbs',
                   'lsf',
                   'll',
                   'sge',
                   'slurm',
                   'condor',
                   'factory',
                   'boinc']

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class JobDistributor:
   def __init__(self,
                configurationDirectory,
                distributorConfigurationFile,
                daemonsConfigurationFile,
                infosConfigurationFile):
      self.logger = logging.getLogger(__name__)

      self.configData = {}
      self.configurationDirectory       = configurationDirectory
      self.distributorConfigurationFile = distributorConfigurationFile
      self.daemonsConfigurationFile     = daemonsConfigurationFile
      self.infosConfigurationFile       = infosConfigurationFile

      self.version              = __version__
      self.operationMode        = 0
      self.session              = 0
      self.submitterClass       = 1
      self.distributorPid       = os.getpid()
      self.jobId                = 0
      self.jobs                 = {}
      self.jobStatistics        = {}
      self.reportMetrics        = False
      self.fpUserLogPath        = None
      self.abortAttempted       = False
      self.disableProbeCheck    = True
      self.quotaLimit           = -1
      self.childPid             = 0
      self.bufferSize           = 4096
      self.dataDirectory        = ""
      self.binDirectory         = ""
      self.disableJobMonitoring = False

      configFilePath = os.path.join(self.configurationDirectory,self.daemonsConfigurationFile)
      self.daemonsInfo       = DaemonsInfo(configFilePath)
      self.jobListenURI      = self.daemonsInfo.getDaemonListenURI('jobMonitor','tcp')
      self.identityListenURI = self.daemonsInfo.getDaemonListenURI('identitiesManager','tcp')
      self.probeListenURI    = self.daemonsInfo.getDaemonListenURI('probeMonitor','tcp')
      self.tunnelListenURI   = self.daemonsInfo.getDaemonListenURI('tunnelMonitor','tcp')
      self.cloudListenURI    = self.daemonsInfo.getDaemonListenURI('cloudMonitor','tcp')

      self.batchCommands = {}
      self.batchCommands['receiveInput']    = RECEIVEINPUTCOMMAND
      self.batchCommands['submitBatchJob']  = SUBMITBATCHJOBCOMMAND
      self.batchCommands['postProcessJob']  = POSTPROCESSJOBCOMMAND
      self.batchCommands['transmitResults'] = TRANSMITRESULTSCOMMAND
      self.batchCommands['cleanupJob']      = CLEANUPJOBCOMMAND
      self.batchCommands['killBatchJob']    = KILLBATCHJOBCOMMAND

      self.batchCommands['preStageJobInput'] = PRESTAGEINPUTCOMMAND
      self.batchCommands['copyJobInput']     = COPYINPUTCOMMAND
      self.batchCommands['copyJobOutput']    = COPYOUTPUTCOMMAND

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.nonBatchJobs       = []
      self.splitExecutionJobs = []
      self.waitForJobsInfo    = {}

      self.remoteMonitors = {}
      self.remoteMonitors['job']      = None
      self.remoteMonitors['identity'] = None
      self.remoteMonitors['probe']    = None
      self.remoteMonitors['tunnel']   = None
      self.remoteMonitors['cloud']    = None

      self.infosInfo                = None
      self.sitesInfo                = None
      self.tapisSitesInfo           = None
      self.fileMoversInfo           = None
      self.cloudsInfo               = None
      self.tunnelsInfo              = None
      self.toolsInfo                = None
      self.appsAccessInfo           = None
      self.managersInfo             = None
      self.toolFilesInfo            = None
      self.dockerImagesInfo         = None
      self.environmentWhitelistInfo = None
      self.submissionScriptsInfo    = None

      self.successfulInstance        = None
      self.maximumSelectedSites      = 1
      self.nRedundant                = 1
      self.userDestinations          = []
      self.inputFiles                = []
      self.outputFiles               = []
      self.isMultiCoreRequest        = False
      self.nCpus                     = 1
      self.ppn                       = ""
      self.wallTime                  = 60
      self.environment               = ""
      self.managerSpecified          = False
      self.managerInfo               = {}
      self.x509SubmitProxy           = ""
      self.stdinput                  = ""
      self.isParametric              = False
      self.parameterNames            = []
      self.parameterCombinations     = None
      self.parameterCombinationsPath = None
      self.runType                   = None
      self.filesToRemove             = []
      self.emptyFilesToRemove        = []
      self.nInstances                = 0
      self.executeInstance           = None
      self.inputsPath                = None
      self.jobPath                   = None

      self.commandParser           = None
      self.enteredCommandArguments = []
      self.tailFiles               = []

      self.runName              = ""
      self.localJobId           = None
      self.hubUserId            = None
      self.hubUserName          = None
      self.progressReport       = 'text'
      self.isClientTTY          = True
      self.useSetup             = None
      self.pegasusVersion       = None
      self.pegasusHome          = None
      self.clientSudoUser       = None
      self.clientWorkDirectory  = None
      self.clientTransferPath   = None
      self.cacheHosts           = None
      self.doubleDashTerminator = False

      self.abortGlobal = {}
      self.abortGlobal['abortAttempted'] = self.abortAttempted
      self.abortGlobal['abortSignal']    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def setSystemEnvironment(self):
      if self.configData['pbsRoot'] != "" and self.configData['pbsRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['pbsRoot'],'bin') + os.pathsep + os.environ['PATH']

      if self.configData['condorRoot'] != "" and self.configData['condorRoot'] != "REMOTE":
         os.environ['PATH'] = os.path.join(self.configData['condorRoot'],'bin') + os.pathsep + \
                              os.path.join(self.configData['condorRoot'],'sbin') + os.pathsep + os.environ['PATH']
         if self.configData['condorConfig'] != "":
            os.environ['CONDOR_CONFIG'] = self.configData['condorConfig']

      os.environ['PATH'] = self.binDirectory + os.pathsep + os.environ['PATH']


   def getSystemEnvironment(self):
      self.hubUserName          = os.getenv("USER")
      self.hubUserId            = os.getuid()
      self.session              = int(os.getenv("SESSION",'0'))
      self.submitterClass       = int(os.getenv("SUBMITTER_CLASS",'1'))
      self.isClientTTY          = bool(int(os.getenv("SUBMIT_ISCLIENTTTY",'0')))
      self.pegasusVersion       = os.getenv("PEGASUS_VERSION")
      self.pegasusHome          = os.getenv("PEGASUS_HOME")
      self.useSetup             = os.getenv("USE_SETUP_SCRIPT")
      self.clientSudoUser       = os.getenv("CLIENT_SUDO_USER","")
      self.clientWorkDirectory  = os.getenv("CLIENT_WORK_DIRECTORY")
      self.clientTransferPath   = os.getenv("CLIENT_TRANSFER_PATH")
      self.cacheHosts           = os.getenv("CACHE_HOSTS")
      if self.cacheHosts:
         self.cacheHosts        = [ cacheHost.strip() for cacheHost in self.cacheHosts.split(',') ]
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'False').lower() == 'true')


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inDistributorSection = False

      configured = False
      try:
         configFilePath = os.path.join(self.configurationDirectory,self.distributorConfigurationFile)
         fpConfig = open(configFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inDistributorSection = (sectionName == 'distributor')
                     if inDistributorSection:
                        self.configData = {'probeMonitoringInstalled':False,
                                           'maximumSelectedSites':3,
                                           'allowedVenueMechanisms':['local','ssh'],
                                           'dataDirectory':os.path.join(os.sep,'opt','submit'),
                                           'binDirectory':os.path.join(os.sep,'opt','submit','bin'),
                                           'mountPointTranslations':{},
                                           'condorRoot':'',
                                           'condorConfig':'',
                                           'pbsRoot':''
                                          }
                  elif inDistributorSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = list(self.configData[key].keys())[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 try:
                                    self.configData[key] = json.loads(value)
                                 except:
                                    self.configData[key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.configData[key] = {}
               
                              if not self.configData[key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
               else:
                  eof = True
                  configured = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (configFilePath)))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (configFilePath)))

      if configured:
         if os.path.isdir(self.configData['dataDirectory']):
            self.dataDirectory = self.configData['dataDirectory']
         else:
            message = "Specified dataDirectory does not exist: %s" % (self.configData['dataDirectory'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            configured = False

         if os.path.isdir(self.configData['binDirectory']):
            self.binDirectory = self.configData['binDirectory']
         else:
            message = "Specified binDirectory does not exist: %s" % (self.configData['binDirectory'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            configured = False

      if configured:
         self.setSystemEnvironment()
         self.getSystemEnvironment()
         self.maximumSelectedSites = self.configData['maximumSelectedSites']

      return(configured)


   def doRun(self):
      return(self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID)


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.logger.log(logging.INFO,getLogMessage("Args are:" + str(sys.argv)))
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR:
         self.__writeToStdout("Submit distributor version: %s\n" % (self.version))

      reportVersion = self.commandParser.getOption('version') or \
                      self.commandParser.getOption('versionServer') or \
                      self.commandParser.getOption('versionDistributor')

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.configData['probeMonitoringInstalled']:
            self.disableProbeCheck = self.commandParser.getOption('disableProbeCheck')
         self.disableJobMonitoring = self.commandParser.getOption('disableJobMonitoring')

         exitCode = self.commandParser.setSweepParameters()
         if exitCode == 0:
            exitCode = self.commandParser.setCSVDataParameters()
            if exitCode == 0:
               if self.commandParser.getParameterCombinationCount() > 0:
                  self.isParametric = True

               if not reportVersion:
                  enteredCommand = self.commandParser.getEnteredCommand()
                  if enteredCommand == "":
                     self.logger.log(logging.ERROR,getLogMessage("Command must be supplied"))
                     self.__writeToStderr("Command must be supplied\n")
                     if not exitCode:
                        exitCode = 1
                  else:
                     if self.commandParser.getOption('runName'):
                        self.runName = self.commandParser.getOption('runName')
                     self.reportMetrics = self.commandParser.getOption('metrics')

      if   not self.commandParser.getOption('progress'):
         self.progressReport = 'silent'
      elif self.commandParser.getOption('progressCurses'):
         self.progressReport = 'curses'
      elif self.commandParser.getOption('progressSubmit'):
         self.progressReport = 'submit'
      elif self.commandParser.getOption('progressText'):
         self.progressReport = 'text'
      elif self.commandParser.getOption('progressPegasus'):
         self.progressReport = 'pegasus'
      else:
         if self.isParametric and self.isClientTTY:
            self.progressReport = 'curses'
         else:
            self.progressReport = 'text'

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      def getJobId(dataDirectory):
         nextLocalJobId = 0

         exitCode = 0
         localJobIdFileName = os.path.join(dataDirectory,'localJobId.dat')
         if not os.path.isfile(localJobIdFileName):
            try:
               fpLocalJobId = open(localJobIdFileName,'w')
               try:
                  fpLocalJobId.write('0')
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("could not write %s" % (localJobIdFileName)))
                  self.__writeToStderr("could not write %s\n" % (localJobIdFileName))
                  exitCode = 1
               finally:
                  fpLocalJobId.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("could not open %s" % (localJobIdFileName)))
               self.__writeToStderr("could not open %s\n" % (localJobIdFileName))
               exitCode = 1

         if not exitCode:
            try:
               fpLocalJobId = open(localJobIdFileName,'r')
               try:
                  previousLocalJobId = int(fpLocalJobId.readline().strip())
                  nextLocalJobId = previousLocalJobId+1
                  try:
                     fpLocalJobId = open(localJobIdFileName,'w')
                     try:
                        fpLocalJobId.write('%d' % (nextLocalJobId))
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (localJobIdFileName)))
                     finally:
                        fpLocalJobId.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("could not open %s for writing" % (localJobIdFileName)))
                     self.__writeToStderr("could not open %s for writing\n" % (localJobIdFileName))
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (localJobIdFileName)))
               finally:
                  fpLocalJobId.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("could not open %s for reading" % (localJobIdFileName)))
               self.__writeToStderr("could not open %s for reading\n" % (localJobIdFileName))

         return(nextLocalJobId)

      exitCode = 0

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.commandParser.getOption('jobId'):
            self.jobId = self.commandParser.getOption('jobId')
         if self.jobId == 0:
            self.jobId = getJobId(self.dataDirectory)

      self.localJobId = "%08d" % (self.jobId)
      logSetJobId(self.jobId)

      if self.operationMode & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.jobId == 0:
            exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setInfo(self):
      configFilePath = os.path.join(self.configurationDirectory,self.infosConfigurationFile)
      self.infosInfo = InfosInfo(configFilePath)

      restrictionUser = self.clientSudoUser if self.clientSudoUser else self.hubUserName
#     self.logger.log(logging.DEBUG,getLogMessage("setInfo(restrictionUser): %s" % (restrictionUser)))

      self.sitesInfo = SitesInfo(self.infosInfo.getInfoPath('sites'),
                                 restrictionUser=restrictionUser,
                                 pegasusTemplateDirectory=self.infosInfo.getInfoPath('pegasusTemplates'),
                                 allowedVenueMechanisms=self.configData['allowedVenueMechanisms'],
                                 pegasusVersion=self.pegasusVersion)

      self.tapisSitesInfo = TapisSitesInfo(self.infosInfo.getInfoPath('tapissites'),
                                           restrictionUser=restrictionUser)
      self.fileMoversInfo = FileMoversInfo(self.infosInfo.getInfoPath('filemovers'),
                                           restrictionUser=restrictionUser)

      self.tunnelsInfo = TunnelsInfo(self.infosInfo.getInfoPath('tunnels'))

      self.toolsInfo = ToolsInfo(self.infosInfo.getInfoPath('tools'))
      self.toolsInfo.applyUserRestriction(restrictionUser)
      self.toolsInfo.applyGroupRestriction(restrictionUser=restrictionUser)

      self.appsAccessInfo = AppsAccessInfo(self.infosInfo.getInfoPath('access'),
                                           restrictionUser=restrictionUser)

      self.managersInfo = ManagersInfo(self.infosInfo.getInfoPath('managers'))

      self.toolFilesInfo = ToolFilesInfo(self.infosInfo.getInfoPath('toolfiles'))

      self.dockerImagesInfo = DockerImagesInfo(self.infosInfo.getInfoPath('dockerimages'))

      self.environmentWhitelistInfo = EnvironmentWhitelistInfo(self.infosInfo.getInfoPath('environment'),
                                                               restrictionUser=restrictionUser)

      self.submissionScriptsInfo = SubmissionScriptsInfo('Distributor',
                                                         submissionScriptRootPath=self.infosInfo.getInfoPath('submissionscripts'))


   def showUsage(self):
      if self.operationMode & self.commandParser.OPERATIONMODEHELPUSAGE:
         self.commandParser.showUsage()

      if self.operationMode & self.commandParser.OPERATIONMODEHELPVENUES:
         venues = []
         for site in self.sitesInfo.getEnabledSites():
            venues.append(site)

         if len(venues) > 0:
            venues.sort()
            self.__writeToStdout("\nCurrently available VENUES are:\n")
            for venue in venues:
               self.__writeToStdout("   %s\n" % (venue))
         del venues

      if self.operationMode & self.commandParser.OPERATIONMODEHELPTOOLS:
         executableToolNames = self.toolsInfo.getToolNames()
         if len(executableToolNames) > 0:
            executableToolNames.sort()
            self.__writeToStdout("\nCurrently available TOOLs are:\n")
            for executableToolName in executableToolNames:
               self.__writeToStdout("   %s\n" % (executableToolName))
         del executableToolNames

      if self.operationMode & self.commandParser.OPERATIONMODEHELPMANAGERS:
         validManagers = self.managersInfo.getManagerNames()
         if len(validManagers) > 0:
            validManagers.sort()
            self.__writeToStdout("\nCurrently available MANAGERs are:\n")
            for validManager in validManagers:
               self.__writeToStdout("   %s\n" % (validManager))
         del validManagers


   def reportExitCondition(self):
      if self.doRun() and not self.disableJobMonitoring:
#        self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(len(self.jobs)): %d" % (len(self.jobs))))
         if len(self.jobs) == 0:
            jobIndices = list(self.jobStatistics.keys())
            maximumJobIndex = max(jobIndices)
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(maximumJobIndex): %d" % (maximumJobIndex)))

            try:
               mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
               remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
               remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
               exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
               cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
               elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
               waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
               nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']
               event          = self.jobStatistics[maximumJobIndex]['event']
               generated      = self.jobStatistics[maximumJobIndex]['generated']

               message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                         (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
               if event:
                  message += " event=%s" % (event)
               if generated:
                  message += " generated=%s" % (generated)
               self.logger.log(logging.INFO,getLogMessage(message))
               message += "\n"
               if self.fpUserLogPath:
                  self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                  self.fpUserLogPath.flush()
               os.write(3,message.encode('utf-8'))
            except:
               self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
               pass
         else:
#           if self.successfulInstance == None:
#              self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(successfulInstance): None"))
#           else:
#              self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(successfulInstance): %d" % (self.successfulInstance)))
            instanceIndex = 0
            instances = list(self.jobs.keys())
            instances.sort()
            for instance in instances:
#              self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(instance): %d" % (instance)))
               if (self.successfulInstance == None) or \
                  (self.successfulInstance != None and (instance != self.successfulInstance)):
                  jobIndices = list(self.jobs[instance].jobStatistics.keys())
                  jobIndices.sort()
                  maximumJobIndex = max(jobIndices)
#                 self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(maximumJobIndex): %d" % (maximumJobIndex)))

                  try:
                     for jobIndex in jobIndices:
                        if jobIndex != 0 or maximumJobIndex == 0:
                           mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                           remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                           remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                           exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                           cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                            self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                           elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                           waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                           nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                           event          = self.jobs[instance].jobStatistics[jobIndex]['event']
                           generated      = self.jobs[instance].jobStatistics[jobIndex]['generated']

                           instanceIndex += 1
                           message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                 (instanceIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                           if event:
                              message += " event=%s" % (event)
                           if generated:
                              message += " generated=%s" % (generated)
                           self.logger.log(logging.INFO,getLogMessage(message))
                           message += "\n"
                           if self.fpUserLogPath:
                              self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                              self.fpUserLogPath.flush()
                           os.write(3,message.encode('utf-8'))
                  except:
                     self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                     pass

            if self.successfulInstance != None:
               instance = self.successfulInstance
               jobIndices = list(self.jobs[instance].jobStatistics.keys())
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               executionMode = waitForJobInfo['executionMode']

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                        if self.jobs[instance].remoteBatchSystem == 'PEGASUS':
                           if mechanism == "" or mechanism == 'Unknown':
                              mechanism = 'PEGASUS'
                        else:
                           if jobIndex != maximumJobIndex:
                              if exitCode == 0:
                                 exitCode = 65533
                        cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                        event          = self.jobs[instance].jobStatistics[jobIndex]['event']
                        generated      = self.jobs[instance].jobStatistics[jobIndex]['generated']

                        if instance == 0 and len(jobIndices) == 1 and \
                           (self.runType == 'pegasus' or self.runType == 'boinc'):
                           message = "venue=0:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                               (              mechanism,remoteId,remoteLocation, \
                                                              exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        else:
                           instanceIndex += 1
                           message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                               (instanceIndex,mechanism,remoteId,remoteLocation, \
                                                              exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        if event:
                           message += " event=%s" % (event)
                        if generated:
                           message += " generated=%s" % (generated)
                        message += " executionMode=%s" % (executionMode)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        message += "\n"
                        if self.fpUserLogPath:
                           self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                           self.fpUserLogPath.flush()
                        os.write(3,message.encode('utf-8'))
               except:
                  self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                  pass

      if self.fpUserLogPath:
         self.fpUserLogPath.close()
         self.fpUserLogPath = None


# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortGlobal['abortAttempted'] = self.abortAttempted
         self.abortGlobal['abortSignal']    = signalNumber

         for instance in self.waitForJobsInfo:
            if 'isBatchJob' in self.waitForJobsInfo[instance]:
               if not self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'R':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def getUserQuota(self):
      tryStandardQuotaCommand = True
      if self.clientSudoUser:
         checkUser = self.clientSudoUser
      else:
         checkUser = self.hubUserName

      try:
         homeDirectory = pwd.getpwnam(checkUser).pw_dir
      except:
         pass
      else:
         dfCommand = ['df','--portability',homeDirectory]
         child = subprocess.Popen(dfCommand,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         dfStdOutput,dfStdError = child.communicate()
         dfExitStatus = child.returncode
         if dfExitStatus == 0:
#Filesystem                             1024-blocks     Used Available Capacity Mounted on
#nanohub.ogre.hubzero.org:/home/nanohub    71959904 31659072  39552064      45% /home/nanohub
            homeDirectoryMount = ''.join(dfStdOutput.decode('utf-8')).strip().split('\n')[1:][0]
#nanohub.ogre.hubzero.org:/home/nanohub    71959904 31659072  39552064      45% /home/nanohub
            device = homeDirectoryMount.split()[0]
#nanohub.ogre.hubzero.org:/home/nanohub
            if ':' in device:
               fileServer = device.split(':')[0]
               ncCommand = ['nc','-w','5',fileServer,'301']
               self.logger.log(logging.INFO,getLogMessage("quotaCommand = " + ' '.join(ncCommand)))
               child = subprocess.Popen(ncCommand,
                                        stdin=subprocess.PIPE,
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        close_fds=True)
               ncCommand = "getquota user=%s\n" % (checkUser)
               ncStdOutput,ncStdError = child.communicate(ncCommand.encode('utf-8'))
               ncExitStatus = child.returncode
               if ncExitStatus == 0:
                  spaceUsed = -1
                  softSpace = -1
# status=good,softspace=10240000000,hardspace=21990232064,space=7355744256,files=139095,remaining=0
                  quotaResponse = ncStdOutput.decode('utf-8').strip().split(',')
                  if quotaResponse:
                     for attribute in quotaResponse:
                        try:
                           key,value = attribute.split('=')
                        except:
                           pass
                        else:
                           if   key == 'space':
                              spaceUsed = int(value)
                           elif key == 'softspace':
                              softSpace = int(value)
                     if (softSpace > -1) and (spaceUsed > -1):
                        spaceAvailable = softSpace-spaceUsed
                        self.quotaLimit = int(spaceAvailable*0.9/1024.0)
                        tryStandardQuotaCommand = False
               else:
                  self.logger.log(logging.ERROR,getLogMessage(ncStdOutput.decode('utf-8')))
                  self.logger.log(logging.ERROR,getLogMessage(ncStdError.decode('utf-8')))
         else:
            self.logger.log(logging.ERROR,getLogMessage(dfStdOutput.decode('utf-8')))
            self.logger.log(logging.ERROR,getLogMessage(dfStdError.decode('utf-8')))

      if tryStandardQuotaCommand:
         quotaCommand = ['quota','-w']
         self.logger.log(logging.INFO,getLogMessage("quotaCommand = " + ' '.join(quotaCommand)))
         exitStatus,stdOutput,stdError = self.executeCommand(quotaCommand)
         if exitStatus == 0:
            try:
               quotaResults = stdOutput.strip().split('\n')[-1].split()
               if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
                  currentUsage = int(quotaResults[1].strip("*"))
                  self.quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
            except:
               self.logger.log(logging.ERROR,getLogMessage(stdOutput))
               self.logger.log(logging.ERROR,getLogMessage(stdError))
               self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))

      if self.quotaLimit >= 0:
         self.logger.log(logging.INFO,getLogMessage("quotaLimit = %d kbytes" % (self.quotaLimit)))


   def getStdInputFile(self,
                       inputFiles):
      stdInputFile = '/dev/null'
      toCheck = []
      stdinFd = sys.stdin.fileno()
      toCheck.append(stdinFd)
      try:
         ready = select.select(toCheck,[],[],0.) # wait for input
      except select.error:
         ready = {}
         ready[0] = []

      if stdinFd in ready[0]:
         content = sys.stdin.read()
         if content != "":
            stdInputFile = ".__%s.stdin" % (self.localJobId)
            try:
               stdinFile = open(stdInputFile,'w')
               try:
                  stdinFile.write(content)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdInputFile)))
               else:
                  inputFiles.append(stdInputFile)
               finally:
                  stdinFile.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdInputFile)))
         del content

      return(stdInputFile)


   def executeCommand(self,
                      command,
                      useShell=False,
                      streamOutput=False):
      if isinstance(command,list):
         child = subprocess.Popen(command,shell=useShell,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      else:
         commandArgs = shlex.split(command)
         child = subprocess.Popen(commandArgs,shell=useShell,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize).decode('utf-8')
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize).decode('utf-8')
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def writeInputBuiltFile(self,
                           inputBuiltPath):
      inputBuiltFileCreated = False
      try:
         fpInputBuilt = open(inputBuiltPath,'w')
         try:
            timeBuilt = int(time.time())
            fpInputBuilt.write("%d\n" % (timeBuilt))
            fpInputBuilt.close()
            inputBuiltFileCreated = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputBuiltPath)))
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputBuiltPath)))

      return(inputBuiltFileCreated)


   def buildJobDescription(self):
      exitCode = 0

      self.logger.log(logging.INFO,getLogMessage("session = %d" % (self.session)))
      if self.commandParser.getOption('quotaCheck'):
         self.getUserQuota()

      userDestinations = []
      inputFiles       = []
      nCpus            = 1
      ppn              = ""
      wallTime         = 60
      environment      = "SESSION=%d " % (self.session)

      if self.commandParser.getOption('destinations'):
         userDestinations = self.commandParser.getOption('destinations')
      if len(userDestinations) > 0:
         reasonsDenied = self.sitesInfo.purgeDisabledSites(userDestinations)
         if len(userDestinations) == 0:
            errorMessage = "Access to all listed venues has been denied.\n"
            for userDestination in reasonsDenied:
               errorMessage += "   %20s %s\n" % (userDestination,reasonsDenied[userDestination])
            errorMessage += "Please select another venue or attempt execution at a later time."
            self.logger.log(logging.ERROR,getLogMessage(errorMessage))
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 7
         del reasonsDenied

      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            inputFiles.append(inputFile)
      self.stdinput = self.getStdInputFile(inputFiles)

      if self.commandParser.getOption('outputFiles'):
         for outputFile in self.commandParser.getOption('outputFiles'):
            self.outputFiles.append(outputFile)

      if self.commandParser.getOption('manager'):
         manager = self.commandParser.getOption('manager')[0]
         if self.managersInfo.managerExists(manager):
            self.managerSpecified = True
            self.managerInfo      = self.managersInfo.getManagerInfo(manager)
         else:
            validManagers = self.managersInfo.getManagerNames()
            message = "Invalid manager %s specified. Valid managers are %s" % (manager,validManagers)
            self.logger.log(logging.ERROR,getLogMessage(message))
            self.__writeToStderr("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
            del validManagers
            if not exitCode:
               exitCode = 1
      else:
         self.managerInfo = self.managersInfo.getDefaultManagerInfo('serial')

      if self.commandParser.getOption('nCpus') != None:
         self.isMultiCoreRequest = True
         nCpus = self.commandParser.getOption('nCpus')

      if self.commandParser.getOption('ppn'):
         if self.isMultiCoreRequest:
            ppn = str(self.commandParser.getOption('ppn'))

      if self.commandParser.getOption('nRedundant'):
         self.nRedundant = min(max(1,self.commandParser.getOption('nRedundant')),self.maximumSelectedSites)

      if self.commandParser.getOption('wallTime'):
         if ":" in self.commandParser.getOption('wallTime'):
            wallTime = hhmmssTomin(self.commandParser.getOption('wallTime'))
         else:
            try:
               wallTime = float(self.commandParser.getOption('wallTime'))
            except ValueError:
               message = "Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes"
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes\n")
               if not exitCode:
                  exitCode = 1
         if wallTime <= 0.:
            self.logger.log(logging.ERROR,getLogMessage("wallTime must be > 0"))
            self.__writeToStderr("wallTime must be > 0\n")
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('userLogPath'):
         userLogPath = self.commandParser.getOption('userLogPath')
         try:
            self.fpUserLogPath = open(userLogPath,'a')
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("User log %s could not be opened." % (userLogPath)))
            self.__writeToStderr("User log %s could not be opened.\n" % (userLogPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('userJobidPath'):
         userJobidPath = self.commandParser.getOption('userJobidPath')
         try:
            fpUserJobId = open(userJobidPath,'w')
            try:
               fpUserJobId.write("%d\n" % (self.jobId))
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (userJobidPath)))
               if not exitCode:
                  exitCode = 1
            finally:
               fpUserJobId.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("User jobId file %s could not be created." % (userJobidPath)))
            self.__writeToStderr("User jobId file %s could not be created.\n" % (userJobidPath))
            if not exitCode:
               exitCode = 1

      if self.commandParser.getOption('x509SubmitProxy'):
         try:
            self.x509SubmitProxy = os.path.expandvars(os.path.expanduser(self.commandParser.getOption('x509SubmitProxy')))
         except:
            pass

      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               message = "Invalid environment variable %s specified." % (environmentVariableValue)
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("%s\n" % (message))
               if not exitCode:
                  exitCode = 1
            else:
               if self.environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
                  environment += environmentVariable + "=" + value + " "
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Environment variable %s could not be set." % (environmentVariable)))
                  self.__writeToStderr("Environment variable %s could not be set.\n" % (environmentVariable))
      environment = environment.strip()

      if self.isMultiCoreRequest:
         if not self.managerSpecified:
            self.managerInfo = self.managersInfo.getDefaultManagerInfo('mpi')

      if not self.disableProbeCheck:
         ignoreProbeSites = self.sitesInfo.getIgnoreProbeSites()
         self.remoteMonitors['probe'] = RemoteProbeMonitor(self.probeListenURI,
                                                           ignoreProbeSites=ignoreProbeSites)

      if len(userDestinations) > 0:
         self.sitesInfo.purgeOfflineSites(userDestinations,self.remoteMonitors['probe'])
         if len(userDestinations) == 0:
            errorMessage = "All specified venues are out of service.\n" + \
                           "Please select another venue or attempt execution at a later time."
            self.logger.log(logging.ERROR,getLogMessage(errorMessage))
            self.__writeToStderr(errorMessage + "\n")
            exitCode = 8

      nUserDestinations = len(userDestinations)
      self.sitesInfo.purgeResourceLimitedSites(userDestinations,nCpus,wallTime)
      if nUserDestinations > 0 and len(userDestinations) == 0:
         errorMessage = "All specified venues are unable to meet the specified resource\n" + \
                        "requirements.  Please select another venue."
         self.logger.log(logging.ERROR,getLogMessage(errorMessage))
         self.__writeToStderr(errorMessage + "\n")
         exitCode = 10

      nUserDestinations = len(userDestinations)
      self.sitesInfo.purgeExecutionModeLimitedSites(userDestinations,self.commandParser.getOption('asynchronous'))
      if nUserDestinations > 0 and len(userDestinations) == 0:
         errorMessage = "All specified venues are unable to meet the specified synchronicity\n" + \
                        "requirement.  Please select another venue."
         self.logger.log(logging.ERROR,getLogMessage(errorMessage))
         self.__writeToStderr(errorMessage + "\n")
         exitCode = 10

      toolDestinations = self.toolsInfo.getAllToolDestinations()
      self.sitesInfo.purgeOfflineSites(toolDestinations,self.remoteMonitors['probe'])
      self.toolsInfo.purgeOfflineSites(toolDestinations)

      self.userDestinations = copy.copy(userDestinations)
      self.inputFiles       = copy.copy(inputFiles)
      self.nCpus            = copy.copy(nCpus)
      self.ppn              = copy.copy(ppn)
      self.wallTime         = copy.copy(wallTime)
      self.environment      = copy.copy(environment)

      self.jobStatistics[0]['exitCode'] = exitCode

      if not exitCode:
         exitCode = self.setRunType()
         if exitCode:
            self.jobStatistics[0]['exitCode'] = exitCode

      if self.commandParser.getOption('tailFiles'):
         tailFiles = self.commandParser.getOption('tailFiles')
         for tailFile in tailFiles:
            if ':' in tailFile:
               fileName,nLines = tailFile.split(':')
               try:
                  nLines = int(nLines)
               except:
                  nLines = 10
            else:
               fileName = tailFile
               nLines = 10
            self.tailFiles.append("%s:%d" % (fileName,nLines))

      nLines = self.commandParser.getOption('tailStdoutLength')
      if nLines:
         self.tailFiles.append("%s:%s" % ('#STDOUT#',nLines[-1]))
      nLines = self.commandParser.getOption('tailStderrLength')
      if nLines:
         self.tailFiles.append("%s:%s" % ('#STDERR#',nLines[-1]))

      return(exitCode)


   def findInputFileOrDirectory(self,
                                currentWorkingDirectory,
                                inputFileOrDirectory):
      fileOrDirectory = ""
      stripedInputFileOrDirectory = inputFileOrDirectory.replace('@:','')
#     if stripedInputFileOrDirectory.startswith(os.sep):
#        if stripedInputFileOrDirectory.startswith('/data/projects/'): 
#           stripedInputFileOrDirectory = os.path.join(os.sep,'sftp',stripedInputFileOrDirectory.lstrip(os.sep))
#        relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
#     else:
#        relativeInputFileOrDirectory = stripedInputFileOrDirectory
      if stripedInputFileOrDirectory.startswith(os.sep):
         for clientMountPoint in self.configData['mountPointTranslations']:
            if stripedInputFileOrDirectory.startswith(clientMountPoint): 
               serverMountPoint = self.configData['mountPointTranslations'][clientMountPoint]['serverMountPoint']
               stripedInputFileOrDirectory = os.path.join(serverMountPoint,stripedInputFileOrDirectory[len(clientMountPoint):])
               break
         relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
      else:
         relativeInputFileOrDirectory = stripedInputFileOrDirectory

      if self.clientWorkDirectory.startswith(os.sep):
         relativeClientWorkDirectory = self.clientWorkDirectory[1:]
      else:
         relativeClientWorkDirectory = self.clientWorkDirectory

      if self.clientTransferPath:
         candidateFileOrDirectory = os.path.abspath(os.path.join(self.clientTransferPath,
                                                                 relativeClientWorkDirectory,
                                                                 relativeInputFileOrDirectory))
         if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
            fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory

      if not fileOrDirectory:
         if self.clientWorkDirectory == currentWorkingDirectory:
            candidateFileOrDirectory = stripedInputFileOrDirectory
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
         else:
            candidateFileOrDirectory = os.path.join(self.clientWorkDirectory,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               fileOrDirectory = candidateFileOrDirectory
            else:
               candidateFileOrDirectory = os.path.join(currentWorkingDirectory,relativeInputFileOrDirectory)
               if   os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
                  fileOrDirectory = candidateFileOrDirectory
               elif stripedInputFileOrDirectory.startswith(os.sep):
                  candidateFileOrDirectory = stripedInputFileOrDirectory
                  if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
                     fileOrDirectory = candidateFileOrDirectory

#     if not fileOrDirectory:
#        self.logger.log(logging.DEBUG,getLogMessage("findInputFileOrDirectory: workDirectory=%s inputDirectory=%s" % \
#                                                          (relativeClientWorkDirectory,relativeInputFileOrDirectory)))

      if fileOrDirectory:
         if '@:' in inputFileOrDirectory:
            fileOrDirectory = '@:' + fileOrDirectory

      return(fileOrDirectory)


   def isClientTransferFileOrDirectory(self,
                                       inputFileOrDirectory):
      clientTransferFileorDirectory = False
      stripedInputFileOrDirectory = inputFileOrDirectory.replace('@:','')
      if stripedInputFileOrDirectory.startswith(os.sep):
         relativeInputFileOrDirectory = stripedInputFileOrDirectory[1:]
      else:
         relativeInputFileOrDirectory = stripedInputFileOrDirectory
      if self.clientWorkDirectory.startswith(os.sep):
         relativeClientWorkDirectory = self.clientWorkDirectory[1:]
      else:
         relativeClientWorkDirectory = self.clientWorkDirectory

      if self.clientTransferPath:
         candidateFileOrDirectory = os.path.abspath(os.path.join(self.clientTransferPath,
                                                                 relativeClientWorkDirectory,
                                                                 relativeInputFileOrDirectory))
         if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
            clientTransferFileorDirectory = True
         else:
            candidateFileOrDirectory = os.path.join(self.clientTransferPath,relativeInputFileOrDirectory)
            if os.path.isfile(candidateFileOrDirectory) or os.path.isdir(candidateFileOrDirectory):
               clientTransferFileorDirectory = True

      return(clientTransferFileorDirectory)


   class CommandError(Exception):
      pass

   class InstancesError(Exception):
      pass

   class InstanceError(Exception):
      pass


   def getUseEnvironment(self):
      useEnvironment = ""
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = list(filter(reChoice.search,os.environ.keys()))
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         includedScripts = []
         useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
         useEnvironment += ". %s\n" % (self.useSetup)
         if "ENVIRON_CONFIG_DIRS" in os.environ:
            useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
         for environmentVar in environmentVars:
            includedScript = os.environ[environmentVar]
            if not includedScript in includedScripts:
               useEnvironment += "use -e -r " + includedScript + "\n"
               includedScripts.append(includedScript)
         del includedScripts

      return(useEnvironment)


   @staticmethod
   def __getManagerEnvironment(managerInfoEnvironment):
      managerEnvironment = ""
      if len(managerInfoEnvironment) > 0:
         for environmentVariableValue in managerInfoEnvironment:
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]

            if environmentVariable != "" and value != "":
               managerEnvironment += environmentVariable + "=" + value + " "
         managerEnvironment = managerEnvironment.strip()

      return(managerEnvironment)


   @staticmethod
   def __getToolEnvironment(toolInfoEnvironment):
      toolEnvironment = ""
      if len(toolInfoEnvironment) > 0:
         for environmentVariableValue in toolInfoEnvironment:
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  value = " "
            else:
               message = "Tool environment variable must be of the form Variable = Value."
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr(message + "\n")

            if environmentVariable != "" and value != "":
               toolEnvironment += environmentVariable + "=" + value + " "
         toolEnvironment = toolEnvironment.strip()

      return(toolEnvironment)


   def setRunType(self):
      exitCode = 0
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("getRunType:os.getcwd(): No such file or directory"))
         exitCode = 1
      else:
         userDestinations         = []
         executable               = ""
         transferExecutable       = None
         executableClassification = ''
         toolInfo                 = None

         self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
         userExecutable = self.enteredCommandArguments[0]
         self.enteredCommandArguments[0] = os.path.expandvars(os.path.expanduser(userExecutable))

         try:
            executable = self.enteredCommandArguments[0]
            if "/" in executable:
               candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
               if candidateFileOrDirectory:
                  self.enteredCommandArguments[0] = candidateFileOrDirectory

            executable = self.enteredCommandArguments[0]
            transferExecutable = True
            if "/" in executable:
               if   not os.path.isfile(executable):
                  self.logger.log(logging.ERROR,getLogMessage("Missing executable: %s" % (executable)))
                  self.__writeToStderr("Missing executable: %s\n" % (executable))
                  if not exitCode:
                     exitCode = 1
                     raise JobDistributor.CommandError
               elif not os.access(executable,os.X_OK):
                  self.logger.log(logging.ERROR,getLogMessage("Permission denied on executable: %s" % (executable)))
                  self.__writeToStderr("Permission denied on executable: %s\n" % (executable))
                  if not exitCode:
                     exitCode = 1
                     raise JobDistributor.CommandError
            else:
               transferExecutable = False

            applicationIsStaged = False
            if transferExecutable:
               submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
               if not submissionAllowed:
                  self.logger.log(logging.ERROR,getLogMessage("Access to %s denied for %s" % (executable,self.hubUserName)))
                  self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
                  if not exitCode:
                     exitCode = 9
                     raise JobDistributor.CommandError
            else:
               if self.toolsInfo.isExecutableTool(executable):
                  if self.toolsInfo.isPermissionGranted(executable):
                     applicationIsStaged      = True
                     executableClassification = 'staged'
               if not applicationIsStaged:
                  submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
                  if not submissionAllowed:
                     self.logger.log(logging.ERROR,getLogMessage("Access to %s denied for %s" % (executable,self.hubUserName)))
                     self.__writeToStderr("Access to %s denied for %s\n" % (executable,self.hubUserName))
                     if not exitCode:
                        exitCode = 9
                        raise JobDistributor.CommandError

            enabledSites = self.sitesInfo.getEnabledSites()
            self.toolsInfo.purgeDisabledSites(enabledSites)
            if not self.isParametric:
               pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
               self.toolsInfo.purgePegasusSites(pegasusSites)

            userMappedDestinations = []
            for userDestination in self.userDestinations:
               if self.sitesInfo.siteExists(userDestination):
                  isUserDestinationValid = True
                  if transferExecutable:
                     if not self.isParametric:
                        if userDestination in pegasusSites:
                           errorMessage = "Requested venue does not have a suitable submission method.\n" + \
                                          "Please select another venue."
                           self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                           self.__writeToStderr(errorMessage + "\n")
                           isUserDestinationValid = False
                  if isUserDestinationValid:
                     userMappedDestinations.append(userDestination)
               else:
                  errorMessage = "Requested venue does not exist.\n" + \
                                 "Please select another venue."
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr(errorMessage + "\n")
                  exitCode = 10
                  raise JobDistributor.CommandError

            if not self.isParametric:
               del pegasusSites

            if len(self.userDestinations) > 0 and len(userMappedDestinations) == 0:
               errorMessage = "All specified venues are unable to meet\n" + \
                              "the submission requirements.  Please select another venue."
               self.logger.log(logging.ERROR,getLogMessage(errorMessage))
               self.__writeToStderr(errorMessage + "\n")
               exitCode = 10
               raise JobDistributor.CommandError

            userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                   executableClassification,userMappedDestinations) + \
                               self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                   '*',userMappedDestinations)

            if len(self.userDestinations) > 0 and len(userDestinations) == 0:
               errorMessage = "All specified venues are unable to meet\n" + \
                              "the access requirements.  Please select another venue."
               self.logger.log(logging.ERROR,getLogMessage(errorMessage))
               self.__writeToStderr(errorMessage + "\n")
               exitCode = 10
               raise JobDistributor.CommandError

         except JobDistributor.CommandError:
            self.jobStatistics[0]['exitCode'] = exitCode
         else:
            toolInfo = self.toolsInfo.getDefaultToolInfo()
            destination  = ""
            destinations = []

            try:
               if not transferExecutable:
                  if self.toolsInfo.isExecutableTool(executable):
                     if self.toolsInfo.isPermissionGranted(executable):
                        expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,
                                                                                    remoteProbeMonitor=self.remoteMonitors['probe'])
                        toolInfo = self.toolsInfo.selectTool(executable,expandedUserDestinations)
                        del expandedUserDestinations
                        if len(toolInfo['destinations']) == 0:
                           errorMessage = "The combination of destination and executable is not properly specified."
                           self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                           self.__writeToStderr("%s\n" % (errorMessage))
                           exitCode = 1
                           raise JobDistributor.InstancesError

                        executable               = toolInfo['executablePath']
                        executableClassification = 'staged'
                     else:
                        self.logger.log(logging.ERROR,getLogMessage("Access to %s is restricted by user or group" % (executable)))
                        self.__writeToStderr("Access to %s is restricted by user or group\n" % (executable))
                        exitCode = 1
                        raise JobDistributor.InstancesError

               if len(toolInfo['destinations']) == 0:
# if tool is not staged and only one user requested site is available.
                  destinations = self.sitesInfo.selectSites(userDestinations)
                  if len(destinations) == 1:
                     destination = destinations[0]
                     if self.isParametric:
                        if   self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                           self.runType = 'pegasus'
                        elif self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'BOINC':
                           self.runType = 'boinc'
                        else:
                           self.runType = 'sweep'
                     else:
                        self.runType = 'redundant'

               if destination == "":
                  if len(toolInfo['destinations']) > 0:
# if tool is staged pick site(s)
                     if not self.isParametric:
                        destinations = self.sitesInfo.selectSites(toolInfo['destinations'])
                        if len(destinations) > 0:
                           destination = random.choice(destinations)
                           if self.isParametric:
                              if   self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                                 self.runType = 'pegasus'
                              elif self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'BOINC':
                                 self.runType = 'boinc'
                              else:
                                 self.runType = 'sweep'
                           else:
                              self.runType = 'redundant'
                     else:
                        pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                           toolInfo['destinations'])
                        boincSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','BOINC',
                                                                         toolInfo['destinations'])
                        parametricSites = pegasusSites+boincSites
                        destinations = self.sitesInfo.selectSites(parametricSites)
                        del boincSites
                        del pegasusSites
                        if len(destinations) > 0:
                           destination = random.choice(destinations)
                           if self.isParametric:
                              if   self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                                 self.runType = 'pegasus'
                              elif self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'BOINC':
                                 self.runType = 'boinc'
                           else:
                              self.runType = 'redundant'
                        else:
                           nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                                    toolInfo['destinations'])
                           nonBoincSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','BOINC',
                                                                                  toolInfo['destinations'])
                           nonParametricSites = [site for site in nonPegasusSites if site in nonBoincSites]
                           destinations = self.sitesInfo.selectSites(nonParametricSites)
                           del nonBoincSites
                           del nonPegasusSites
                           if len(destinations) > 0:
                              destination = 'Sweep'
                              self.runType = 'sweep'

               if destination == "":
                  if len(userDestinations) > 1:
# if user requested site is available
                     if not self.isParametric:
                        destinations = userDestinations
                        destination = random.choice(destinations)
                        self.runType = 'redundant'
                     else:
                        pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                           userDestinations)
                        boincSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','BOINC',
                                                                         userDestinations)
                        parametricSites = pegasusSites+boincSites
                        destinations = self.sitesInfo.selectSites(parametricSites)
                        del boincSites
                        del pegasusSites
                        if len(destinations) > 0:
                           destination = random.choice(destinations)
                           if   self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                              self.runType = 'pegasus'
                           elif self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'BOINC':
                              self.runType = 'boinc'
                        else:
                           nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                                    userDestinations)
                           nonBoincSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','BOINC',
                                                                                  userDestinations)
                           nonParametricSites = [site for site in nonPegasusSites if site in nonBoincSites]
                           destinations = self.sitesInfo.selectSites(nonParametricSites)
                           del nonBoincSites
                           del nonPegasusSites
                           if len(destinations) > 0:
                              destination = 'Sweep'
                              self.runType = 'sweep'

               if not self.isParametric:
                  if   self.sitesInfo.siteExists(destination):
# selected an existing site
                     pass
                  else:
# select site from those declared with undeclaredSiteSelectionWeight
                     enabledSites = self.sitesInfo.getEnabledSites()
                     self.sitesInfo.purgeOfflineSites(enabledSites,self.remoteMonitors['probe'])
                     destinations = self.sitesInfo.selectUndeclaredSites(enabledSites,
                                                                         executableClassification,
                                                                         self.nRedundant)
                     if len(destinations) > 0:
                        self.runType = 'redundant'
                     else:
                        self.logger.log(logging.ERROR,getLogMessage("Currently no venues are available"))
                        self.__writeToStderr("Currently no venues are available\n")
                        exitCode = 1
                        raise JobDistributor.InstancesError
               else:
                  if   self.sitesInfo.siteExists(destination):
# selected an existing site
                     pass
                  elif destination == 'Sweep':
# non pegausus/boinc sweep using one or more destinations
                     pass
                  else:
# select site from those declared with undeclaredSiteSelectionWeight
                     enabledSites = self.sitesInfo.getEnabledSites()
                     self.sitesInfo.purgeOfflineSites(enabledSites,self.remoteMonitors['probe'])
                     if self.pegasusVersion:
                        pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                           enabledSites)
                     else:
                        pegasusSites = []
                     boincSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','BOINC',
                                                                      enabledSites)
                     parametricSites = pegasusSites+boincSites
                     destination = self.sitesInfo.selectUndeclaredSite(parametricSites,
                                                                       executableClassification)
                     del boincSites
                     del pegasusSites
                     if destination:
                        if   self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                           self.runType = 'pegasus'
                        elif self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'BOINC':
                           self.runType = 'boinc'
                     else:
                        nonPegasusSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                                 enabledSites)
                        nonBoincSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','BOINC',
                                                                               enabledSites)
                        nonParametricSites = [site for site in nonPegasusSites if site in nonBoincSites]
                        destinations = self.sitesInfo.selectUndeclaredSites(nonParametricSites,
                                                                            executableClassification,
                                                                            self.nRedundant)
                        del nonBoincSites
                        del nonPegasusSites
                        if len(destinations) > 0:
                           destination = 'Sweep'
                           self.runType = 'sweep'
                        else:
                           self.logger.log(logging.ERROR,getLogMessage("Currently no venues are available"))
                           self.__writeToStderr("Currently no venues are available\n")
                           exitCode = 1
                           raise JobDistributor.InstancesError

            except JobDistributor.InstancesError:
               self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def getInputFiles(self,
                     currentWorkingDirectory,
                     substitutions=None):
      userInputFiles = []
      for inputFile in self.inputFiles:
         if substitutions:
            template = ParameterTemplate(inputFile)
            try:
               inputFile = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,inputFile)
         if candidateFileOrDirectory:
            if not candidateFileOrDirectory in userInputFiles:
               userInputFiles.append(candidateFileOrDirectory)

      for environmentVariableValue in self.environment.split():
         environmentVariable,value = environmentVariableValue.split('=')
         if substitutions:
            template = ParameterTemplate(value)
            try:
               value = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,value)
         if candidateFileOrDirectory:
            if not candidateFileOrDirectory in userInputFiles:
               userInputFiles.append(candidateFileOrDirectory)

      if len(self.enteredCommandArguments) > 1:
         for arg in self.enteredCommandArguments[1:]:
            if substitutions:
               template = ParameterTemplate(arg)
               try:
                  arg = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
            if not arg.startswith('-'):
               if '=' in arg:
                  arglets = arg.split('=')
                  for arglet in arglets[1:]:
                     candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                     if candidateFileOrDirectory:
                        if not candidateFileOrDirectory in userInputFiles:
                           userInputFiles.append(candidateFileOrDirectory)
               else:
                  candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
                  if candidateFileOrDirectory:
                     if not candidateFileOrDirectory in userInputFiles:
                        userInputFiles.append(candidateFileOrDirectory)
            else:
               lexArgs = shlex.split(arg)
               for lexArg in lexArgs:
                  if '=' in lexArg:
                     arglets = lexArg.split('=')
                     for arglet in arglets[1:]:
                        candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                        if candidateFileOrDirectory:
                           if not candidateFileOrDirectory in userInputFiles:
                              userInputFiles.append(candidateFileOrDirectory)
                  else:
                     candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                     if candidateFileOrDirectory:
                        if not candidateFileOrDirectory in userInputFiles:
                           userInputFiles.append(candidateFileOrDirectory)

      if len(userInputFiles) > 1:
         filteredInputFiles = []
         for inputFile in userInputFiles:
            if '@:' in inputFile:
               filteredInputFile = '@:' + inputFile.replace('@:','')
               if not filteredInputFile in filteredInputFiles:
                  filteredInputFiles.append(filteredInputFile)
         for inputFile in userInputFiles:
            if not '@:' in inputFile:
               filteredInputFile = '@:' + inputFile
               if not filteredInputFile in filteredInputFiles:
                  filteredInputFiles.append(inputFile)
         del userInputFiles
         userInputFiles = filteredInputFiles

      return(userInputFiles)


   def getArguments(self,
                    siteInfo,
                    currentWorkingDirectory,
                    tarFiles,
                    substitutions=None):
      remoteArgs = []
      if siteInfo['stageFiles']:
         rarg = self.enteredCommandArguments[0]
         if substitutions:
            template = ParameterTemplate(rarg)
            try:
               rarg = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError
         remoteArgs.append(rarg)

         if len(self.enteredCommandArguments) > 1:
            for arg in self.enteredCommandArguments[1:]:
               if substitutions:
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError as e:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
               if not arg.startswith('-'):
                  if '=' in arg:
                     arglets = arg.split('=')
                     rarglets = [arglets[0]]
                     for arglet in arglets[1:]:
                        argletIsFile = False
                        for tarFile in tarFiles:
                           if re.match(".*" + tarFile.replace(' ','\ ') + "$",arglet):
                              rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                              rarglets.append(rarglet)
                              argletIsFile = True
                              isClientTransfer = self.isClientTransferFileOrDirectory(arglet)
                              if isClientTransfer and siteInfo['sharedUserSpace']:
                                 clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                                 if not os.path.isfile(rarglet):
                                    if '@:' in clientTransferPath:
                                       clientTransferPath = clientTransferPath.replace('@:','')
                                    shutil.copy2(clientTransferPath,rarglet)
                              break
                        if not argletIsFile:
                           rarglets.append(arglet)
                     rarg = '='.join(rarglets)
                  else:
                     argIsFile = False
                     for tarFile in tarFiles:
                        if re.match(".*" + tarFile.replace(' ','\ ') + "$",arg):
                           rarg = os.path.join('.',tarFile.replace(' ','\ '))
                           argIsFile = True
                           isClientTransfer = self.isClientTransferFileOrDirectory(arg)
                           if isClientTransfer and siteInfo['sharedUserSpace']:
                              clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,arg)
                              if not os.path.isfile(rarg):
                                 if '@:' in clientTransferPath:
                                    clientTransferPath = clientTransferPath.replace('@:','')
                                 shutil.copy2(clientTransferPath,rarg)
                           break
                     if not argIsFile:
                        rarg = arg
               else:
                  lexArgs = shlex.split(arg)
                  rlexArgs = []
                  for lexArg in lexArgs:
                     if '=' in lexArg:
                        arglets = lexArg.split('=')
                        rarglets = [arglets[0]]
                        for arglet in arglets[1:]:
                           argletIsFile = False
                           for tarFile in tarFiles:
                              if re.match(".*" + tarFile.replace(' ','\ ') + "$",arglet):
                                 rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                                 rarglets.append(rarglet)
                                 argletIsFile = True
                                 isClientTransfer = self.isClientTransferFileOrDirectory(arglet)
                                 if isClientTransfer and siteInfo['sharedUserSpace']:
                                    clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,arglet)
                                    if not os.path.isfile(rarglet):
                                       if '@:' in clientTransferPath:
                                          clientTransferPath = clientTransferPath.replace('@:','')
                                       shutil.copy2(clientTransferPath,rarglet)
                                 break
                           if not argletIsFile:
                              rarglets.append(arglet)
                        rlexArgs.append('='.join(rarglets))
                     else:
                        argletIsFile = False
                        for tarFile in tarFiles:
                           if re.match(".*" + tarFile.replace(' ','\ ') + "$",lexArg):
                              rarglet = os.path.join('.',tarFile.replace(' ','\ '))
                              argletIsFile = True
                              isClientTransfer = self.isClientTransferFileOrDirectory(lexArg)
                              if isClientTransfer and siteInfo['sharedUserSpace']:
                                 clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,lexArg)
                                 if not os.path.isfile(rarglet):
                                    if '@:' in clientTransferPath:
                                       clientTransferPath = clientTransferPath.replace('@:','')
                                    shutil.copy2(clientTransferPath,rarglet)
                              break
                        if argletIsFile:
                           rlexArgs.append(rarglet)
                        else:
                           rlexArgs.append(lexArg)
                  rarg = ' '.join(rlexArgs)
               remoteArgs.append(rarg)
      else:
         for arg in self.enteredCommandArguments:
            if substitutions:
               template = ParameterTemplate(arg)
               try:
                  arg = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
            if '@:' in arg:
               arg = arg.replace('@:','')
            remoteArgs.append(arg)

      remoteCommand = " ".join(remoteArgs)
      self.logger.log(logging.INFO,getLogMessage("remoteCommand " + remoteCommand))

      arguments = remoteArgs[1:]

      return(arguments)


   def getEnvironment(self,
                      siteInfo,
                      currentWorkingDirectory,
                      tarFiles,
                      substitutions=None):
      jobEnvironments = []
      environment = copy.copy(self.environment)
      if environment:
         if siteInfo['stageFiles']:
            for environmentVariableValue in environment.split(' '):
               environmentVariable,value = environmentVariableValue.split('=')
               if substitutions:
                  template = ParameterTemplate(value)
                  try:
                     value = template.substitute(substitutions)
                  except KeyError as e:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
               valueIsFile = False
               for tarFile in tarFiles:
                  if re.match(".*" + tarFile.replace(' ','\ ') + "$",value):
                     rValue = os.path.join('.',tarFile.replace(' ','\ '))
                     valueIsFile = True
                     isClientTransfer = self.isClientTransferFileOrDirectory(value)
                     if isClientTransfer and siteInfo['sharedUserSpace']:
                        clientTransferPath = self.findInputFileOrDirectory(currentWorkingDirectory,value)
                        if not os.path.isfile(rValue):
                           if '@:' in clientTransferPath:
                              clientTransferPath = clientTransferPath.replace('@:','')
                           shutil.copy2(clientTransferPath,rValue)
                     break
               if not valueIsFile:
                  rValue = value
               jobEnvironments.append('='.join([environmentVariable,rValue]))
         else:
            for environmentVariableValue in environment.split(' '):
               environmentVariable,value = environmentVariableValue.split('=')
               if substitutions:
                  template = ParameterTemplate(value)
                  try:
                     value = template.substitute(substitutions)
                  except KeyError as e:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                     exitCode = 1
                     raise JobDistributor.InstanceError
               if '@:' in value:
                  value = value.replace('@:','')
               jobEnvironments.append('='.join([environmentVariable,value]))

      jobEnvironment = ' '.join(jobEnvironments)

      return(jobEnvironment)


   def getRedundantInstanceDestination(self,
                                       currentWorkingDirectory,
                                       previouslyUsedDestinations):
      destination              = ""
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = self.enteredCommandArguments[0]
      self.enteredCommandArguments[0] = os.path.expandvars(os.path.expanduser(userExecutable))

      executable = self.enteredCommandArguments[0]
      if "/" in executable:
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
         if candidateFileOrDirectory:
            self.enteredCommandArguments[0] = candidateFileOrDirectory

      executable = self.enteredCommandArguments[0]
      transferExecutable = "/" in executable

      applicationIsStaged = False
      if transferExecutable:
         submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
      else:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               applicationIsStaged      = True
               executableClassification = 'staged'
         if not applicationIsStaged:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)

      if applicationIsStaged:
         event = ':' + userExecutable + ':'
      else:
         event = '/' + os.path.basename(userExecutable)

      enabledSites = self.sitesInfo.getEnabledSites()
      self.toolsInfo.purgeDisabledSites(enabledSites)

      enabledSites = self.sitesInfo.getExpandedSiteNames(enabledSites,
                                                         remoteProbeMonitor=self.remoteMonitors['probe'])
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 executableClassification,enabledSites) + \
                             self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 '*',enabledSites)

      userMappedDestinations = []
      for userDestination in self.userDestinations:
         if self.sitesInfo.siteExists(userDestination):
            userMappedDestinations.append(userDestination)

      userMappedDestinations = self.sitesInfo.getExpandedSiteNames(userMappedDestinations,
                                                                   remoteProbeMonitor=self.remoteMonitors['probe'])
      userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             executableClassification,userMappedDestinations) + \
                         self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             '*',userMappedDestinations)

      for previouslyUsedDestination in previouslyUsedDestinations:
         try:
            eligibleDestinations.remove(previouslyUsedDestination)
         except:
            pass
         try:
            userDestinations.remove(previouslyUsedDestination)
         except:
            pass

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination = ""

      if not transferExecutable:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               if len(userDestinations) > 0:
                  toolInfo = self.toolsInfo.selectTool(executable,userDestinations)
               else:
                  toolInfo = self.toolsInfo.selectTool(executable,eligibleDestinations)

               if len(toolInfo['destinations']) > 0:
                  destination              = random.choice(toolInfo['destinations'])
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'

      if destination == "":
         if len(userDestinations) > 0:
            destination = random.choice(userDestinations)
         else:
            destination = self.sitesInfo.selectUndeclaredSite(eligibleDestinations,
                                                              executableClassification)

      return(destination,executable,transferExecutable,executableClassification,event,toolInfo)


   def buildRedundantJobInstances(self,
                                  currentWorkingDirectory):
      enteredCommand = self.commandParser.getEnteredCommand()

      userInputFiles = self.getInputFiles(currentWorkingDirectory)

      runDirectoryChecked = False
      runDirectoryExisted = False

      managerSpecifiedByUser = self.managerSpecified
      nInstanceIdDigits = max(2,int(math.log10(max(1,self.nRedundant))+1))
      instance = 0
      previouslyUsedDestinations = []
      destination = 'There must be at least one'
      cacheHit = False
      while (instance < self.nRedundant) and destination and not cacheHit:
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)

         self.managerSpecified = managerSpecifiedByUser
         cacheRequestFile = self.commandParser.getOption('requestCache')
         cacheDestination,resultFile,cacheResponseContent = self.requestCacheResult(currentWorkingDirectory,cacheRequestFile)
         if cacheDestination:
            destination        = cacheDestination
            executable         = ""
            transferExecutable = False
            event              = "cacheHit"
            toolInfo           = self.toolsInfo.getDefaultToolInfo()
         else:
            destination,executable,transferExecutable,executableClassification,event,toolInfo = \
                               self.getRedundantInstanceDestination(currentWorkingDirectory,
                                                                    previouslyUsedDestinations)
         if destination:
            previouslyUsedDestinations.append(destination)
            if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
               if self.managersInfo.managerExists(toolInfo['remoteManager']):
                  self.managerSpecified = True
                  self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
               else:
                  message = "Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable)
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self.__writeToStderr("%s\n" % (message))
                  exitCode = 1
                  raise JobDistributor.InstancesError

            toolFilesInfo   = None
            dockerImageInfo = None
            if 'toolFiles' in toolInfo:
               toolFilesName = toolInfo['toolFiles']
               if toolFilesName:
                  toolFilesInfo = self.toolFilesInfo.getToolFilesInfo(toolFilesName)
                  if toolFilesInfo['toolFilesName'] == toolFilesName:
                     if 'dockerImage' in toolFilesInfo:
                        dockerImageInfo = self.dockerImagesInfo.getDockerImageInfo(toolFilesInfo['dockerImage'])
            exitCode = 0

            if cacheDestination:
               inputFiles = []
               scratchDirectory = self.sitesInfo.getSiteKeyValue(destination,'remoteScratchDirectory')
               tempDirectory = os.path.join(os.sep,scratchDirectory,"%s_%s" % (self.localJobId,instanceId))
               os.makedirs(tempDirectory)
               resultPath = os.path.join(tempDirectory,resultFile)
               try:
                  fpCacheResult = open(resultPath,'w')
                  try:
                     fpCacheResult.write(cacheResponseContent)
                     fpCacheResult.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (resultPath)))
                  else:
                     inputFiles.append(resultPath)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (resultPath)))
            else:
               inputFiles    = copy.copy(userInputFiles)

            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            siteInfo      = self.sitesInfo.getDefaultSiteInfo()
            tapisSiteInfo = self.tapisSitesInfo.getDefaultTapisSiteInfo()
            fileMoverInfo = self.fileMoversInfo.getDefaultFileMoverInfo()

            gridsite        = "Unknown"
            pegasusSiteInfo = {}
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if siteInfo['tapisSite']:
                     tapisSiteInfo = self.tapisSitesInfo.getTapisSiteInfo(siteInfo['tapisSite'])
                  if siteInfo['fileMover']:
                     fileMoverInfo = self.fileMoversInfo.getFileMoverInfo(siteInfo['fileMover'])
                  if self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                     gridsite = self.sitesInfo.getSitePegasusSites(destination)
                     pegasusSite = gridsite.split(',')[0]
                     arch      = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'arch')
                     osFlavor  = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osFlavor')
                     osVersion = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osVersion')
                     osRelease = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osRelease')
                     pegasusSiteInfo['name']      = pegasusSite
                     pegasusSiteInfo['arch']      = arch
                     pegasusSiteInfo['osFlavor']  = osFlavor
                     pegasusSiteInfo['osVersion'] = osVersion
                     pegasusSiteInfo['osRelease'] = osRelease


                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for venue %s." % (siteInfo['remoteManager'],destination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("%s\n" % (message))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
                  else:
                     ppn = str(int(math.ceil(float(nCpus) / float(nNodes))))
               else:
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if siteInfo['remoteBatchSystem'] != 'SCRIPT':
                  if not runDirectoryChecked:
                     runDirectory = os.path.join(currentWorkingDirectory,self.runName)
                     runDirectoryExisted = os.path.exists(runDirectory)
                     runDirectoryChecked = True
                     if runDirectoryExisted:
                        self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (runDirectory)))
                        self.__writeToStderr("Run directory %s exists\n" % (runDirectory))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  else:
                     if runDirectoryExisted:
                        exitCode = 1
                        raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected venue does not meet the specified\n" + \
                                    "application access requirements.  Please select another venue."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               touchCommand = "touch " + localJobIdFile
               self.logger.log(logging.INFO,getLogMessage("command = " + touchCommand))
               touchExitStatus,touchStdOutput,touchStdError = self.executeCommand(touchCommand)
               if touchExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(touchStdError))
                  self.logger.log(logging.ERROR,getLogMessage(touchStdOutput))
                  self.__writeToStderr(touchStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               else:
                  inputFiles.append(localJobIdFile)
                  self.filesToRemove.append(localJobIdFile)
               remoteJobIdFile = "%s.%s_%s" % (REMOTEJOBID,self.localJobId,instanceId)
               self.filesToRemove.append(remoteJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               if cacheDestination:
                  stageInTarFile = "%s_%s_output.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  else:
                     tarFiles = tarStdOutput.strip().split('\n')
                  shutil.rmtree(tempDirectory,True)
                  cacheHit = True
               else:
                  if siteInfo['stageFiles']:
                     stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                     tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     else:
                        tarFiles = tarStdOutput.strip().split('\n')

                     self.filesToRemove.append(stageInTarFile)
                  else:
# if not shared work directory?
                     stageInTarFile = ""
                     tarFiles = []

               arguments = self.getArguments(siteInfo,currentWorkingDirectory,tarFiles)

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               environment = self.getEnvironment(siteInfo,currentWorkingDirectory,tarFiles)

               managerInfoEnvironment = managerInfo['environment']
               managerEnvironment = self.__getManagerEnvironment(managerInfoEnvironment)

               toolInfoEnvironment = toolInfo['environment']
               toolEnvironment = self.__getToolEnvironment(toolInfoEnvironment)

               additionalEnvironmentVariableValues = ""
               for environmentVariableValue in managerEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               for environmentVariableValue in toolEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               environment += " " + additionalEnvironmentVariableValues
               environment = environment.strip()

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,instanceId)
               timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,instanceId)
               timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)

               jobScriptFiles        = None
               auxiliaryFiles        = []
               stageAuxiliaryTarFile = False
               auxiliaryTarFile      = "%s_%s_auxiliary.tar" % (self.localJobId,instanceId)
               if   siteInfo['venueMechanism'] == 'cache':
                  self.jobs[instance] = VenueMechanismCache(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            transferExecutable,executable,
                                                            self.appsAccessInfo,event,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            transferExecutable,executable,
                                                            self.appsAccessInfo,event,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,auxiliaryTarFile,
                                                          transferExecutable,executable,
                                                          self.appsAccessInfo,event,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,self.disableJobMonitoring,
                                                          siteInfo,tapisSiteInfo,fileMoverInfo,
                                                          toolFilesInfo,dockerImageInfo,
                                                          self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,
                                                          self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'tapis':
                  venueMechanism = VenueMechanismSsh(self.remoteMonitors,
                                                     self.hubUserName,self.hubUserId,self.submitterClass,
                                                     currentWorkingDirectory,self.session,self.distributorPid,
                                                     self.batchCommands,self.isParametric,
                                                     self.runName,self.localJobId,instanceId,destination,
                                                     self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                     self.tunnelsInfo,
                                                     cloudInstanceId,
                                                     stageInTarFile,auxiliaryTarFile,
                                                     transferExecutable,executable,
                                                     self.appsAccessInfo,event,
                                                     self.stdinput,arguments,useEnvironment,environment,
                                                     self.isMultiCoreRequest,self.disableJobMonitoring,
                                                     siteInfo,tapisSiteInfo,fileMoverInfo,
                                                     toolFilesInfo,dockerImageInfo,
                                                     self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                     self.x509SubmitProxy,self.disableProbeCheck,
                                                     self.quotaLimit,
                                                     timeHistoryLogs,self.useSetup,
                                                     self.pegasusVersion,self.pegasusHome)
                  exitCode,jobScriptFiles = venueMechanism.createScripts()
                  for key in tapisSiteInfo:
                     if key in siteInfo:
                        siteInfo[key] = tapisSiteInfo[key]
                  if   tapisSiteInfo['venueMechanism'] == 'ssh':
                     self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.submitterClass,
                                                             currentWorkingDirectory,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,instanceId,destination,
                                                             self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                             self.tunnelsInfo,
                                                             cloudInstanceId,
                                                             stageInTarFile,auxiliaryTarFile,
                                                             transferExecutable,executable,
                                                             self.appsAccessInfo,event,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,self.disableJobMonitoring,
                                                             siteInfo,tapisSiteInfo,fileMoverInfo,
                                                             toolFilesInfo,dockerImageInfo,
                                                             self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                             self.x509SubmitProxy,self.disableProbeCheck,
                                                             self.quotaLimit,
                                                             timeHistoryLogs,self.useSetup,
                                                             self.pegasusVersion,self.pegasusHome)
                  elif tapisSiteInfo['venueMechanism'] == 'local':
                     self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                               self.hubUserName,self.hubUserId,self.submitterClass,
                                                               currentWorkingDirectory,self.session,self.distributorPid,
                                                               self.batchCommands,self.isParametric,
                                                               self.runName,self.localJobId,instanceId,destination,
                                                               self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                               stageInTarFile,auxiliaryTarFile,
                                                               transferExecutable,executable,
                                                               self.appsAccessInfo,event,
                                                               self.stdinput,arguments,useEnvironment,environment,
                                                               self.isMultiCoreRequest,self.disableJobMonitoring,
                                                               siteInfo,tapisSiteInfo,fileMoverInfo,
                                                               toolFilesInfo,dockerImageInfo,
                                                               self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                               self.quotaLimit,
                                                               self.x509SubmitProxy,self.disableProbeCheck,
                                                               timeHistoryLogs,self.useSetup,
                                                               self.pegasusVersion,self.pegasusHome)

               if not exitCode:
                  exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if jobScriptFiles:
                  auxiliaryFiles += scriptFiles
                  del scriptFiles
                  scriptFiles = jobScriptFiles

               if siteInfo['stageFiles']:
                  inputBuiltPath = os.path.join(currentWorkingDirectory,timeHistoryLogs['timestampInputBuilt'])
                  if self.writeInputBuiltFile(inputBuiltPath):
                     scriptFiles += [inputBuiltPath]
                     self.filesToRemove.append(inputBuiltPath)
                  else:
                     self.__writeToStderr("Creation of %s failed\n" % (inputBuiltPath))
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  self.filesToRemove.append(stageInTarFile)

               if siteInfo['executionMode'] == 'split':
                  if len(auxiliaryFiles) > 0:
                     tarCommand = buildCreateTarCommand(auxiliaryTarFile,auxiliaryFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                     niceCommand = "nice -n 19 gzip " + auxiliaryTarFile
                     self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                     gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                     if gzipExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                        self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                        self.__writeToStderr(gzipStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     auxiliaryTarFile += '.gz'
                     self.filesToRemove.append(auxiliaryTarFile)
                     stageAuxiliaryTarFile = True

               self.jobs[instance].stageFilesToInstanceDirectory(stageAuxiliaryTarFile=stageAuxiliaryTarFile)

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['executionMode']         = waitForJobInfo['executionMode']
               self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
               self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
               self.waitForJobsInfo[instance]['instanceToken']         = waitForJobInfo['instanceToken']
               self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
               self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
               self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory
               self.waitForJobsInfo[instance]['recentJobStatus']       = '?'

            except JobDistributor.InstanceError:
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s.stderr" % (self.runName)
            self.emptyFilesToRemove.append(stdFile)
            for batchQueueType in BATCHQUEUETYPES:
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueueType,self.runName,instanceId,outFile)
                  self.emptyFilesToRemove.append(batchQueueOutputFile)

      if len(self.waitForJobsInfo) > 0:
         exitCode = 0
         self.nInstances = len(self.waitForJobsInfo)
      else:
         exitCode = 1
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def registerRedundantJobInstances(self):
      self.nonBatchJobs       = []
      self.splitExecutionJobs = []
      for instance in self.waitForJobsInfo:
         if not self.waitForJobsInfo[instance]['isBatchJob']:
            self.nonBatchJobs.append(instance)
         if self.waitForJobsInfo[instance]['executionMode'] == 'split':
            self.splitExecutionJobs.append(instance)
         self.jobs[instance].registerJob()
         self.waitForJobsInfo[instance]['state'] = 'held'

      if self.progressReport == 'text':
         if   self.nInstances == 1:
            self.__writeToStdout("Run %d registered %d job instance. %s\n" % (self.jobId,self.nInstances,time.ctime()))
         elif self.nInstances > 1:
            self.__writeToStdout("Run %d registered %d job instances. %s\n" % (self.jobId,self.nInstances,time.ctime()))


   def startRedundantSplitJobInstances(self):
      while self.splitExecutionJobs and (self.successfulInstance == None):
         instance = random.choice(self.splitExecutionJobs)
         if self.waitForJobsInfo[instance]['isBatchJob']:
            if self.waitForJobsInfo[instance]['state'] == 'held':
               if self.jobs[instance].isJobReleased():
                  self.waitForJobsInfo[instance]['state'] = 'released'
                  if self.progressReport == 'text':
                     self.__writeToStdout("Run %d instance %d (%s) released for submission. %s\n" % \
                                    (self.jobId,instance, \
                                     self.waitForJobsInfo[instance]['instanceToken'],time.ctime()))
                  if self.jobs[instance].stashJobInputs():
                     if self.jobs[instance].stageJob():
                        self.successfulInstance = instance
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                  self.splitExecutionJobs.remove(instance)
               else:
                  time.sleep(1)


   def startRedundantBatchJobInstances(self):
      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['isBatchJob']:
            if self.waitForJobsInfo[instance]['state'] == 'held':
               if self.jobs[instance].isJobReleased():
                  self.waitForJobsInfo[instance]['state'] = 'released'
                  if self.progressReport == 'text':
                     self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                                             (self.jobId,instance,time.ctime()))
                  if self.jobs[instance].sendFiles():
                     if self.jobs[instance].executeJob():
                        waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                        self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                        self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                        self.waitForJobsInfo[instance]['instanceToken']   = waitForJobInfo['instanceToken']
                        self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                     else:
                        self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                  else:
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'


   def runRedundantJobInstances(self):
      nBatchJobsHeld = self.nInstances-len(self.nonBatchJobs)
      while self.nonBatchJobs and (self.successfulInstance == None):
         instance = random.choice(self.nonBatchJobs)
         if self.waitForJobsInfo[instance]['state'] == 'held':
            if self.jobs[instance].isJobReleased():
               self.waitForJobsInfo[instance]['state'] = 'released'
               if self.progressReport == 'text':
                  self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                                          (self.jobId,instance,time.ctime()))
               if self.jobs[instance].sendFiles():
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'R'
                  if self.jobs[instance].executeJob():
                     self.jobs[instance].postProcess()
                     self.jobs[instance].retrieveFiles()
                     self.jobs[instance].cleanupFiles()
                     self.successfulInstance = instance
                  else:
                     self.jobs[instance].cleanupFiles()
               else:
                  self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               self.nonBatchJobs.remove(instance)

      if len(self.splitExecutionJobs) > 0:
         if (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
            self.startRedundantSplitJobInstances()
      else:
         jobsRunning = False
         if self.successfulInstance == None:
            while (jobsRunning or nBatchJobsHeld > 0) and \
                  (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
               self.startRedundantBatchJobInstances()
               completeRemoteJobIndexes = self.remoteMonitors['job'].waitForRedundantJobs(self.waitForJobsInfo,
                                                                                          self.progressReport,
                                                                                          self.abortGlobal)

               nBatchJobsHeld = 0
               jobsRunning = False
               for instance in self.waitForJobsInfo:
                  if self.waitForJobsInfo[instance]['state'] == 'released':
                     if 'recentJobSite' in self.waitForJobsInfo[instance]:
                        executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                        self.jobs[instance].updateVenue(executionVenue)
                        if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                           jobsRunning = True
                  else:
                     nBatchJobsHeld += 1

               for instance in completeRemoteJobIndexes:
                  self.jobs[instance].postProcess()
                  self.jobs[instance].retrieveFiles()
                  self.jobs[instance].cleanupFiles()
                  self.jobs[instance].recordJobStatistics()
                  if self.jobs[instance].wasJobSuccessful():
                     self.successfulInstance = instance

         if self.abortGlobal['abortAttempted']:
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs('redundant',
                                                              self.waitForJobsInfo,
                                                              self.progressReport)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                     self.jobs[instance].cleanupFiles()
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

         if self.successfulInstance != None:
            nJobsToKill = 0
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     self.jobs[instance].killScripts(signal.SIGUSR1)
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
                     nJobsToKill += 1
            if nJobsToKill > 0:
               self.remoteMonitors['job'].waitForKilledBatchJobs('redundant',
                                                                 self.waitForJobsInfo,
                                                                 self.progressReport)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if   self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                     self.jobs[instance].cleanupFiles()
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                  elif instance != self.successfulInstance:
                     self.jobs[instance].cleanupFiles()

         if self.successfulInstance != None:
            self.jobs[self.successfulInstance].recoverFiles(False)
            fileToMove = "%s_%02d.stdout" % (self.runName,self.successfulInstance)
            if os.path.exists(fileToMove):
               fileNewName = "%s.stdout" % (self.runName)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.stderr" % (self.runName,self.successfulInstance)
            if os.path.exists(fileToMove):
               fileNewName = "%s.stderr" % (self.runName)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.SUCCESS" % (self.runName,self.successfulInstance)
            if os.path.exists(fileToMove):
               fileNewName = "%s.SUCCESS" % (self.runName)
               os.rename(fileToMove,fileNewName)
         else:
            for instance in self.waitForJobsInfo:
               self.jobs[instance].recoverStdFiles()
            if len(self.jobs) > 0:
               recoverInstance = random.choice(list(self.jobs.keys()))
               self.jobs[recoverInstance].recoverFiles(True)

      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['state'] == 'held':
            self.jobs[instance].removeJobRegistration()

      for instance in self.waitForJobsInfo:
         self.jobs[instance].removeInstanceDirectory()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)


   def getPegasusInstanceDestination(self,
                                     currentWorkingDirectory):
      destination              = ""
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = self.enteredCommandArguments[0]
      self.enteredCommandArguments[0] = os.path.expandvars(os.path.expanduser(userExecutable))

      executable = self.enteredCommandArguments[0]
      if "/" in executable:
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
         if candidateFileOrDirectory:
            self.enteredCommandArguments[0] = candidateFileOrDirectory

      executable = self.enteredCommandArguments[0]
      transferExecutable = "/" in executable

      applicationIsStaged = False
      if transferExecutable:
         submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
      else:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               applicationIsStaged      = True
               executableClassification = 'staged'
         if not applicationIsStaged:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)

      if applicationIsStaged:
         event = ':' + userExecutable + ':'
      else:
         event = '/' + os.path.basename(userExecutable)

      enabledSites = self.sitesInfo.getEnabledSites()
      self.toolsInfo.purgeDisabledSites(enabledSites)

      enabledSites = self.sitesInfo.getExpandedSiteNames(enabledSites,
                                                         remoteProbeMonitor=self.remoteMonitors['probe'])
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 executableClassification,enabledSites) + \
                             self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 '*',enabledSites)
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                                 eligibleDestinations)

      userMappedDestinations = []
      for userDestination in self.userDestinations:
         if self.sitesInfo.siteExists(userDestination):
            userMappedDestinations.append(userDestination)

      userMappedDestinations = self.sitesInfo.getExpandedSiteNames(userMappedDestinations,
                                                                   remoteProbeMonitor=self.remoteMonitors['probe'])
      userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             executableClassification,userMappedDestinations) + \
                         self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             '*',userMappedDestinations)
      userDestinations = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',
                                                             userDestinations)

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination = ""

      if not transferExecutable:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               if len(userDestinations) > 0:
                  toolInfo = self.toolsInfo.selectTool(executable,userDestinations)
               else:
                  toolInfo = self.toolsInfo.selectTool(executable,eligibleDestinations)

               if len(toolInfo['destinations']) > 0:
                  destination              = random.choice(toolInfo['destinations'])
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'

      if destination == "":
         if len(userDestinations) > 0:
            destination = random.choice(userDestinations)
         else:
            destination = self.sitesInfo.selectUndeclaredSite(eligibleDestinations,
                                                              executableClassification)

      return(destination,executable,transferExecutable,executableClassification,event,toolInfo)


   def buildPegasusJobInstances(self,
                                currentWorkingDirectory):
      try:
         destination,executable,transferExecutable,executableClassification,event,toolInfo = \
                                 self.getPegasusInstanceDestination(currentWorkingDirectory)
         if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
            if self.managersInfo.managerExists(toolInfo['remoteManager']):
               self.managerSpecified = True
               self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
            else:
               message = "Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable)
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("%s\n" % (message))
               exitCode = 1
               raise JobDistributor.InstancesError

         toolFilesInfo   = None
         dockerImageInfo = None
         if 'toolFiles' in toolInfo:
            toolFilesName = toolInfo['toolFiles']
            if toolFilesName:
               toolFilesInfo = self.toolFilesInfo.getToolFilesInfo(toolFilesName)
               if toolFilesInfo['toolFilesName'] == toolFilesName:
                  if 'dockerImage' in toolFilesInfo:
                     dockerImageInfo = self.dockerImagesInfo.getDockerImageInfo(toolFilesInfo['dockerImage'])

         self.jobPath = os.path.join(currentWorkingDirectory,self.runName)
         if os.path.exists(self.jobPath):
            self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (self.jobPath)))
            self.__writeToStderr("Run directory %s exists\n" % (self.jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(self.jobPath):
            os.mkdir(self.jobPath)
         if self.progressReport == 'curses' or self.reportMetrics:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'remoteParameterCombinations.csv')
         else:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(self.parameterCombinationsPath)

         auxiliaryTarFile = ""

         instanceError = False
         self.inputsPath   = os.path.join(self.jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = self.getInputFiles(currentWorkingDirectory,substitutions=substitutions)

            siteInfo      = self.sitesInfo.getDefaultSiteInfo()
            tapisSiteInfo = self.tapisSitesInfo.getDefaultTapisSiteInfo()
            fileMoverInfo = self.fileMoversInfo.getDefaultFileMoverInfo()

            gridsite        = "Unknown"
            pegasusSiteInfo = {}
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if siteInfo['tapisSite']:
                     tapisSiteInfo = self.tapisSitesInfo.getTapisSiteInfo(siteInfo['tapisSite'])
                  if siteInfo['fileMover']:
                     fileMoverInfo = self.fileMoversInfo.getFileMoverInfo(siteInfo['fileMover'])
                  gridsite = self.sitesInfo.getSitePegasusSites(destination)
                  pegasusSite = gridsite.split(',')[0]
                  arch      = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'arch')
                  osFlavor  = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osFlavor')
                  osVersion = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osVersion')
                  osRelease = self.sitesInfo.getSitePegasusSiteAttributeValue(destination,pegasusSite,'osRelease')
                  pegasusSiteInfo['name']      = pegasusSite
                  pegasusSiteInfo['arch']      = arch
                  pegasusSiteInfo['osFlavor']  = osFlavor
                  pegasusSiteInfo['osVersion'] = osVersion
                  pegasusSiteInfo['osRelease'] = osRelease

                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for venue %s." % (siteInfo['remoteManager'],destination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("%s\n" % (message))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
                  else:
                     ppn = str(int(math.ceil(float(nCpus) / float(nNodes))))
               else:
                  errorMessage = "Invalid destination selection: %s" % (destination)
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected venue does not meet the specified\n" + \
                                    "application access requirements.  Please select another venue."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               touchCommand = "touch " + localJobIdFile
               self.logger.log(logging.INFO,getLogMessage("command = " + touchCommand))
               touchExitStatus,touchStdOutput,touchStdError = self.executeCommand(touchCommand)
               if touchExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(touchStdError))
                  self.logger.log(logging.ERROR,getLogMessage(touchStdOutput))
                  self.__writeToStderr(touchStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               else:
                  inputFiles.append(localJobIdFile)
                  self.filesToRemove.append(localJobIdFile)
               remoteJobIdFile = "%s.%s_%s" % (REMOTEJOBID,self.localJobId,instanceId)
               self.filesToRemove.append(remoteJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     try:
                        fpInputFile = open(inputFile,'r')
                        try:
                           inputText = fpInputFile.readlines()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                        else:
                           template = ParameterTemplate(''.join(inputText))
                           try:
                              inputText = template.substitute(substitutions)
                           except KeyError as e:
                              self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                              exitCode = 1
                              raise JobDistributor.InstanceError
                           instanceInputsPath = os.path.join(self.inputsPath,instanceId)
                           if not os.path.isdir(instanceInputsPath):
                              os.makedirs(instanceInputsPath)
                           inputName = os.path.basename(inputFile)
                           inputPath = os.path.join(instanceInputsPath,inputName)
                           try:
                              fpInputPath = open(inputPath,'w')
                              try:
                                 fpInputPath.writelines(inputText)
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                              else:
                                 substitutedInputFiles.append(inputPath)
                              finally:
                                 fpInputPath.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                        finally:
                           fpInputFile.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  else:
                     tarFiles = tarStdOutput.strip().split('\n')

                  self.filesToRemove.append(stageInTarFile)
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  tarFiles = []

               arguments = self.getArguments(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               environment = self.getEnvironment(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               managerInfoEnvironment = managerInfo['environment']
               managerEnvironment = self.__getManagerEnvironment(managerInfoEnvironment)

               toolInfoEnvironment = toolInfo['environment']
               toolEnvironment = self.__getToolEnvironment(toolInfoEnvironment)

               additionalEnvironmentVariableValues = ""
               for environmentVariableValue in managerEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               for environmentVariableValue in toolEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               environment += " " + additionalEnvironmentVariableValues
               environment = environment.strip()

               template = ParameterTemplate(environment)
               try:
                  environment = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,instanceId)
               timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,instanceId)
               timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            transferExecutable,executable,
                                                            self.appsAccessInfo,event,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,auxiliaryTarFile,
                                                          transferExecutable,executable,
                                                          self.appsAccessInfo,event,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,self.disableJobMonitoring,
                                                          siteInfo,tapisSiteInfo,fileMoverInfo,
                                                          toolFilesInfo,dockerImageInfo,
                                                          self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,
                                                          self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  inputBuiltPath = os.path.join(currentWorkingDirectory,timeHistoryLogs['timestampInputBuilt'])
                  if self.writeInputBuiltFile(inputBuiltPath):
                     scriptFiles += [inputBuiltPath]
                     self.filesToRemove.append(inputBuiltPath)
                  else:
                     self.__writeToStderr("Creation of %s failed\n" % (inputBuiltPath))
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  self.filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['executionMode']   = waitForJobInfo['executionMode']
               self.waitForJobsInfo[instance]['instanceToken']   = waitForJobInfo['instanceToken']
               self.waitForJobsInfo[instance]['recentJobStatus'] = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         self.nInstances      = instance
         self.executeInstance = 0

         timeHistoryLogs = {}
         timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,str(self.executeInstance))

         enteredCommand = self.commandParser.getEnteredCommand()

         if   siteInfo['venueMechanism'] == 'local':
            wfInstanceId = "WF;%d" % (self.nInstances)
            self.jobs[self.executeInstance] = VenueMechanismLocal(self.remoteMonitors,
                                                                  self.hubUserName,self.hubUserId,self.submitterClass,
                                                                  currentWorkingDirectory,self.session,self.distributorPid,
                                                                  self.batchCommands,self.isParametric,
                                                                  self.runName,self.localJobId,wfInstanceId,destination,
                                                                  self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                                  stageInTarFile,auxiliaryTarFile,
                                                                  transferExecutable,executable,
                                                                  self.appsAccessInfo,"[sweep]",
                                                                  self.stdinput,arguments,useEnvironment,environment,
                                                                  self.isMultiCoreRequest,self.disableJobMonitoring,
                                                                  siteInfo,tapisSiteInfo,fileMoverInfo,
                                                                  toolFilesInfo,dockerImageInfo,
                                                                  self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                                  self.quotaLimit,
                                                                  self.x509SubmitProxy,self.disableProbeCheck,
                                                                  timeHistoryLogs,self.useSetup,
                                                                  self.pegasusVersion,self.pegasusHome)
         elif siteInfo['venueMechanism'] == 'ssh':
            wfInstanceId = "WF;%d" % (self.nInstances)
            self.jobs[self.executeInstance] = VenueMechanismSsh(self.remoteMonitors,
                                                                self.hubUserName,self.hubUserId,self.submitterClass,
                                                                currentWorkingDirectory,self.session,self.distributorPid,
                                                                self.batchCommands,self.isParametric,
                                                                self.runName,self.localJobId,wfInstanceId,destination,
                                                                self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                                self.tunnelsInfo,
                                                                cloudInstanceId,
                                                                stageInTarFile,auxiliaryTarFile,
                                                                transferExecutable,executable,
                                                                self.appsAccessInfo,"[sweep]",
                                                                self.stdinput,arguments,useEnvironment,environment,
                                                                self.isMultiCoreRequest,self.disableJobMonitoring,
                                                                siteInfo,tapisSiteInfo,fileMoverInfo,
                                                                toolFilesInfo,dockerImageInfo,
                                                                self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                                self.x509SubmitProxy,self.disableProbeCheck,
                                                                self.quotaLimit,
                                                                timeHistoryLogs,self.useSetup,
                                                                self.pegasusVersion,self.pegasusHome)

         exitCode,scriptFiles = self.jobs[self.executeInstance].createScripts()
         if exitCode:
            raise JobDistributor.InstancesError

         self.waitForJobsInfo[self.executeInstance] = {}
         waitForJobInfo = self.jobs[self.executeInstance].getWaitForJobInfo()
         self.waitForJobsInfo[self.executeInstance]['isBatchJob']            = waitForJobInfo['isBatchJob']
         self.waitForJobsInfo[self.executeInstance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
         self.waitForJobsInfo[self.executeInstance]['executionMode']         = waitForJobInfo['executionMode']
         self.waitForJobsInfo[self.executeInstance]['instanceToken']         = waitForJobInfo['instanceToken']
         self.waitForJobsInfo[self.executeInstance]['remoteBatchSystem']     = self.jobs[self.executeInstance].remoteBatchSystem
         self.waitForJobsInfo[self.executeInstance]['instanceDirectory']     = self.jobs[self.executeInstance].instanceDirectory
         self.waitForJobsInfo[self.executeInstance]['scratchDirectory']      = self.jobs[self.executeInstance].scratchDirectory
         self.waitForJobsInfo[self.executeInstance]['recentJobStatus']       = '?'

         stdFile = "%s.stderr" % (self.runName)
         self.emptyFilesToRemove.append(stdFile)
         for batchQueueType in BATCHQUEUETYPES:
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s.%s" % (batchQueueType,self.runName,outFile)
               self.emptyFilesToRemove.append(batchQueueOutputFile)
      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def registerPegasusJobInstances(self):
      self.nonBatchJobs       = []
      self.splitExecutionJobs = []
      if not self.waitForJobsInfo[self.executeInstance]['isBatchJob']:
         self.nonBatchJobs.append(self.executeInstance)
      if self.waitForJobsInfo[self.executeInstance]['executionMode'] == 'split':
         self.splitExecutionJobs.append(self.executeInstance)
      self.jobs[self.executeInstance].registerJob()
      self.waitForJobsInfo[self.executeInstance]['state'] = 'held'
      if self.progressReport == 'text':
         self.__writeToStdout("Run %d registered 1 job instance. %s\n" % (self.jobId,time.ctime()))


   def startPegasusBatchJobInstances(self):
      if self.waitForJobsInfo[self.executeInstance]['isBatchJob']:
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'held':
            if self.jobs[self.executeInstance].isJobReleased():
               self.waitForJobsInfo[self.executeInstance]['state'] = 'released'
               if self.progressReport == 'text':
                  self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                              (self.jobId,self.executeInstance,time.ctime()))
               if self.jobs[self.executeInstance].sendFiles():
                  if self.jobs[self.executeInstance].executeJob():
                     waitForJobInfo = self.jobs[self.executeInstance].getWaitForJobInfo()
                     self.waitForJobsInfo[self.executeInstance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                     self.waitForJobsInfo[self.executeInstance]['knownSite']       = waitForJobInfo['knownSite']
                     self.waitForJobsInfo[self.executeInstance]['instanceToken']   = waitForJobInfo['instanceToken']
                     self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = '?'
                  else:
                     self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'


   def runPegasusJobInstances(self):
      nBatchJobsHeld = 1-len(self.nonBatchJobs)
      jobsRunning = False
      while (jobsRunning or nBatchJobsHeld > 0) and \
            (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
         self.startPegasusBatchJobInstances()
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForPegasusWorkflowJobs(self.waitForJobsInfo,
                                                                                          self.nInstances,
                                                                                          self.parameterCombinationsPath,
                                                                                          self.progressReport,self.abortGlobal)
         nBatchJobsHeld = 0
         jobsRunning = False
         for instance in [self.executeInstance]:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()
            if self.jobs[instance].wasJobSuccessful():
               self.successfulInstance = instance

      if self.abortGlobal['abortAttempted']:
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] != 'D':
               self.jobs[self.executeInstance].killScripts(self.abortGlobal['abortSignal'])
               self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs('pegasus',
                                                           self.waitForJobsInfo,
                                                           self.progressReport,
                                                           self.parameterCombinationsPath)
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if 'recentJobSite' in self.waitForJobsInfo[self.executeInstance]:
               executionVenue = self.waitForJobsInfo[self.executeInstance]['recentJobSite']
               self.jobs[self.executeInstance].updateVenue(executionVenue)
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] == 'KD':
               self.jobs[self.executeInstance].cleanupFiles()
               self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'

      if self.successfulInstance != None:
         for instance in iterRange(1,self.nInstances+1):
            self.jobs[instance].jobSubmitted = True

      for instance in iterRange(1,self.nInstances+1):
         if 'recentJobSite' in self.waitForJobsInfo[instance]:
            executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
            self.jobs[instance].updateVenue(executionVenue)
         self.jobs[instance].postProcess()
         self.jobs[instance].retrieveFiles()
         self.jobs[instance].cleanupFiles()
         self.jobs[instance].recordJobStatistics()

      if self.waitForJobsInfo[self.executeInstance]['state'] == 'held':
         self.jobs[self.executeInstance].removeJobRegistration()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isdir(self.inputsPath):
         shutil.rmtree(self.inputsPath,True)

      if self.progressReport == 'text' or self.progressReport == 'submit' or self.progressReport == 'pegasus':
         self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (self.jobPath))

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)


   def getBoincInstanceDestination(self,
                                   currentWorkingDirectory):
      destination              = ""
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = self.enteredCommandArguments[0]
      self.enteredCommandArguments[0] = os.path.expandvars(os.path.expanduser(userExecutable))

      executable = self.enteredCommandArguments[0]
      if "/" in executable:
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
         if candidateFileOrDirectory:
            self.enteredCommandArguments[0] = candidateFileOrDirectory

      executable = self.enteredCommandArguments[0]
      transferExecutable = "/" in executable

      applicationIsStaged = False
      if transferExecutable:
         submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
      else:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               applicationIsStaged      = True
               executableClassification = 'staged'
         if not applicationIsStaged:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)

      if applicationIsStaged:
         event = ':' + userExecutable + ':'
      else:
         event = '/' + os.path.basename(userExecutable)

      enabledSites = self.sitesInfo.getEnabledSites()
      self.toolsInfo.purgeDisabledSites(enabledSites)

      enabledSites = self.sitesInfo.getExpandedSiteNames(enabledSites,
                                                         remoteProbeMonitor=self.remoteMonitors['probe'])
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 executableClassification,enabledSites) + \
                             self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 '*',enabledSites)
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','BOINC',
                                                                 eligibleDestinations)

      userMappedDestinations = []
      for userDestination in self.userDestinations:
         if self.sitesInfo.siteExists(userDestination):
            userMappedDestinations.append(userDestination)

      userMappedDestinations = self.sitesInfo.getExpandedSiteNames(userMappedDestinations,
                                                                   remoteProbeMonitor=self.remoteMonitors['probe'])
      userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             executableClassification,userMappedDestinations) + \
                         self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             '*',userMappedDestinations)
      userDestinations = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','BOINC',
                                                             userDestinations)

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination = ""

      if not transferExecutable:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               if len(userDestinations) > 0:
                  toolInfo = self.toolsInfo.selectTool(executable,userDestinations)
               else:
                  toolInfo = self.toolsInfo.selectTool(executable,eligibleDestinations)

               if len(toolInfo['destinations']) > 0:
                  destination              = random.choice(toolInfo['destinations'])
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'

      if destination == "":
         if len(userDestinations) > 0:
            destination = random.choice(userDestinations)
         else:
            destination = self.sitesInfo.selectUndeclaredSite(eligibleDestinations,
                                                              executableClassification)

      return(destination,executable,transferExecutable,executableClassification,event,toolInfo)


   def buildBoincJobInstances(self,
                              currentWorkingDirectory):
      try:
         destination,executable,transferExecutable,executableClassification,event,toolInfo = \
                                   self.getBoincInstanceDestination(currentWorkingDirectory)
         if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
            if self.managersInfo.managerExists(toolInfo['remoteManager']):
               self.managerSpecified = True
               self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
            else:
               message = "Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable)
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.__writeToStderr("%s\n" % (message))
               exitCode = 1
               raise JobDistributor.InstancesError

         toolFilesInfo   = None
         dockerImageInfo = None
         if 'toolFiles' in toolInfo:
            toolFilesName = toolInfo['toolFiles']
            if toolFilesName:
               toolFilesInfo = self.toolFilesInfo.getToolFilesInfo(toolFilesName)
               if toolFilesInfo['toolFilesName'] == toolFilesName:
                  if 'dockerImage' in toolFilesInfo:
                     dockerImageInfo = self.dockerImagesInfo.getDockerImageInfo(toolFilesInfo['dockerImage'])

         self.jobPath = os.path.join(currentWorkingDirectory,self.runName)
         if os.path.exists(self.jobPath):
            self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (self.jobPath)))
            self.__writeToStderr("Run directory %s exists\n" % (self.jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(self.jobPath):
            os.mkdir(self.jobPath)
         if self.progressReport == 'curses' or self.reportMetrics:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'remoteParameterCombinations.csv')
         else:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(self.parameterCombinationsPath)

         auxiliaryTarFile = ""

         nCacheHits = 0
         instanceError = False
         self.inputsPath   = os.path.join(self.jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = self.getInputFiles(currentWorkingDirectory,substitutions=substitutions)

            checkCachePath = ""
            cacheRequestFile = self.commandParser.getOption('requestCache')
            if cacheRequestFile:
               if '@:' in cacheRequestFile:
                  cacheRequestFile = cacheRequestFile.replace('@:','')
               template = ParameterTemplate(cacheRequestFile)
               try:
                  cacheRequestFile = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError
               else:
                  cacheRequestPath = self.findInputFileOrDirectory(currentWorkingDirectory,cacheRequestFile)
            else:
               cacheRequestPath = ""

            substitutedInputFiles = []
            for inputFile in inputFiles:
               if '@:' in inputFile:
                  inputFile = inputFile.replace('@:','')
                  try:
                     fpInputFile = open(inputFile,'r')
                     try:
                        inputText = fpInputFile.readlines()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                     else:
                        template = ParameterTemplate(''.join(inputText))
                        try:
                           inputText = template.substitute(substitutions)
                        except KeyError as e:
                           self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                           exitCode = 1
                           raise JobDistributor.InstanceError
                        instanceInputsPath = os.path.join(self.inputsPath,instanceId)
                        if not os.path.isdir(instanceInputsPath):
                           os.makedirs(instanceInputsPath)
                        inputName = os.path.basename(inputFile)
                        inputPath = os.path.join(instanceInputsPath,inputName)
                        try:
                           fpInputPath = open(inputPath,'w')
                           try:
                              fpInputPath.writelines(inputText)
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                           else:
                              substitutedInputFiles.append(inputPath)
                              if inputFile == cacheRequestPath:
                                 checkCachePath = inputPath
                           finally:
                              fpInputPath.close()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                     finally:
                        fpInputFile.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
               else:
                  substitutedInputFiles.append(inputFile)
                  if inputFile == cacheRequestPath:
                     checkCachePath = cacheRequestPath

            del inputFiles
            inputFiles = substitutedInputFiles

            instanceDestination        = destination
            instanceExecutable         = executable
            instanceTransferExecutable = transferExecutable
            instanceEvent              = event
            instanceToolInfo           = toolInfo
            cacheDestination           = ""
            if checkCachePath:
               cacheDestination,resultFile,cacheResponseContent = self.requestCacheResult(currentWorkingDirectory,checkCachePath)
               if cacheDestination:
                  instanceDestination        = cacheDestination
                  instanceExecutable         = ""
                  instanceTransferExecutable = False
                  instanceEvent              = "cacheHit"
                  instanceToolInfo           = self.toolsInfo.getDefaultToolInfo()

                  del inputFiles
                  inputFiles = []
                  scratchDirectory = self.sitesInfo.getSiteKeyValue(instanceDestination,'remoteScratchDirectory')
                  tempDirectory = os.path.join(os.sep,scratchDirectory,"%s_%s" % (self.localJobId,instanceId))
                  os.makedirs(tempDirectory)
                  resultPath = os.path.join(tempDirectory,resultFile)
                  try:
                     fpCacheResult = open(resultPath,'w')
                     try:
                        fpCacheResult.write(cacheResponseContent)
                        fpCacheResult.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (resultPath)))
                     else:
                        inputFiles.append(resultPath)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (resultPath)))
                  nCacheHits += 1

            siteInfo      = self.sitesInfo.getDefaultSiteInfo()
            tapisSiteInfo = self.tapisSitesInfo.getDefaultTapisSiteInfo()
            fileMoverInfo = self.fileMoversInfo.getDefaultFileMoverInfo()

            gridsite        = "Unknown"
            pegasusSiteInfo = {}
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(instanceDestination):
                  siteInfo = self.sitesInfo.getSiteInfo(instanceDestination)
                  if siteInfo['tapisSite']:
                     tapisSiteInfo = self.tapisSitesInfo.getTapisSiteInfo(siteInfo['tapisSite'])
                  if siteInfo['fileMover']:
                     fileMoverInfo = self.fileMoversInfo.getFileMoverInfo(siteInfo['fileMover'])
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for venue %s." % (siteInfo['remoteManager'],instanceDestination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("%s\n" % (message))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
                  else:
                     ppn = str(int(math.ceil(float(nCpus) / float(nNodes))))
               else:
                  errorMessage = "Invalid destination selection: %s" % (instanceDestination)
                  self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                  self.__writeToStderr("%s\n" % (errorMessage))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if instanceTransferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected venue does not meet the specified\n" + \
                                    "application access requirements.  Please select another venue."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               touchCommand = "touch " + localJobIdFile
               self.logger.log(logging.INFO,getLogMessage("command = " + touchCommand))
               touchExitStatus,touchStdOutput,touchStdError = self.executeCommand(touchCommand)
               if touchExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(touchStdError))
                  self.logger.log(logging.ERROR,getLogMessage(touchStdOutput))
                  self.__writeToStderr(touchStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               else:
                  inputFiles.append(localJobIdFile)
                  self.filesToRemove.append(localJobIdFile)
               remoteJobIdFile = "%s.%s_%s" % (REMOTEJOBID,self.localJobId,instanceId)
               self.filesToRemove.append(remoteJobIdFile)

               if instanceTransferExecutable:
                  inputFiles.append(instanceExecutable)

               if cacheDestination:
                  stageInTarFile = "%s_%s_output.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  else:
                     tarFiles = tarStdOutput.strip().split('\n')
                  shutil.rmtree(tempDirectory,True)
               else:
                  if siteInfo['stageFiles']:
                     stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                     tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     else:
                        tarFiles = tarStdOutput.strip().split('\n')

                     self.filesToRemove.append(stageInTarFile)
                  else:
# if not shared work directory?
                     stageInTarFile = ""
                     tarFiles = []

               arguments = self.getArguments(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               environment = self.getEnvironment(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               managerInfoEnvironment = managerInfo['environment']
               managerEnvironment = self.__getManagerEnvironment(managerInfoEnvironment)

               toolInfoEnvironment = instanceToolInfo['environment']
               toolEnvironment = self.__getToolEnvironment(toolInfoEnvironment)

               additionalEnvironmentVariableValues = ""
               for environmentVariableValue in managerEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               for environmentVariableValue in toolEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               environment += " " + additionalEnvironmentVariableValues
               environment = environment.strip()

               template = ParameterTemplate(environment)
               try:
                  environment = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,instanceId)
               timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,instanceId)
               timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'cache':
                  self.jobs[instance] = VenueMechanismCache(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,instanceDestination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            instanceTransferExecutable,instanceExecutable,
                                                            self.appsAccessInfo,instanceEvent,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,instanceDestination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            instanceTransferExecutable,instanceExecutable,
                                                            self.appsAccessInfo,instanceEvent,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,instanceDestination,
                                                          self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,auxiliaryTarFile,
                                                          instanceTransferExecutable,instanceExecutable,
                                                          self.appsAccessInfo,instanceEvent,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,self.disableJobMonitoring,
                                                          siteInfo,tapisSiteInfo,fileMoverInfo,
                                                          toolFilesInfo,dockerImageInfo,
                                                          self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,
                                                          self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if siteInfo['stageFiles']:
                  inputBuiltPath = os.path.join(currentWorkingDirectory,timeHistoryLogs['timestampInputBuilt'])
                  if self.writeInputBuiltFile(inputBuiltPath):
                     scriptFiles += [inputBuiltPath]
                     self.filesToRemove.append(inputBuiltPath)
                  else:
                     self.__writeToStderr("Creation of %s failed\n" % (inputBuiltPath))
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  self.filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               if self.disableJobMonitoring:
                  self.jobs[instance].writeHarvestInformationFile()

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['executionMode']      = waitForJobInfo['executionMode']
               self.waitForJobsInfo[instance]['instanceToken']      = waitForJobInfo['instanceToken']
               if siteInfo['venueMechanism'] == 'cache':
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[instance]['recentJobStatus'] = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s_%s.stderr" % (self.runName,instanceId)
            self.emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stdFile))
            for batchQueueType in BATCHQUEUETYPES:
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueueType,self.runName,instanceId,outFile)
                  self.emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,batchQueueOutputFile))

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError

         self.nInstances      = instance
         self.executeInstance = 0

         timeHistoryLogs = {}
         timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,str(self.executeInstance))
         timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,str(self.executeInstance))

         enteredCommand = self.commandParser.getEnteredCommand()

         siteInfo = self.sitesInfo.getSiteInfo(destination)
         if siteInfo['tapisSite']:
            tapisSiteInfo = self.tapisSitesInfo.getTapisSiteInfo(siteInfo['tapisSite'])
         else:
            tapisSiteInfo = self.tapisSitesInfo.getDefaultTapisSiteInfo()
         if siteInfo['fileMover']:
            fileMoverInfo = self.fileMoversInfo.getFileMoverInfo(siteInfo['fileMover'])
         else:
            fileMoverInfo = self.fileMoversInfo.getDefaultFileMoverInfo()
         if   siteInfo['venueMechanism'] == 'local':
            wfInstanceId = "WF;%d" % (self.nInstances)
            self.jobs[self.executeInstance] = VenueMechanismLocal(self.remoteMonitors,
                                                                  self.hubUserName,self.hubUserId,self.submitterClass,
                                                                  currentWorkingDirectory,self.session,self.distributorPid,
                                                                  self.batchCommands,self.isParametric,
                                                                  self.runName,self.localJobId,wfInstanceId,destination,
                                                                  self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                                  stageInTarFile,auxiliaryTarFile,
                                                                  transferExecutable,executable,
                                                                  self.appsAccessInfo,"[sweep]",
                                                                  self.stdinput,arguments,useEnvironment,environment,
                                                                  self.isMultiCoreRequest,self.disableJobMonitoring,
                                                                  siteInfo,tapisSiteInfo,fileMoverInfo,
                                                                  toolFilesInfo,dockerImageInfo,
                                                                  self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                                  self.quotaLimit,
                                                                  self.x509SubmitProxy,self.disableProbeCheck,
                                                                  timeHistoryLogs,self.useSetup,
                                                                  self.pegasusVersion,self.pegasusHome)
            instancesInitialJobStatus = []
            for instance in self.waitForJobsInfo:
               instancesInitialJobStatus.append(self.waitForJobsInfo[instance]['recentJobStatus'])
            self.jobs[self.executeInstance].setInstancesInitialJobStatus(instancesInitialJobStatus)
         elif siteInfo['venueMechanism'] == 'ssh':
            wfInstanceId = "WF;%d" % (self.nInstances)
            self.jobs[self.executeInstance] = VenueMechanismSsh(self.remoteMonitors,
                                                                self.hubUserName,self.hubUserId,self.submitterClass,
                                                                currentWorkingDirectory,self.session,self.distributorPid,
                                                                self.batchCommands,self.isParametric,
                                                                self.runName,self.localJobId,wfInstanceId,destination,
                                                                self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                                self.tunnelsInfo,
                                                                cloudInstanceId,
                                                                stageInTarFile,auxiliaryTarFile,
                                                                transferExecutable,executable,
                                                                self.appsAccessInfo,"[sweep]",
                                                                self.stdinput,arguments,useEnvironment,environment,
                                                                self.isMultiCoreRequest,self.disableJobMonitoring,
                                                                siteInfo,tapisSiteInfo,fileMoverInfo,
                                                                toolFilesInfo,dockerImageInfo,
                                                                self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                                self.x509SubmitProxy,self.disableProbeCheck,
                                                                self.quotaLimit,
                                                                timeHistoryLogs,self.useSetup,
                                                                self.pegasusVersion,self.pegasusHome)

         exitCode,scriptFiles = self.jobs[self.executeInstance].createScripts()
         if exitCode:
            raise JobDistributor.InstancesError

         self.waitForJobsInfo[self.executeInstance] = {}
         waitForJobInfo = self.jobs[self.executeInstance].getWaitForJobInfo()
         if nCacheHits == self.nInstances:
            self.waitForJobsInfo[self.executeInstance]['isBatchJob']         = False
         else:
            self.waitForJobsInfo[self.executeInstance]['isBatchJob']         = waitForJobInfo['isBatchJob']
         self.waitForJobsInfo[self.executeInstance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
         self.waitForJobsInfo[self.executeInstance]['instanceToken']         = waitForJobInfo['instanceToken']
         self.waitForJobsInfo[self.executeInstance]['executionMode']         = waitForJobInfo['executionMode']
         self.waitForJobsInfo[self.executeInstance]['remoteBatchSystem']     = self.jobs[self.executeInstance].remoteBatchSystem
         self.waitForJobsInfo[self.executeInstance]['instanceDirectory']     = self.jobs[self.executeInstance].instanceDirectory
         self.waitForJobsInfo[self.executeInstance]['scratchDirectory']      = self.jobs[self.executeInstance].scratchDirectory
         self.waitForJobsInfo[self.executeInstance]['recentJobStatus']       = '?'

         stdFile = "%s.stderr" % (self.runName)
         self.emptyFilesToRemove.append(stdFile)
         for batchQueueType in BATCHQUEUETYPES:
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s.%s" % (batchQueueType,self.runName,outFile)
               self.emptyFilesToRemove.append(batchQueueOutputFile)

         if self.disableJobMonitoring:
            remoteJobIdCompleteFile = "%s.%s_%s" % (REMOTEJOBIDCOMPLETE,self.localJobId,self.executeInstance)
            self.emptyFilesToRemove.append(os.path.join(currentWorkingDirectory,self.runName,remoteJobIdCompleteFile))
      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def registerBoincJobInstances(self):
      self.nonBatchJobs       = []
      self.splitExecutionJobs = []
      if self.waitForJobsInfo[self.executeInstance]['isBatchJob']:
         if self.waitForJobsInfo[self.executeInstance]['executionMode'] == 'split':
            self.splitExecutionJobs.append(self.executeInstance)
         self.jobs[self.executeInstance].registerJob()
         self.waitForJobsInfo[self.executeInstance]['state'] = 'held'
         if self.progressReport == 'text':
            self.__writeToStdout("Run %d registered 1 job instance. %s\n" % (self.jobId,time.ctime()))
      else:
         self.nonBatchJobs.append(self.executeInstance)
         self.successfulInstance = self.executeInstance
         self.waitForJobsInfo[self.executeInstance]['state'] = 'released'
         self.jobs[self.executeInstance].jobStatistics[self.jobs[self.executeInstance].jobIndex]['exitCode'] = 0


   def startBoincBatchJobInstances(self):
      if self.waitForJobsInfo[self.executeInstance]['isBatchJob']:
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'held':

            while not self.jobs[self.executeInstance].isJobReleased():
               time.sleep(5)

            if self.jobs[self.executeInstance].isJobReleased():
               self.waitForJobsInfo[self.executeInstance]['state'] = 'released'
               if self.progressReport == 'text':
                  self.__writeToStdout("Run %d instance %d released for submission. %s\n" % \
                                              (self.jobId,self.executeInstance,time.ctime()))
               if self.jobs[self.executeInstance].sendFiles():
                  if self.jobs[self.executeInstance].executeJob():
                     waitForJobInfo = self.jobs[self.executeInstance].getWaitForJobInfo()
                     self.waitForJobsInfo[self.executeInstance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                     self.waitForJobsInfo[self.executeInstance]['knownSite']       = waitForJobInfo['knownSite']
                     self.waitForJobsInfo[self.executeInstance]['instanceToken']   = waitForJobInfo['instanceToken']
                     self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = '?'

                     siteMonitorDesignator = self.waitForJobsInfo[self.executeInstance]['siteMonitorDesignator']
                     remoteJobId           = self.waitForJobsInfo[self.executeInstance]['remoteJobId']
                     currentJobStatus,runProgressMessage = \
                                          self.remoteMonitors['job'].updateBoincWorkflowStatus(siteMonitorDesignator,
                                                                                               remoteJobId,
                                                                                               self.nInstances,
                                                                                               self.parameterCombinationsPath)
                     if not self.disableJobMonitoring:
                        if self.progressReport == 'submit':
                           if runProgressMessage:
                              self.__writeToStdout("%s timestamp=%.1f\n" % (runProgressMessage,time.time()))
                     else:
                        self.jobs[self.executeInstance].writeHarvestInformationFile(self.runType,
                                                                                    self.filesToRemove,
                                                                                    self.emptyFilesToRemove)
                  else:
                     self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'


   def runBoincJobInstances(self):
      nBatchJobsHeld = 1-len(self.nonBatchJobs)
      jobsRunning = False
      while (jobsRunning or nBatchJobsHeld > 0) and \
            (self.successfulInstance == None) and (not self.abortGlobal['abortAttempted']):
         self.startBoincBatchJobInstances()
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForBoincWorkflowJobs(self.waitForJobsInfo,
                                                                                        self.nInstances,
                                                                                        self.parameterCombinationsPath,
                                                                                        self.progressReport,self.abortGlobal)
         nBatchJobsHeld = 0
         jobsRunning = False
         for instance in [self.executeInstance]:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()
            if self.jobs[instance].wasJobSuccessful():
               self.successfulInstance = instance

      if self.abortGlobal['abortAttempted']:
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] != 'D':
               self.jobs[self.executeInstance].killScripts(self.abortGlobal['abortSignal'])
               self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs('boinc',
                                                           self.waitForJobsInfo,
                                                           self.progressReport,
                                                           self.parameterCombinationsPath)
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if 'recentJobSite' in self.waitForJobsInfo[self.executeInstance]:
               executionVenue = self.waitForJobsInfo[self.executeInstance]['recentJobSite']
               self.jobs[self.executeInstance].updateVenue(executionVenue)
         if self.waitForJobsInfo[self.executeInstance]['state'] == 'released':
            if self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] == 'KD':
               self.jobs[self.executeInstance].cleanupFiles()
               self.waitForJobsInfo[self.executeInstance]['recentJobStatus'] = 'D'

      if self.successfulInstance != None:
         for instance in iterRange(1,self.nInstances+1):
            if self.jobs[instance].remoteBatchSystem == 'CACHE':
               self.jobs[instance].jobStatistics[self.jobs[instance].jobIndex]['exitCode'] = 0
            else:
               self.jobs[instance].jobSubmitted = True

      for instance in iterRange(1,self.nInstances+1):
         if 'recentJobSite' in self.waitForJobsInfo[instance]:
            executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
            self.jobs[instance].updateVenue(executionVenue)
         self.jobs[instance].postProcess()
         self.jobs[instance].retrieveFiles()
         self.jobs[instance].cleanupFiles()
         self.jobs[instance].recordJobStatistics()

      if self.waitForJobsInfo[self.executeInstance]['state'] == 'held':
         self.jobs[self.executeInstance].removeJobRegistration()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isdir(self.inputsPath):
         shutil.rmtree(self.inputsPath,True)

      if self.progressReport == 'text' or self.progressReport == 'submit':
         self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (self.jobPath))

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)


   def getSweepInstanceDestination(self,
                                   currentWorkingDirectory):
      destination              = ""
      executable               = ""
      transferExecutable       = None
      executableClassification = ''
      toolInfo                 = None

      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
      userExecutable = self.enteredCommandArguments[0]
      self.enteredCommandArguments[0] = os.path.expandvars(os.path.expanduser(userExecutable))

      executable = self.enteredCommandArguments[0]
      if "/" in executable:
         candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,executable)
         if candidateFileOrDirectory:
            self.enteredCommandArguments[0] = candidateFileOrDirectory

      executable = self.enteredCommandArguments[0]
      transferExecutable = "/" in executable

      applicationIsStaged = False
      if transferExecutable:
         submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)
      else:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               applicationIsStaged      = True
               executableClassification = 'staged'
         if not applicationIsStaged:
            submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(executable)

      if applicationIsStaged:
         event = ':' + userExecutable + ':'
      else:
         event = '/' + os.path.basename(userExecutable)

      enabledSites = self.sitesInfo.getEnabledSites()
      self.toolsInfo.purgeDisabledSites(enabledSites)

      enabledSites = self.sitesInfo.getExpandedSiteNames(enabledSites,
                                                         remoteProbeMonitor=self.remoteMonitors['probe'])
      eligibleDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 executableClassification,enabledSites) + \
                             self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                                 '*',enabledSites)
      eligibleDestinations = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                    eligibleDestinations)
      eligibleDestinations = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','BOINC',
                                                                    eligibleDestinations)

      userMappedDestinations = []
      for userDestination in self.userDestinations:
         if self.sitesInfo.siteExists(userDestination):
            userMappedDestinations.append(userDestination)

      userMappedDestinations = self.sitesInfo.getExpandedSiteNames(userMappedDestinations,
                                                                   remoteProbeMonitor=self.remoteMonitors['probe'])
      userDestinations = self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             executableClassification,userMappedDestinations) + \
                         self.sitesInfo.getSitesWithKeyValue('executableClassificationsAllowed',
                                                             '*',userMappedDestinations)
      userDestinations = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','PEGASUS',
                                                                userDestinations)
      userDestinations = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','BOINC',
                                                                userDestinations)

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination = ""

      if not transferExecutable:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               if len(userDestinations) > 0:
                  toolInfo = self.toolsInfo.selectTool(executable,userDestinations)
               else:
                  toolInfo = self.toolsInfo.selectTool(executable,eligibleDestinations)

               if len(toolInfo['destinations']) > 0:
                  destination              = random.choice(toolInfo['destinations'])
                  executable               = toolInfo['executablePath']
                  executableClassification = 'staged'

      if destination == "":
         if len(userDestinations) > 0:
            destination = random.choice(userDestinations)
         else:
            destination = self.sitesInfo.selectUndeclaredSite(eligibleDestinations,
                                                              executableClassification)

      return(destination,executable,transferExecutable,executableClassification,event,toolInfo)


   def buildSweepJobInstances(self,
                              currentWorkingDirectory):
      try:
         self.jobPath = os.path.join(currentWorkingDirectory,self.runName)
         if os.path.exists(self.jobPath):
            self.logger.log(logging.ERROR,getLogMessage("Run directory %s exists" % (self.jobPath)))
            self.__writeToStderr("Run directory %s exists\n" % (self.jobPath))
            exitCode = 1
            raise JobDistributor.InstancesError

         if not os.path.isdir(self.jobPath):
            os.mkdir(self.jobPath)
         if self.progressReport == 'curses' or self.reportMetrics:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'remoteParameterCombinations.csv')
         else:
            self.parameterCombinationsPath = os.path.join(self.jobPath,'parameterCombinations.csv')
         self.commandParser.writeParameterCombinations(self.parameterCombinationsPath)

         managerSpecifiedByUser = self.managerSpecified
         instanceError = False
         self.inputsPath   = os.path.join(self.jobPath,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(max(1,self.commandParser.getParameterCombinationCount()))+1))
         instance = 0
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)

            self.managerSpecified = managerSpecifiedByUser
            destination,executable,transferExecutable,executableClassification,event,toolInfo = \
                                      self.getSweepInstanceDestination(currentWorkingDirectory)
            if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
               if self.managersInfo.managerExists(toolInfo['remoteManager']):
                  self.managerSpecified = True
                  self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
               else:
                  message = "Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable)
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self.__writeToStderr("%s\n" % (message))
                  exitCode = 1
                  raise JobDistributor.InstancesError

            toolFilesInfo   = None
            dockerImageInfo = None
            if 'toolFiles' in toolInfo:
               toolFilesName = toolInfo['toolFiles']
               if toolFilesName:
                  toolFilesInfo = self.toolFilesInfo.getToolFilesInfo(toolFilesName)
                  if toolFilesInfo['toolFilesName'] == toolFilesName:
                     if 'dockerImage' in toolFilesInfo:
                        dockerImageInfo = self.dockerImagesInfo.getDockerImageInfo(toolFilesInfo['dockerImage'])

            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError as e:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
               exitCode = 1
               raise JobDistributor.InstanceError

            inputFiles = self.getInputFiles(currentWorkingDirectory,substitutions=substitutions)

            siteInfo      = self.sitesInfo.getDefaultSiteInfo()
            tapisSiteInfo = self.tapisSitesInfo.getDefaultTapisSiteInfo()
            fileMoverInfo = self.fileMoversInfo.getDefaultFileMoverInfo()

            gridsite        = "Unknown"
            pegasusSiteInfo = {}
            nNodes          = "1"
            arguments       = []
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if siteInfo['tapisSite']:
                     tapisSiteInfo = self.tapisSitesInfo.getTapisSiteInfo(siteInfo['tapisSite'])
                  if siteInfo['fileMover']:
                     fileMoverInfo = self.fileMoversInfo.getFileMoverInfo(siteInfo['fileMover'])
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        message = "Invalid manager %s specified for venue %s." % (siteInfo['remoteManager'],destination)
                        self.logger.log(logging.ERROR,getLogMessage(message))
                        self.__writeToStderr("%s\n" % (message))
                        exitCode = 1
                        raise JobDistributor.InstanceError
                  if nCpus == 0:
                     nCpus = int(siteInfo['remotePpn'])
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
                  else:
                     ppn = str(int(math.ceil(float(nCpus) / float(nNodes))))
               else:
                  if transferExecutable:
                     errorMessage = "Invalid destination selection: %s" % (destination)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               if transferExecutable:
                  if (not '*' in siteInfo['executableClassificationsAllowed']) and \
                     (not executableClassification in siteInfo['executableClassificationsAllowed']):
                     errorMessage = "The selected venue does not meet the specified\n" + \
                                    "application access requirements.  Please select another venue."
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               touchCommand = "touch " + localJobIdFile
               self.logger.log(logging.INFO,getLogMessage("command = " + touchCommand))
               touchExitStatus,touchStdOutput,touchStdError = self.executeCommand(touchCommand)
               if touchExitStatus:
                  self.logger.log(logging.ERROR,getLogMessage(touchStdError))
                  self.logger.log(logging.ERROR,getLogMessage(touchStdOutput))
                  self.__writeToStderr(touchStdError)
                  exitCode = 1
                  raise JobDistributor.InstanceError
               else:
                  inputFiles.append(localJobIdFile)
                  self.filesToRemove.append(localJobIdFile)
               remoteJobIdFile = "%s.%s_%s" % (REMOTEJOBID,self.localJobId,instanceId)
               self.filesToRemove.append(remoteJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     try:
                        fpInputFile = open(inputFile,'r')
                        try:
                           inputText = fpInputFile.readlines()
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (inputFile)))
                        else:
                           template = ParameterTemplate(''.join(inputText))
                           try:
                              inputText = template.substitute(substitutions)
                           except KeyError as e:
                              self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                              exitCode = 1
                              raise JobDistributor.InstanceError
                           instanceInputsPath = os.path.join(self.inputsPath,instanceId)
                           if not os.path.isdir(instanceInputsPath):
                              os.makedirs(instanceInputsPath)
                           inputName = os.path.basename(inputFile)
                           inputPath = os.path.join(instanceInputsPath,inputName)
                           try:
                              fpInputPath = open(inputPath,'w')
                              try:
                                 fpInputPath.writelines(inputText)
                              except (IOError,OSError):
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (inputPath)))
                              else:
                                 substitutedInputFiles.append(inputPath)
                              finally:
                                 fpInputPath.close()
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputPath)))
                        finally:
                           fpInputFile.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (inputFile)))
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                     self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                     self.__writeToStderr(tarStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  else:
                     tarFiles = tarStdOutput.strip().split('\n')

                  self.filesToRemove.append(stageInTarFile)
               else:
# if not shared work directory?
                  stageInTarFile = ""
                  tarFiles = []

               arguments = self.getArguments(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               if siteInfo['passUseEnvironment']:
                  useEnvironment = self.getUseEnvironment()
               else:
                  useEnvironment = ""

               environment = self.getEnvironment(siteInfo,currentWorkingDirectory,tarFiles,substitutions)

               managerInfoEnvironment = managerInfo['environment']
               managerEnvironment = self.__getManagerEnvironment(managerInfoEnvironment)

               toolInfoEnvironment = toolInfo['environment']
               toolEnvironment = self.__getToolEnvironment(toolInfoEnvironment)

               additionalEnvironmentVariableValues = ""
               for environmentVariableValue in managerEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               for environmentVariableValue in toolEnvironment.split(" "):
                  if environmentVariableValue:
                     environmentVariable,value = environmentVariableValue.split('=')
                     if environment.find(environmentVariable + '=') < 0:
                        additionalEnvironmentVariableValues += environmentVariable + "=" + value + " "

               environment += " " + additionalEnvironmentVariableValues
               environment = environment.strip()

               template = ParameterTemplate(environment)
               try:
                  environment = template.substitute(substitutions)
               except KeyError as e:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
                  exitCode = 1
                  raise JobDistributor.InstanceError

               if self.isMultiCoreRequest and managerInfo['computationMode'] == 'serial':
                  if nCpus == 1:
                     self.isMultiCoreRequest = False
                  else:
                     errorMessage = "Serial computation is not compatible with multiple core(%d) request." % (nCpus)
                     self.logger.log(logging.ERROR,getLogMessage(errorMessage))
                     self.__writeToStderr("%s\n" % (errorMessage))
                     exitCode = 1
                     raise JobDistributor.InstanceError

               timeHistoryLogs = {}
               timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT,self.localJobId,instanceId)
               timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED,self.localJobId,instanceId)
               timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)

               jobScriptFiles        = None
               auxiliaryFiles        = []
               stageAuxiliaryTarFile = False
               auxiliaryTarFile      = "%s_%s_auxiliary.tar" % (self.localJobId,instanceId)
               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,self.submitterClass,
                                                            currentWorkingDirectory,self.session,self.distributorPid,
                                                            self.batchCommands,self.isParametric,
                                                            self.runName,self.localJobId,instanceId,destination,
                                                            self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                            stageInTarFile,auxiliaryTarFile,
                                                            transferExecutable,executable,
                                                            self.appsAccessInfo,event,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,self.disableJobMonitoring,
                                                            siteInfo,tapisSiteInfo,fileMoverInfo,
                                                            toolFilesInfo,dockerImageInfo,
                                                            self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,
                                                            timeHistoryLogs,self.useSetup,
                                                            self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,self.submitterClass,
                                                          currentWorkingDirectory,self.session,self.distributorPid,
                                                          self.batchCommands,self.isParametric,
                                                          self.runName,self.localJobId,instanceId,destination,
                                                          self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,auxiliaryTarFile,
                                                          transferExecutable,executable,
                                                          self.appsAccessInfo,event,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,self.disableJobMonitoring,
                                                          siteInfo,tapisSiteInfo,fileMoverInfo,
                                                          toolFilesInfo,dockerImageInfo,
                                                          self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.useSetup,
                                                          self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'tapis':
                  venueMechanism = VenueMechanismSsh(self.remoteMonitors,
                                                     self.hubUserName,self.hubUserId,self.submitterClass,
                                                     currentWorkingDirectory,self.session,self.distributorPid,
                                                     self.batchCommands,self.isParametric,
                                                     self.runName,self.localJobId,instanceId,destination,
                                                     self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                     self.tunnelsInfo,
                                                     cloudInstanceId,
                                                     stageInTarFile,auxiliaryTarFile,
                                                     transferExecutable,executable,
                                                     self.appsAccessInfo,event,
                                                     self.stdinput,arguments,useEnvironment,environment,
                                                     self.isMultiCoreRequest,self.disableJobMonitoring,
                                                     siteInfo,tapisSiteInfo,fileMoverInfo,
                                                     toolFilesInfo,dockerImageInfo,
                                                     self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                     self.x509SubmitProxy,self.disableProbeCheck,
                                                     self.quotaLimit,
                                                     timeHistoryLogs,self.useSetup,
                                                     self.pegasusVersion,self.pegasusHome)
                  exitCode,jobScriptFiles = venueMechanism.createScripts()
                  for key in tapisSiteInfo:
                     if key in siteInfo:
                        siteInfo[key] = tapisSiteInfo[key]
                  if   tapisSiteInfo['venueMechanism'] == 'ssh':
                     self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,self.submitterClass,
                                                             currentWorkingDirectory,self.session,self.distributorPid,
                                                             self.batchCommands,self.isParametric,
                                                             self.runName,self.localJobId,instanceId,destination,
                                                             self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                             self.tunnelsInfo,
                                                             cloudInstanceId,
                                                             stageInTarFile,auxiliaryTarFile,
                                                             transferExecutable,executable,
                                                             self.appsAccessInfo,event,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,self.disableJobMonitoring,
                                                             siteInfo,tapisSiteInfo,fileMoverInfo,
                                                             toolFilesInfo,dockerImageInfo,
                                                             self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                             self.x509SubmitProxy,self.disableProbeCheck,
                                                             self.quotaLimit,
                                                             timeHistoryLogs,self.useSetup,
                                                             self.pegasusVersion,self.pegasusHome)
                  elif tapisSiteInfo['venueMechanism'] == 'local':
                     self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                               self.hubUserName,self.hubUserId,self.submitterClass,
                                                               currentWorkingDirectory,self.session,self.distributorPid,
                                                               self.batchCommands,self.isParametric,
                                                               self.runName,self.localJobId,instanceId,destination,
                                                               self.tailFiles,enteredCommand,gridsite,pegasusSiteInfo,
                                                               stageInTarFile,auxiliaryTarFile,
                                                               transferExecutable,executable,
                                                               self.appsAccessInfo,event,
                                                               self.stdinput,arguments,useEnvironment,environment,
                                                               self.isMultiCoreRequest,self.disableJobMonitoring,
                                                               siteInfo,tapisSiteInfo,fileMoverInfo,
                                                               toolFilesInfo,dockerImageInfo,
                                                               self.submissionScriptsInfo,managerInfo,nCpus,nNodes,ppn,wallTime,
                                                               self.quotaLimit,
                                                               self.x509SubmitProxy,self.disableProbeCheck,
                                                               timeHistoryLogs,self.useSetup,
                                                               self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise JobDistributor.InstanceError

               if jobScriptFiles:
                  auxiliaryFiles += scriptFiles
                  del scriptFiles
                  scriptFiles = jobScriptFiles

               if siteInfo['stageFiles']:
                  inputBuiltPath = os.path.join(currentWorkingDirectory,timeHistoryLogs['timestampInputBuilt'])
                  if self.writeInputBuiltFile(inputBuiltPath):
                     scriptFiles += [inputBuiltPath]
                     self.filesToRemove.append(inputBuiltPath)
                  else:
                     self.__writeToStderr("Creation of %s failed\n" % (inputBuiltPath))
                     exitCode = 1
                     raise JobDistributor.InstanceError

                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                     self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                     self.__writeToStderr(gzipStdError)
                     exitCode = 1
                     raise JobDistributor.InstanceError
                  stageInTarFile += '.gz'
                  self.filesToRemove.append(stageInTarFile)

               if siteInfo['executionMode'] == 'split':
                  if len(auxiliaryFiles) > 0:
                     tarCommand = buildCreateTarCommand(auxiliaryTarFile,auxiliaryFiles)
                     self.logger.log(logging.INFO,getLogMessage("command = " + str(tarCommand)))
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                        self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                        self.__writeToStderr(tarStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError

                     niceCommand = "nice -n 19 gzip " + auxiliaryTarFile
                     self.logger.log(logging.INFO,getLogMessage("command = " + niceCommand))
                     gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                     if gzipExitStatus:
                        self.logger.log(logging.ERROR,getLogMessage(gzipStdError))
                        self.logger.log(logging.ERROR,getLogMessage(gzipStdOutput))
                        self.__writeToStderr(gzipStdError)
                        exitCode = 1
                        raise JobDistributor.InstanceError
                     auxiliaryTarFile += '.gz'
                     self.filesToRemove.append(auxiliaryTarFile)
                     stageAuxiliaryTarFile = True

               self.jobs[instance].stageFilesToInstanceDirectory(stageAuxiliaryTarFile=stageAuxiliaryTarFile)

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['executionMode']         = waitForJobInfo['executionMode']
               self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
               self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
               self.waitForJobsInfo[instance]['instanceToken']         = waitForJobInfo['instanceToken']
               self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
               self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory
               self.waitForJobsInfo[instance]['scratchDirectory']      = self.jobs[instance].scratchDirectory
               self.waitForJobsInfo[instance]['recentJobStatus']       = '?'

            except JobDistributor.InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s_%s.stderr" % (self.runName,instanceId)
            self.emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,stdFile))
            for batchQueueType in BATCHQUEUETYPES:
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueueType,self.runName,instanceId,outFile)
                  self.emptyFilesToRemove.append(os.path.join(self.jobs[instance].instanceDirectory,batchQueueOutputFile))

         if instanceError:
            exitCode = 1
            raise JobDistributor.InstancesError
         else:
            self.nInstances = len(self.waitForJobsInfo)
      except JobDistributor.InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def registerSweepJobInstances(self):
      self.nonBatchJobs       = []
      self.splitExecutionJobs = []
      for instance in self.waitForJobsInfo:
         if not self.waitForJobsInfo[instance]['isBatchJob']:
            self.nonBatchJobs.append(instance)
         if self.waitForJobsInfo[instance]['executionMode'] == 'split':
            self.splitExecutionJobs.append(instance)
         self.jobs[instance].registerJob()
         self.waitForJobsInfo[instance]['state'] = 'held'
      if self.progressReport == 'text':
         if self.nInstances == 1:
            self.__writeToStdout("Run %d registered %d job. %s\n" % (self.jobId,self.nInstances,time.ctime()))
         else:
            self.__writeToStdout("Run %d registered %d jobs. %s\n" % (self.jobId,self.nInstances,time.ctime()))


   def startSweepBatchJobInstances(self):
      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['isBatchJob']:
            if self.waitForJobsInfo[instance]['state'] == 'held':
               if self.jobs[instance].isJobReleased():
                  self.waitForJobsInfo[instance]['state'] = 'released'
                  if self.progressReport == 'text':
                     self.__writeToStdout("Run %d job %d released for submission. %s\n" % \
                                                        (self.jobId,instance,time.ctime()))
                  if self.jobs[instance].sendFiles():
                     if self.jobs[instance].executeJob():
                        waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                        self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                        self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                        self.waitForJobsInfo[instance]['instanceToken']   = waitForJobInfo['instanceToken']
                        self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                     else:
                        self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                  else:
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'


   def runSweepJobInstances(self):
      nBatchJobsHeld = self.nInstances-len(self.nonBatchJobs)
      jobsRunning = False
      while (jobsRunning or nBatchJobsHeld > 0) and (not self.abortGlobal['abortAttempted']):
         self.startSweepBatchJobInstances()
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForSweepJobs(self.waitForJobsInfo,
                                                                                self.parameterCombinationsPath,
                                                                                self.progressReport,self.abortGlobal)
         nBatchJobsHeld = 0
         jobsRunning = False
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()

      if self.abortGlobal['abortAttempted']:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                  self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
         self.remoteMonitors['job'].waitForKilledBatchJobs('sweep',
                                                           self.waitForJobsInfo,
                                                           self.progressReport,
                                                           self.parameterCombinationsPath)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

      for instance in self.waitForJobsInfo:
         if self.waitForJobsInfo[instance]['state'] == 'held':
            self.jobs[instance].removeJobRegistration()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if os.path.isdir(self.inputsPath):
         shutil.rmtree(self.inputsPath,True)

      if self.progressReport == 'text' or self.progressReport == 'submit' or self.progressReport == 'pegasus':
         self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (self.jobPath))

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)


   def buildJobInstances(self):
      exitCode = 0
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("buildJobInstances:os.getcwd(): No such file or directory"))
         exitCode = 1
      else:
         self.remoteMonitors['job']      = RemoteJobMonitor(self.jobListenURI)
         self.remoteMonitors['identity'] = RemoteIdentityManager(self.identityListenURI)
         self.remoteMonitors['tunnel']   = RemoteTunnelMonitor(self.tunnelListenURI)

         if   self.runType == 'redundant':
            exitCode = self.buildRedundantJobInstances(currentWorkingDirectory)
         elif self.runType == 'pegasus':
            exitCode = self.buildPegasusJobInstances(currentWorkingDirectory)
         elif self.runType == 'boinc':
            exitCode = self.buildBoincJobInstances(currentWorkingDirectory)
         elif self.runType == 'sweep':
            exitCode = self.buildSweepJobInstances(currentWorkingDirectory)
         else:
            self.logger.log(logging.ERROR,getLogMessage("buildJobInstances: runType has not been properly set."))
            exitCode = 1

      return(exitCode)


   def requestCacheResult(self,
                          currentWorkingDirectory,
                          cacheRequestFile,
                          substitutions=None):
      cacheDestination     = ""
      resultFile           = ""
      cacheResponseContent = ""

      if cacheRequestFile and self.cacheHosts:
         if substitutions:
            template = ParameterTemplate(cacheRequestFile)
            try:
               cacheRequestFile = template.substitute(substitutions)
            except KeyError as e:
               cacheRequestFile = ""
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (e.args[0]))
         if cacheRequestFile:
            candidateFileOrDirectory = self.findInputFileOrDirectory(currentWorkingDirectory,cacheRequestFile)
            self.logger.log(logging.DEBUG,getLogMessage("Checking cache %s for %s" % (self.cacheHosts,candidateFileOrDirectory)))
            if candidateFileOrDirectory:
               cacheRequestContent = ""
               try:
                  fpCacheRequest = open(candidateFileOrDirectory,'r')
                  try:
                     cacheRequestContent = fpCacheRequest.read()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (candidateFileOrDirectory)))
                  finally:
                     fpCacheRequest.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (candidateFileOrDirectory)))

               if cacheRequestContent:
                  fileNamePattern = re.compile(b"<filename>(.*)</filename>")
                  sessionPattern  = re.compile(b"<session>(.*)</session>")

                  for cacheHost in self.cacheHosts:
                     cacheRequestHost,cacheRequestPort = cacheHost.split(':')
                     cacheRequestPort = int(cacheRequestPort)
                     destination = self.sitesInfo.getSiteWithVenueAndPort(cacheRequestHost,cacheRequestPort)
                     if destination:
                        protocol = self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem')
                        try:
                           requestResponse = requests.post(protocol + '://' + cacheHost + '/cache/request',data=cacheRequestContent)
                        except:
                           self.logger.log(logging.ERROR,getLogMessage("connection failed posting %s to %s" % \
                                                                         (candidateFileOrDirectory,cacheHost)))
                        else:
                           if requestResponse.status_code == 200:
                              cacheResponseContent = requestResponse.content
                              resultPathXML = fileNamePattern.search(cacheResponseContent)
                              if resultPathXML:
                                 resultPath = resultPathXML.group(1)
                                 resultFile = os.path.basename(resultPath)
                              else:
                                 now = datetime.datetime.now()
                                 epoch = int((time.mktime(now.timetuple())+(now.microsecond/1000000.))*1000000.)
                                 resultFile = "run%d.xml" % (epoch)
                              sessionXML = sessionPattern.search(cacheResponseContent)
                              if sessionXML:
                                 executionSession = sessionXML.group(1)
                              else:
                                 executionSession = ""
                              cacheDestination = destination
                              self.logger.log(logging.DEBUG,getLogMessage("Cache request resultFile %s, %d bytes, session %s" % \
                                                                        (resultFile,len(cacheResponseContent),executionSession)))
                              break
                           else:
                              self.logger.log(logging.DEBUG,getLogMessage("Cache request status_code %d" % \
                                                                             (requestResponse.status_code)))

      return(cacheDestination,resultFile,cacheResponseContent)


   def registerJobInstances(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("registerJobInstances:os.getcwd(): No such file or directory"))
      else:
         if   self.runType == 'redundant':
            self.registerRedundantJobInstances()
         elif self.runType == 'pegasus':
            self.registerPegasusJobInstances()
         elif self.runType == 'boinc':
            self.registerBoincJobInstances()
         elif self.runType == 'sweep':
            self.registerSweepJobInstances()


   def runJobInstances(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("runJobInstances:os.getcwd(): No such file or directory"))
      else:
         if   self.runType == 'redundant':
            self.runRedundantJobInstances()
         elif self.runType == 'pegasus':
            self.runPegasusJobInstances()
         elif self.runType == 'boinc':
            if self.disableJobMonitoring:
               self.startBoincBatchJobInstances()
            else:
               self.runBoincJobInstances()
         elif self.runType == 'sweep':
            self.runSweepJobInstances()


   def processCommandArguments(self):
      exitCode = self.parseCommandArguments()
      if not exitCode:
         exitCode = self.setJobId()
         if not exitCode:
            if not self.runName:
               self.runName = self.localJobId
            self.setInfo()
            self.showUsage()

      return(exitCode)


   def buildJobs(self):
      if self.doRun():
         exitCode =  self.buildJobDescription()
         if not exitCode:
            exitCode = self.buildJobInstances()
      else:
         exitCode = 1

      return(exitCode)


   def registerJobs(self):
      if self.doRun():
         self.registerJobInstances()


   def runJobs(self):
      if self.doRun():
         self.runJobInstances()


