# @package      hubzero-submit-distributor
# @file         SitesInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import glob
import random
import copy
import socket
import logging
from xml.dom import minidom, Node
import yaml

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership   import GroupMembership
from hubzero.submit.ParameterTemplate import ParameterTemplate

PEGASUSTEMPLATEEXTENSIONS = ['rc','xml','txt','yml']

class SitesInfo:
   def __init__(self,
                sitesPath,
                restrictionUser=None,
                pegasusTemplateDirectory=None,
                allowedVenueMechanisms=[],
                pegasusVersion=None):
      self.logger = logging.getLogger(__name__)

      self.sites = {}

      if os.path.isdir(sitesPath):
         for sitesInfoPath in glob.iglob(os.path.join(sitesPath,'*')):
            self.readSitesInfoFile(sitesInfoPath)
      else:
         for sitesInfoPath in glob.iglob(sitesPath):
            self.readSitesInfoFile(sitesInfoPath)

      for site in self.sites:
         arbitraryExecutableAllowed = self.sites[site]['arbitraryExecutableAllowed']
         executableClassificationsAllowed = self.sites[site]['executableClassificationsAllowed']
         if len(executableClassificationsAllowed) == 1:
            if executableClassificationsAllowed[0] == '*':
               if not arbitraryExecutableAllowed:
                  self.sites[site]['executableClassificationsAllowed'] = ['staged', 'apps']
         del self.sites[site]['arbitraryExecutableAllowed']

      if not '*' in allowedVenueMechanisms:
         for site in self.sites:
            venueMechanism = self.sites[site]['venueMechanism']
            if venueMechanism != '':
               if not venueMechanism in allowedVenueMechanisms:
                  self.sites[site]['state'] = 'restrictedByVenueMechanism'

      if restrictionUser:
         for site in self.sites:
            restrictedToUsers = self.sites[site]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not restrictionUser in restrictedToUsers:
                  self.sites[site]['state'] = 'restrictedByUser'

         groupMembership = GroupMembership(restrictionUser=restrictionUser)
         for site in self.sites:
            if self.sites[site]['state'] == 'enabled':
               restrictedToGroups = self.sites[site]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if groupMembership.isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.sites[site]['state'] = 'restrictedByGroup'

      if not pegasusVersion:
         for site in self.sites:
            if self.sites[site]['state'] == 'enabled':
               if self.sites[site]['remoteBatchSystem'] == 'PEGASUS':
                  self.sites[site]['state'] = 'restrictedByPegasus'

      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            members = self.sites[site]['members']
            if len(members) > 0:
               markedForDeletion = []
               for member in members:
                  if member in self.sites:
                     if self.sites[member]['state'] != 'enabled':
                        markedForDeletion.append(member)
               for member in markedForDeletion:
                  self.sites[site]['members'].remove(member)
               del markedForDeletion
               if len(members) == 0:
                  self.sites[site]['state'] = 'restrictedEmptyMemberList'

      for site in self.sites:
         failoverSite = self.sites[site]['failoverSite']
         if failoverSite != 'None':
            if failoverSite in self.sites:
               if self.sites[failoverSite]['state'] != 'enabled':
                  self.sites[site]['failoverSite'] = 'None'
            else:
               self.logger.log(logging.INFO,getLogMessage("%s(failoverSite) %s does not exist." % (site,failoverSite)))
               self.sites[site]['failoverSite'] = 'None'

      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            probeSiteName = self.sites[site]['probeSite']
            if probeSiteName:
               if probeSiteName == 'MEMBERS':
                  members = self.sites[site]['members']
                  if len(members) <= 0:
                     self.logger.log(logging.INFO,getLogMessage("%s(probeSite) %s does not exist." % (site,probeSiteName)))
               else:
                  if not probeSiteName in self.sites:
                     self.logger.log(logging.INFO,getLogMessage("%s(probeSite) %s does not exist." % (site,probeSiteName)))
            else:
               self.sites[site]['probeSite'] = site
         else:
            self.sites[site]['probeSite'] = site

      for site in self.sites:
         pegasusTemplates = copy.copy(self.sites[site]['pegasusTemplates'])
         self.sites[site]['pegasusTemplates'] = {}
         self.sites[site]['pegasusTemplates']['rc']    = ""
         self.sites[site]['pegasusTemplates']['sites'] = ""
         self.sites[site]['pegasusTemplates']['tc']    = ""
         if self.sites[site]['state'] == 'enabled':
            if pegasusTemplateDirectory:
               for pegasusTemplate in pegasusTemplates:
                  foundPegasusTemplate = False
                  for pegasusTemplateExtension in PEGASUSTEMPLATEEXTENSIONS:
                     pegasusTemplateFile = "%s.%s" % (pegasusTemplate,pegasusTemplateExtension)
                     pegasusTemplatePath = os.path.join(pegasusTemplateDirectory,
                                                        "pegasus-%s" % (pegasusVersion),pegasusTemplateFile)
                     if os.path.exists(pegasusTemplatePath):
                        if   pegasusTemplateExtension == 'rc':
                           self.sites[site]['pegasusTemplates']['rc']    = pegasusTemplatePath
                        elif pegasusTemplateExtension == 'xml':
                           self.sites[site]['pegasusTemplates']['sites'] = pegasusTemplatePath
                           pegasusSites = {}
                           siteDoc = minidom.parse(pegasusTemplatePath)
                           rootNode = siteDoc.documentElement
                           children = rootNode.getElementsByTagName("site")
                           for child in children:
                              handle = child.getAttribute('handle')
                              if handle != 'local':
                                 pegasusSites[handle] = {}
                                 try:
                                    arch      = child.getAttribute('arch')
                                 except:
                                    pass
                                 else:
                                    pegasusSites[handle]['arch'] = arch

                                 try:
                                    osFlavor  = child.getAttribute('os')
                                 except:
                                    pass
                                 else:
                                    pegasusSites[handle]['osFlavor'] = osFlavor

                                 try:
                                    osVersion = child.getAttribute('osversion')
                                 except:
                                    pass
                                 else:
                                    pegasusSites[handle]['osVersion'] = osVersion

                                 try:
                                    osRelease = child.getAttribute('osrelease')
                                 except:
                                    pass
                                 else:
                                    pegasusSites[handle]['osRelease'] = osRelease

                                 gridstartDistribute = False
                                 for grandChild in child.getElementsByTagName("profile"):
                                    if grandChild.nodeType == Node.ELEMENT_NODE:
                                       if grandChild.getAttribute("namespace") == "pegasus" and \
                                          grandChild.getAttribute("key") == "gridstart":
                                          if grandChild.hasChildNodes():
                                             for greatGrandChild in grandChild.childNodes:
                                                gridstart = greatGrandChild.nodeValue
                                                if gridstart == 'Distribute':
                                                   gridstartDistribute = True
                                 pegasusSites[handle]['gridstartDistribute'] = gridstartDistribute
                           self.sites[site]['pegasusSites'] = pegasusSites
                        elif pegasusTemplateExtension == 'txt':
                           self.sites[site]['pegasusTemplates']['tc'] = pegasusTemplatePath
                        elif pegasusTemplateExtension == 'yml':
                           with open(pegasusTemplatePath,'r') as fpTemplate:
                              templateDictionary = yaml.load(fpTemplate,Loader=yaml.CSafeLoader)
                           if   'sites' in templateDictionary:
                              self.sites[site]['pegasusTemplates']['sites'] = pegasusTemplatePath
                              pegasusSites = {}
                              for templateSite in templateDictionary['sites']:
                                 handle = templateSite['name']
                                 if handle != 'local':
                                    pegasusSites[handle] = {}
                                    try:
                                       arch      = templateSite['arch']
                                    except:
                                       pass
                                    else:
                                       pegasusSites[handle]['arch'] = arch

                                    try:
                                       osFlavor  = templateSite['os.type']
                                    except:
                                       pass
                                    else:
                                       pegasusSites[handle]['osFlavor'] = osFlavor

                                    try:
                                       osVersion = templateSite['os.version']
                                    except:
                                       pass
                                    else:
                                       pegasusSites[handle]['osVersion'] = osVersion

                                    try:
                                       osRelease = templateSite['os.release']
                                    except:
                                       pass
                                    else:
                                       pegasusSites[handle]['osRelease'] = osRelease

                                    try:
                                       gridstart = templateSite['profiles']['pegasus']['gridstart']
                                    except:
                                       pegasusSites[handle]['gridstartDistribute'] = False
                                    else:
                                       if gridstart == 'Distribute':
                                          pegasusSites[handle]['gridstartDistribute'] = True
                                       else:
                                          pegasusSites[handle]['gridstartDistribute'] = False
                                       
                              self.sites[site]['pegasusSites'] = pegasusSites
                           elif 'transformations' in templateDictionary:
                              self.sites[site]['pegasusTemplates']['tc'] = pegasusTemplatePath
                        foundPegasusTemplate = True
                        break

                  if not foundPegasusTemplate:
                     self.logger.log(logging.ERROR,getLogMessage("Pegasus template file %s does not exist." % (pegasusTemplate)))
         del pegasusTemplates


   def readSitesInfoFile(self,
                         sitesInfoPath):
      sitePattern     = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      siteName        = ""

      if os.path.exists(sitesInfoPath):
         try:
            fpInfo = open(sitesInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   sitePattern.match(record):
                        siteName = sitePattern.match(record).group(2)
                        self.sites[siteName] = {'venues':[],
                                                'venuePort':22,
                                                'sshOptions':"",
                                                'gsiHost':socket.gethostname(),
                                                'cloudDesignator':'',
                                                'tunnelDesignator':'',
                                                'siteMonitorDesignator':'',
                                                'identityManagers':[],
                                                'venueMechanism':'',
                                                'venueGatekeeper':',,'.split(','),
                                                'remoteUser':'',
                                                'remoteBatchAccount':'',
                                                'remoteBatchSystem':'',
                                                'remoteBatchQueue':'',
                                                'remoteBatchPartition':'',
                                                'remoteBatchPartitionSize':'',
                                                'remoteBatchConstraints':'',
                                                'parallelEnvironment':'',
                                                'submissionScriptCommands':[],
                                                'remoteBinDirectory':os.path.join('${HOME}','bin'),
                                                'remoteApplicationRootDirectory':'',
                                                'remoteScratchDirectory':'',
                                                'remotePpn':'',
                                                'remoteManager':'',
                                                'remoteHostAttribute':'',
                                                'stageFiles':True,
                                                'passUseEnvironment':False,
                                                'environment':[],
                                                'arbitraryExecutableAllowed':True,
                                                'executableClassificationsAllowed':['*'],
                                                'members':[],
                                                'state':'enabled',
                                                'failoverSite':'None',
                                                'checkProbeResult':True,
                                                'probeSite':'',
                                                'restrictedToUsers':[],
                                                'restrictedToGroups':[],
                                                'logUserRemotely':False,
                                                'undeclaredSiteSelectionWeight':0.,
                                                'minimumWallTime':0.,
                                                'maximumWallTime':0.,
                                                'minimumCores':1,
                                                'maximumCores':0,
                                                'minimumCoreWallTime':0.,
                                                'maximumCoreWallTime':0.,
                                                'pegasusTemplates':[],
                                                'tapisSite':'',
                                                'executionMode':'block',
                                                'fileMover':''
                                               }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.sites[siteName]:
                           if   isinstance(self.sites[siteName][key],list):
                              self.sites[siteName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.sites[siteName][key],bool):
                              self.sites[siteName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.sites[siteName][key],float):
                              self.sites[siteName][key] = float(value)
                           elif isinstance(self.sites[siteName][key],int):
                              self.sites[siteName][key] = int(value)
                           elif isinstance(self.sites[siteName][key],dict):
                              try:
                                 sampleKey   = self.sites[siteName][key].keys()[0]
                                 sampleValue = self.sites[siteName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.sites[siteName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.sites[siteName][key][dictKey] = dictValue
                           else:
                              self.sites[siteName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for site %s" % (key,value,siteName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Sites configuration file %s could not be read" % \
                                                                                             (sitesInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Sites configuration file %s could not be opened" % \
                                                                                            (sitesInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Sites configuration file %s is missing" % \
                                                                                (sitesInfoPath)))


   def getEnabledSites(self):
      enabledSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            enabledSites.append(site)

      return(enabledSites)


   def purgeDisabledSites(self,
                          siteNames):
      reasonsDenied = {}
      markedForDeletion = []
      for siteName in siteNames:
         try:
            site  = self.sites[siteName]
            state = site['state']
            if state != 'enabled':
               markedForDeletion.append(siteName)
               reasonsDenied[siteName] = state
         except:
            pass
      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      del markedForDeletion

      return(reasonsDenied)


   def __checkProbeResult(self,
                          siteName):
      try:
         site = self.sites[siteName]
      except:
         checkProbeResult = True
      else:
         if site['state'] == 'enabled':
            probeSiteName = site['probeSite']
            if probeSiteName == 'MEMBERS':
               members = site['members']
               if len(members) > 0:
                  checkProbeResult = False
                  for member in members:
                     try:
                        probeSiteName = self.sites[member]['probeSite']
                     except:
                        checkProbeResult = True
                     else:
                        try:
                           probeSite = self.sites[probeSiteName]
                        except:
                           checkProbeResult = True
                        else:
                           if probeSite['checkProbeResult']:
                              checkProbeResult = True
                              break
               else:
                  checkProbeResult = True
            else:
               try:
                  probeSite = self.sites[probeSiteName]
               except:
                  checkProbeResult = True
               else:
                  checkProbeResult = probeSite['checkProbeResult']
         else:
            checkProbeResult = True

      return(checkProbeResult)


   def getIgnoreProbeSites(self):
      ignoreProbeSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            if not self.__checkProbeResult(site):
               ignoreProbeSites.append(site)

      return(ignoreProbeSites)


   def getProbeResult(self,
                      siteName,
                      remoteProbeMonitor):
      probeResult = False
      probeResultSiteName = None
      try:
         site = self.sites[siteName]
      except:
         pass
      else:
         if site['state'] == 'enabled':
            probeSiteName = site['probeSite']
            if probeSiteName == 'MEMBERS':
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     probeSiteName = self.sites[member]['probeSite']
                     probeResultSiteName = probeSiteName
                     if probeSiteName in self.sites:
                        if remoteProbeMonitor.isSiteAvailable(probeSiteName):
                           probeResult = True
                           break
            else:
               if probeSiteName in self.sites:
                  probeResult = remoteProbeMonitor.isSiteAvailable(probeSiteName)
                  probeResultSiteName = probeSiteName

      return(probeResultSiteName,probeResult)


   def purgeOfflineSites(self,
                         siteNames,
                         remoteProbeMonitor):
      markedForDeletion = []
      substituteSites   = []
      for siteName in siteNames:
         if siteName in self.sites:
            if remoteProbeMonitor:
               if self.__checkProbeResult(siteName):
                  probeResultSiteName,probeResult = self.getProbeResult(siteName,remoteProbeMonitor)
                  if not probeResult:
                     markedForDeletion.append(siteName)
                     try:
                        site         = self.sites[siteName]
                        failoverSite = site['failoverSite']
                        if failoverSite != 'None':
                           probeResultSiteName,probeResult = self.getProbeResult(failoverSite,remoteProbeMonitor)
                           if probeResult:
                              substituteSites.append(failoverSite)
                     except:
                        pass
         else:
            markedForDeletion.append(siteName)

      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      if len(siteNames) == 0:
         if len(substituteSites) > 0:
            for siteName in substituteSites:
               siteNames.append(siteName)


   def purgeResourceLimitedSites(self,
                                 siteNames,
                                 nCores,
                                 jobWallTime):
      markedForDeletion = []
      for siteName in self.sites:
         site = self.sites[siteName]
         if site['state'] == 'enabled':
            members = site['members']
            if len(members) > 0:
               membersMarkedForDeletion = []
               for member in members:
                  if member in self.sites:
                     memberSite = self.sites[member]
                     if memberSite['state'] == 'enabled':
                        if nCores == 0:
                           try:
                              nMemberSiteCores = int(memberSite['remotePpn'])
                           except:
                              nMemberSiteCores = 0
                        else:
                           nMemberSiteCores = nCores
                        if   nMemberSiteCores > 0:
                           if nMemberSiteCores < memberSite['minimumCores']:
                              membersMarkedForDeletion.append(member)
                           if memberSite['maximumCores'] > 0:
                              if nMemberSiteCores > memberSite['maximumCores']:
                                 membersMarkedForDeletion.append(member)
                        elif nMemberSiteCores < 0:
                           membersMarkedForDeletion.append(member)

                        if float(jobWallTime) < memberSite['minimumWallTime']:
                           membersMarkedForDeletion.append(member)
                        if memberSite['maximumWallTime'] > 0.:
                           if float(jobWallTime) > memberSite['maximumWallTime']:
                              membersMarkedForDeletion.append(member)

                        jobCoreWallTime = nMemberSiteCores * float(jobWallTime)
                        if jobCoreWallTime < memberSite['minimumCoreWallTime']:
                           membersMarkedForDeletion.append(member)
                        if site['maximumCoreWallTime'] > 0.:
                           if jobCoreWallTime > memberSite['maximumCoreWallTime']:
                              membersMarkedForDeletion.append(member)

               for member in membersMarkedForDeletion:
                  if member in site['members']:
                     site['members'].remove(member)
               del membersMarkedForDeletion
               if len(site['members']) == 0:
                  markedForDeletion.append(siteName)
            else:
               if nCores == 0:
                  try:
                     nSiteCores = int(site['remotePpn'])
                  except:
                     nSiteCores = 0
               else:
                  nSiteCores = nCores
               if   nSiteCores > 0:
                  if nSiteCores < site['minimumCores']:
                     markedForDeletion.append(siteName)
                  if site['maximumCores'] > 0:
                     if nSiteCores > site['maximumCores']:
                        markedForDeletion.append(siteName)
               elif nSiteCores < 0:
                  markedForDeletion.append(siteName)

               if float(jobWallTime) < site['minimumWallTime']:
                  markedForDeletion.append(siteName)
               if site['maximumWallTime'] > 0.:
                  if float(jobWallTime) > site['maximumWallTime']:
                     markedForDeletion.append(siteName)

               jobCoreWallTime = nSiteCores * float(jobWallTime)
               if jobCoreWallTime < site['minimumCoreWallTime']:
                  markedForDeletion.append(siteName)
               if site['maximumCoreWallTime'] > 0.:
                  if jobCoreWallTime > site['maximumCoreWallTime']:
                     markedForDeletion.append(siteName)

      for siteName in markedForDeletion:
         site = self.sites[siteName]
         site['state'] = 'resourceLimit'
         if siteName in siteNames:
            siteNames.remove(siteName)

      del markedForDeletion


   def purgeExecutionModeLimitedSites(self,
                                      siteNames,
                                      asynchronous):
      if not asynchronous is None:
         if asynchronous:
            executionModes = ['split']
         else:
            executionModes = ['block']

         markedForDeletion = []
         for siteName in self.sites:
            site = self.sites[siteName]
            if site['state'] == 'enabled':
               members = site['members']
               if len(members) > 0:
                  membersMarkedForDeletion = []
                  for member in members:
                     if member in self.sites:
                        memberSite = self.sites[member]
                        if memberSite['state'] == 'enabled':
                           if not memberSite['executionMode'] in executionModes:
                              membersMarkedForDeletion.append(member)

                  for member in membersMarkedForDeletion:
                     if member in site['members']:
                        site['members'].remove(member)
                  del membersMarkedForDeletion
                  if len(site['members']) == 0:
                     markedForDeletion.append(siteName)
               else:
                  if not site['executionMode'] in executionModes:
                     markedForDeletion.append(siteName)

         for siteName in markedForDeletion:
            site = self.sites[siteName]
            site['state'] = 'executionModeLimit'
            if siteName in siteNames:
               siteNames.remove(siteName)

         del markedForDeletion


   def selectUndeclaredSites(self,
                             siteNames,
                             executableClassification,
                             maximumSelectedSites):
      selectedUndeclaredSites = []

      if len(siteNames) > 0:
         siteAdded = True
         while siteAdded and (len(selectedUndeclaredSites) < maximumSelectedSites):
            siteAdded = False
            undeclaredSiteSelectionWeights = {}
            possibleSites = []
            for siteName in siteNames:
               if not siteName in selectedUndeclaredSites:
                  try:
                     site = self.sites[siteName]
                     if site['undeclaredSiteSelectionWeight'] >= 0.:
                        members = site['members']
                        if len(members) > 0:
                           addMembers = True
                           for member in members:
                              if member in selectedUndeclaredSites:
                                 addMembers = False
                                 break
                           if addMembers:
                              undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                              for member in members:
                                 if not member in possibleSites:
                                    possibleSites.append(member)
                                    undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                                 else:
                                    undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
                        else:
                           undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                           if not siteName in possibleSites:
                              possibleSites.append(siteName)
                              undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                           else:
                              undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
                  except:
                     pass

            if len(possibleSites) > 0:
               markedForDeletion = []
               for possibleSite in possibleSites:
                  try:
                     site = self.sites[possibleSite]
                     if (not '*' in site['executableClassificationsAllowed']) and \
                        (not executableClassification in site['executableClassificationsAllowed'] ):
                        markedForDeletion.append(possibleSite)
                  except:
                     markedForDeletion.append(possibleSite)
               for possibleSite in markedForDeletion:
                  possibleSites.remove(possibleSite)
               del markedForDeletion

               if len(possibleSites) > 0:
                  nWeights = 0
                  undeclaredSelectionWeightSum = 0.
                  for possibleSite in possibleSites:
                     if undeclaredSiteSelectionWeights[possibleSite] > 0.:
                        nWeights += 1
                        undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]

                  if nWeights == 0:
                     possibleIndex = random.randint(0,len(possibleSites)-1)
                     selectedSite = possibleSites[possibleIndex]
                     selectedUndeclaredSites.append(selectedSite)
                     siteAdded = True
                  else:
                     randomWeight = random.random()
                     weightSum = 0.
                     for possibleSite in possibleSites:
                        undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                        if undeclaredSiteSelectionWeight > 0.:
                           weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                           if weightSum >= randomWeight:
                              selectedSite = possibleSite
                              selectedUndeclaredSites.append(selectedSite)
                              siteAdded = True
                              break

            del possibleSites
            del undeclaredSiteSelectionWeights

      return(selectedUndeclaredSites)


   def selectUndeclaredSite(self,
                            siteNames,
                            executableClassification):
      selectedSite = ""
      undeclaredSiteSelectionWeights = {}

      if len(siteNames) > 0:
         possibleSites = []
         for siteName in siteNames:
            try:
               site = self.sites[siteName]
               if site['undeclaredSiteSelectionWeight'] >= 0.:
                  members = site['members']
                  if len(members) > 0:
                     undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                     for member in members:
                        if not member in possibleSites:
                           possibleSites.append(member)
                           undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                        else:
                           undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
                  else:
                     if not siteName in possibleSites:
                        undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                        possibleSites.append(siteName)
                        undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                     else:
                        undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
            except:
               pass

         if len(possibleSites) > 0:
            markedForDeletion = []
            for possibleSite in possibleSites:
               try:
                  site = self.sites[possibleSite]
                  if (not '*' in site['executableClassificationsAllowed']) and \
                     (not executableClassification in site['executableClassificationsAllowed'] ):
                     markedForDeletion.append(possibleSite)
               except:
                  markedForDeletion.append(possibleSite)
            for possibleSite in markedForDeletion:
               possibleSites.remove(possibleSite)
            del markedForDeletion

            if len(possibleSites) > 0:
               undeclaredSelectionWeightSum = 0.
               for possibleSite in possibleSites:
                  undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]

               if undeclaredSelectionWeightSum == 0.:
                  possibleIndex = random.randint(0,len(possibleSites)-1)
                  selectedSite = possibleSites[possibleIndex]
               else:
                  randomWeight = random.random()
                  weightSum = 0.
                  for possibleSite in possibleSites:
                     undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                     if undeclaredSiteSelectionWeight > 0.:
                        weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                        if weightSum >= randomWeight:
                           selectedSite = possibleSite
                           break

      del undeclaredSiteSelectionWeights

      return(selectedSite)


   def selectSites(self,
                   siteNames):
      selectedSites = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if not siteName in self.sites:
               selectedSites.append(siteName)

         if len(selectedSites) == 0:
            for siteName in siteNames:
               if siteName in self.sites:
                  members = self.sites[siteName]['members']
                  if len(members) > 0:
                     for member in members:
                        selectedSites.append(member)
                  else:
                     selectedSites.append(siteName)

      return(selectedSites)


   def getSiteKeyValue(self,
                       siteName,
                       key):
      value = ""

      if siteName in self.sites:
         site    = self.sites[siteName]
         members = site['members']
         if len(members) > 0:
            for member in members:
               if member in self.sites:
                  site = self.sites[member]
                  if key in site:
                     value = site[key]
                     break
         else:
            if key in site:
               value = site[key]

      return(value)


   def getSiteVenues(self,
                     siteName):
      venues = []
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'venues' in site:
            venues = site['venues']

      return(venues)


   def getSiteMembers(self,
                      siteName):
      members = []
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'members' in site:
            members = site['members']

      return(members)


   def getSitePegasusSites(self,
                           siteName):
      sitePegasusSites = ""
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'pegasusSites' in site:
            if len(site['pegasusSites']) > 0:
               pegasusSites = []
               for pegasusSite in site['pegasusSites']:
                  pegasusSites.append((site['pegasusSites'][pegasusSite]['gridstartDistribute'],pegasusSite))
               pegasusSites.sort()
               orderedPegasusSites = [ pegasusSite for gridstartDistribute,pegasusSite in pegasusSites ]
               sitePegasusSites = ','.join(orderedPegasusSites)

      return(sitePegasusSites)


   def getSitePegasusSiteAttributeValue(self,
                                        siteName,
                                        pegasusSite,
                                        attribute):
      sitePegasusSiteAttributeValue = ""
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'pegasusSites' in site:
            pegasusSites = site['pegasusSites']
            if pegasusSite in pegasusSites:
               if attribute in pegasusSites[pegasusSite]:
                  sitePegasusSiteAttributeValue = pegasusSites[pegasusSite][attribute]

      return(sitePegasusSiteAttributeValue)


   def getSitesWithKeyValue(self,
                            key,
                            value,
                            siteNames):
      sitesWithKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if   isinstance(site[key],list):
                              if value in site[key]:
                                 sitesWithKeyValue.append(member)
                           elif site[key] == value:
                              sitesWithKeyValue.append(member)
               else:
                  if key in site:
                     if   isinstance(site[key],list):
                        if value in site[key]:
                           sitesWithKeyValue.append(siteName)
                     elif site[key] == value:
                        sitesWithKeyValue.append(siteName)

      return(sitesWithKeyValue)


   def getSitesWithoutKeyValue(self,
                               key,
                               value,
                               siteNames):
      sitesWithoutKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  nMembersWithoutKeyValue = 0
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if   isinstance(site[key],list):
                              if not value in site[key]:
                                 sitesWithoutKeyValue.append(member)
                                 nMembersWithoutKeyValue += 1
                           elif site[key] != value:
                              sitesWithoutKeyValue.append(member)
                              nMembersWithoutKeyValue += 1
                        else:
                           sitesWithoutKeyValue.append(siteName)
                  if nMembersWithoutKeyValue == len(members):
                     sitesWithoutKeyValue.append(siteName)
               else:
                  if key in site:
                     if   isinstance(site[key],list):
                        if not value in site[key]:
                           sitesWithoutKeyValue.append(siteName)
                     elif site[key] != value:
                        sitesWithoutKeyValue.append(siteName)
                  else:
                     sitesWithoutKeyValue.append(siteName)

      return(sitesWithoutKeyValue)


   def getSiteWithVenueAndPort(self,
                               venue,
                               venuePort):
      siteWithVenueAndPort = ""

      for siteName in self.sites:
         venues = self.sites[siteName]['venues']
         if venue in venues:
            if venuePort == self.sites[siteName]['venuePort']:
               siteWithVenueAndPort = siteName
               break

      return(siteWithVenueAndPort)


   def siteExists(self,
                  siteName):
      return(siteName in self.sites)


   def getExpandedSiteNames(self,
                            siteNames,
                            remoteProbeMonitor=None):
      expandedSiteNames = set()

      for siteName in siteNames:
         if siteName in self.sites:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               for member in members:
                  if member in self.sites:
                     if remoteProbeMonitor and self.__checkProbeResult(member):
                        probeSiteName = self.sites[member]['probeSite']
                        if remoteProbeMonitor.isSiteAvailable(probeSiteName):
                           expandedSiteNames.add(member)
                     else:
                        expandedSiteNames.add(member)
                  else:
                     expandedSiteNames.add(siteName)
            else:
               if remoteProbeMonitor and self.__checkProbeResult(siteName):
                  probeSiteName = site['probeSite']
                  if remoteProbeMonitor.isSiteAvailable(probeSiteName):
                     expandedSiteNames.add(siteName)
               else:
                  expandedSiteNames.add(siteName)
         else:
            expandedSiteNames.add(siteName)

      expandedSiteNames = list(expandedSiteNames)
      random.shuffle(expandedSiteNames)

      return(expandedSiteNames)


   def getDefaultSiteInfo(self):
      siteInfo = {}
      siteInfo['siteName']                         = "default"
      siteInfo['probeSite']                        = "default"
      siteInfo['venues']                           = []
      siteInfo['venueIndex']                       = -1
      siteInfo['venue']                            = "Unknown"
      siteInfo['venuePort']                        = 22
      siteInfo['sshOptions']                       = ""
      siteInfo['venueMechanism']                   = ''
      siteInfo['cloudDesignator']                  = ""
      siteInfo['tunnelDesignator']                 = ""
      siteInfo['siteMonitorDesignator']            = ""
      siteInfo['identityManagers']                 = []
      siteInfo['venueGatekeeper']                  = ""
      siteInfo['remoteUser']                       = ""
      siteInfo['sharedUserSpace']                  = False
      siteInfo['stageFiles']                       = True
      siteInfo['passUseEnvironment']               = False
      siteInfo['environment']                      = []
      siteInfo['logUserRemotely']                  = False
      siteInfo['remoteBatchAccount']               = ""
      siteInfo['remoteBatchSystem']                = ""
      siteInfo['remoteBatchQueue']                 = ""
      siteInfo['remoteBatchPartition']             = ""
      siteInfo['remoteBatchPartitionSize']         = ""
      siteInfo['remoteBatchConstraints']           = ""
      siteInfo['parallelEnvironment']              = ""
      siteInfo['submissionScriptCommands']         = []
      siteInfo['remoteBinDirectory']               = ""
      siteInfo['remoteApplicationRootDirectory']   = ""
      siteInfo['remoteScratchDirectory']           = ""
      siteInfo['remoteManager']                    = ""
      siteInfo['hostAttributes']                   = ""
      siteInfo['executableClassificationsAllowed'] = ['*']
      siteInfo['remotePpn']                        = "1"
      siteInfo['pegasusTemplates']                 = []
      siteInfo['tapisSite']                        = ""
      siteInfo['executionMode']                    = 'block'
      siteInfo['fileMover']                        = ""

      return(siteInfo)


   def getSiteInfo(self,
                   siteName):
      siteInfo = {}

      site = self.sites[siteName]
      siteInfo['siteName']                         = siteName
      siteInfo['probeSite']                        = siteName
      siteInfo['venues']                           = site['venues']
      if len(siteInfo['venues']) > 0:
         siteInfo['venueIndex']                    = random.randint(0,len(siteInfo['venues'])-1)
         siteInfo['venue']                         = siteInfo['venues'][siteInfo['venueIndex']]
      else:
         siteInfo['venueIndex']                    = -1
         siteInfo['venue']                         = ""
      siteInfo['venuePort']                        = site['venuePort']
      siteInfo['sshOptions']                       = site['sshOptions']
      siteInfo['gsiHost']                          = site['gsiHost']

      siteInfo['venueMechanism']                   = site['venueMechanism']
      siteInfo['cloudDesignator']                  = site['cloudDesignator']
      siteInfo['tunnelDesignator']                 = site['tunnelDesignator']
      siteInfo['siteMonitorDesignator']            = site['siteMonitorDesignator']
      siteInfo['identityManagers']                 = site['identityManagers']
      siteInfo['venueGatekeeper']                  = site['venueGatekeeper']
      siteInfo['remoteUser']                       = site['remoteUser']
      if   siteInfo['remoteUser'].startswith('USER:'):
         siteInfo['sharedUserSpace']               = True
      elif siteInfo['remoteUser'].startswith('USER'):
         siteInfo['sharedUserSpace']               = True
      else:
         siteInfo['sharedUserSpace']               = False
      siteInfo['stageFiles']                       = site['stageFiles']
      siteInfo['passUseEnvironment']               = site['passUseEnvironment']
      siteInfo['environment']                      = site['environment']
      siteInfo['logUserRemotely']                  = site['logUserRemotely']
      siteInfo['remoteBatchAccount']               = site['remoteBatchAccount']
      siteInfo['remoteBatchSystem']                = site['remoteBatchSystem']
      siteInfo['remoteBatchQueue']                 = site['remoteBatchQueue']
      siteInfo['remoteBatchPartition']             = site['remoteBatchPartition']
      siteInfo['remoteBatchPartitionSize']         = site['remoteBatchPartitionSize']
      siteInfo['remoteBatchConstraints']           = site['remoteBatchConstraints']
      siteInfo['parallelEnvironment']              = site['parallelEnvironment']
      siteInfo['submissionScriptCommands']         = site['submissionScriptCommands']
      siteInfo['remoteBinDirectory']               = site['remoteBinDirectory']
      siteInfo['remoteApplicationRootDirectory']   = site['remoteApplicationRootDirectory']
      siteInfo['remoteScratchDirectory']           = site['remoteScratchDirectory']
      siteInfo['remoteManager']                    = site['remoteManager']
      siteInfo['hostAttributes']                   = site['remoteHostAttribute']
      siteInfo['executableClassificationsAllowed'] = site['executableClassificationsAllowed']
      siteInfo['remotePpn']                        = site['remotePpn']
      siteInfo['pegasusTemplates']                 = site['pegasusTemplates']
      siteInfo['tapisSite']                        = site['tapisSite']
      siteInfo['executionMode']                    = site['executionMode']
      siteInfo['fileMover']                        = site['fileMover']

      return(siteInfo)


